///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/cad.h"
using namespace rheolef;
using namespace std;

// -----------------------------------------------------------------------
// gnuplot output : support Bezier curves
// -----------------------------------------------------------------------
static
inline
cad::size_type
make_subdiv (cad::size_type subdiv, cad::size_type deg, bool adapt_degree)
{
    if (deg == 1)  return 1;
    if (! adapt_degree) return subdiv;
    return subdiv*deg;
}
int
cad_rep::gnuplot(string basename, 
	 bool execute, bool clean, bool verbose,
	 size_type subdiv, bool adapt_degree) const
{
    check_macro (dimension() == 2, "cad: gnuplot output supports only 2D");
    //
    // output on file
    //
    string plotname = basename + ".plot";
    string file_list = plotname;
    ofstream plot (plotname.c_str());
    if (verbose) clog << "! file \"" << plotname << "\" created.\n";
    int digits10 = numeric_limits<Float>::digits10;
    plot << setprecision(digits10);

    Float delta0 = 0.1*(_xmax[0] - _xmin[0]);
    Float delta1 = 0.1*(_xmax[1] - _xmin[1]);

    plot << "#!gnuplot" << endl
         << "set title \"" << basename << "\"" << endl
         << "set size square" << endl
         << "plot [" 
         << _xmin[0]-delta0 << ":" << _xmax[0]+delta0 << "]["
         << _xmin[1]-delta1 << ":" << _xmax[1]+delta1 << "] \\" 
         << endl;

    for (size_type idom = 0; idom < _e.size(); idom++) {
        const cad_domain& d = _e[idom];

        string gdatname = basename + "-" + d.name() + ".gdat";
        file_list = file_list + " " + gdatname;
        ofstream gdat (gdatname.c_str());
	gdat << setprecision(digits10);
	plot << "\t\"" << gdatname << "\" smooth bezier title \"" << d.name() << "\"";
        if (idom != _e.size()-1) plot << ", \\";
        plot << endl;

        if (verbose) clog << "! file \"" << gdatname << "\" created.\n";

	for (vector<cad_element>::const_iterator q = d.begin(); q != d.end(); q++) {
	    const cad_element& S = (*q);
    	    check_macro (S.type() == cad_element::e, "unexpected cad element type");

            size_type subdeg = make_subdiv (subdiv, S.degree(), adapt_degree);
	    for (size_type i = 0; i < subdeg+1; i++) {
		point x_loc = eval (S, S.ref_node_from_degree(i, subdeg, subdeg));
		gdat << x_loc[0] << " " << x_loc[1] << endl;
	    }
	    gdat << endl;
        }
	gdat.close();
    }
    plot << "pause 1000 \"<ctrl-C>\"" << endl;
    plot.close();
    //
    // run gnuplot
    //
    string command;
    if (execute) {
        command = "gnuplot " + plotname;
        if (verbose) clog << "! " << command << endl;
	// hit <ctrl-C> to exit from gnuplot => exit status = 1
        int status = system (command.c_str());
    }
    //
    // clear data
    //
    if (clean) {
        command = "/bin/rm -f " + file_list;
        if (verbose) clog << "! " << command << endl;
        int status = system (command.c_str());
    }
    return 0;
}
