///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// geo gmsh inputs / outputs
//
// author:
//      Pierre.Saramito@imag.fr
//
// date: 19 oct 2009
//

// ============================================================================
//  includes
// ============================================================================

#include "rheolef/georep.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h" // i/o utility
#include "rheolef/field.h"
#include "rheolef/form.h"
#include "rheolef/tiny_element.h"
using namespace rheolef;
using namespace std;

// --------------------------------------------------------------------
// local utility (static linkage)
// --------------------------------------------------------------------
static
string
gmsh_strip_name (string name) {
  string new_name;
  for (size_t i = 0; i < name.size(); i++) {
    if (name[i] == '"') continue;
    switch (name[i]) {
     case ' ' : new_name += "_"; break;
     default  : new_name += name[i]; break;
    }
  }
  return new_name;
}
static
size_t
add_domain_map (georep::domlist_type& domlist,
    map<size_t, string>& domain_name,
    map<size_t, domain>& domain_map,
    size_t d, size_t shift) {
  size_t k = shift;
  for (map<size_t, domain>::iterator pdom = domain_map.begin();
      pdom != domain_map.end(); pdom++, k++) {
    size_t  dom_idx = (*pdom).first;
    domain dom = (*pdom).second;
    string name = gmsh_strip_name(domain_name[dom_idx]);
    if (name == "") name = "domain_" + itos(dom_idx);
    dom.set_name(name);
    dom.set_dimension(d);
    domlist.at(k) = dom;
  }
  return domain_map.size();
}
static 
void
gmsh_put_element (ostream& s, size_t idx, size_t dom_idx, const geo_element& K)
{
  size_t gmsh_type = 0;
  switch (K.name()) {
    case 'p' : gmsh_type = 15; break;
    case 'e' : gmsh_type =  1; break;
    case 't' : gmsh_type =  2; break;
    case 'q' : gmsh_type =  3; break;
    case 'T' : gmsh_type =  4; break;
    case 'P' : gmsh_type =  6; break;
    case 'H' : gmsh_type =  5; break;
    default: error_macro ("unexpected "<<K.type()<<" element type");
  }
  s << idx << " " << gmsh_type << " "
    << " 3 " << dom_idx << " " << dom_idx << " 0";
  for (size_t i = 0; i < K.size(); i++) {
    s << " " << K[i]+1;
  }
  s << endl;
}
// --------------------------------------------------------------------
// gmsh input
// --------------------------------------------------------------------
istream&
georep::get_gmsh(istream& s)
{
  // => this is only a version 1 mesh file format type
  //    not consistent for P2 elements
  _version = 1;

  if (!s) error_macro("bad input stream for geo.");
  bool verbose = iorheo::getverbose(s);
  if (verbose) warning_macro ("gmsh mesh needs upgrade to geo version 2");
  check_macro (scatch(s,"$MeshFormat"),
        "input stream does not contains a gmsh mesh file ($MeshFormat not found).");
  Float gmsh_fmt_version;
  size_t file_type, float_data_size;
  s >> gmsh_fmt_version >> file_type >> float_data_size;
  if (gmsh_fmt_version > 2.1) {
    warning_macro("gmsh format version " << gmsh_fmt_version << " founded ; expect version 2.1");
  }
  check_macro (file_type == 0, "unsupported gmsh non-ascii format");
  check_macro (scatch(s,"$EndMeshFormat"), "gmsh input error: $EndMeshFormat not found.");
  //
  // get domain names
  //
  check_macro (scatch(s,"$"), "gmsh input error: no more label found.");
  map<size_t, string> domain_name;
  string label;
  s >> label;
  size_type n_names = 0;
  if (label == "PhysicalNames") {
    s >> n_names;
    for (size_t i = 0; i < n_names; i++) {
      size_t d, idx;
      string name;
      if (gmsh_fmt_version >= 2.1) s >> d; // additional domain dimension in 2.1 (not used here)
      s >> idx >> name;
      domain_name[idx] = name;
    }
    check_macro (scatch(s,"$EndPhysicalNames"), "gmsh input error ($EndPhysicalNames not found).");
    check_macro (scatch(s,"$"), "gmsh input error: no more label found.");
    s >> label;
  }
  // when only one domain: merge all in mesh and output no domains
  bool output_all_in_mesh = (n_names <= 1);
  //
  // get coordinates
  //
  check_macro (label == "Nodes", "$Nodes not found in gmsh file");
  size_type nverts;
  s >> nverts;
  _x.resize(nverts);
  for (size_type j = 0; j < 3; j++) {
    _xmin[j] =  numeric_limits<Float>::max();
    _xmax[j] = -numeric_limits<Float>::max();
  }
  for (size_t i = 0; i < nverts && s.good(); i++) {
    size_t idx;
    point xi;
    s >> idx >> xi;
    idx--;
    check_macro (idx < nverts, "gmsh node index " << idx << " out of range");
    _x.at(idx) = xi;
    for (size_type j = 0 ; j < 3; j++) {
      _xmin[j] = ::min(xi[j], _xmin[j]);
      _xmax[j] = ::max(xi[j], _xmax[j]);
    }
  }
  _count_geo [0] = _count_element [0] = nverts;
  if (!s.good()) error_macro ("a problem occurs while loading a gmsh mesh");
  //
  // dimension is deduced from bounding box
  //
  _dim = 3;
  if (_xmax[2] == _xmin[2]) {
    _dim = 2;
    if (_xmax[1] == _xmin[1]) _dim = 1;
  }
  //
  // get elements
  //
  check_macro (scatch(s,"$Elements"), "$Elements not found in gmsh file");
  size_type ncells_gmsh;
  s >> ncells_gmsh;
  resize(ncells_gmsh);
  iterator K = begin();
  size_t K_idx = 0;
  map<size_t, domain> 
    point_domain_map, edge_domain_map, face_domain_map, volume_domain_map;
  for (size_t i = 0; i < ncells_gmsh && s.good(); i++) {
      geo_element& Ki = K[K_idx];
      size_t K_idx_gmsh, element_type_gmsh;
      s >> K_idx_gmsh >> element_type_gmsh;
      switch (element_type_gmsh) {
       case 15: Ki.set_name('p'); break;
       case  1: Ki.set_name('e'); break;
       case  2: Ki.set_name('t'); break;
       case  3: Ki.set_name('q'); break;
       case  4: Ki.set_name('T'); break;
       case  5: Ki.set_name('H'); break;
       case  6: Ki.set_name('P'); break;
       default:
	error_macro ("unsupported gmsh element type " << element_type_gmsh);
      }
      Ki.set_index(K_idx);
      size_type n_tag_gmsh;
      s >> n_tag_gmsh;
      size_type domain_idx = 0;
      for (size_type j = 0 ; j < n_tag_gmsh; j++) {
	// the first tag is the physical domain index
        // the second tag is the object index, defined for all elements
	// the third is zero (in all examples)
	size_type tag_dummy;
        s >> tag_dummy;
	if (j == 0) {
	  domain_idx = tag_dummy;
        }
      }
      for (size_type j = 0 ; j < Ki.size(); j++) {
	s >> Ki[j];
        Ki[j]--;
	check_macro (Ki[j] < nverts,
	    "gmsh input: element " << K_idx_gmsh << ": " << j+1 << "-th vertex index "
              << Ki[j]+1 << " out of range 1:" << nverts);
      }
      if (domain_idx != 0) {
        switch (Ki.dimension()) {
  	 case 0:   point_domain_map[domain_idx].push_back(Ki); break;
  	 case 1:    edge_domain_map[domain_idx].push_back(Ki); break;
  	 case 2:    face_domain_map[domain_idx].push_back(Ki); break;
  	 default: volume_domain_map[domain_idx].push_back(Ki); break;
        }
      }
      if (!output_all_in_mesh && Ki.dimension() < _dim) {
	// treated as a domain only
	continue;
      }
      _count_geo    [Ki.dimension()]++;
      _count_element[Ki.type()]++;
      K_idx++;
  }
  resize(K_idx);
  //
  // domains
  //
  if (output_all_in_mesh) {
    // no domains: all is in mesh (e.g. surface mesh in 3d)
    _domlist.resize(0);
    return s;
  }
  size_type ndom = point_domain_map.size()
                  + edge_domain_map.size()
                  + face_domain_map.size()
                  + volume_domain_map.size();
  _domlist.resize(ndom);
  size_t shift = 0;
  shift += add_domain_map (_domlist, domain_name,  point_domain_map, 0, shift);
  shift += add_domain_map (_domlist, domain_name,   edge_domain_map, 1, shift);
  if (_dim > 2 || face_domain_map.size() > 1) {
    shift += add_domain_map (_domlist, domain_name,   face_domain_map, 2, shift);
  } else if (_dim == 2 && face_domain_map.size() == 1) {
    _domlist.pop_back();
  }
  if (_dim == 3 && volume_domain_map.size() > 1) {
    shift += add_domain_map (_domlist, domain_name, volume_domain_map, 3, shift);
  } else if (_dim == 3 && volume_domain_map.size() == 1) {
    _domlist.pop_back();
  }
  return s;
}
// --------------------------------------------------------------------
// gmsh output
// --------------------------------------------------------------------
ostream&
georep::put_gmsh(ostream& s, bool only_boundary) const
{
  s << setprecision(numeric_limits<Float>::digits10)
    << "$MeshFormat" << endl
    << "2.1 0 8" << endl
    << "$EndMeshFormat" << endl
    << "$PhysicalNames" << endl
    << _domlist.size() << endl;
  size_t all_dom_size = 0;
  for (size_t i = 0; i < _domlist.size(); i++) {
    const domain& dom = _domlist.at(i);
    s << dom.dimension() << " " << i+1 << " " << "\"" << dom.name() << "\"" << endl;
    all_dom_size += dom.size();
  }
  s << "$EndPhysicalNames" << endl
    << "$Nodes" << endl
    << n_vertex() << endl;
  geo::const_iterator_vertex p = begin_vertex();
  for (size_t i = 0; i < n_vertex(); i++) {
    s << i+1 << " " << *p++ << endl;
  }
  size_t total_size = all_dom_size;
  if (!only_boundary) total_size += size();
  s << "$EndNodes" << endl
    << "$Elements" << endl
    << total_size << endl;
  size_t idx = 0;
  for (size_t i = 0; i < _domlist.size(); i++) {
    const domain& dom = _domlist.at(i);
    for (size_t j = 0; j < dom.size(); j++, idx++) {
      gmsh_put_element (s, idx+1, i+1, dom.at(j));
    }
  }
  if (!only_boundary) {
    size_t whole_idx = _domlist.size();
    geo::const_iterator q = begin();
    for (size_t i = 0; i < size(); i++, q++, idx++) {
      gmsh_put_element (s, idx+1, whole_idx+1, *q);
    }
  }
  s << "$EndElements" << endl;
  return s;
}
// --------------------------------------------------------------------
// gmsh+stl 2d surface mesh output
// --------------------------------------------------------------------
ostream&
georep::put_stl (ostream& out, const domain& dom) const
{
  out << setprecision(numeric_limits<Float>::digits10)
      << "solid created by rheolef" << endl;
  for (domain::const_iterator p = dom.begin(); p != dom.end(); p++) {
    const geo_element& S = *p;
    out << "facet normal " << normal (S) << endl
        << "\touter loop" << endl;
    for (size_t i = 0; i < S.size(); i++) {
      out << "\tvertex " << vertex(S[i]) << endl;
    }
    out << "\tendloop" << endl
        << "endfacet" << endl;
  }
  out << "endsolid created by rheolef" << endl;
  return out;
}
#ifdef TO_CLEAN
// .stl does not need all the volume vertices list..
// but loose the domainnames !
ostream&
georep::put_gmshcad (std::ostream& cad) const
{
  size_t n_dom_bdry = 0;
  for (georep::const_iterator_domain p = begin_domain(); p != end_domain(); p++) {
    const domain& dom = *p;
    if (dom.dimension() != dimension()-1) continue;
    string dom_name = name() + "-" + dom.name() + ".stl";
    ofstream out (dom_name.c_str());
    cerr << "! file \"" << dom_name << "\" created" << endl;
    put_stl (out, dom);
    cad << "Merge \"" << dom_name << "\";" << endl;
    n_dom_bdry++;
  }
  cad << "Surface Loop(1) = {";
  for (size_t i = 0; i < n_dom_bdry; i++) {
    if (i != 0) cad << ",";
    cad << i+1;
  }
  cad << "};" << endl
      << "Volume(1) = {1};" << endl;
  return cad;
}
#endif // TO_CLEAN
// more simple with a .msh bdry mesh than .stl
// and remember the domain name
// but copy all the volume vertex in the bdry mesh
ostream&
georep::put_gmshcad (std::ostream& cad, string bdry_name) const
{
  if (bdry_name == "") bdry_name = name() + "-bdry";
  ofstream bdry ((bdry_name+".msh").c_str());
  cerr << "! file \"" << bdry_name << ".msh\" created" << endl;
  put_gmsh (bdry, true);
  bdry.close();
  cad << "Mesh.Algorithm   = 5;" << endl
      << "Mesh.Algorithm3D = 1; // tetgen (comment if unavailable)" << endl
      << "Merge \"" << bdry_name << ".msh\";" << endl
      << "Surface Loop(" << n_domain()+1 << ") = {";
  size_t i = 0;
  bool first = true;
  for (georep::const_iterator_domain p = begin_domain(); p != end_domain(); p++, i++) {
    const domain& dom = *p;
    if (dom.dimension() != dimension()-1) continue;
    if (!first) cad << ",";
    first = false;
    cad << i+1;
  }
  i = n_domain()+1;
  cad << "};" << endl
      << "Volume("<<n_domain()+2<<") = {"<<n_domain()+1<<"};" << endl
      << "Physical Volume(" << n_domain()+3 <<") = {"<<n_domain()+2<<"};" << endl;
  return cad;
}
