///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// visualisation of geometries using plotmtv
//
// authors:
//	Nicolas.Roquet@imag.fr
//    	Pierre.Saramito@imag.fr
//
// date: 12 may 1997
//
#include "rheolef/georep.h"
#include "rheolef/geo-visu-aux.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h" // i/o utility
using namespace rheolef;
using namespace std;

ostream& georep::put_mtv_domains (ostream& mtv, size_type max_dim) const
{
  size_type color = 2;
  size_type plot_dim = (dimension() == 1) ? 2 : dimension();
  for (size_type dim = max_dim; dim+1 >= 1; dim--) {
    domlist_type::const_iterator last = _domlist.end();
    for (domlist_type::const_iterator iter = _domlist.begin(); iter != last; iter++) {
      const domain& dom = *iter;
      if (dom.dimension() != dim) continue;
      mtv << "% linelabel  = \"" << dom.name() << "\"" << endl;
      for (domain::const_iterator p = dom.begin(); p != dom.end(); p++) {
	  plotmtv_element(mtv, *p, begin_node(), plot_dim, color);
      }
      color++;
    }
  }
  return mtv;
}

int georep::plotmtv (const string& basename, bool execute, bool clean, bool verbose) const
{
  string mtvname = basename + ".mtv";
  ofstream mtv (mtvname.c_str());
  if (verbose) clog << "! file \"" << mtvname << "\" created.\n";
  int digits10 = numeric_limits<Float>::digits10;
  mtv << setprecision(digits10);
  size_type plot_dim = (dimension() == 1) ? 2 : dimension();
  //
  // header
  //
  mtv << "#!plotmtv\n"
      << "$ DATA = CURVE" << plot_dim << "D\n"
      << "% toplabel   = \"" << basename << ": " 
       		<< size() << " elements, " << n_node() << " vertices\"\n"
      << "% subtitle   = \"\"\n"
      << "% equalscale\n"
      << "% linelabel  = " << name() << endl;
  if (dimension() >= 1) {
    Float delta = 0.1*(_xmax[0] - _xmin[0]);
    mtv << "% xmin  = " << _xmin[0]-delta << endl
        << "% xmax  = " << _xmax[0]+delta << endl;
  }
  if (dimension() >= 2) {
    Float delta = 0.1*(_xmax[1] - _xmin[1]);
    mtv << "% ymin  = " << _xmin[1]-delta << endl
        << "% ymax  = " << _xmax[1]+delta << endl;
  }
  if (dimension() >= 3) {
    Float delta = 0.1*(_xmax[2] - _xmin[2]);
    mtv << "% ymin  = " << _xmin[2]-delta << endl
        << "% ymax  = " << _xmax[2]+delta << endl;
  }
  //
  // plot elements
  //
  const_iterator iter_e = begin();
  const_iterator last_e = end();
  while (iter_e != last_e)
      gnuplot_element (mtv, *iter_e++, begin_node(), plot_dim);
  //
  // plot domains
  //
  put_mtv_domains(mtv, dimension());
  mtv.close();
  //
  // run plotmtv
  //
  int status = 0;
  if (execute) {
      string option = "-colorps -noframe -nodate ";
      if (dimension() == 3) option += "-3d ";
      string command = "plotmtv " + option + basename + ".mtv ";
      if (verbose) {
          command = command + "1>&2";
	  clog << "! " << command << endl;
      } else {
          command = command + "> /dev/null";
      }
      status = system (command.c_str());
  }
  //
  // clear plotmtv data
  //
  if (clean) {
      string command = "/bin/rm -f " + basename + ".mtv";
      if (verbose) clog << "! " << command << endl;
      status = system (command.c_str());
  }
  return status;
}
// ---------------------------------------------------------------------------------
// atom
// ---------------------------------------------------------------------------------
int georep::atom (const string& basename, bool execute, bool clean, bool verbose) const
{
  size_type dim = dimension();

  char filename [1000];
  sprintf (filename, "%s.atom", basename.c_str());
  ofstream atom (filename);
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  int digits10 = numeric_limits<Float>::digits10;
  atom << setprecision(digits10);
  //
  // first: plot domains: elements and nodes
  //
  Float r_atom = 0.0125;
  Float r_edge = 0.005;
  atom << "@clear\n";
  const_iterator_node          p    = begin_node();
  domlist_type::const_iterator iter = _domlist.begin();
  domlist_type::const_iterator last = _domlist.end();
  size_type dom_count = 1;
  while (iter != last) {
      const domain& dom = *iter;
      for (domain::const_iterator p_side = dom.begin(); p_side != dom.end(); p_side++) {
	  size_type c = dom_count+1;
          atom_element (atom, (*p_side), begin_node(), dim, r_edge, c);
	  for (size_type j = 0; j < (*p_side).size(); j++) {
              p[(*p_side)[j]].put (atom, dim);
              for (size_type l = dim; l < 3; l++) cout << " 0";
              atom << " r=" << r_atom 
                   << " c=" << c 
		   << " t=" << (*p_side)[j] << endl;
          }
      }
      ++dom_count;
      ++iter;
  }
  //
  // second: plot all vertices as atoms
  // (domain vertices have color priority, as first input)
  //
  for (size_type i=0 ; i < n_node(); i++ ) {

      p[i].put (atom, dim);
      for (size_type l = dim; l < 3; l++) cout << " 0";
      atom << " r=" << r_atom << " c=1 t=" << i << endl;
  }
  //
  // third: plot elements by edges
  //
  const_iterator iter_e = begin();
  const_iterator last_e = end();
  while (iter_e != last_e)
      atom_element (atom, *iter_e++, begin_node(), dim, r_edge, 1);
  atom << endl;
  atom.close();
  //
  // run PlotM from lassp tools
  //
  int status = 0;
  char command [1000];
  if (execute) {
      sprintf (command, "PlotM < %s.atom", basename.c_str());
      if (verbose) clog << "! " << command << endl;
      status = system (command);
  }
  //
  // clear atom data
  //
  if (clean) {
      sprintf (command, "/bin/rm -f %s.atom", basename.c_str());
      if (verbose) clog << "! " << command << endl;
      status = system (command);
  }
  return status;
}
// ---------------------------------------------------------------------------------
// x3d
// ---------------------------------------------------------------------------------
int georep::x2d(const string& basename, bool execute, bool clean, bool verbose) const
{
  char filename [1000];
  sprintf (filename, "%s.poly", basename.c_str());
  ofstream poly (filename);
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  int digits10 = numeric_limits<Float>::digits10;
  poly << setprecision(digits10);
  //
  // plot edges
  //
  point  x0 = (_xmax + _xmin)/2;
  point  dx = (_xmax - _xmin);
  const Float scale = 3000;
  const_iterator iter_e = begin();
  const_iterator last_e = end();
  while (iter_e != last_e)
      x3d_element (poly, *iter_e++, begin_node(), dimension(), scale, x0, dx);
  poly.close();
  //
  // run hull and x3d
  //
  int status = 0;
  char command [1000];
  if (execute) {
      sprintf (command, "hull %s.poly %s.x3d 100", basename.c_str(), basename.c_str());
      if (verbose) clog << "! " << command << endl;
      status = system (command);
      sprintf (command, "x3d %s.x3d", basename.c_str());
      if (verbose) clog << "! " << command << endl;
      status |= system (command);
  }
  //
  // clear data
  //
  if (clean) {
      sprintf (command, "/bin/rm -f %s.poly %s.x3d", basename.c_str(), basename.c_str());
      if (verbose) clog << "! " << command << endl;
      status = system (command);
  }
  return status;
}
