///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

#undef _RHEOLEF_PARANO
#include "rheolef/geo_tree.h"
using namespace rheolef;
using namespace std;

// =========================================================================
// high level predicates
// =========================================================================

// test box[x0,x0+h0] inter box[x-d,x+d] is non-empty
bool
geo_tree::box_intersect_box_i (
    const ipoint&           ix0,
    const big_integer_type& h0,
    const ipoint&           ix,
    const big_integer_type& d) const
{
    for (size_type i = 0; i < _dim; i++)
	if ((ix[i]+d < ix0[i]) || (ix[i]-d > ix0[i]+h0))
	    return false;
    return true;
}
// test box[xmin,xmax] inter box[x-d,x+d] is non-empty
bool
geo_tree::box_intersect_box_i (
    const ipoint&           ixmin,
    const ipoint&           ixmax,
    const ipoint&           ix,
    const big_integer_type& d) const
{
    for (size_type i = 0; i < _dim; i++)
	if ((ix[i]+d < ixmin[i]) || (ix[i]-d > ixmax[i]))
	    return false;
    return true;
}
// test a in box[x,x+h]
bool
geo_tree::belongs_to_box_i (const ipoint& ia,
	const ipoint& ix, const big_integer_type& h) const
{
    for (size_type i = 0; i < _dim; i++)
        if (ia[i] < ix[i] || ia[i] > ix[i]+h)
            return false;
    return true;
}
// K subset box[x,x+h] 
bool
geo_tree::element_subset_box_i (const geo_element& K,
          const ipoint& ix, const big_integer_type& h) const
{
    // since K is convex, all vertices may belong to box(x,x+h):
    for (size_type i = 0; i < K.size(); i++)
	if ( ! belongs_to_box_i (_ivertex[K[i]], ix, h))
	    return false;
    return true;
}
// compute bbox(K) = box[xmin,xmax]
void
geo_tree::bbox_element_i (const geo_element& K,
          ipoint& ixmin, ipoint& ixmax) const
{
  for (size_type j = 0; j < _dim; j++) {
    ixmin[j] =  coord_max;
    ixmax[j] =  0;
  }
  for (size_type i = 0; i < K.size(); i++) {
    for (size_type j = 0 ; j < _dim; j++) {
      ixmin[j] = min(_ivertex[K[i]][j], ixmin[j]);
      ixmax[j] = max(_ivertex[K[i]][j], ixmax[j]);
    }
  }
}
// bbox(K) inter box[x,x+h] 
bool
geo_tree::bbox_element_intersect_box_i (const geo_element& K,
          const ipoint& ix, const big_integer_type& h) const
{
  ipoint ixmin, ixmax;
  bbox_element_i (K, ixmin, ixmax);
  return box_intersect_box_i (ixmin, ixmax, ix, h);
}
// K inter box[x,x+h] HAS A BUG, at least in 3D
bool
geo_tree::element_intersect_box_i (const geo_element& K,
          const ipoint& ix, const big_integer_type& h) const
{
    // TODO: see graphic gems, vol 1.
    fatal_macro ("element_intersect_box: not yet");
#ifdef TODO
    trace_macro ("element_intersect_box [D0] K.name = `"
	<< K.name() << "' K.dim = " << K.dimension());
    // if at least one vertice of K belongs to box(x,h) : ok
    for (size_type i = 0; i < K.size(); i++) {
      if (belongs_to_box_i (_ivertex[K[i]], ix, h)) {
          trace_macro ("element_intersect_box [F0]...");
	  return true;
      }
    }
    trace_macro ("element_intersect_box [D1]...");
    // if at least one vertice of box(x,h) belongs to K : ok
    if (K.dimension() == _dim) {
      switch (_dim) {
        case 1: {
	    ipoint x1 (x[0]+h);
	    if (belongs_to_element (x,  K) || 
	        belongs_to_element (x1, K)) {
                    trace_macro ("element_intersect_box [F1.d1]...");
		    return true;
	    }
	    break;
        }
        case 2: {
	    ipoint x1 (x[0]+h, x[1]);
	    ipoint x2 (x[0]+h, x[1]+h);
	    ipoint x3 (x[0],   x[1]+h);
	    if (belongs_to_element (x,  K) || belongs_to_element (x1, K) ||
	        belongs_to_element (x2, K) || belongs_to_element (x3, K)) {
                    trace_macro ("element_intersect_box [F1.d2]...");
		    return true;
		}
		break;
        }
	default: {
	    ipoint x1 (x[0]+h, x[1],   x[2]);
	    ipoint x2 (x[0]+h, x[1]+h, x[2]);
	    ipoint x3 (x[0],   x[1]+h, x[2]);
	    ipoint x4 (x[0],   x[1],   x[2]+h);
	    ipoint x5 (x[0]+h, x[1],   x[2]+h);
	    ipoint x6 (x[0]+h, x[1]+h, x[2]+h);
	    ipoint x7 (x[0],   x[1]+h, x[2]+h);
	    if (belongs_to_element (x,  K) || belongs_to_element (x1, K) ||
	        belongs_to_element (x2, K) || belongs_to_element (x3, K) ||
	        belongs_to_element (x4, K) || belongs_to_element (x4, K) ||
	        belongs_to_element (x6, K) || belongs_to_element (x7, K)) {
                    trace_macro ("element_intersect_box [F1.d2]...");
		    return true;
	    }
	    break;
        }
      }
    }
    if (K.dimension() <= 1 || _dim <= 1) return false;

    // otherwise, since K is convex, one side S may intersect bbox
    trace_macro ("element_intersect_box [D2]...");
    for (size_type j = 0; j < K.n_subgeo(_dim-1); j++) {
	geo_element S;
	K.build_subgeo(K.dimension()-1, j, S);
        trace_macro ("element_intersect_box [D2] K.name = `"
	      << K.name() << ": side " << j << "...");
	if (element_intersect_box (S, x, h)) {
            trace_macro ("element_intersect_box [F2] K.name = `"
	       << K.name() << ": side " << j << ": intersect !");
	    return true;
        }
        trace_macro ("element_intersect_box [F2] K.name = `"
	    << K.name() << ": side " << j << ": not intersect, continue...");
    }
    trace_macro ("element_intersect_box [F3]");
#endif // TODO
    return false;
}
// x in K
bool
geo_tree::belongs_to_element (const point& x, const geo_element& K) const
{
    switch (_dim) {
      case 1: {
        check_macro (K.type() == geo_element::e, 
	    "belongs_to_element: type `" << K.name() << "' not implemented.");
	bool res = belongs_to_segment_1d (x[0], _fvertex[K[0]][0], _fvertex[K[1]][0]);
	return res;
      }
      case 2: {
        check_macro (K.type() == geo_element::t, 
	    "belongs_to_element: type `" << K.name() << "' not implemented.");
	bool res = belongs_to_triangle_2d (x, _fvertex[K[0]], _fvertex[K[1]], _fvertex[K[2]]);
	return res;
      }
      case 3: {
        check_macro (K.type() == geo_element::T, 
	    "belongs_to_element: type `" << K.name() << "' not implemented.");
	bool res = belongs_to_tetra_3d (x, _fvertex[K[0]], _fvertex[K[1]],
		_fvertex[K[2]], _fvertex[K[3]]);
	return res;
      }
      default: {
	error_macro ("belongs_to_element: " << _dim << "D not implemented yet");
      }
    }
}
// =========================================================================
// 1d predicates
// =========================================================================

// x in [a,b]
bool
geo_tree::belongs_to_segment_1d (
    const Float& x,
    const Float& a, const Float& b) 
{
    return (x >=a && x <= b);
}
// =========================================================================
// 2d predicates
// =========================================================================
// notation: box(x,x+h) = [x0,x0+h] x [x1,x1+h]
// triangle(a,b,c) = closed triangle, i.e. including the boundary
//
// sgn(ab^ac)
int
geo_tree::sgn_mixt_2d (
    const point& a, 
    const point& b,
    const point& c)
{
    Float r = orient2d (b, c, a);
    return  (r < 0) ? -1 : ((r > 0) ? 1 : 0);
}
// x in triangle(a,b,c)
bool
geo_tree::belongs_to_triangle_2d (
    const point& x,
    const point& a, 
    const point& b,
    const point& c)
{
    return sgn_mixt_2d(a,b,x) >= 0 && 
	   sgn_mixt_2d(b,c,x) >= 0 &&
	   sgn_mixt_2d(c,a,x) >= 0;
}
// =========================================================================
// 3d predicates
// =========================================================================
// sgn(mixt(ab,ac,ad))
int
geo_tree::sgn_mixt_3d (
    const point& a, 
    const point& b,
    const point& c,
    const point& d)
{
    Float r = orient3d (b, c, d, a);
    return  (r < 0) ? -1 : ((r > 0) ? 1 : 0);
}
// x in tetra(a,b,c,d)
bool
geo_tree::belongs_to_tetra_3d (
    const point& x,
    const point& a, 
    const point& b,
    const point& c,
    const point& d)
{
    return sgn_mixt_3d(a,b,c,x) >= 0 && 
	   sgn_mixt_3d(a,d,b,x) >= 0 &&
	   sgn_mixt_3d(a,c,d,x) >= 0 &&
	   sgn_mixt_3d(b,d,c,x) >= 0;
}
