///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
#include "rheolef/form.h"
#include "rheolef/form_element.h"

#include "form_assembly.icc"

namespace rheolef { 
using namespace std;

template<class T>
form_basic<T>::form_basic (
    const space_type& X, 
    const space_type& Y,
    const std::string& name)
  : _X(X), 
    _Y(Y), 
    _uu(),
    _ub(),
    _bu(),
    _bb()
{
#ifdef TODO
    _X.freeze();
    _Y.freeze();
#endif // TODO
    form_element<T> form_e (
	name,
	_X.get_element(),
	_Y.get_element(),
	_X.get_geo());
    assembly (get_geo(), form_e);
}
// ----------------------------------------------------------------------------
// blas2
// ----------------------------------------------------------------------------
template<class T>

field_basic<T>
form_basic<T>::operator* (const field_basic<T>& xh) const
{
#ifdef TO_CLEAN
    size_t n = 5;
    vec<Float> x(n, 2.);
    vec<Float> y(n, 3.);

    // set size and then assign:
    vec<Float> z(n);
    z = 2*x - y;
    pcout << "z = " << z.par_size() << endl;
#endif // TO_CLEAN

    field yh (_Y, T(0));
    yh.u() = uu()*xh.u() + ub()*xh.b();
    yh.b() = bu()*xh.u() + bb()*xh.b();
#ifdef TODO
#endif // TODO
    return yh;
}
// ----------------------------------------------------------------------------
// output: print all four csr as a large sparse matrix in matrix-market format
// ----------------------------------------------------------------------------

struct id {
  typedef form_basic<Float>::size_type size_type;
  size_type operator() (size_type i) { return i; }
};
template<class T, class Permutation1, class Permutation2>
static
void
merge (
    asr<T>& a, 
    const csr<T>& m,
    Permutation1 par_im2par_idof,
    Permutation2 par_jm2par_jdof)
{
    typedef typename form_basic<T>::size_type size_type;
    size_type i0 = m.row_ownership().first_index();
    size_type j0 = m.col_ownership().first_index();
    typename csr<T>::const_iterator ia = m.begin(); 
    for (size_type im = 0, nrow = m.nrow(); im < nrow; im++) {
      size_type par_im = im + i0;
      size_type par_idof = par_im2par_idof (par_im);
      for (typename csr<T>::const_data_iterator p = ia[im]; p != ia[im+1]; p++) {
	const size_type& jm  = (*p).first;
	const T&         val = (*p).second;
	size_type par_jm     = jm + j0;
	size_type par_jdof   = par_jm2par_jdof (par_jm);
        a.entry (par_idof, par_jdof) = val;
      }
    }
#ifdef _RHEOLEF_HAVE_MPI
    typename csr<T>::const_iterator ext_ia = m.ext_begin(); 
    for (size_type im = 0, nrow = m.nrow(); im < nrow; im++) {
      size_type par_im = im + i0;
      size_type par_idof = par_im2par_idof (par_im);
      long int ext_size_im = std::distance(ext_ia[im],ext_ia[im+1]);
      for (typename csr<T>::const_data_iterator p = ext_ia[im]; p != ext_ia[im+1]; p++) {
	const size_type& jext = (*p).first;
	const T&         val  = (*p).second;
	size_type par_jm      = m.jext2par_j (jext);
	size_type par_jdof    = par_jm2par_jdof (par_jm);
        a.entry (par_idof, par_jdof) = val;
      }
    }
#endif // _RHEOLEF_HAVE_MPI
}
template<class T>
oparstream& 
form_basic<T>::put (oparstream& ops, bool show_partition) const
{
    // put all on io_proc 
    size_type par_nrow = get_second_space().par_size();
    size_type par_ncol =  get_first_space().par_size();
    size_type io_proc = oparstream::io_proc();
    size_type my_proc = comm().rank();
    distributor io_row_ownership (par_nrow, comm(), (my_proc == io_proc ? par_nrow : 0));
    distributor io_col_ownership (par_ncol, comm(), (my_proc == io_proc ? par_ncol : 0));
    asr<T> a (io_row_ownership, io_col_ownership);

    if (show_partition) {
        merge (a, uu(), id(), id());
        merge (a, ub(), id(), id());
        merge (a, bu(), id(), id());
        merge (a, bb(), id(), id());
    } else {
        error_macro ("not yet");
    }
    a.assembly();
    ops << "%%MatrixMarket matrix coordinate real general" << std::endl
        << par_nrow << " " << par_ncol << " " << a.par_nnz() << std::endl
        << a;
    return ops;
}
template <class T>
void
form_basic<T>::dump (std::string name) const
{
    _uu.dump (name + "-uu");
    _ub.dump (name + "-ub");
    _bu.dump (name + "-bu");
    _bb.dump (name + "-bb");
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class form_basic<Float>;

}// namespace rheolef
