# ifndef _RHEOLEF_FORM_H
# define _RHEOLEF_FORM_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/csr.h"
#include "rheolef/field.h"
#include "rheolef/form_element.h"
#include "rheolef/scalar_traits.h"
namespace rheolef { 

/*Class:form
NAME:  @code{form} - representation of a finite element bilinear form
DESCRIPTION:       
  The form class groups four sparse matrix, associated to 
  a bilinear form on two finite element spaces:
  @example
       a: U*V   ----> IR
         (u,v)  |---> a(u,v)
  @end example
  The operator @code{A} associated to the bilinear form is
  defined by:
  @example
       A: U  ----> V'
          u  |---> A(u)
  @end example
  where @code{A(u)} is such that @code{a(u,v)=<A(u),v>}
  for all u in U and v in V and where @code{<.,.>} denotes
  the duality product between V and V'. Since V is a
  finite dimensional spaces, the duality product is
  the euclidian product in IR^dim(V).

  Since both U and V are finite dimensional spaces,
  the linear operator can be represented by a matrix.
  The @code{form} class is represented by four sparse matrix
  in @code{csr} format (@pxref{csr class}),
  associated to unknown and blocked degrees of freedom 
  of origin and destination spaces (@pxref{space class}).
EXAMPLE:
  
The operator A associated to a bilinear form a(.,.) by
  the relation (Au,v) = a(u,v) could be applied by using
  a sample matrix notation A*u, as shown by the following code:
  @example
      geo omega("square");
      space V (omega,"P1");
      form a (V,V,"grad_grad");
      field uh = interpolate (fct, V);
      field vh = a*uh;
      cout << v;
  @end example

  The block-matrix @code{v=a*u} operation is equivalent to:
  @example
     vh.u = a.uu*uh.u + a.ub*uh.b;
     vh.b = a.bu*uh.u + a.bb*uh.b;
  @end example

SEE ALSO: "space"(3), "field"(3), "csr"(3)
AUTHOR: Pierre.Saramito@imag.fr
DATE:   2 july 1997
METHODS: @form
End:
*/

//<form:
template<class T>
class form_basic {
public :
// typedefs:

    typedef typename csr<T>::size_type      size_type;
    typedef T                               value_type;
    typedef typename scalar_traits<T>::type float_type; // TODO: when T=complex<scalar_type>
    typedef space_basic<float_type>         space_type;
    typedef geo_basic<float_type>           geo_type;

// allocator/deallocator:

    form_basic ();
    form_basic (const space_type& X, const space_type& Y, const std::string& name);

// accessors:

    const space_type& get_first_space() const;
    const space_type& get_second_space() const;
    const geo_type&   get_geo() const;

    const communicator& comm() const;

    csr<T>& uu();
    csr<T>& ub();
    csr<T>& bu();
    csr<T>& bb();
    const csr<T>& uu() const;
    const csr<T>& ub() const;
    const csr<T>& bu() const;
    const csr<T>& bb() const;

// linear algebra:

    form_basic<T>  operator+ (const form_basic<T>& b) const;
    form_basic<T>  operator- (const form_basic<T>& b) const;
    field_basic<T> operator* (const field_basic<T>& xh) const;

#ifdef TODO
    float_type operator () (const field_basic<T>& uh, const field_basic<T>& vh) const;
#endif // TODO

// io:

    oparstream& put (oparstream& ops, bool show_partition = true) const;
    void dump (std::string name) const;

// data
protected:
    space_type  _X;
    space_type  _Y;
    csr<T>      _uu;
    csr<T>      _ub;
    csr<T>      _bu;
    csr<T>      _bb;

// internals:
    void assembly (const geo_type& omega, const form_element<T>& form_e); 
};
typedef form_basic<Float> form;
//>form:

// ------------ inline'd -----------------------------------

template<class T>
inline
form_basic<T>::form_basic ()
: _X(), _Y(), _uu(), _ub(), _bu(), _bb()
{
}
template<class T>
inline
const typename form_basic<T>::space_type&
form_basic<T>::get_first_space() const
{
  return _X;
}
template<class T>
inline
const typename form_basic<T>::space_type&
form_basic<T>::get_second_space() const
{
  return _Y;
}
template<class T>
inline
const typename form_basic<T>::geo_type&
form_basic<T>::get_geo() const
{
  return _X.get_geo();
}
template<class T>
inline
csr<T>&
form_basic<T>::uu() 
{
  return _uu;
}
template<class T>
inline
csr<T>&
form_basic<T>::ub() 
{
  return _ub;
}
template<class T>
inline
csr<T>&
form_basic<T>::bu() 
{
  return _bu;
}
template<class T>
inline
csr<T>&
form_basic<T>::bb() 
{
  return _bb;
}
template<class T>
inline
const csr<T>&
form_basic<T>::uu() const
{
  return _uu;
}
template<class T>
inline
const csr<T>&
form_basic<T>::ub() const
{
  return _ub;
}
template<class T>
inline
const csr<T>&
form_basic<T>::bu() const
{
  return _bu;
}
template<class T>
inline
const csr<T>&
form_basic<T>::bb() const
{
  return _bb;
}
template<class T>
inline
const communicator&
form_basic<T>::comm() const
{
  return get_geo().comm();
}
template<class T>
inline
form_basic<T>
form_basic<T>::operator+ (const form_basic<T>& b) const
{
  form_basic<T> c;
  c.uu() = uu() + b.uu();
  c.ub() = ub() + b.ub();
  c.bu() = bu() + b.bu();
  c.bb() = bb() + b.bb();
  return c;
}
template<class T>
inline
form_basic<T>
form_basic<T>::operator- (const form_basic<T>& b) const
{
  form_basic<T> c;
  c.uu() = uu() - b.uu();
  c.ub() = ub() - b.ub();
  c.bu() = bu() - b.bu();
  c.bb() = bb() - b.bb();
  return c;
}

}// namespace rheolef
# endif /* _RHEOLEF_FORM_H */
