///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/form_element.h"
#include "rheolef/tensor.h"

namespace rheolef {
using namespace std;

// ==========================================================================
// part 1) allocator
// ==========================================================================
template<class T>
void
form_element_rep<T>::initialize_all ()
{
  initialize();
#ifdef TODO
  check_macro (_X.get_global_geo() == _Y.get_global_geo(),
	"form assembly: global spaces based on different meshes: unsupported");
#endif // TODO

  size_type quad_order = get_first_basis().degree() + get_second_basis().degree();
#ifdef TODO
  if (coordinate_system_type() != fem_helper::cartesian) quad_order++;
  if (_is_weighted) quad_order += get_weight_basis().degree();
#endif // TODO
  quad_order -= n_derivative();
  _quad.set_order (quad_order);
  _bx_table.set (_quad, get_first_basis());
  _by_table.set (_quad, get_second_basis());
  _tr_p1_table.set (_quad, get_p1_transformation());
#ifdef TODO
  if (_is_weighted) {
    check_macro (get_weight().get_space().get_global_geo() == get_global_geo(),
	"weighted form assembly: weight space based on a different mesh: not supported");
    _bw_table.set (_quad, get_weight_basis());
  }
#endif // TODO
  _initialized = true;
}
// ==========================================================================
// part 2) piola transform
// ==========================================================================
template<class T>
T
form_element_rep<T>::det_jacobian_piola_transformation (const tensor& DF, size_type map_dim) const
{
    if (coordinate_dimension() == map_dim) {
      return DF.determinant (map_dim);
    }
    /* surface jacobian: references:
     * Spectral/hp element methods for CFD
     * G. E. M. Karniadakis and S. J. Sherwin
     * Oxford university press
     * 1999
     * page 165
     */
    switch (map_dim) {
      case 0: return 1;
      case 1: return norm(DF.col(0));
      case 2: return norm(vect(DF.col(0), DF.col(1)));
      default:
        error_macro ("det_jacobian_piola_transformation: unsupported element dimension "
	    << map_dim << " in " << coordinate_dimension() << "D mesh.");
	return 0;
    }
}
// ----------------------------------------------------------
// DF (hat_xq)
// ----------------------------------------------------------
template<class T>
void
form_element_rep<T>::jacobian_piola_transformation (const geo_element& K, size_type q, tensor& DF) const
{
    size_type nt = get_p1_transformation().size(K);
    check_macro (nt == K.size(), "unsupported transformation " << get_p1_transformation().name());
    DF.fill (0);
    basis_on_quadrature::const_iterator_grad grad_tr
     = _tr_p1_table.begin_grad (K, q);
    size_type coord_d = coordinate_dimension();
    for (size_type k = 0; k < nt; k++, grad_tr++)
      my_cumul_otimes (DF, par_vertex(K[k]), *grad_tr, coord_d, K.dimension());
}
// The pseudo inverse extend inv(DF) for face in 3d or edge in 2d
//  i.e. usefull for Laplacian-Beltrami and others surfacic forms.
//
// pinvDF (hat_xq) = inv DF, if tetra in 3d, tri in 2d, etc
//                  = pseudo-invese, when tri in 3d, edge in 2 or 3d
// e.g. on 3d face : pinvDF*DF = [1, 0, 0; 0, 1, 0; 0, 0, 0]
//
// let DF = [u, v, w], where u, v, w are the column vectors of DF
// then det(DF) = mixt(u,v,w)
// and det(DF)*inv(DF)^T = [v^w, w^u, u^v] where u^v = vect(u,v)
//
// application:
// if K=triangle(a,b,c) then u=ab=b-a, v=ac=c-a and w = n = u^v/|u^v|.
// Thus DF = [ab,ac,n] and det(DF)=|ab^ac|
// and inv(DF)^T = [ac^n/|ab^ac|, -ab^n/|ab^ac|, n]
// The pseudo-inverse is obtained by remplacing the last column n by zero.
//
template<class T>
tensor
form_element_rep<T>::pseudo_inverse_jacobian_piola_transformation (
    const tensor& DF,
    size_type map_dim) const
{
    size_t d = coordinate_dimension();
    if (d == map_dim) {
      return inv(DF, map_dim);
    }
    tensor invDF;
    switch (map_dim) {
      case 0: { // point in 1D
          invDF(0,0) = 1;
          return invDF;
      }
      case 1: { // segment in 2D
          point t = DF.col(0);
          invDF.set_row (t/norm2(t), 0, d);
          return invDF;
      }
      case 2: {
          point t0 = DF.col(0);
          point t1 = DF.col(1);
          point n = vect(t0,t1);
          T det2 = norm2(n);
          point v0 =   vect(t1,n)/det2;
          point v1 = - vect(t0,n)/det2;
          invDF.set_row (v0, 0, d);
          invDF.set_row (v1, 1, d);
          return invDF;
      }
      default:
          error_macro ("pseudo_inverse_jacobian_piola_transformation: unsupported element dimension "
	    << map_dim << " in " << coordinate_dimension() << "D mesh.");
          return invDF;
    }
}
// ==========================================================================
// part 3) some few algebra on elementary matrices
// ==========================================================================

// ----------------------------------------------------------
// t(u,v) = (t*u,v) with map_d <= 3
// ----------------------------------------------------------
static
Float
tensor_apply (
  const tensor& t,
  const point& u,
  const point& v,
  tensor::size_type map_d)
{
  typedef tensor::size_type size_type;
  Float sum = 0;
  for (size_type k = 0; k < map_d; k++) {
    for (size_type l = 0; l < map_d; l++) {
      sum += v[k]*t(k,l)*u[l];
    }
  }
  return sum;
}
// ----------------------------------------------------------
// add to elementary matrix : M += (phi otimes psi) * w
// ----------------------------------------------------------
template<class T>
void
form_element_rep<T>::cumul_otimes (
    ublas::matrix<T>& m, 
    const T& w,
    basis_on_quadrature::const_iterator phi,
    basis_on_quadrature::const_iterator last_phi,
    basis_on_quadrature::const_iterator first_psi,
    basis_on_quadrature::const_iterator last_psi)
{
    for (size_type i = 0; phi != last_phi; phi++, i++) {
      basis_on_quadrature::const_iterator psi = first_psi;
      T phi_w = (*phi)*w;
      for (size_type j = 0; psi != last_psi; psi++, j++)
        m(i,j) += (*psi)*phi_w;
    }
}
// --------------------------------------------------------------------
// Mij += trans(grad_phi_i) * Dw * grad_psi_j
// --------------------------------------------------------------------
template<class T>
void
form_element_rep<T>::cumul_otimes (
    ublas::matrix<T>& m, 
    const tensor& Dw,
    basis_on_quadrature::const_iterator_grad grad_phi,
    basis_on_quadrature::const_iterator_grad last_grad_phi,
    basis_on_quadrature::const_iterator_grad first_grad_psi,
    basis_on_quadrature::const_iterator_grad last_grad_psi,
    size_type map_d)
{
    for (size_type i = 0; grad_phi != last_grad_phi; grad_phi++, i++) {
      basis_on_quadrature::const_iterator_grad grad_psi = first_grad_psi;
      for (size_type j = 0; grad_psi != last_grad_psi; grad_psi++, j++) {
	m(i,j) += tensor_apply (Dw, *grad_psi, *grad_phi, map_d);
      }
    }
}
static void my_cumul_otimes (tensor& t, const point& a, const point& b, size_t na, size_t nb) {
  cumul_otimes (t, a, b, na, nb);
}
// ==========================================================================
// part 4) predefined forms
// ==========================================================================

// ----------------------------------------------------------
// elementary mass form:
// ----------------------------------------------------------
template<class T>
void
form_element_rep<T>::build_scalar_mass (const geo_element& K, ublas::matrix<T>& m) const
{
  reference_element hat_K (K.type());
  m.resize (get_second_basis().size(hat_K), get_first_basis().size(hat_K));
  m.clear();
  quadrature::const_iterator first = _quad.begin(hat_K);
  quadrature::const_iterator last  = _quad.end  (hat_K);
  tensor DF;
  for (size_type q = 0; first != last; first++, q++) {
    T wq = 1;
#ifdef TODO
    wq *= is_weighted() ? weight(K,q,weight_i_comp) : T(1);
    wq *= weight_coordinate_system (K, q);
#endif // TODO
    jacobian_piola_transformation (K, q, DF);
    wq *= det_jacobian_piola_transformation (DF, K.dimension());
    wq *= (*first).w; 
    cumul_otimes (m, wq, 
    	_by_table.begin (hat_K, q), _by_table.end (hat_K, q),
    	_bx_table.begin (hat_K, q), _bx_table.end (hat_K, q));
  }
}
// ----------------------------------------------------------
// elementary grad_grad form:
// --------------------------------------------------------------------
template<class T>
void 
form_element_rep<T>::build_scalar_grad_grad (const geo_element& K, ublas::matrix<T>& a) const
{
  reference_element hat_K (K.type());
  size_type map_d = K.dimension();
  a.resize (get_second_basis().size(hat_K), get_first_basis().size(hat_K));
  a.clear();
  quadrature::const_iterator first = _quad.begin(hat_K);
  quadrature::const_iterator last  = _quad.end  (hat_K);
  tensor DF;
  tensor invDF;
  tensor Dw;
  tensor W;
  for (size_type q = 0; first != last; first++, q++) {
    jacobian_piola_transformation (K, q, DF);
    invDF = pseudo_inverse_jacobian_piola_transformation (DF, map_d);
    T wq = 1;
#ifdef TODO
    wq *= weight_coordinate_system (K, q);
#endif // TODO
    wq *= det_jacobian_piola_transformation (DF, K.dimension());
    wq *= (*first).w;
#ifdef TODO
    if (scalar_diffusion) {
       wq *= (is_weighted() ? weight(K,q) : T(1));
#endif // TODO
       Dw = wq*invDF*trans(invDF);
#ifdef TODO
    } else { // tensorial diffusion
       weight (K, q, W);
       Dw = wq*invDF*W*trans(invDF);
    }
#endif // TODO
    cumul_otimes (a, Dw,
      _by_table.begin_grad (hat_K, q), _by_table.end_grad (hat_K, q),
      _bx_table.begin_grad (hat_K, q), _bx_table.end_grad (hat_K, q),
      map_d);
  }
}
// ==========================================================================
// instanciation in library
// ==========================================================================
template class form_element_rep<Float>;

} // namespace rheolef
