#ifndef _RHEOLEF_GEO_H
#define _RHEOLEF_GEO_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

#include "rheolef/array.h"
#include "rheolef/polymorphic_array.h"
#include "rheolef/geo_element.h"
#include "rheolef/basic_point.h"
#include "rheolef/point_serialize.h"
#include "rheolef/domain.h"

namespace rheolef {

// =========================================================================
/// @brief pair(idx,point) mini-struct
// =========================================================================
/**
 * vertex are re-numbered for partitionning. The original numbering
 * is conserved in an associated index. Original numbering is important
 * when numeric data i/o, that may not depend upon partition algorithm
 * and number of processor.
 */
template <class T>
struct pair_idx_point : std::pair<typename basic_point<T>::size_type, basic_point<T> > {
    typedef typename basic_point<T>::size_type       size_type;
    typedef std::pair<size_type, basic_point<T> >    data_type;

    pair_idx_point () : data_type(std::numeric_limits<size_type>::max(), basic_point<T>()) {}

    size_type             par_old_iv() const { return data_type::first; }
    const basic_point<T>& vertex()     const { return data_type::second; }

    friend std::ostream&  operator<< (std::ostream& os, const pair_idx_point<T>& x) {
	return os << x.second; }
    template<class Archive>
    void serialize (Archive& ar, const unsigned int version) {
	ar & data_type::first; ar & data_type::second; }
};
/// @brief point input helper
template <class T>
struct _point_get {
  typedef typename basic_point<T>::size_type size_type;
  _point_get (size_type d1) : d(d1) {}
  size_type d;
  std::istream& operator() (std::istream& is, pair_idx_point<T>& x) { return x.second.get (is, d); }
}; 
/// @brief point output helper
template <class T>
struct _point_put {
  typedef typename basic_point<T>::size_type size_type;
  _point_put (size_type d1) : d(d1) {}
  size_type d;
  std::ostream& operator() (std::ostream& os, const pair_idx_point<T>& x) { return x.second.put (os, d); }
};

} // namespace rheolef

#ifdef _RHEOLEF_HAVE_MPI
// Some serializable types, have a fixed amount of data stored at fixed field positions.
// When this is the case, boost::mpi can optimize their serialization and transmission to avoid extraneous 
// copy operations.
// To enable this optimization, we specialize the type trait is_mpi_datatype, e.g.:
namespace boost {
 namespace mpi {
  template <> struct is_mpi_datatype<rheolef::pair_idx_point<double> > : mpl::true_ { };
 } // namespace mpi
} // namespace boost
#endif // _RHEOLEF_HAVE_MPI

namespace rheolef {

// forward declaration:
template <class T, class M> class geo_basic;

// =========================================================================
/// @brief base class for M=sequential or distributed meshes representations
// =========================================================================
// NOTE: since geo_seq_rep contains sequential arrays for vertices and elts,
//  the geo_mpi_rep cannot derive from geo_seq_rep. The solution is to
//  derive both from a generic base class that takes the memory model (seq or mpi).
template <class T, class M>
class geo_rep : public polymorphic_array<geo_element,M> {
public:
// typedefs:

    typedef typename polymorphic_array<geo_element,M>::size_type      size_type;
    typedef typename polymorphic_array<geo_element,M>::iterator       iterator;
    typedef typename polymorphic_array<geo_element,M>::const_iterator const_iterator;
    typedef basic_point<T>                                       vertex_type;
    typedef pair_idx_point<T>                                    idx_vertex_type;
    typedef typename array<idx_vertex_type>::const_iterator      const_iterator_idx_vertex;
// allocators:

    geo_rep (size_type par_ne = 0, size_type par_nv = 0);
    geo_rep (const geo_rep<T,M>&);
    ~geo_rep ();

// accessors & modifiers:
  
    std::string name() const { return _name; }
    void set_name(std::string name) { _name = name; }

    const distributor&   ownership() const { return polymorphic_array<geo_element,M>::ownership(); }
    const distributor&   vertex_ownership() const { return _idx_vertex.ownership(); }
    const communicator&       comm() const { return ownership().comm(); }
    size_type                 size() const { return polymorphic_array<geo_element,M>::size(); }
    size_type             par_size() const { return polymorphic_array<geo_element,M>::par_size(); }
    size_type             n_vertex() const { return _idx_vertex.size(); }
    size_type         par_n_vertex() const { return _idx_vertex.par_size(); }
    size_type            dimension() const { return _dimension; }
    size_type        map_dimension() const { return _dimension; }
    size_type         par_n_face() const { return _faces.par_size(); }
    size_type         par_n_edge() const { return _edges.par_size(); }

    const_iterator_idx_vertex begin_idx_vertex() const { return _idx_vertex.begin(); }

    const geo_element&	operator[] (size_type ie) const {
	return polymorphic_array<geo_element,M>::operator[](ie);
    }
	  geo_element&	operator[] (size_type ie)	     { 
	return polymorphic_array<geo_element,M>::operator[](ie);
    }
    const geo_element&	element (size_type ie) const { return operator[] (ie); }
	  geo_element&	element (size_type ie)	     { return operator[] (ie); }

    const basic_point<T>&  vertex  (size_type iv) const { return _idx_vertex[iv].second; }
	  basic_point<T>&  vertex  (size_type iv)	{ return _idx_vertex[iv].second; }

          iterator begin()       { return polymorphic_array<geo_element,M>::begin(); }
          iterator end()         { return polymorphic_array<geo_element,M>::end(); }
    const_iterator begin() const { return polymorphic_array<geo_element,M>::begin(); }
    const_iterator end()   const { return polymorphic_array<geo_element,M>::end(); }

    const geo_element&	edge (size_type iedg) const { return _edges [iedg]; }
    const_iterator begin_edge() const { return _edges.begin(); }
    const_iterator end_edge()   const { return _edges.end(); }

    const geo_element&	face (size_type ifac) const { return _faces [ifac]; }
    const_iterator begin_face() const { return _faces.begin(); }
    const_iterator end_face()   const { return _faces.end(); }

    distributor subgeo_ownership     (size_type dim) const;
    const geo_element& subgeo (size_type dim, size_type isg) const;

    const domain_basic<T,M>& operator[] (const std::string& name) const;

// data:
protected:
    array<idx_vertex_type, M>            _idx_vertex; // vertices & old idx before part.
    polymorphic_array<geo_element,M>     _edges;
    polymorphic_array<geo_element,M>     _faces;
    size_type                            _version;
    size_type                            _dimension;
    std::string                          _name;
    std::vector<domain_basic<T,M> >      _domains;
};
template <class T, class M>
inline
geo_rep<T,M>::geo_rep (size_type par_ne, size_type par_nv)
 : polymorphic_array<geo_element,M>(par_ne),
   _idx_vertex(par_nv),
   _edges(),
   _faces(),
   _version(0),
   _dimension(0),
   _name("none"),
   _domains()
{
}
// =========================================================================
/// @brief sequential mesh representation
// =========================================================================
template <class T>
class geo_seq_rep : public geo_rep<T,sequential> {
public:
// typedefs:

    typedef geo_rep<T,sequential>                    base;
    typedef typename base::size_type                 size_type;
    typedef typename base::vertex_type               vertex_type;
    typedef typename base::const_iterator_idx_vertex const_iterator_idx_vertex;

// allocators:

    geo_seq_rep();

// accessors:

    const vertex_type& par_vertex (size_type par_iv) const { return base::vertex (par_iv); }

    const distributor& vertex_ownership() const { return base::vertex_ownership(); }
          distributor  subgeo_ownership(size_type dim) const;

    const geo_element&	operator[] (size_type ie) const {
	return base::operator[](ie);
    }
	  geo_element&	operator[] (size_type ie)	     { 
	return base::operator[](ie);
    }
    const geo_element& subgeo (size_type dim, size_type isg) const;

    const domain_basic<T,sequential>& operator[] (const std::string& name) const {
			return base::operator[] (name); }

// i/o:

    iparstream& get (iparstream&, const geo_basic<T,sequential>&);
    oparstream& put (oparstream&) const;
    void dump (std::string name) const;
};
template <class T>
inline
geo_seq_rep<T>::geo_seq_rep ()
 : geo_rep<T,sequential>()
{
}
template <class T>
inline
distributor
geo_seq_rep<T>::subgeo_ownership (size_type dim) const
{
    return base::subgeo_ownership (dim);
}
template <class T>
inline
const geo_element&
geo_seq_rep<T>::subgeo (size_type dim, size_type isg) const
{
    return base::subgeo (dim, isg);
}

#ifdef _RHEOLEF_HAVE_MPI
// =========================================================================
/// @brief distributed mesh representation
// =========================================================================
template <class T>
class geo_mpi_rep : public geo_rep<T,distributed> {
public:
// typedefs:

    typedef geo_rep<T,distributed>                   base;
    typedef typename base::size_type                 size_type;
    typedef typename base::vertex_type               vertex_type;
    typedef typename base::idx_vertex_type           idx_vertex_type;
    typedef typename base::const_iterator_idx_vertex const_iterator_idx_vertex;
    typedef std::map <size_type, idx_vertex_type, std::less<size_type>,
            heap_allocator<std::pair<size_type,idx_vertex_type> > >    idx_vertex_map_type;

// allocators:

    geo_mpi_rep (size_type par_ne = 0, size_type par_nv = 0);

// accessors:

    size_type size () const;
    size_type map_dimension() const;
    const distributor& vertex_ownership() const;
          distributor  subgeo_ownership(size_type dim) const;

    const_iterator_idx_vertex begin_idx_vertex() const {
	return geo_rep<T,distributed>::begin_idx_vertex();
    }
    const geo_element&	operator[] (size_type ie) const {
	return geo_rep<T,distributed>::operator[](ie);
    }
	  geo_element&	operator[] (size_type ie)	{
	return geo_rep<T,distributed>::operator[](ie);
    }
    const geo_element&	element (size_type ie) const { return operator[] (ie); }
	  geo_element&	element (size_type ie)	     { return operator[] (ie); }

    const geo_element& subgeo (size_type dim, size_type isg) const;

    const domain_basic<T,distributed>& operator[] (const std::string& name) const {
			return base::operator[] (name); }

    const vertex_type& par_vertex (size_type par_iv) const;
    size_type       iv2par_old_iv (size_type     iv) const;
    size_type   par_iv2par_old_iv (size_type par_iv) const;

    const idx_vertex_map_type& ext_par_iv2par_old_iv_vertex() const { return _ext_par_iv2par_old_iv_vertex; }
    distributor subgeo_old_ownership (size_type dim) const;
    size_type old_isubgeo2par_isubgeo (size_type dim, size_type old_isg) const;
    size_type isubgeo2par_old_isubgeo (size_type dim, size_type isg) const;

// i/o:

    iparstream& get (iparstream&, const geo_basic<T,distributed>&);
    oparstream& put (oparstream&) const;
    void dump (std::string name) const;

    void load (std::string name, const communicator& comm,
		const geo_basic<T,distributed>& smart_ptr_this);

protected:
// data:
    idx_vertex_map_type       _ext_par_iv2par_old_iv_vertex; // external vertex, at partition bdry: pair(par_iv,pair(par_old_iv,point))
    array<size_type>          _old_ie2par_ie;     // reverse permutation for elements
    array<size_type>          _old_iv2par_iv;     // for vertices
    array<size_type>          _old_iedg2par_iedg; // for edges
    array<size_type>          _old_ifac2par_ifac; // for faces
};
template <class T>
inline
geo_mpi_rep<T>::geo_mpi_rep (size_type par_ne, size_type par_nv)
 : geo_rep<T,distributed>(par_ne,par_nv),
   _ext_par_iv2par_old_iv_vertex(),
   _old_ie2par_ie(),
   _old_iv2par_iv(),
   _old_iedg2par_iedg(),
   _old_ifac2par_ifac()
{
}
template <class T>
inline
typename geo_mpi_rep<T>::size_type
geo_mpi_rep<T>::size () const
{
    return base::size();
}
template <class T>
inline
typename geo_mpi_rep<T>::size_type
geo_mpi_rep<T>::map_dimension () const
{
    return base::map_dimension();
}
template <class T>
inline
typename geo_mpi_rep<T>::size_type
geo_mpi_rep<T>::iv2par_old_iv (size_type iv) const
{
    return base::_idx_vertex [iv].first;
}
template <class T>
inline
const distributor&
geo_mpi_rep<T>::vertex_ownership() const
{
    return base::vertex_ownership();
}
template <class T>
inline
distributor
geo_mpi_rep<T>::subgeo_ownership (size_type dim) const
{
    return base::subgeo_ownership (dim);
}
template <class T>
inline
const geo_element&
geo_mpi_rep<T>::subgeo (size_type dim, size_type isg) const
{
    return base::subgeo (dim, isg);
}
#endif // _RHEOLEF_HAVE_MPI

// =========================================================================
/// @brief generic mesh with rerefence counting
// =========================================================================
template <class T, class M = rheo_default_memory_model>
class geo_basic {
public:
    typedef M memory_type;
};
/*Class:geo
NAME:  geo - finite element mesh (@PACKAGE@-@VERSION@)
SYNOPSYS:       
 Distributed finite element mesh.
SEE ALSO: "geo_element"(3)
AUTHORS: Pierre.Saramito@imag.fr
DATE:   10 december 2010
End:
*/
// =========================================================================
/// @brief sequential mesh with reference counting
// =========================================================================
//<verbatim:
template <class T>
class geo_basic<T,sequential> : public smart_pointer<geo_seq_rep<T> > {
public:
    typedef sequential                              memory_type;
    typedef geo_seq_rep<T>                          rep;
    typedef smart_pointer<rep>                      base;
    typedef typename rep::size_type                 size_type;
    typedef typename rep::vertex_type               vertex_type;
    typedef typename rep::const_iterator_idx_vertex const_iterator_idx_vertex;

    geo_basic ()
      : base (new_macro(rep)) {}

    const communicator& comm() const { return base::data().comm(); }
    const distributor& ownership() const { return base::data().ownership(); }
    const distributor& vertex_ownership() const { return base::data().vertex_ownership(); }
          distributor  subgeo_ownership(size_type dim) const;

    std::string       name() const { return base::data().name(); }
    void set_name(std::string name) { base::data().set_name (name); }
    size_type         size() const { return base::data().size(); }
    size_type     par_size() const { return base::data().par_size(); }
    size_type     n_vertex() const { return base::data().n_vertex(); }
    size_type par_n_vertex() const { return base::data().par_n_vertex(); }
    size_type    dimension() const { return base::data().dimension(); }

    const vertex_type&     vertex(size_type     iv)const{ return base::data().vertex(iv); }
    const geo_element&	operator[] (size_type ie) const { return base::data().operator[](ie); }
	  geo_element&	operator[] (size_type ie)	{ return base::data().operator[](ie); }
    size_type     iv2par_old_iv (size_type     iv) const { return iv; }
    size_type par_iv2par_old_iv (size_type par_iv) const { return par_iv; }
    const vertex_type& par_vertex(size_type par_iv)const{ return base::data().par_vertex(par_iv); }

    const geo_element& subgeo (size_type dim, size_type isg) const;

    iparstream& get (iparstream& ips)	    { return base::data().get (ips,*this); }
    oparstream& put (oparstream& ops) const { return base::data().put (ops); }
    void dump (std::string name) const      {        base::data().dump (name); }
};
//>verbatim:
template <class T>
inline
distributor
geo_basic<T,sequential>::subgeo_ownership (size_type dim) const
{
    return base::data().subgeo_ownership (dim);
}
template <class T>
inline
const geo_element&
geo_basic<T,sequential>::subgeo (size_type dim, size_type isg) const
{
    return base::data().subgeo (dim, isg);
}
#ifdef _RHEOLEF_HAVE_MPI
// =========================================================================
/// @brief distributed mesh with rerefence counting
// =========================================================================
//<verbatim:
template <class T>
class geo_basic<T,distributed> : public smart_pointer<geo_mpi_rep<T> > {
public:
    typedef distributed                             memory_type;
    typedef geo_mpi_rep<T>                          rep;
    typedef smart_pointer<rep>                      base;
    typedef typename rep::size_type                 size_type;
    typedef typename rep::vertex_type               vertex_type;
    typedef typename rep::idx_vertex_type           idx_vertex_type;
    typedef typename rep::const_iterator_idx_vertex const_iterator_idx_vertex;
    typedef typename rep::idx_vertex_map_type       idx_vertex_map_type;

    geo_basic()
      : base (new_macro(rep)) {}
    geo_basic (std::string name, const communicator& comm = communicator())
      : base (new_macro(rep)) { base::data().load (name,comm,*this); }

    const communicator& comm()     const { return base::data().comm(); }
    const distributor& ownership() const { return base::data().ownership(); }
    const distributor& vertex_ownership() const { return base::data().vertex_ownership(); }
          distributor  subgeo_ownership(size_type dim) const;

    std::string       name() const { return base::data().name(); }
    void set_name(std::string name) { base::data().set_name (name); }
    size_type         size() const { return base::data().size(); }
    size_type     par_size() const { return base::data().par_size(); }
    size_type     n_vertex() const { return base::data().n_vertex(); }
    size_type par_n_vertex() const { return base::data().par_n_vertex(); }
    size_type    dimension() const { return base::data().dimension(); }
    
    const vertex_type&     vertex(size_type     iv)const{ return base::data().vertex(iv); }
    const vertex_type& par_vertex(size_type par_iv)const{ return base::data().par_vertex(par_iv); }
    const_iterator_idx_vertex begin_idx_vertex() const          { return base::data().begin_idx_vertex(); }
    const geo_element&	operator[] (size_type ie) const { return base::data().operator[](ie); }
	  geo_element&	operator[] (size_type ie)	{ return base::data().operator[](ie); }
    const geo_element&	element (size_type ie) const   { return operator[] (ie); }
	  geo_element&	element (size_type ie)	       { return operator[] (ie); }

    const geo_element& subgeo (size_type dim, size_type isg) const;

    const domain_basic<T,distributed>& operator[] (const std::string& name) const {
	  return base::data().operator[](name); }

    size_type     iv2par_old_iv (size_type     iv) const { return base::data().iv2par_old_iv(iv); }
    size_type par_iv2par_old_iv (size_type par_iv) const { return base::data().par_iv2par_old_iv(par_iv); }
    const idx_vertex_map_type& ext_par_iv2par_old_iv_vertex() const {
    				return base::data().ext_par_iv2par_old_iv_vertex(); }
    iparstream& get (iparstream& ips)	    { return base::data().get (ips, *this); }
    oparstream& put (oparstream& ops) const { return base::data().put (ops); }
};
//>verbatim:
#endif // _RHEOLEF_HAVE_MPI

/// @brief geo - the default mesh class
typedef geo_basic<Float,rheo_default_memory_model> geo;

// ==============================================================================
// inlined
// ==============================================================================
template <class T>
inline
distributor
geo_basic<T,distributed>::subgeo_ownership (size_type dim) const
{
    return base::data().subgeo_ownership (dim);
}
template <class T>
inline
const geo_element&
geo_basic<T,distributed>::subgeo (size_type dim, size_type isg) const
{
    return base::data().subgeo (dim, isg);
}
template <class T, class M>
inline 
iparstream&
operator>> (iparstream& ips, geo_basic<T,M>& omega)
{
    return omega.get (ips);
}
template <class T, class M>
inline 
oparstream&
operator<< (oparstream& ops, const geo_basic<T,M>& omega)
{
    return omega.put (ops);
}

} // namespace rheolef
#endif // _RHEOLEF_GEO_H
