///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/space.h"

namespace rheolef {

// ---------------------------------------------------------------
// space_rep
// ---------------------------------------------------------------
template <class T, class M>
inline
space_rep<T,M>::space_rep (
    const geo_basic<T,M>& omega,
    std::string approx)
  : _constit(omega,approx), 
    _is_freezed(false),
    _idof2blk_iub(),
    _iu_ownership(),
    _ib_ownership()
{
    if (approx == "") return; // empty element => default cstor
    // implements only P1 without blocked domains:
    // P1 ownership follows vertex ownership
    warning_macro ("cstor: par_n_vertex  = " << get_geo().par_n_vertex() << ", n_vertex  = " << get_geo().n_vertex());
    size_type par_ndof = get_geo().par_n_vertex();
    size_type ndof     = get_geo().n_vertex();
    distributor idof_ownership (par_ndof, comm(), ndof);
    warning_macro ("cstor: 0) par_ndof  = " << par_ndof << ", ndof  = " << ndof);
    warning_macro ("cstor: 1) par_size  = " << idof_ownership.par_size() << ", size  = " << idof_ownership.size());
    _idof2blk_iub.resize (idof_ownership);
    warning_macro ("cstor: 2) par_size  = " << ownership().par_size() << ", size  = " << ownership().size());
    check_macro (idof_ownership.par_size() == par_ndof, "invalid par_ndof count");
    check_macro (idof_ownership.size()     == ndof,     "invalid ndof count");
}
template <class T, class M>
void
space_rep<T,M>::base_freeze_body () const
{
    warning_macro ("base_freeze_body...");
    check_macro (get_element().name() != "", "space with undefined finite element method cannot be used");
    // -----------------------------------------------------------------------
    // loop on domains: mark blocked dofs
    // -----------------------------------------------------------------------
    warning_macro ("base_freeze_body[1]: n_dom="<<_constit.size());
    const geo_basic<T,M>& omega = get_geo();
    size_type first_par_idof = ownership().first_index();
    size_type last_par_idof  = ownership().last_index();
    warning_macro ("first_par_idof = " << first_par_idof);
    warning_macro ("last_par_idof  = " << last_par_idof);
    array<size_type,M> blocked_flag (ownership(), 0); // array<bool,M> not supported
    for (typename space_constitution<T,M>::const_iterator
	iter = _constit.begin(),
	last = _constit.end(); iter != last; iter++) {
      const domain_basic<T,M>&          dom = (*iter).get_domain();
      typename space_act<T,M>::act_type act = (*iter).get_act();
      size_type dom_dim = dom.dimension();
      distributor isg_ownership = omega.subgeo_ownership (dom_dim);
      size_type   first_par_isg = isg_ownership.first_index();
      size_type    last_par_isg = isg_ownership.last_index();
      bool blk = (act == space_act<T,M>::block) ? true : false;
      warning_macro ("block domain["<<dom.name()<<"]...");
      std::vector<size_type> par_idof;
      geo_element_p P;
      for (size_type ioisg = 0, noisg = dom.size(); ioisg < noisg; ioisg++) {
        size_type par_isg = dom.oisg (ioisg).index();
        check_macro (par_isg >= first_par_isg && par_isg < last_par_isg, "par_isg="<<par_isg<<" out of range");
	// TODO: shift par_isg to isg into domain
        size_type isg = par_isg - first_par_isg;
        if (dom_dim > 0) {
          const geo_element& S = omega.subgeo (dom_dim, isg);
          warning_macro ("block domain["<<dom.name()<<"].side["<<ioisg<<"] = element["<<par_isg<<"] = " << S);
          set_par_dof (S, par_idof);
        } else {
          // fix for the d=0 domains of vertices:
          P[0] = par_isg;
          set_par_dof (P, par_idof);
        }
        warning_macro ("block domain["<<dom.name()<<"].side["<<ioisg<<"].nloc = " << par_idof.size());
        for (size_type iloc = 0, nloc = par_idof.size(); iloc < nloc; iloc++) {
          if (par_idof[iloc] >= first_par_idof && par_idof[iloc] < last_par_idof) {
            size_type idof = par_idof [iloc] - first_par_idof;
            warning_macro ("block domain["<<dom.name()<<"].side["<<ioisg<<"]["<<iloc<<"]: block local par_idof="<<par_idof[iloc]);
            blocked_flag[idof] = blk;
	  } else {
            warning_macro ("block domain["<<dom.name()<<"].side["<<ioisg<<"]["<<iloc<<"]: block NONLOCAL par_idof="<<par_idof[iloc]);
            blocked_flag.set (par_idof[iloc], blk);
          }
        }
      }
    }
    blocked_flag.assembly();
    // copy the blocked_flag into the iub array, as the "blocked" bit:
    for (size_type idof = 0, ndof = blocked_flag.size(); idof < ndof; idof++) {
      _idof2blk_iub [idof].set_blocked (blocked_flag[idof]);
    }
    // -----------------------------------------------------------------------
    // init numbering
    // -----------------------------------------------------------------------
    warning_macro ("base_freeze_body[2]: init numbering...");
    size_type n_unknown = 0;
    size_type n_blocked = 0;
    for (size_type idof = 0, ndof = _idof2blk_iub.size(); idof < ndof; idof++) {
     
        bool blk = _idof2blk_iub [idof].is_blocked();
        if (! blk) {
          _idof2blk_iub[idof].set_iub (n_unknown);
          n_unknown++;
        } else {
          _idof2blk_iub[idof].set_iub (n_blocked);
          n_blocked++;
        }
        warning_macro ("par_idof2blk_iub["<<first_par_idof + idof<<"] = " << _idof2blk_iub[idof]);
    }
    size_type par_n_unknown = mpi::all_reduce (comm(), n_unknown, std::plus<T>());
    size_type par_n_blocked = mpi::all_reduce (comm(), n_blocked, std::plus<T>());
    _iu_ownership = distributor (par_n_unknown, comm(), n_unknown);
    _ib_ownership = distributor (par_n_blocked, comm(), n_blocked);

#ifndef TO_CLEAN
    warning_macro ("par_n  = " << ownership().par_size() << ", n  = " << ownership().size());
    warning_macro ("par_nu = " << par_n_unknown << ", nu = " << n_unknown);
    warning_macro ("par_nb = " << par_n_blocked << ", nb = " << n_blocked);
    
    check_macro (ownership().par_size() == _iu_ownership.par_size() + _ib_ownership.par_size(),
		"invalid par count: "
    		<<   ownership().par_size() << " = "
	        << _iu_ownership.par_size() << " + "
		<< _ib_ownership.par_size());

    check_macro (ownership().size() == _iu_ownership.size() + _ib_ownership.size(),
		"invalid count: "
    		<<   ownership().size() << " = "
	        << _iu_ownership.size() << " + "
		<< _ib_ownership.size());

#endif // TO_CLEAN

    warning_macro ("base_freeze_body done");
}
// ----------------------------------------------------------------------------
// used by field & form_assembly
// ----------------------------------------------------------------------------
template <class T, class M>
void
space_rep<T,M>::set_par_dof (const geo_element& K, std::vector<size_type>& par_idof) const
{
    // TODO: only for P1 element here
    freeze_guard();
    par_idof.resize (K.size());
    for (size_type iloc = 0, nloc = K.size(); iloc < nloc; iloc++) {
        par_idof [iloc] = K[iloc];
    }
}
// ---------------------------------------------------------------
// space_seq_rep
// ---------------------------------------------------------------
template <class T>
space_seq_rep<T>::space_seq_rep (
    const geo_basic<T,sequential>& omega,
    std::string approx)
  : space_rep<T,sequential>::space_rep (omega, approx)
{
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class space_rep<Float,sequential>;
template class space_seq_rep<Float>;

#ifdef _RHEOLEF_HAVE_MPI
template class space_rep<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
