#ifndef _RHEO_MSG_UTIL_H
#define _RHEO_MSG_UTIL_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
// message exchange basic algorithms
//   implemented with MPI
//
// author: Pierre.Saramito@imag.fr
//
// date: 17 december 1998
//
# include "rheolef/parallel.h"
namespace rheolef {

// bugs in MPI...
#ifndef RHEO_MPI_PRUDENCE
#define RHEO_MPI_PRUDENCE 0
#endif

// there is a non-standard SGI extension like that:
template <class T1, class T2>
struct select1st : std::unary_function<std::pair<T1,T2>, T1> {
  T1 operator() (const std::pair<T1,T2>& x) const { return x.first; }
};
template <class T1, class T2>
struct select2nd : std::unary_function<std::pair<T1,T2>, T2> {
  T2 operator() (const std::pair<T1,T2>& x) const { return x.second; }
};

// always true predicate
template <class T>
struct always_true : std::unary_function<T,bool> {
    bool operator()(const T& x) const { return true; }
};
// operator :=
template <class T1, class T2>
struct set_op : std::binary_function<T1,T2,T1> {
    T1& operator()(T1& x, const T2& y) const { return x = y; }
};  
// operator +=
template <class T1, class T2>
struct set_add_op : std::binary_function<T1,T2,T1> {
    T1& operator()(T1& x, const T2& y) const { return x += y; }
};  
// dummy iterator, when an iterator is required but
// we have no usage.
template <class T, class Distance = ptrdiff_t>
class dummy_iterator : public std::iterator<std::input_iterator_tag, T, Distance, const T*, const T&> {
public:
	dummy_iterator& operator++() { return *this; }
	dummy_iterator& operator++(int) { return *this; }
	T& operator*() { return _dummy; }
	const T& operator*() const { return _dummy; }
	dummy_iterator() : _dummy() {}
protected:
	T _dummy;
};
// index iterator, simulates array[i] = i
template <class Size, class Distance = ptrdiff_t>
class index_iterator : public std::iterator<std::input_iterator_tag, Size, Distance, const Size*, const Size&> {
public:
	index_iterator& operator++() { _i++; return *this; }
	index_iterator operator++(int) { 
	    index_iterator<Size,Distance> tmp = *this;
            _i++;
            return tmp;
	}
	const Size& operator*() const { return _i; }
	const Size& operator[](const Size& i) const { return i; }
	bool operator==(const index_iterator<Size,Distance>& x) const{
	    return x._i == _i; }
	bool operator!=(const index_iterator<Size,Distance>& x) const{
	    return !(x._i == _i); }
	index_iterator(Size i0 = 0) : _i(i0) {}
protected:
	Size _i;
};

// f1(pair x) = x.first
template <class Pair>
struct first_op
: public std::unary_function<Pair,typename Pair::first_type> {
    typename std::unary_function<Pair,typename Pair::first_type>::result_type
    operator()(const Pair& x) const {
	return x.first; }
};
// f2(pair x) = x.second
template <class Pair>
struct second_op
  : public std::unary_function<Pair, typename Pair::second_type> {
    typename std::unary_function<Pair,typename Pair::second_type>::result_type
    operator()(const Pair& x) const {
	return x.second; }
};
// pair<const uint, T> and pair<uint, T> are not compatible
// for some C++; so convert it explicitly:
template<class Pair1, class Pair2>
struct pair_identity : public std::unary_function<Pair1, Pair2> {
    Pair2 operator()(const Pair1& x) const {
	return Pair2(x.first, x.second); }
};
// wrapper iterator class that applies an operator
template <class Iterator, class Operator>
class apply_iterator : public std::iterator_traits<Iterator> {
public:
    typedef typename Operator::result_type value_type;
    apply_iterator(Iterator i1, Operator op1)
      : i(i1), op(op1) {}
    apply_iterator& operator++() { 
	i++; return *this; }
    apply_iterator operator++(int) { 
	apply_iterator t = *this; i++; return t; }
    value_type operator*() const { return op(*i); }
    bool operator== (apply_iterator<Iterator,Operator> b) const{ return (i == b.i); }
    bool operator!= (apply_iterator<Iterator,Operator> b) const{ return (i != b.i); }
protected:
    Iterator i;
    Operator op;
};
template <class Iterator, class Operator>
inline
apply_iterator<Iterator,Operator>
make_apply_iterator(Iterator i, Operator op) {
    return apply_iterator<Iterator,Operator>(i,op); 
}
// some c++ cannot convert pair<const I,T> to pair<I,T>:
template <class InputIterator, class OutputIterator>
OutputIterator
msg_pair_copy(InputIterator input, InputIterator last,
OutputIterator result) {
    while (input != last) {
	(*result).first = (*input).first;
  	(*result++).second = (*input++).second;
    }
    return result;
}
} // namespace rheolef
#endif // _RHEO_MSG_UTIL_H
