#ifndef BOOST_PP_IS_ITERATING
#ifndef _RHEOLEF_POLYMORPHIC_ARRAY_H
#define _RHEOLEF_POLYMORPHIC_ARRAY_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

#include "rheolef/polymorphic_traits.h"
#include "rheolef/polymorphic_data_vector.h"
#include "rheolef/polymorphic_data_map.h"
#include "rheolef/polymorphic_map.h"
#include "rheolef/polymorphic_scatter_message.h"
#include "rheolef/mpi_scatter_init.h"
#include <list>

namespace rheolef {
namespace mpl = boost::mpl;

template <class T, class V, int NV>
struct polymorphic_array_seq_rep {};

/// @brief array element output helper
template <class T>
struct _polymorphic_array_put_element_type {
  std::ostream& operator() (std::ostream& os, const T& x) { return os << x; }
};

#ifdef _RHEOLEF_HAVE_MPI

template <class T, class V, int NV>
struct polymorphic_array_mpi_rep {};

template <class Stash, class Message, int NV>
struct mpi_polymorphic_assembly_begin_t {};

template <class Container, class Message, int NV>
struct mpi_polymorphic_assembly_end_t {};

template <class Container, class Message, class Buffer, class Tag, int NV>
struct mpi_polymorphic_scatter_begin_t {};

template <class Container, class Message, class Tag, int NV>
struct mpi_polymorphic_scatter_begin_global_t {};

template <class Container, class Message, class Buffer, int NV>
struct mpi_polymorphic_scatter_end_t {};

#endif // _RHEOLEF_HAVE_MPI

// ----------------------------------------------------------
// run here the recursive inclusion of this file:
// ----------------------------------------------------------
// _RHEOLEF_POLYMORPHIC_MAX_SIZE was defined by "polymorphic_data_vector.h"
#define BOOST_PP_ITERATION_LIMITS (0, _RHEOLEF_POLYMORPHIC_MAX_SIZE)
#define BOOST_PP_FILENAME_1    "rheolef/polymorphic_array.h" // this file
#include BOOST_PP_ITERATE()

} // namespace rheolef
#endif // _RHEOLEF_POLYMORPHIC_ARRAY_H
#else  // BOOST_PP_IS_ITERATING
// ----------------------------------------------------------
// here, this file is recursively included with growing "N" :
// ----------------------------------------------------------
#define N    BOOST_PP_ITERATION()

/*Class:polymorphic_array
NAME:  polymorphic_array - an array of derived classes (@PACKAGE@-@VERSION@)
SYNOPSYS:
  @example
  template <class T, class V> class polymorphic_array_seq_rep;
  @end example
DESCRIPTION:       
 Geometric entities of finite elements are implemented by using
 a polymorphic hierarchy of classes: the base class @code{geo_element} 
 is a pure virtual methods. Then, derived classes, such as 
 @code{geo_element_t} for a triangle, are introduced with a storage
 zone for the indexes of its vertices in the mesh (See "geo"(3)).

 Each element has a fixed size storage zone: a buffer of a
 particular element type can be efficiently exchanged with MPI.

 Here is an atempt to generalize the distributed array<T>,
 with reference counting, to the polymorphic case. The base
 class is denoted as T and the derived ones are included as
 a vector of type V, e.g. V=mpl::vctor<T0,T1,T2>.

TODO:
 Here, class Ti are suposed to have a method :
	size_type variant() const;
 that returns the class index i=0..n_variant-1.
 A more general approch should us run-time type identification.

 Finally, the container should honor an optional Allocator template class parameter.
SEE ALSO: "geo"(3) "geo_element"(3)
AUTHORS:
   LJK-IMAG, 38041 Grenoble cedex 9, France
   | Pierre.Saramito@imag.fr
DATE:   15 december 2010
End:
*/

// -------------------------------------------------------------
// representation classes
// -------------------------------------------------------------
template <class T, class V>
class polymorphic_array_seq_rep<T,V,N> {
public:

// typedefs:

    static const size_t                                   _n_variant  = mpl::size<V>::value; // should be = N
    typedef typename std::vector<int>::size_type          size_type;
    typedef T                                             element_type;
    typedef          polymorphic_data_map<T,V,N>          data_type;   // extensible area
    typedef          polymorphic_data_vector<T,V,N>       buffer_type; // contiguous area ; for mpi

#define _RHEOLEF_typedef(z,k,unused)                                    \
    typedef typename mpl::at_c<V,k>::type                 T##k;		\
    typedef typename data_type::map##k##_type             map##k##_type;
    BOOST_PP_REPEAT(N, _RHEOLEF_typedef, ~)
#undef _RHEOLEF_typedef

    struct const_reference {
        const_reference (const polymorphic_pair_type<T,V,N>& x) : _x(x) {}
        operator const T& () const { return _x.get_reference(); }
      // data:
      protected:
        const polymorphic_pair_type<T,V,N>& _x;
    };
    struct const_iterator {
      typedef typename std::vector<polymorphic_pair_type<T,V,N> >::const_iterator  raw_iterator;
      typedef std::bidirectional_iterator_tag        iterator_category;
      typedef typename raw_iterator::value_type      value_type;
      typedef typename raw_iterator::difference_type difference_type;
      typedef typename raw_iterator::pointer         pointer;
      typedef typename polymorphic_array_seq_rep<T,V,N>::const_reference reference;

      const_iterator (raw_iterator ri) : _ri(ri) {}
      const_reference operator* () const { return const_reference(*_ri); }
      const_iterator& operator++ () { ++_ri; return *this; }
      const_iterator& operator-- () { --_ri; return *this; }
      const_iterator  operator++ (int) {
		const_iterator tmp = *this;
		++_ri;
		return tmp;
      }
      const_iterator  operator-- (int) {
		const_iterator tmp = *this;
		--_ri;
		return tmp;
      }
      bool operator== (const_iterator j) const { return _ri == j._ri; }
      bool operator!= (const_iterator j) const { return _ri != j._ri; }
    protected:
      raw_iterator  _ri;
    };
    struct reference {
      reference (polymorphic_array_seq_rep<T,V,N>& x, size_type index)
       : _x(x), _index(index) {}
      operator const T& () const { return _x.rhs (_index); }
      operator       T& ()       { return _x.rhs_non_const (_index); }

      size_type variant() const { return _x.variant(_index); }

#define _RHEOLEF_assign(z,k,unused)                            	\
      reference& operator= (const T##k& value) {		\
        _x.assign (_index, value);				\
        return *this;						\
      }
      BOOST_PP_REPEAT(N, _RHEOLEF_assign, ~)
#undef _RHEOLEF_assign

#ifdef TODO
      reference& operator= (const_reference ref);
#endif // TODO
    // data:
    protected:
      polymorphic_array_seq_rep<T,V,N>& _x;
      size_type                         _index;
    };
    struct iterator {
      typedef std::bidirectional_iterator_tag iterator_category;
      typedef T                               value_type;
      typedef std::ptrdiff_t                  difference_type;
      typedef T&                              pointer;
      typedef typename polymorphic_array_seq_rep<T,V,N>::reference reference;

      iterator (polymorphic_array_seq_rep<T,V,N>& x, size_type index)
         : _x(x), _index(index) {}
      operator const_iterator() const { return const_iterator (_x._ptr.begin() + _index); }
      reference operator* () const { return reference(_x,_index); }
      iterator& operator++ () { ++_index; return *this; }
      iterator& operator-- () { --_index; return *this; }
      iterator  operator++ (int) {
		iterator tmp = *this;
		++_index;
		return tmp;
      }
      iterator  operator-- (int) {
		iterator tmp = *this;
		--_index;
		return tmp;
      }
      bool operator== (iterator j) const { return _index == j._index; }
      bool operator!= (iterator j) const { return _index != j._index; }
      // data:
    protected:
      polymorphic_array_seq_rep<T,V,N>& _x;
      size_type                         _index;
    };

// allocators:

    polymorphic_array_seq_rep (const distributor& ownership);
    polymorphic_array_seq_rep (size_type loc_size = 0);
    polymorphic_array_seq_rep (const array_seq_rep<T>& x);
    void resize (const distributor& ownership);
    void resize (size_type loc_size);

// accessors

  size_type     size() const { return _ptr.size(); }
  size_type par_size() const { return size(); }
  reference       operator[] (size_type index)       { return       reference (*this, index); }
  const_reference operator[] (size_type index) const { return const_reference (_ptr[index]); }
  size_type       variant    (size_type index) const { return _ptr[index].variant(); }
  const_iterator begin () const { return const_iterator (_ptr.begin()); }
  const_iterator end   () const { return const_iterator (_ptr.end()); }
  iterator begin () { return iterator (*this, 0); }
  iterator end   () { return iterator (*this, _ptr.size()); }
  const distributor&  ownership() const { return _ownership; }
  const communicator&  comm() const { return ownership().comm(); }

#define _RHEOLEF_set(z,k,unused)                            			\
  void set    (size_type index, const T##k& value) { assign (index, value); }
  BOOST_PP_REPEAT(N, _RHEOLEF_set, ~)
#undef _RHEOLEF_set

// i/o

  iparstream& get (iparstream& ips);
  oparstream& put (oparstream& ops) const;
  template <class PutFunction> oparstream& put (oparstream& opsa, PutFunction put_element) const;
  void dump (std::string name) const;

// internal:
protected:
  const T& rhs     (size_type index) const { return _ptr [index].get_reference(); }
  T& rhs_non_const (size_type index)       { return _ptr [index].get_reference(); }
  void erase (size_type index);
  void update_ptr();

#define _RHEOLEF_load(z,k,unused)                  			\
  void load (const std::vector<std::pair<size_type,T##k> >& buffer);	\
  void assign (size_type index, const T##k& value);
  BOOST_PP_REPEAT(N, _RHEOLEF_load, ~)
#undef _RHEOLEF_load

// data:

protected:
  friend class polymorphic_map_rep<T,V,N>;
  distributor                                 _ownership;
  std::vector<polymorphic_pair_type<T,V,N> >  _ptr;
  data_type                                   _data;
};
#ifdef _RHEOLEF_HAVE_MPI
template <class T, class V>
class polymorphic_array_mpi_rep<T,V,N> : public polymorphic_array_seq_rep<T,V,N> {
public:

// typedefs:

  static const size_t  _n_variant  = mpl::size<V>::value; // should be = N
  typedef polymorphic_array_seq_rep<T,V,N>   seq;
  typedef typename seq::size_type        size_type;
  typedef T                              element_type;
  typedef typename seq::data_type        data_type;
  typedef typename seq::buffer_type      buffer_type;
  typedef typename seq::reference        reference;
  typedef typename seq::const_reference  const_reference;
  typedef typename seq::iterator         iterator;
  typedef typename seq::const_iterator   const_iterator;

#define _RHEOLEF_typedef(z,k,unused)                                    \
    typedef typename mpl::at_c<V,k>::type                 T##k;		\
    typedef typename data_type::map##k##_type             map##k##_type;
    BOOST_PP_REPEAT(N, _RHEOLEF_typedef, ~)
#undef _RHEOLEF_typedef

// allocators:

  polymorphic_array_mpi_rep (const distributor& ownership);
  polymorphic_array_mpi_rep (size_type par_size = 0, size_type loc_size = distributor::decide); 
  polymorphic_array_mpi_rep (const polymorphic_array_mpi_rep<T,V,N>& x);
  void resize (const distributor& ownership);
  void resize (size_type par_size = 0, size_type loc_size = distributor::decide);

// accessors & modifiers:

  size_type size()     const { return polymorphic_array_seq_rep<T,V,N>::size(); }
  size_type par_size() const { return ownership().par_size(); }

  const distributor&  ownership() const { return polymorphic_array_seq_rep<T,V,N>::ownership(); }
  const communicator& comm() const { return ownership().comm(); }

  const_reference operator[] (size_type index) const {
	return polymorphic_array_seq_rep<T,V,N>::operator[] (index);
  }
  reference operator[] (size_type index) {
  	return polymorphic_array_seq_rep<T,V,N>::operator[] (index);
  }
  const_iterator begin () const { return seq::begin(); }
  const_iterator end   () const { return seq::end(); }
        iterator begin ()       { return seq::begin(); }
        iterator end   ()       { return seq::end(); }

#define _RHEOLEF_set(z,k,unused)                            	\
  void set    (size_type par_index, const T##k& value);
  BOOST_PP_REPEAT(N, _RHEOLEF_set, ~)
#undef _RHEOLEF_set

  void assembly_begin ();
  void assembly_end ();

  void repartition (					      // old_numbering for *this
        const array_mpi_rep<size_type>&     partition,	      // old_ownership
        polymorphic_array_mpi_rep<T,V,N>&   new_array,	      // new_ownership (created)
        array_mpi_rep<size_type>&           old_numbering,    // new_ownership
        array_mpi_rep<size_type>&           new_numbering) const; // old_ownership

  template<class Set>
  void scatter (const Set& ext_idx_set, polymorphic_map_rep<T,V,N>& ext_idx_map) const;

// i/o

  iparstream& get (iparstream& ips);
  oparstream& put (oparstream& ops) const;
  template <class Permutation>
  oparstream& permuted_put (oparstream& ops, const Permutation& perm) const;
  template <class Permutation, class PutFunction>
  oparstream& permuted_put (oparstream& ops, const Permutation& perm,
				PutFunction put_element) const;
  void dump (std::string name) const;

protected:
  typedef polymorphic_data_map<T,V,_n_variant>   stash_type;
  struct message_type {
    boost::array<std::list<std::pair<size_t,mpi::request> >,_n_variant>   waits;
    polymorphic_data_vector<T,V,_n_variant>                               data;
  };
// data:
  stash_type                          _stash;
  message_type                        _send;
  message_type                        _receive;
  boost::array<size_type,_n_variant>  _receive_max_size;
};
#endif // _RHEOLEF_HAVE_MPI

#ifndef _RHEOLEF_POLYMORPHIC_ARRAY
#define _RHEOLEF_POLYMORPHIC_ARRAY
// -------------------------------------------------------------
// wrapper class: sequential polymorphic_array 
// -------------------------------------------------------------
template <class T, class M = rheo_default_memory_model, class V = typename polymorphic_traits<T>::derived_type>
struct polymorphic_array {};

//<polymorphic_array:
template <class T, class V>
class polymorphic_array<T,sequential,V> : public smart_pointer<polymorphic_array_seq_rep<T,V,mpl::size<V>::value> > {
public:

// typedefs:

    static const size_t _n_variant  = mpl::size<V>::value;

    typedef sequential                                    memory_type;
    typedef polymorphic_array_seq_rep<T,V,_n_variant>     rep;
    typedef smart_pointer<rep>                            base;
    typedef typename rep::size_type                       size_type;
    typedef typename rep::reference                       reference;
    typedef typename rep::const_reference                 const_reference;
    typedef typename rep::iterator                        iterator;
    typedef typename rep::const_iterator                  const_iterator;

// constructors:

    polymorphic_array (size_type n = 0);
    polymorphic_array (const distributor& ownership);
    void resize (size_type n);
    void resize (const distributor& ownership);

// accessors & modifiers:

    size_type       size () const;
    size_type       par_size() const;
    reference       operator[] (size_type i);
    const_reference operator[] (size_type i) const;
    const_iterator begin () const;
    const_iterator end   () const;
          iterator begin ();
          iterator end   ();

    const distributor&  ownership() const;
    const communicator& comm() const;

// i/o:

    iparstream& get (iparstream& ips);
    oparstream& put (oparstream& ops) const;
    template <class PutFunction> oparstream& put (oparstream& ops, PutFunction put_element) const;
    void dump (std::string name) const;
};
template <class T, class V>
oparstream& operator<< (oparstream& ops, const polymorphic_array<T,sequential,V>& x);

template <class T, class V>
iparstream& operator>> (iparstream& ips, polymorphic_array<T,sequential,V>& x);
//>polymorphic_array:

template <class T, class V>
inline
polymorphic_array<T,sequential,V>::polymorphic_array (const distributor& ownership)
 : base (new_macro(rep(ownership)))
{
}
template <class T, class V>
inline
polymorphic_array<T,sequential,V>::polymorphic_array (size_type n)
 : base (new_macro(rep(n)))
{
}
template <class T, class V>
inline
void
polymorphic_array<T,sequential,V>::resize (size_type n)
{
    return base::data().resize(n);
}
template <class T, class V>
inline
void
polymorphic_array<T,sequential,V>::resize (const distributor& ownership)
{
    return base::data().resize(ownership);
}
template <class T, class V>
inline
typename polymorphic_array<T,sequential,V>::size_type
polymorphic_array<T,sequential,V>::size () const
{
    return base::data().size();
}
template <class T, class V>
inline
typename polymorphic_array<T,sequential,V>::size_type
polymorphic_array<T,sequential,V>::par_size () const
{
    return base::data().par_size();
}
template <class T, class V>
inline
typename polymorphic_array<T,sequential,V>::reference
polymorphic_array<T,sequential,V>::operator[] (size_type i) 
{
    return base::data().operator[] (i);
}
template <class T, class V>
inline
typename polymorphic_array<T,sequential,V>::const_reference
polymorphic_array<T,sequential,V>::operator[] (size_type i) const
{
    return base::data().operator[] (i);
}
template <class T, class V>
inline
typename polymorphic_array<T,sequential,V>::iterator
polymorphic_array<T,sequential,V>::begin ()
{
    return base::data().begin();
}
template <class T, class V>
inline
typename polymorphic_array<T,sequential,V>::iterator
polymorphic_array<T,sequential,V>::end ()
{
    return base::data().end();
}
template <class T, class V>
inline
typename polymorphic_array<T,sequential,V>::const_iterator
polymorphic_array<T,sequential,V>::begin () const
{
    return base::data().begin();
}
template <class T, class V>
inline
typename polymorphic_array<T,sequential,V>::const_iterator
polymorphic_array<T,sequential,V>::end () const
{
    return base::data().end();
}
template <class T, class V>
inline
const distributor&
polymorphic_array<T,sequential,V>::ownership () const
{
    return base::data().ownership();
}
template <class T, class V>
inline
const communicator&
polymorphic_array<T,sequential,V>::comm() const
{
    return base::data().comm();
}
template <class T, class V>
inline
iparstream&
polymorphic_array<T,sequential,V>::get(iparstream& ips)
{
    return base::data().get(ips);
}
template <class T, class V>
template <class PutFunction>
inline
oparstream&
polymorphic_array<T,sequential,V>::put(oparstream& ops, PutFunction put_element) const
{
    return base::data().put (ops, put_element);
}
template <class T, class V>
inline
oparstream&
polymorphic_array<T,sequential,V>::put(oparstream& ops) const
{
    return base::data().put(ops);
}
template <class T, class V>
inline
void
polymorphic_array<T,sequential,V>::dump(std::string name) const
{
    return base::data().dump (name);
}
template <class T, class V>
inline
oparstream& 
operator<< (oparstream& ops, const polymorphic_array<T,sequential,V>& x) {
     return x.put (ops);
}
template <class T, class V>
inline
iparstream& 
operator>> (iparstream& ips, polymorphic_array<T,sequential,V>& x) {
     return x.get (ips);
}
// -------------------------------------------------------------
// wrapper class: distributed polymorphic_array 
// -------------------------------------------------------------
#ifdef _RHEOLEF_HAVE_MPI

//<polymorphic_array:
template <class T, class V>
class polymorphic_array<T,distributed,V> : public smart_pointer<polymorphic_array_mpi_rep<T,V,mpl::size<V>::value> > {
public:

// typedefs:

    static const size_t _n_variant  = mpl::size<V>::value;

    typedef distributed                                   memory_type;
    typedef polymorphic_array_mpi_rep<T,V,_n_variant>     rep;
    typedef smart_pointer<rep>                            base;
    typedef typename rep::size_type                       size_type;
    typedef typename rep::reference                       reference;
    typedef typename rep::const_reference                 const_reference;
    typedef typename rep::iterator                        iterator;
    typedef typename rep::const_iterator                  const_iterator;

// constructors:

    polymorphic_array (size_type n = 0);
    polymorphic_array (const distributor& ownership);
    void resize (size_type n);
    void resize (const distributor& ownership);

// accessors & modifiers:

    size_type       size () const;
    size_type       par_size () const;
    reference       operator[] (size_type i);
    const_reference operator[] (size_type i) const;
    const_iterator begin () const;
    const_iterator end   () const;
          iterator begin ();
          iterator end   ();

    const distributor&  ownership() const;
    const communicator& comm() const;

    void repartition (					      // old_numbering for *this
        const array<size_type,distributed>& partition,	      // old_ownership
        polymorphic_array<T,distributed,V>& new_array,	      // new_ownership (created)
        array<size_type,distributed>&       old_numbering,    // new_ownership
        array<size_type,distributed>&       new_numbering) const; // old_ownership

  template<class Set>
  void scatter (const Set& ext_idx_set, polymorphic_map<T,V>& ext_idx_map) const;

// i/o:

    iparstream& get (iparstream& ips);
    oparstream& put (oparstream& ops) const;
    template <class Permutation>
    oparstream& permuted_put (oparstream& ops, const Permutation& perm) const;
    template <class Permutation, class PutFunction>
    oparstream& permuted_put (oparstream& ops, const Permutation& perm,
				PutFunction put_element) const;
    void dump (std::string name) const;
};
template <class T, class V>
oparstream& operator<< (oparstream& ops, const polymorphic_array<T,distributed,V>& x);

template <class T, class V>
iparstream& operator>> (iparstream& ips, polymorphic_array<T,distributed,V>& x);
//>polymorphic_array:

template <class T, class V>
inline
polymorphic_array<T,distributed,V>::polymorphic_array (size_type n)
 : base (new_macro(rep(n)))
{
}
template <class T, class V>
inline
polymorphic_array<T,distributed,V>::polymorphic_array (const distributor& ownership)
 : base (new_macro(rep(ownership)))
{
}
template <class T, class V>
inline
void
polymorphic_array<T,distributed,V>::resize (size_type n)
{
    return base::data().resize(n);
}
template <class T, class V>
inline
void
polymorphic_array<T,distributed,V>::resize (const distributor& ownership)
{
    return base::data().resize(ownership);
}
template <class T, class V>
inline
typename polymorphic_array<T,distributed,V>::size_type
polymorphic_array<T,distributed,V>::size () const
{
    return base::data().size();
}
template <class T, class V>
inline
typename polymorphic_array<T,distributed,V>::size_type
polymorphic_array<T,distributed,V>::par_size () const
{
    return base::data().par_size();
}
template <class T, class V>
inline
typename polymorphic_array<T,distributed,V>::reference
polymorphic_array<T,distributed,V>::operator[] (size_type i) 
{
    return base::data().operator[] (i);
}
template <class T, class V>
inline
typename polymorphic_array<T,distributed,V>::const_reference
polymorphic_array<T,distributed,V>::operator[] (size_type i) const
{
    return base::data().operator[] (i);
}
template <class T, class V>
inline
typename polymorphic_array<T,distributed,V>::iterator
polymorphic_array<T,distributed,V>::begin ()
{
    return base::data().begin();
}
template <class T, class V>
inline
typename polymorphic_array<T,distributed,V>::iterator
polymorphic_array<T,distributed,V>::end ()
{
    return base::data().end();
}
template <class T, class V>
inline
typename polymorphic_array<T,distributed,V>::const_iterator
polymorphic_array<T,distributed,V>::begin () const
{
    return base::data().begin();
}
template <class T, class V>
inline
typename polymorphic_array<T,distributed,V>::const_iterator
polymorphic_array<T,distributed,V>::end () const
{
    return base::data().end();
}
template <class T, class V>
inline
const distributor&
polymorphic_array<T,distributed,V>::ownership () const
{
    return base::data().ownership();
}
template <class T, class V>
inline
const communicator&
polymorphic_array<T,distributed,V>::comm() const
{
    return base::data().comm();
}
template <class T, class V>
inline
iparstream&
polymorphic_array<T,distributed,V>::get(iparstream& ips)
{
    return base::data().get(ips);
}
template <class T, class V>
inline
oparstream&
polymorphic_array<T,distributed,V>::put(oparstream& ops) const
{
    return base::data().put(ops);
}
template <class T, class V>
template <class Permutation>
inline
oparstream&
polymorphic_array<T,distributed,V>::permuted_put (
    oparstream& ops, 
    const Permutation& perm) const
{
    return base::data().permuted_put (ops, perm);
}
template <class T, class V>
template <class Permutation, class PutFunction>
inline
oparstream&
polymorphic_array<T,distributed,V>::permuted_put (
    oparstream& ops, 
    const Permutation& perm,
    PutFunction put_element) const
{
    return base::data().permuted_put (ops, perm, put_element);
}
template <class T, class V>
inline
void
polymorphic_array<T,distributed,V>::dump(std::string name) const
{
    return base::data().dump (name);
}
template <class T, class V>
inline
oparstream& 
operator<< (oparstream& ops, const polymorphic_array<T,distributed,V>& x) {
     return x.put (ops);
}
template <class T, class V>
inline
iparstream& 
operator>> (iparstream& ips, polymorphic_array<T,distributed,V>& x) {
     return x.get (ips);
}
template <class T, class V>
inline
void
polymorphic_array<T,distributed,V>::repartition (
        const array<size_type,distributed>& partition,	
        polymorphic_array<T,distributed,V>& new_array,	 
        array<size_type,distributed>&       old_numbering,
        array<size_type,distributed>&       new_numbering) const
{
    base::data().repartition (
        partition.data(),
        new_array.data(),
        old_numbering.data(),
        new_numbering.data());
}
template <class T, class V>
template<class Set>
inline
void
polymorphic_array<T,distributed,V>::scatter (
	const Set&            ext_idx_set,
	polymorphic_map<T,V>& ext_idx_map) const
{
    base::data().scatter (ext_idx_set, ext_idx_map.data());
}
#endif // _RHEOLEF_HAVE_MPI


#endif // _RHEOLEF_POLYMORPHIC_ARRAY
// -------------------------------------------------------------
// not inlined : longer code
// -------------------------------------------------------------
#include "rheolef/polymorphic_array_seq.icc"
#include "rheolef/polymorphic_array_mpi.icc"

#endif // BOOST_PP_IS_ITERATING
