///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "div.h"
using namespace rheolef;
using namespace std;

void
_div::operator() (const geo_element& K, ublas::matrix<Float>& m) const
{
  size_type map_d = K.dimension();
  check_macro (coordinate_dimension() == K.dimension(),
    "unsupported `div' form on element type `" << K.name()
    << " in " << coordinate_dimension() << "D geometry");
  reference_element hat_K = K;
  size_type nx = get_first_basis().size(hat_K);
  size_type ny = get_second_basis().size(hat_K);
  m.resize (ny, map_d*nx);
  m.clear();
  quadrature::const_iterator first = _quad.begin(hat_K);
  quadrature::const_iterator last  = _quad.end  (hat_K);
  tensor DF;
  tensor invDF;
  tensor invDFt_wq;
  for (size_type q = 0; first != last; first++, q++) {
    jacobian_piola_transformation (K,q,DF);
    invDF = inv(DF,map_d);
    Float wq = is_weighted() ? weight(K,q) : Float(1);
    wq *= weight_coordinate_system (K, q);
    wq *= det_jacobian_piola_transformation (DF, K.dimension());
    wq *= (*first).w;
    invDFt_wq = trans(invDF) * wq;
    
    basis_on_quadrature::const_iterator phi      = _by_table.begin(hat_K, q);
    basis_on_quadrature::const_iterator last_phi = _by_table.end(hat_K, q);
    basis_on_quadrature::const_iterator_grad first_grad_psi = _bx_table.begin_grad (hat_K, q);
    basis_on_quadrature::const_iterator_grad last_grad_psi  = _bx_table.end_grad (hat_K, q);
    for (size_type i = 0; phi != last_phi; phi++, i++) {
      basis_on_quadrature::const_iterator_grad grad_psi = first_grad_psi;
      Float phi_i = *phi;
      for (size_type j = 0; grad_psi != last_grad_psi; grad_psi++, j++) {
	const point& grad_psi_j = *grad_psi;
        for (size_type k = 0; k < map_d; k++) {
	  Float sum = 0;
          for (size_type l = 0; l < map_d; l++) {
            sum += phi_i * invDFt_wq(k,l) * grad_psi_j[l];
	  }
	  m(i,j+k*nx) += sum;
	}
      }
    }
  }
  if (coordinate_system_type() == fem_helper::cartesian) return;
  // --------------------------------------
  // otherwise, add the (ur/r) r dr dz term
  //   as : ur dr dz
  // i.e. skipping the `r' extra weight
  // --------------------------------------
  ublas::matrix<Float> mr;
  set_use_coordinate_system_weight (false);
  build_scalar_mass (K, mr);
  set_use_coordinate_system_weight (true);
  size_type xoffset = 0;
  if (coordinate_system_type() == fem_helper::axisymmetric_zr)
    xoffset += nx;
  for (size_type i = 0; i < ny; i++)
    for (size_type j = 0; j < nx; j++)
      m(i,j+xoffset) += mr(i,j);
}
_div::size_type
_div::n_derivative() const
{
  return 1;
}
void
_div::check_after_initialize () const
{
  size_type d = coordinate_dimension();
  check_macro (
	d == get_first_space().n_component(),
	"unsupported non-vectorial first space for `div' form");
  check_macro (
	1 == get_second_space().n_component(),
	"unsupported non-scalar second space for `div' form");
}
