#ifndef _RHEOLEF_DOMAIN_H
#define _RHEOLEF_DOMAIN_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

#include "rheolef/geo.h"
#include "rheolef/domain_indirect.h"

namespace rheolef {

/*Class:domain
NAME: @code{geo_domain_indirect_rep} - a named part of a finite element mesh
@cindex  mesh boundary
@clindex geo_domain_indirect_rep
DESCRIPTION:
  @noindent
  The @code{geo_domain_indirect_rep} class defines a container for a part of a
  finite element mesh.
  This describes the connectivity of edges or faces.
  This class is usefull for boundary condition setting.
IMPLEMENTATION NOTE:
  The @code{geo_domain_indirect_rep} class is splitted into two parts.
  The first one is the @code{domain_indirect} class, that contains the main
  renumbering features: it acts as an indirection on a @code{geo} class(@pxref{geo class}).
  The second one is the @code{geo} class itself, named here the background geo.
  Thus, the @code{geo_domain_indirect} class develops a complete @code{geo}-like interface, 
  via the @code{geo_abstract_rep} pure virtual class derivation,
  and can be used by the @code{space} class (@pxref{space class}).

  The split between @code{domain_indirect} and @code{geo_domain_indirect} is necessary,
  because the @code{geo} class contains a list of domain_indirect.
  The @code{geo} class cannot contains a list of @code{geo_domain_indirect} classes, that refers
  to the @code{geo} class itself: a loop in reference counting
  leads to a blocking situation in the automatic deallocation.

DATE: 20 february 2011
End:
*/
typedef geo domain;

// ========================================================================
// shared by seq & mpi rep
// ========================================================================
template <class T, class M>
class geo_domain_indirect_base_rep : public geo_abstract_rep<T,M> {
public:
// typedefs:
    typedef geo_abstract_rep<T,M>          base;
    typedef typename base::size_type       size_type;
    typedef typename base::const_reference const_reference;
    typedef typename base::const_iterator  const_iterator;

// allocators:

    geo_domain_indirect_base_rep ();
    geo_domain_indirect_base_rep (const domain_indirect_basic<M>& indirect, const geo_basic<T,M>& omega);

// accessors: the geo_abstract seq interface is defined here

    size_type variant() const { return geo_abstract_base_rep<T>::geo_domain_indirect; }
    std::string name() const { return _indirect.name(); }
    size_type dimension() const { return _omega.dimension(); }
    size_type map_dimension() const { return _indirect.map_dimension(); }

    const size_type* size_by_dimension()     const { return _omega.size_by_dimension(); }
    const size_type* size_by_variant()       const { return _omega.size_by_variant(); }
    const size_type* dis_size_by_dimension() const { return _omega.dis_size_by_dimension(); }
    const size_type* dis_size_by_variant()   const { return _omega.dis_size_by_variant(); }
    const size_type* ios_size_by_dimension() const { return _omega.ios_size_by_dimension(); }
    const size_type* ios_size_by_variant()   const { return _omega.ios_size_by_variant(); }

    const distributor& geo_element_ownership (size_type dim) const;
    const_reference get_geo_element (size_type dim, size_type ige) const;
    const_iterator begin (size_type dim) const;
    const_iterator end   (size_type dim) const;

    const basic_point<T>& vertex (size_type iv) const { return _omega.vertex (iv); }

    size_type                     n_domain_indirect() const;
    const domain_indirect_basic<M>& get_domain_indirect (size_type i) const;
    const domain_indirect_basic<M>& get_domain_indirect (const std::string& name) const;

    odiststream& put (odiststream& ops) const;

// accessors:

    const domain_indirect_basic<M>& get_indirect()        const { return _indirect; }
    const geo_basic<T,M>&           get_background_geo()  const { return _omega; }

protected:
// data:
    domain_indirect_basic<M>   _indirect;
    geo_basic<T,M>             _omega;
};
template <class T, class M>
inline
geo_domain_indirect_base_rep<T,M>::geo_domain_indirect_base_rep()
  : geo_abstract_rep<T,M>(),   
    _indirect(),
    _omega()
{
}
template <class T, class M>
inline
geo_domain_indirect_base_rep<T,M>::geo_domain_indirect_base_rep (const domain_indirect_basic<M>& indirect, const geo_basic<T,M>& omega)
  : geo_abstract_rep<T,M>(),   
    _indirect (indirect),
    _omega (omega)
{
}
// ========================================================================
// sequential rep
// ========================================================================
template <class T, class M>
class geo_domain_indirect_rep {};

template <class T>
class geo_domain_indirect_rep<T,sequential> : public geo_domain_indirect_base_rep<T,sequential> {
public:
    typedef geo_domain_indirect_base_rep<T,sequential>		base;

    geo_domain_indirect_rep () : base() {}
    geo_domain_indirect_rep (const domain_indirect_basic<sequential>& indirect, const geo_basic<T,sequential>& omega)
	: base(indirect,omega) {}
};
// ========================================================================
// distributed rep
// ========================================================================
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
class geo_domain_indirect_rep<T,distributed> : public geo_domain_indirect_base_rep<T,distributed> {
public:
    typedef geo_domain_indirect_base_rep<T,distributed>	base;
    typedef typename base::size_type                    size_type;
    typedef typename base::vertex_type                  vertex_type;

    geo_domain_indirect_rep() : base() {}
    geo_domain_indirect_rep (const domain_indirect_basic<distributed>& indirect, const geo_basic<T,distributed>& omega)
	: base(indirect,omega) {}

// accessors: the geo_abstract mpi interface is defined here

    const vertex_type& dis_vertex (size_type dis_iv) const { return base::_omega.dis_vertex (dis_iv); }
    distributor geo_element_ios_ownership (size_type dim) const;
    size_type             ige2ios_dis_ige (size_type dim, size_type     ige) const;
    size_type         dis_ige2ios_dis_ige (size_type dim, size_type dis_ige) const;
    size_type             ios_ige2dis_ige (size_type dim, size_type ios_ige) const;
};
#endif // _RHEOLEF_HAVE_MPI
// ========================================================================
// geo_basic allocator and accessor related to domains
// ========================================================================
#define _RHEOLEF_geo_cstor(M)					\
template<class T>						\
inline								\
geo_basic<T,M>::geo_basic (					\
    const domain_indirect_basic<M>& dom, 				\
    const geo_basic<T,M>& omega)				\
 : base (new_macro((geo_domain_indirect_rep<T,M>)(dom,omega)))	\
{								\
}
#define _RHEOLEF_geo_dmn_by_idx(M)				\
template<class T>						\
inline								\
geo_basic<T,M>							\
geo_basic<T,M>::get_domain (size_type i) const			\
{								\
    const domain_indirect_basic<M>& dm = base::data().get_domain_indirect (i); \
    return geo_basic<T,M> (dm, *this);				\
}
#define _RHEOLEF_geo_dmn_by_name(M)				\
template<class T>						\
inline								\
geo_basic<T,M>							\
geo_basic<T,M>::operator[] (const std::string& name) const	\
{								\
    const domain_indirect_basic<M>& dm = base::data().get_domain_indirect (name); \
    return geo_basic<T,M> (dm, *this);				\
}
_RHEOLEF_geo_cstor(sequential)
_RHEOLEF_geo_dmn_by_idx(sequential)
_RHEOLEF_geo_dmn_by_name(sequential)

#ifdef _RHEOLEF_HAVE_MPI
_RHEOLEF_geo_cstor(distributed)
_RHEOLEF_geo_dmn_by_idx(distributed)
_RHEOLEF_geo_dmn_by_name(distributed)
#endif // _RHEOLEF_HAVE_MPI

#undef _RHEOLEF_geo_cstor
#undef _RHEOLEF_geo_dmn_by_idx
#undef _RHEOLEF_geo_dmn_by_name

} // namespace rheolef
#endif // _RHEOLEF_DOMAIN_H
