// Copyright 2015 The rkt Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package main

import (
	"strings"
)

const (
	// fileDepMkTemplate holds a template for file-based
	// dependency tracking make file.
	fileDepMkTemplate = `# generated by !!!DEPS_GEN_APP_NAME!!!

define _DEPS_GEN_GET_ALL_FILES_
!!!DEPS_GEN_FILES_GENERATOR!!!
endef

_DEPS_GEN_ALL_FILES_STORED_ := !!!DEPS_GEN_ALL_FILES_STORED!!!

_DEPS_GEN_SPACE_ :=
_DEPS_GEN_SPACE_ +=

_DEPS_GEN_F1_ := $(sort $(strip $(_DEPS_GEN_ALL_FILES_STORED_)))
_DEPS_GEN_F2_ := $(sort $(strip $(call _DEPS_GEN_GET_ALL_FILES_)))

ifeq ($(_DEPS_GEN_F1_),$(_DEPS_GEN_F2_))

!!!DEPS_GEN_TARGET!!!: $(_DEPS_GEN_ALL_FILES_STORED_)

else

$(info Invalidating !!!DEPS_GEN_TARGET!!!)
$(info Prerequisites stored in deps file that are not in currently on disk: $(filter-out $(_DEPS_GEN_F2_),$(_DEPS_GEN_F1_)))
$(info Prerequisites currently on disk that are not stored in deps file: $(filter-out $(_DEPS_GEN_F1_),$(_DEPS_GEN_F2_)))

# invalidate the target
!!!DEPS_GEN_TARGET!!!: _DEPS_GEN_INVALIDATE_
.PHONY: _DEPS_GEN_INVALIDATE_

endif

$(call undefine-namespaces,_DEPS_GEN)
`

	// kvStoredPrefix is used to prefix the variable name we want
	// to store (so actual variable checked at runtime is FOO and
	// its stored value from previous run is $(kvStoredPrefix)FOO.
	kvStoredPrefix = "_DEPS_GEN_STORED_"
	// kvDepMkTemplate holds a template for key-value-based
	// dependency tracking make file.
	kvDepMkTemplate = `# generated by !!!DEPS_GEN_APP_NAME!!!
_DEPS_GEN_VARIABLES_ := !!!DEPS_GEN_VARIABLES!!!
!!!DEPS_GEN_STORED_VALUES!!!
_DEPS_GEN_DO_INVALIDATE_ := no

$(foreach v,$(_DEPS_GEN_VARIABLES_), \
        $(if $(call equal,$($v),$(!!!DEPS_GEN_STORED_PREFIX!!!$v)), \
                , \
                $(eval _DEPS_GEN_DO_INVALIDATE_ := yes) \
                $(eval _DEPS_GEN_DIFFS_ += $v)))

ifeq ($(_DEPS_GEN_DO_INVALIDATE_),yes)

$(info Invalidating !!!DEPS_GEN_TARGET!!!)
$(foreach k,$(_DEPS_GEN_DIFFS_), \
        $(info Differing key: $k) \
        $(info Value stored in deps file: $(!!!DEPS_GEN_STORED_PREFIX!!!$k)) \
        $(info Value currently used: $($k)))

# invalidate the target
!!!DEPS_GEN_TARGET!!!: _DEPS_GEN_INVALIDATE_
.PHONY: _DEPS_GEN_INVALIDATE_

endif

$(call undefine-namespaces,_DEPS_GEN)
`
)

// GenerateFileDeps returns contents of make file describing
// dependencies of given target on given files and checking if files
// weren't added or removed in directories where given files are.
func GenerateFileDeps(target, filesGenerator string, files []string) string {
	return replacePlaceholders(fileDepMkTemplate,
		"DEPS_GEN_APP_NAME", appName(),
		"DEPS_GEN_FILES_GENERATOR", filesGenerator,
		"DEPS_GEN_ALL_FILES_STORED", strings.Join(files, " "),
		"DEPS_GEN_TARGET", target,
	)
}

func GenerateKvDeps(target string, keysValues map[string]string) string {
	variables := make([]string, 0, len(keysValues))
	storedValues := make([]string, 0, len(keysValues))
	for k, v := range keysValues {
		variables = append(variables, k)
		storedValues = append(storedValues, kvStoredPrefix+k+" := "+v)
	}

	return replacePlaceholders(kvDepMkTemplate,
		"DEPS_GEN_APP_NAME", appName(),
		"DEPS_GEN_VARIABLES", strings.Join(variables, " "),
		"DEPS_GEN_STORED_VALUES", strings.Join(storedValues, "\n"),
		"DEPS_GEN_STORED_PREFIX", kvStoredPrefix,
		"DEPS_GEN_TARGET", target,
	)
}
