/* main.c: main(), initialisation and cleanup*/

/*  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License , or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; see the file COPYING.  If not, write to
    the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

    You may contact the author by e-mail:  hlub@knoware.nl
*/

#include "rlwrap.h"

/* global vars */
int master_pty_fd;                           /* master pty (rlwrap uses this to communicate with client) */
int slave_pty_fd;                            /* slave pty (client uses this to communicate with rlwrap,
					     	we keep it open after forking in order to keep track of
						client's terminal settings */
FILE *debug_fp;                              /* filehandle of debugging log (!= stderr) */
char *program_name, *command_name;           /* "rlwrap" and (base-)name of command */
int within_line_edit = FALSE;                /* TRUE while user is editing input */
int always_readline = FALSE;                 /* -a option */
char *password_prompt_search_string = NULL;  /* argument of -a option (cf manpage) */
int complete_filenames = FALSE;              /* -c option */
int child_pid = 0;                           /* pid of child (client) */
int i_am_child = FALSE;                      /* after forking, child will set this to TRUE */
int nowarn = FALSE;                          /* argument of -n option (suppress warnings) */
int debug = 0;                               /* debugging mask (0 = no debugging) */


/* private vars */

static char *history_filename;
static int histsize=300;
static char *completion_filename, *default_completion_filename;

#ifdef GETOPT_GROKS_OPTIONAL_ARGS
  static char optstring[] =  "+a::b:cC:d::f:hl:nrs:v";
#else
  static char optstring[] =  "+a:b:cC:d:f:hl:nrs:v";
#endif

#ifdef HAVE_GETOPT_LONG
static struct option longopts[] = {
  {"always-readline", optional_argument, NULL, 'a'},
  {"break-chars", required_argument, NULL, 'b'},
  {"complete-filenames", no_argument, NULL, 'c'},
  {"command-name", required_argument, NULL, 'C'},
  {"debug", optional_argument, NULL, 'd' },
  {"file", required_argument, NULL, 'f'}, 
  {"help", no_argument, NULL, 'h'},
  {"logfile", required_argument, NULL, 'l'},
  {"no-warnings", no_argument,  NULL, 'n'},
  {"remember", no_argument, NULL, 'r'},
  {"version", no_argument, NULL, 'v'},
  {"histsize", required_argument, NULL, 's'},
  {0,0,0,0}
};
#endif  




int
main (int argc, char **argv)
{
  int pid, c;
  extern char *optarg;
  extern int optind;
  char *opt_C = NULL;
  int  opt_b = FALSE;
  char *completion_filename_optarg=NULL;

  
  program_name = mysavestring(mybasename(argv[0])); /* normally "rlwrap"; needed by myerror() */   
  rl_basic_word_break_characters = " \t\n\r(){}[].,+-=&^%$#@\";|\\"; 

  
  while (1) {
#ifdef HAVE_GETOPT_LONG
    c = getopt_long(argc, argv, optstring, longopts, NULL );
#else
    c = getopt(argc, argv, optstring);
#endif

    if (c == EOF)
      break;
    
    switch(c) {
    case 'a':
      always_readline = TRUE;
      if (optarg)
	password_prompt_search_string = mysavestring(optarg);
      break;
    case 'b':
      rl_basic_word_break_characters = add3strings("\r\n \t", optarg, "");
      opt_b = TRUE;
      break;
    case 'c':
      complete_filenames =  TRUE;
      break;
    case 'C':
      opt_C = mysavestring(optarg);
      break;
    case 'd':
#ifdef DEBUG
      if (optarg)
	debug = atoi(optarg);
      else
	debug = DEBUG_ALL;
#else
      mywarn("To use debugging, configure %s with --enable-debug and rebuild", program_name);
      debug = 1;
#endif      
      break;
    case 'f':
      use_completion_list = TRUE;
      completion_filename_optarg = optarg;
      break;
    case 'h':
      usage(); /* will call exit() */
    case 'l':
      open_logfile(optarg);
      break;
    case 'n':
      nowarn = TRUE;
      break;
    case 'r':
      use_completion_list = TRUE;
      remember_for_completion = TRUE;
      break;
    case 's':
      histsize = atoi(optarg);
      break;
    case 'v':
      printf ("rlwrap %s\n", VERSION);
      exit (1);
    default: 
      usage();
    }
  }
  /* do 'f' option work only after "case 'b'" logic above is  guaranteed to be finished
     (so -b switch always handled prior to -f)  --Luke Call <lac@onemodel.org> Nov 2003 */
  if (use_completion_list && completion_filename_optarg!=NULL)
    init_completer(completion_filename_optarg);

  if (!complete_filenames && !opt_b) {
    rl_basic_word_break_characters =  add3strings(rl_basic_word_break_characters,"/.","");
  }
  
  if (optind >= argc) 
    usage();

  if (opt_C) {
    int countback = atoi(opt_C);						/* try whether -C option is numeric */

    if (countback > 0) {     							/* e.g -C 1 or -C 12 */
      if (argc - countback <= optind)  						/* -C 66 */                  
	myerror("when using -C %d you need at least %d non-option arguments",
		countback, countback + 1);
      else if (argv[argc-countback][0] == '-') 					/* -C 2 perl -d blah.pl */
	myerror("the last argument minus %d appears to be an option!", countback);
      else									/* -C 1 perl test.cgi */ 
	command_name = mysavestring(mybasename(argv[argc - countback]));  
    } else if (countback == 0) { 						/* -C name1 name2 or -C 0 */
      if (opt_C[0] == '0' && opt_C[1] == '\0' ) 				/* -C 0 */
	myerror("-C 0 makes no sense");
      else if (strlen(mybasename(opt_C)) != strlen(opt_C))			/* -C dir/name */
	myerror("-C option argument should not contain directory components");
      else if (opt_C[0] == '-')							/* -C -d  (?) */
	myerror("-C option needs argument");
      else									/* -C name */
	command_name = opt_C;							
    } else  {									/* -C -2 */
      myerror("-C option needs string or positive number as argument, perhaps you meant -C %d?", -countback);
    }
  } else {                                                                      /* no -C option given, use command name */
    command_name = mysavestring(mybasename(argv[optind]));                     
  }
  
  assert(command_name != NULL);
  init_rlwrap();
  
  pid = my_pty_fork (&master_pty_fd, &saved_terminal_settings, &window_size);
  if (pid > 0) { /* parent: */
    child_pid = pid;
    main_loop (master_pty_fd);
  } else { /* child: */
    DPRINTF1(DEBUG_TERMIO, "preparing to execute %s", argv[optind]);
    if (execvp(argv[optind], &argv[optind]) < 0) {
          myerror("Cannot execute %s", argv[optind]); /* myerror called from child */
	  exit(EXIT_FAILURE);	
    }
  }
  return 0;
}

void
main_loop (int pty_fd)                     /* this function never returns */
{ 
  int nfds; 		                   /* number of ready  file descriptors  */
  fd_set readfds;	                   /* bitmap of readable  file descriptors */
  int nread;                               /* # of chars read */
  struct timeval immediately = {0,0};      /* zero timeout when child is dead */
  struct timeval *forever = NULL;
  char buf[BUFFSIZE], *last_nl;
  char *prompt = saved_rl_state.prompt;
  int messysigs[] = {SIGWINCH, SIGTSTP,0}; /* we don't want to catch these
  					      while readline is processing */
  int termwidth;                           /* width of terminal window */
  int promptlen = 0, removed;              /* used when prompt is wider than terminal */
  
  /* setbuf(stdout, NULL);  */
  init_readline("");
  set_echo(FALSE); /* This will also put the terminal in CBREAK mode */
  while (TRUE) {
    /* listen on both stdin and pty_fd */
    FD_ZERO(&readfds);
    FD_SET(STDIN_FILENO, &readfds);
    FD_SET(pty_fd,&readfds);

    unblock_signals(messysigs);     /* Handle SIGWINCH and SIGTSTP now (and only now) */
    nfds = select( 1+pty_fd,
		   & readfds,
		   (fd_set*) NULL,
		   (fd_set *) NULL,
		   (child_is_dead ? &immediately : forever));
    block_signals(messysigs);       /* Dont't disturb now */

    DPRINTF3(DEBUG_TERMIO, "select(..) = %d (%s), within_line_edit=%d", nfds, (FD_ISSET(pty_fd, &readfds) ? "pty" : "stdin"), within_line_edit);
    
    if (nfds < 0) {       /* exception  */
      if (errno == EINTR) /* interrupted by signal, don't worry */
	continue;
      else
	myerror("select received exception");
    } else if (nfds == 0) { /* child has died ..*/
      if (promptlen > 0)    /* ... its last words were not terminated by \n ... */
	my_putchar('\n');   /* print the \n ourself */
      DPRINTF2(DEBUG_SIGNALS,"select returned 0, child_is_dead=%d, childs_exit_status=%d" , child_is_dead,childs_exit_status);
      cleanup_rlwrap_and_exit(EXIT_SUCCESS);	  
    } else if (nfds > 0) { /* something to read */
      if (FD_ISSET(pty_fd, &readfds)) {
	/* read pty */

	
	if ((nread = read(pty_fd, buf, BUFFSIZE - 1 )) <= 0) {
	  if (nread == EINTR) /* interrupted by signal, don't worry */
	    continue;
	  else if(child_is_dead || nread < 0) {
	    if (promptlen > 0)    /* commands dying words not terminated by \n ... */
	      my_putchar('\n');   /* provide the missing \n */
	    cleanup_rlwrap_and_exit(EXIT_SUCCESS);	  	
	  } else
	    myerror("read error on master pty");
	}
	buf[nread] = '\0';

	if (within_line_edit)  /* client output arrives while we're editing keyboard input  */
	  save_rl_state (&saved_rl_state);        
       
	
	if (remember_for_completion)
	  feed_into_completion_list(buf);
	
        write(STDOUT_FILENO, buf, strlen(buf)); 
        DPRINTF2(DEBUG_READLINE, "wrote %d bytes: %s'", strlen(buf), mangle_string_for_debug_log(buf,40));
        
        write_logfile(buf);
	
	/* now determine the text *after* the last newline. This wil be the
	     "prompt" for the readline input routine: */
	last_nl = strrchr(buf, '\n'); 
	if (last_nl != NULL) {
	  /* newline seen, will get new prompt: */
	  strncpy(prompt, last_nl+1, BUFFSIZE-1);
	} else {
	  /* no newline, extend old prompt: */
	  strncat(prompt, buf, BUFFSIZE-1 - strlen(prompt));
	}

	/* If the prompt is wider than the current terminal width,
	   assume that it wrapped around and take only the last line of it to be the
	   new prompt. This may go wrong if the prompt contains BS or control
	   sequences (which is unusual for programs needing rlwrap)  @@@FIXME? */
	
	termwidth = window_size.ws_col;
	promptlen = strlen(prompt);
        if (termwidth &&      /* this might sometimes be 0 on some weird systems */
	    promptlen > termwidth) {
	  removed = (promptlen/termwidth) * termwidth ;
	   memmove(prompt, prompt + removed, strlen(prompt) + 1 - removed);
	}
	
	DPRINTF1(DEBUG_READLINE, "Prompt now: '%s'", prompt);
	prompt[BUFFSIZE-1] = '\0'; /* just to make sure.... */
        if (within_line_edit) 
	   restore_rl_state (&saved_rl_state);
	  
      }
      
      if (FD_ISSET(STDIN_FILENO, &readfds)) { /* key pressed */
	 nread = read (STDIN_FILENO, buf, 1);  /* read next byte of input   */
	  if (nread < 0)
	    myerror("Unexpected error");
	  else if (nread == 0) /* EOF on stdin */
	    cleanup_rlwrap_and_exit(EXIT_SUCCESS);
	  
	if (slave_is_in_raw_mode() && ! always_readline) { /* just pass it on */
 	  buf[nread] = '\0'; /* only necessary for DPRINTF, 
                                the '\0' won't be written to master_pty */
          DPRINTF1(DEBUG_READLINE, "Read '%s' in transparent mode", buf);
	  write(master_pty_fd, buf, nread); 
	} else {	                         /* hand it over to readline */
	  if (! within_line_edit) {              /* start a new line edit    */
	    DPRINTF0(DEBUG_READLINE, "Starting line edit");
	    within_line_edit = TRUE; 	   
	    restore_rl_state (&saved_rl_state); 
	  }
  
	  rl_pending_input = buf[0];            /* stuff it back in readlines input queue */
	  
	  DPRINTF2(DEBUG_READLINE, "Character %d (%c)", rl_pending_input, (isalpha(rl_pending_input) ?  rl_pending_input : '.'));

	  if (buf[0] == term_eof && strlen(rl_line_buffer) == 0 )    /* kludge: readline on Solaris does not interpret CTRL-D as EOF */ 
	    line_handler(NULL);
	  else	
	    rl_callback_read_char ();
	  
	}
      }
    }   /* if (ndfs > 0)         */
      
  }              /* while (1)             */
}                /* void main_loop()      */
 



void
init_rlwrap()
{

  char *homedir_prefix;

#ifdef DEBUG
  if (debug) {
    debug_fp = fopen(DEBUG_FILENAME, "w");
    if (!debug_fp)
      myerror("Couldn't open debug file %s", DEBUG_FILENAME);
    setbuf(debug_fp,  NULL);
  }
#endif
  
  DPRINTF0(DEBUG_TERMIO, "Initializing");
  init_terminal();
  


   
  
  /* Where should history and completion files go? */	
  homedir_prefix = (getenv("RLWRAP_HOME") ? 
		    add2strings (getenv("RLWRAP_HOME"), "/") :  /* use $RLWRAP_HOME/<command>_history */
		    add2strings (getenv("HOME"), "/."));        /* use ~/.<command>_history */
  
  history_filename            = add3strings(homedir_prefix, command_name, "_history");
  completion_filename         = add3strings(homedir_prefix, command_name, "_completions");
  default_completion_filename = add3strings(DATADIR,"/rlwrap/", command_name);
  free(homedir_prefix);
  
  
  
  /* check history file existence and permissions */

  if (access(history_filename, F_OK) == 0) { /* already exists, can we read/write it? */
    if (access(history_filename, R_OK|W_OK) != 0) {	
      myerror("cannot read and write %s", history_filename);
    }
  } else { /* doesn't exist, can we create it? */
    char *histdir = (getenv("RLWRAP_HOME") ? getenv("RLWRAP_HOME") : getenv("HOME"));
    if (access( histdir, W_OK) != 0) {
      myerror("cannot create history file in %s", histdir);
    }
  }

  
  /* Initialize history */
  using_history ();
  stifle_history (histsize);
  read_history (history_filename); /* ignore errors here: history file may not yet exist */ 

  /* Get last input line from history and put it in previous_line
      The last added history_entry is always at history_length + history_base - 1 */
  if (history_length) {
    strncpy (previous_line, history_get(history_length + history_base - 1) -> line, BUFFSIZE-1);
  }

 
  rl_readline_name = command_name;
 
  /* Initialize completion list (if <completion_filename> is readable) */  
  if (access(completion_filename, R_OK) == 0) {
    use_completion_list = TRUE;
    init_completer(completion_filename); 
           /* misnomer,  this actually *adds* to the completion list
	      (and doesn't reset it) */
  } else if (access(default_completion_filename, R_OK) == 0) {
    use_completion_list = TRUE;
    init_completer(default_completion_filename);
  }
}




void
cleanup_rlwrap_and_exit(int status)
{
  DPRINTF0(DEBUG_TERMIO, "Cleaning up");
  if (history_total_bytes() > 0)     /* avoid creating files like .nsloolup_history after typo*/ 
    write_history (history_filename); /* ignore errors */
  close_logfile();	
  if (terminal_settings_saved &&  (tcsetattr(STDIN_FILENO, TCSAFLUSH, &saved_terminal_settings) < 0)) /* reset terminal */
    ; /*     mywarn ("tcsetattr error on stdin"); */  /* NOT myerror, as this would call cleanup_rlwrap_and_exit, causing stack overflow */ 
  exit(status != EXIT_SUCCESS ? status : WEXITSTATUS(childs_exit_status)); /* signal failure or else propagate child's exit status */
}

