/* main.c: main() , signal handling and error functions */

/*  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License , or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; see the file COPYING.  If not, write to
    the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

    You may contact the author by:
       e-mail:  hlub@knoware.nl
*/

#include "rlwrap.h"

FILE *log_fh;

static void
utils_warn(const char * message, va_list ap)
{

  int saved_errno = errno;
  char buffer[BUFFSIZE];
  

#ifdef  HAVE_SNPRINTF
  snprintf(buffer, sizeof(buffer) - 1, "%s: ", program_name);
  vsnprintf(buffer + strlen(buffer),sizeof(buffer)-strlen(buffer)-1,
	    message, ap);
  if (saved_errno)
    snprintf(buffer + strlen(buffer) ,sizeof(buffer)-strlen(buffer)-1,
	     ": %s", strerror(saved_errno));
#else
  sprintf(buffer, "%s: ", program_name);
  vsprintf(buffer + strlen(buffer),
	    message, ap);
  if (saved_errno)
    sprintf(buffer + strlen(buffer) ,
	     ": %s", strerror(saved_errno));
#endif
  strcat(buffer, "\n");
  
  fflush(stdout);
  fputs(buffer, stderr);
  fflush(stderr);
  DPRINTF1(DEBUG_ALL, "Error/warning: %s", buffer); /* we want this because sometimes error messages (esp. from client) are dropped) */
  
}

void
mywarn(const char * message, ...)
{
  va_list ap;
  char *warning_message;
  if (nowarn)
    return;
  warning_message = add2strings("warning: ", message);
  va_start(ap, message);
  utils_warn(warning_message, ap);
  va_end(ap); 
}

void
myerror(const char * message, ...)
{
  va_list ap;
  char *error_message = add2strings("error: ", message);
  va_start(ap, message);
  utils_warn(error_message, ap);
  va_end(ap);
  if(! i_am_child)
    cleanup_rlwrap_and_exit(EXIT_FAILURE);
  else
    exit(-1);
}


char *
mybasename (char *name)
{ /* determine basename of "name" */
  char *p;
  /* find last '/' in name (if any) */
  for (p=name + strlen(name) - 1; p > name; p--)
    if (*(p-1) == '/')
      break;     	
  return p;
}

void * 
mymalloc(size_t size) 
{ /* malloc with error handling (i.e. bail out when out of memory) */
  void *ptr;
  ptr = malloc(size);
  if (ptr == NULL) 
    myerror("Tried in vain to allocate %d bytes", size);
  return ptr;
}

char *
mysavestring(const char* string)
{  
  return mystrndup(string, strlen(string));
}

char *
mystrndup(const char* string, int len)
{
  /* allocate copy of string on the heap */ 
  char *buf;
  buf = (char *) mymalloc (len + 1);
  strncpy (buf, string, len);
  buf[len] = '\0';
  return buf;
}
  
char* 
add3strings(const char *str1, const char *str2, const char *str3) 
{ 
  int len = strlen(str1) + strlen(str2) + strlen(str3);
  char *buf = (char *) mymalloc (len+1);
  strcpy  (buf, str1);
  strcat (buf, str2);
  strcat (buf, str3);
  return buf;
}


void 
open_logfile(const char *filename)
{
  time_t now;
  log_fh = fopen(filename, "a");
  if (!log_fh) 
    myerror ("Cannot write to logfile %s", filename);
  now = time(NULL);
  fprintf(log_fh, "\n\n[rlwrap] %s\n", ctime(&now)); 
}

void
write_logfile(const char* str)
{
  if (log_fh) 
    fputs(str, log_fh);
}

void  
close_logfile()
{
  if (log_fh)
    fclose (log_fh);
}

static char mangle_buf[BUFFSIZE];

char *
mangle_string_for_debug_log(char *string, int len) {
  char *tmp = mangle_buf;
  char oct[4];
  int i, j;
  
  assert(len + 3 < BUFFSIZE - 1); 

 
  for(i = j = 0; j < len  && string[i] ; ) {
    if (isprint(string[i]) &&  string[i] != '\n')
      tmp[j++] = string[i++];
    else {
      tmp[j++]='\\';
      sprintf(oct, "%03o", string[i++]);
      strncpy(&tmp[j], oct,3);
      j+=3;
    }
  } 
  tmp[j] = '\0';
  
  return tmp;
}


/* ugly #ifdefs, should find some better way @@@ */
   
void
usage()
{
  fprintf(stderr,
"Usage: rlwrap [options] command ...\n"
"\n"
"Options:\n"	  
#ifdef HAVE_GETOPT_LONG
"   -a, --always-readline[=password:] %0.0s\n" /* %0.0s is ugly kludge to get rid of argument */
"   -b, --break-chars <chars>\n" 
"   -c, --complete-filenames\n"
"   -C, --command-name <name|N>\n"	  
#ifdef DEBUG
"   -d, --debug[=mask] (output sent to to %s)\n"
#endif
"   -f, --file <file>\n"
"   -h, --help\n"
"   -l, --logfile <file>\n"
"   -n, --no-warnings\n"
"   -r, --remember\n"
"   -v, --version\n"
"   -s, --histsize <size>\n" ,
#else
"   -a%s always readline\n"      /*
"   -a[:password] always etc. "   */  
"   -b <chars>    break chars\n" 
"   -c            complete filenames\n"
"   -C <name|N>   use <name> as command name\n"	  
#ifdef DEBUG	  
"   -d[mask]      debug (output sent to to %s)\n"
#endif
"   -f <file>     completion file \n"
"   -h            help\n"
"   -l <file>     logfile\n"
"   -n            don't warn\n"
"   -r            remember\n"
"   -v            print version\n"
"   -s <size>     history size\n" ,
#endif
	  
#ifdef GETOPT_GROKS_OPTIONAL_ARGS
	  "[password:]",
#else
	  "<password:>",
#endif
	   DEBUG_FILENAME); /* DEBUG_FILENAME defined in rlwrap.h */

#ifdef DEBUG
fprintf(stderr,
"             \n"
"The debugging mask is a bitmask obtaining by ORing:\n"
"    %3d    if you want to debug termio\n"
"    %3d    if you want to debug signals\n"
"    %3d    if you want to debug readline handling\n"
	"    omitting the mask means: debug everything \n"
	, DEBUG_TERMIO, DEBUG_SIGNALS, DEBUG_READLINE);
#endif
 
fprintf(stderr, 
"\n"
"bug reports, suggestions, updates:\n"
	"http://utopia.knoware.nl/~hlub/uck/rlwrap/\n");

exit(EXIT_SUCCESS);
}




























