//pcm_roar.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010-2011
 *      Copyright (C) Hans-Kristian 'maister' Arntzen - 2010-2011
 *
 *  This file is part of libroar a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  libroar is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 *  NOTE for everyone want's to change something and send patches:
 *  read README and HACKING! There a addition information on
 *  the license of this document you need to read before you send
 *  any patches.
 *
 *  NOTE for uses of non-GPL (LGPL,...) software using libesd, libartsc
 *  or libpulse*:
 *  The libs libroaresd, libroararts and libroarpulse link this lib
 *  and are therefore GPL. Because of this it may be illigal to use
 *  them with any software that uses libesd, libartsc or libpulse*.
 */

//#define DEBUG
#include "roar.h"

// Equvivalent to prepare(). Starts a stream. Might/will be called several times during a program!
/////////////////////////////////////////////////
// Status: Should be mostly complete. 
// Condition for invalid fh is not solved here.
////////////////////////////////////////////////
static int roar_pcm_start (snd_pcm_ioplug_t * io) {
 struct roar_alsa_pcm * self = io->private_data;
 int fh;
 int val;
 int err;

 ROAR_DBG("roar_pcm_start(*) = ?");

 // If start is called several times in a row, just ignore it.
 if (self->stream_opened)
  return 0;

 if ( (self->vss = roar_vs_new_from_con(&(self->roar.con), &err)) == NULL ) {
  roar_err_set(err);
  roar_err_update();
  return -errno;
 }

 if ( roar_vs_stream(self->vss, &(self->info),
                     io->stream == SND_PCM_STREAM_PLAYBACK ? ROAR_DIR_PLAY : ROAR_DIR_MONITOR,
                     &err) == -1 ) {
  roar_vs_close(self->vss, ROAR_VS_TRUE, NULL);
  self->vss = NULL;
  roar_err_set(err);
  roar_err_update();
  return -errno;
 }

 // ALSA really expects there to be self->bufsize writable bytes.
 if ( roar_vs_buffer(self->vss, self->bufsize + 1, &err) == -1 ) {
  roar_vs_close(self->vss, ROAR_VS_TRUE, NULL);
  self->vss = NULL;
  roar_err_set(err);
  roar_err_update();
  return -errno;
 }

 val = ROAR_VS_ASYNCLEVEL_AUTO;
 if ( roar_vs_ctl(self->vss, ROAR_VS_CMD_SET_ASYNC, &val, &err) == -1 ) {
  roar_vs_close(self->vss, ROAR_VS_TRUE, NULL);
  self->vss = NULL;
  roar_err_set(err);
  roar_err_update();
  return -errno;
 }

 if ( self->stream_role != ROAR_ROLE_UNKNOWN ) {
  if ( roar_vs_role(self->vss, self->stream_role, &err) == -1 ) {
   ROAR_WARN("roar_pcm_start(*): Can not set stream role: %s(%i): %s",
             roar_role2str(self->stream_role), self->stream_role, roar_error2str(roar_error));
  }
 }

 if ( roar_vio_ctl(roar_vs_vio_obj(self->vss, NULL), 
    io->stream == SND_PCM_STREAM_PLAYBACK ? ROAR_VIO_CTL_GET_SELECT_WRITE_FH :
    ROAR_VIO_CTL_GET_SELECT_READ_FH, &fh) != 1 ) {
  io->poll_fd = fh;
  io->poll_events = io->stream == SND_PCM_STREAM_PLAYBACK ? POLLOUT : POLLIN;
 }

 // Oh, no, what should we do here if roar_vio_ctl() fails to grab a valid fh to poll?
 // In any case, ALSA will error out the next time it tries to poll() if we don't give it a valid fh.

 snd_pcm_ioplug_reinit_status(io);

 // Stream is now active, yay.
 self->stream_opened = 1;

 ROAR_DBG("roar_pcm_start(*) = 0");
 return 0;
}

void roar_plugin_reset(struct roar_alsa_pcm *self) {
 if ( !self->stream_opened )
  return;

 roar_vs_close(self->vss, ROAR_VS_TRUE, NULL);
 self->vss = NULL;
 self->stream_opened = 0;
 self->last_ptr = 0;
}



// Simply stopping the stream. Will need to be restarted to play more.
// Will be called several times together with roar_pcm_start()
///////////////////////////////////////////////////
// Status: Still needs some error checking for the pthread calls, but
// should work.
//////////////////////////////////////////////////
static int roar_pcm_stop (snd_pcm_ioplug_t *io) {
 struct roar_alsa_pcm * self = io->private_data;

 // If this is called several times in a row, just ignore.

 ROAR_DBG("roar_pcm_stop(*) = 0");

 roar_plugin_reset(self);

 return 0;
}

///////////////////////////////
// Status: Should be complete.
///////////////////////////////
static int roar_hw_constraint(struct roar_alsa_pcm * self) {
 snd_pcm_ioplug_t *io = &(self->io);
 static const snd_pcm_access_t access_list[] = {
  SND_PCM_ACCESS_RW_INTERLEAVED
 };
 static const unsigned int formats[] = {
  SND_PCM_FORMAT_S8,
  SND_PCM_FORMAT_U8,
  SND_PCM_FORMAT_A_LAW,
  SND_PCM_FORMAT_MU_LAW,
  SND_PCM_FORMAT_S16_LE,
  SND_PCM_FORMAT_S16_BE,
  SND_PCM_FORMAT_U16_LE,
  SND_PCM_FORMAT_U16_BE,
  SND_PCM_FORMAT_S32_LE,
  SND_PCM_FORMAT_S32_BE,
  SND_PCM_FORMAT_U32_LE,
  SND_PCM_FORMAT_U32_BE,
  SND_PCM_FORMAT_S24_3LE,
  SND_PCM_FORMAT_S24_3BE,
  SND_PCM_FORMAT_U24_3LE,
  SND_PCM_FORMAT_U24_3BE,
 };
 int ret;

 ROAR_DBG("roar_hw_constraint(*) = ?");

 if ( (ret = snd_pcm_ioplug_set_param_list(io, SND_PCM_IOPLUG_HW_ACCESS,
     _as(access_list), access_list)) < 0 )
  return ret;

 if ( (ret = snd_pcm_ioplug_set_param_list(io, SND_PCM_IOPLUG_HW_FORMAT,
     _as(formats), formats)) < 0 )
  return ret;

 if ( (ret = snd_pcm_ioplug_set_param_minmax(io, SND_PCM_IOPLUG_HW_CHANNELS,
     1, ROAR_MAX_CHANNELS)) < 0 )
  return ret;

 if ( (ret = snd_pcm_ioplug_set_param_minmax(io, SND_PCM_IOPLUG_HW_RATE, 8000, 192000)) < 0 )
  return ret;

 // We shouldn't let ALSA use extremely low or high values, it will kill a kitty most likely. :v
 if ( (ret = snd_pcm_ioplug_set_param_minmax(io, SND_PCM_IOPLUG_HW_PERIOD_BYTES, 1 << 6, 1 << 18)) < 0 )
  return ret;

 if ( (ret = snd_pcm_ioplug_set_param_minmax(io, SND_PCM_IOPLUG_HW_PERIODS, 1, 1024)) < 0 )
  return ret;

 if ( (ret = snd_pcm_ioplug_set_param_minmax(io, SND_PCM_IOPLUG_HW_BUFFER_BYTES, 1 << 13, 1 << 24)) < 0 )
  return ret;

 ROAR_DBG("roar_hw_constraint(*) = 0");

 return 0;
}

// Referring to alsa-lib/src/pcm/pcm_ioplug.c : snd_pcm_ioplug_hw_ptr_update
///////////////////////////////////////////////////////////
// Status: Mostly complete, but uses a really nasty hack!
///////////////////////////////////////////////////////////
static snd_pcm_sframes_t roar_pcm_pointer(snd_pcm_ioplug_t *io) {
 struct roar_alsa_pcm * self = io->private_data;
 int ptr;
 int buffered;
 ssize_t avail;

 ROAR_DBG("roar_pcm_pointer(*) = ?");

 // Did ALSA just call snd_pcm_reset() or something like that without calling the plugin? 
 // We should restart our stream as well.
 if ( io->appl_ptr < self->last_ptr ) {
  roar_pcm_stop(io);
  roar_pcm_start(io);
 }

 // ALSA expects return value to be equal to the amount of data written on the hardware side.
 // It uses this to calculate writable amount. (Because hey, just calling a get_avail() is too hard.)
 // Return value is thus io->appl_ptr - buffered_amount;

 while ( roar_vs_iterate(self->vss, ROAR_VS_NOWAIT, NULL) == 2 );

 avail = roar_vs_get_avail_write(self->vss, NULL);
 ROAR_DBG("roar_pcm_pointer(*): avail=%lli", (long long int)avail);
 buffered = snd_pcm_bytes_to_frames(io->pcm, self->bufsize - avail);

 ptr = io->appl_ptr - buffered;
 self->last_ptr = io->appl_ptr;

 ROAR_DBG("roar_pcm_pointer(*) appl_ptr (frames): %i, buffered (frames): %i", (int)io->appl_ptr, (int)buffered);
 ROAR_DBG("roar_pcm_pointer(*) = %i", ptr);
 return ptr;
}

// TODO: FIXME: add support for reading data!
//////////////////////////////////////////////////
// Status: For writing, this should be complete.
//////////////////////////////////////////////////
static snd_pcm_sframes_t roar_pcm_transfer(snd_pcm_ioplug_t *io,
  const snd_pcm_channel_area_t *areas,
  snd_pcm_uframes_t offset,
  snd_pcm_uframes_t size) {
 struct roar_alsa_pcm * self = io->private_data;
 char * buf;
 size_t len = size * self->info.channels * self->info.bits / 8;
 ssize_t ret;

 ROAR_DBG("roar_pcm_transfer(*) = ?");
 ROAR_DBG("roar_pcm_transfer(*): len=%lu", (long unsigned int) len);

 // Weird ALSA stuff.
 buf = (char *)areas->addr + (areas->first + areas->step * offset) / 8;

 while (len) {
  ret = roar_vs_write(self->vss, buf, len, NULL);
/*
 * <ph3-der-loewe> is it supposed to be EIO?
 * <ph3-der-loewe> or can we return the true error code from roar_vs_write()?
 * <maister> ALSA tends to give -EIO errors
 * <maister> It shouldn't cause problems to pass roar_vs errnos.
 * <maister> But the error strings might be a bit confusing perhaps
 */
  if ( ret == -1 )
   return -EIO;
  len -= ret;
  buf += ret;
 }

 while ( roar_vs_iterate(self->vss, ROAR_VS_NOWAIT, NULL) == 2 );

 ROAR_DBG("roar_pcm_transfer(*) = %lli", (long long int)size);
 return size;
}

///////////////////////////////////////////////////////////////////
// Status: Still missing proper delay measurements from the roar server. 
// Only uses a blind timer now. In ideal conditions, this will work well.
///////////////////////////////////////////////////////////////////
static int roar_pcm_delay(snd_pcm_ioplug_t *io, snd_pcm_sframes_t *delayp) {
 struct roar_alsa_pcm * self = io->private_data;
 roar_mus_t lat;

 ROAR_DBG("roar_pcm_delay(*) = ?");

 lat = roar_vs_latency2(self->vss, ROAR_VS_BACKEND_DEFAULT, ROAR_VS_NOWAIT, NULL);
 *delayp = (lat * self->info.rate) / 1000000;

 return 0;
}

////////////////////
// Status: Complete
////////////////////
static int roar_pcm_prepare(snd_pcm_ioplug_t *io) {
 ROAR_DBG("roar_pcm_prepare(*) = ?");

 return roar_pcm_start(io);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Status: This should be mostly complete.
// I'm not sure if hw_params can be called several times during one stream without stop() start() in between. 
// This will mean a memory leak, and possibly breakage should the buffer size change itself.
///////////////////////////////////////////////////////////////////////////////////////////////////////////////
static int roar_pcm_hw_params(snd_pcm_ioplug_t *io, snd_pcm_hw_params_t *params) {
 struct roar_alsa_pcm * self = io->private_data;
 snd_pcm_uframes_t buffersize;
 int err;

 ROAR_DBG("roar_pcm_hw_params(*) = ?");

 self->info.channels = io->channels;
 self->info.rate     = io->rate;

 switch (io->format) {
  case SND_PCM_FORMAT_S8:
   self->info.codec = ROAR_CODEC_PCM_U_LE;
   self->info.bits  = 8;
   break;
  case SND_PCM_FORMAT_U8:
   self->info.codec = ROAR_CODEC_PCM_U_LE;
   self->info.bits  = 8;
   break;
  case SND_PCM_FORMAT_A_LAW:
   self->info.codec = ROAR_CODEC_ALAW;
   self->info.bits  = 8;
   break;
  case SND_PCM_FORMAT_MU_LAW:
   self->info.codec = ROAR_CODEC_MULAW;
   self->info.bits  = 8;
   break;
  case SND_PCM_FORMAT_S16_LE:
   self->info.codec = ROAR_CODEC_PCM_S_LE;
   self->info.bits  = 16;
   break;
  case SND_PCM_FORMAT_S16_BE:
   self->info.codec = ROAR_CODEC_PCM_S_BE;
   self->info.bits  = 16;
   break;
  case SND_PCM_FORMAT_U16_LE:
   self->info.codec = ROAR_CODEC_PCM_U_LE;
   self->info.bits  = 16;
   break;
  case SND_PCM_FORMAT_U16_BE:
   self->info.codec = ROAR_CODEC_PCM_U_BE;
   self->info.bits  = 16;
   break;
  case SND_PCM_FORMAT_S32_LE:
   self->info.codec = ROAR_CODEC_PCM_S_LE;
   self->info.bits  = 32;
   break;
  case SND_PCM_FORMAT_S32_BE:
   self->info.codec = ROAR_CODEC_PCM_S_BE;
   self->info.bits  = 32;
   break;
  case SND_PCM_FORMAT_U32_LE:
   self->info.codec = ROAR_CODEC_PCM_U_LE;
   self->info.bits  = 32;
   break;
  case SND_PCM_FORMAT_U32_BE:
   self->info.codec = ROAR_CODEC_PCM_U_BE;
   self->info.bits  = 32;
   break;
  case SND_PCM_FORMAT_S24_3LE:
   self->info.codec = ROAR_CODEC_PCM_S_LE;
   self->info.bits  = 24;
   break;
  case SND_PCM_FORMAT_S24_3BE:
   self->info.codec = ROAR_CODEC_PCM_S_BE;
   self->info.bits  = 24;
   break;
  case SND_PCM_FORMAT_U24_3LE:
   self->info.codec = ROAR_CODEC_PCM_U_LE;
   self->info.bits  = 24;
   break;
  case SND_PCM_FORMAT_U24_3BE:
   self->info.codec = ROAR_CODEC_PCM_U_BE;
   self->info.bits  = 24;
   break;
  default:
   return -EINVAL;
 }

 if ((err = snd_pcm_hw_params_get_buffer_size(params, &buffersize) < 0))
  return err;

 ROAR_DBG("roar_pcm_hw_params(*) buffersize (frames): %i", (int)buffersize);

 self->bufsize = self->info.bits * self->info.channels * buffersize / 8;

 ROAR_DBG("roar_pcm_hw_params(*) Setting buffersize (bytes): %i", (int)self->bufsize);
 ROAR_DBG("roar_pcm_hw_params(*) = 0");
 return 0;
}

///////////////////////////////////
// Status: This should be complete. 
// This is the last cleanup function to be called by ALSA.
///////////////////////////////////
static int roar_pcm_close (snd_pcm_ioplug_t * io) {
 struct roar_alsa_pcm * self = io->private_data;

 ROAR_DBG("roar_pcm_close(*) = ?");

 roar_plugin_reset(self);
 roar_disconnect(&(self->roar.con));

 free(self);

 return 0;
}

static snd_pcm_ioplug_callback_t roar_pcm_callback = {
 .start                  = roar_pcm_start,
 .stop                   = roar_pcm_stop,
 .pointer                = roar_pcm_pointer,
 .transfer               = roar_pcm_transfer,
 .delay                  = roar_pcm_delay,
 .prepare                = roar_pcm_prepare,
 .hw_params              = roar_pcm_hw_params,
 .close                  = roar_pcm_close,
};

SND_PCM_PLUGIN_DEFINE_FUNC(roar) {
 struct roar_alsa_pcm * self;
 snd_config_iterator_t i, next;
 snd_config_t * n;
 const char   * para;
 const char   * server = NULL;
 const char   * tmp;
 int            role = ROAR_ROLE_UNKNOWN;
 int            ret;

 (void)root;

 ROAR_DBG("SND_PCM_PLUGIN_DEFINE_FUNC(roar) = ?");

 snd_config_for_each(i, next, conf) {
  n = snd_config_iterator_entry(i);
  if ( snd_config_get_id(n, &para) < 0 )
   continue;

  if ( !strcmp(para, "type") || !strcmp(para, "comment") || !strcmp(para, "hint") )
   continue;

  if ( !strcmp(para, "server") ) {
   if (snd_config_get_string(n, &server) < 0) {
    return -EINVAL;
   }
  } else if ( !strcmp(para, "role") ) {
   if (snd_config_get_string(n, &tmp) < 0) {
    return -EINVAL;
   }
   if ( (role = roar_str2role(tmp)) == -1 ) {
    return -EINVAL;
   }
  } else {
   return -EINVAL;
  }
 }

 errno = ENOSYS;

 if ( (self = malloc(sizeof(struct roar_alsa_pcm))) == NULL )
  return -errno;

 memset(self, 0, sizeof(struct roar_alsa_pcm));

 self->stream_role = role;

 errno = ENOSYS;
 if ( roar_simple_connect(&(self->roar.con), (char*)server, "ALSA Plugin") == -1 ) {
  free(self);
  return -errno;
 }

 self->io.version      = SND_PCM_IOPLUG_VERSION;
 self->io.name         = "RoarAudio Plugin";
 self->io.poll_fd      =  -1;
 self->io.poll_events  =  POLLOUT;
 self->io.mmap_rw      =  0;
 self->io.callback     = &roar_pcm_callback;
 self->io.private_data =  self;

 if ( (ret = snd_pcm_ioplug_create(&(self->io), name, stream, mode)) < 0 ) {
  roar_disconnect(&(self->roar.con));
  free(self);
  return ret;
 }

 if ( (ret = roar_hw_constraint(self)) < 0 ) {
  snd_pcm_ioplug_delete(&(self->io));
  roar_disconnect(&(self->roar.con));
  free(self);
  return ret;
 }

 *pcmp = self->io.pcm;

 ROAR_DBG("SND_PCM_PLUGIN_DEFINE_FUNC(roar) = 0");

 return 0;
}

SND_PCM_PLUGIN_SYMBOL(roar);

//ll
