//hash.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010-2013
 *
 *  This file is part of libroar a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  libroar is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 *  NOTE for everyone want's to change something and send patches:
 *  read README and HACKING! There a addition information on
 *  the license of this document you need to read before you send
 *  any patches.
 *
 *  NOTE for uses of non-GPL (LGPL,...) software using libesd, libartsc
 *  or libpulse*:
 *  The libs libroaresd, libroararts and libroarpulse link this lib
 *  and are therefore GPL. Because of this it may be illigal to use
 *  them with any software that uses libesd, libartsc or libpulse*.
 */

#include "libroar.h"

#ifdef ROAR_HAVE_LIBGCRYPT
#include <gcrypt.h>
#endif

struct roar_hash_state {
 struct roar_hash_cmds * cmds;
 void * state;
};

static const struct hashes {
 const roar_hash_t    id;
 const char         * name;
 const ssize_t        dlen;
} _libroar_hashes[] = {
/*
grep '^  +HT_' doc/new-cmds | sed 's/ *#(.*)$//; s/^  +HT_//; s/ *=.*$//' | while read n; do printf " {ROAR_HT_%-12s \"%-12s -1   },\n" $n, $n\",; done
*/
 {ROAR_HT_NONE,        "NONE",       -1   },
 {ROAR_HT_MD5,         "MD5",         16  },
 {ROAR_HT_SHA1,        "SHA1",        20  },
 {ROAR_HT_RIPEMD160,   "RIPEMD160",   20  },
 {ROAR_HT_MD2,         "MD2",        -1   },
 {ROAR_HT_TIGER,       "TIGER",       24  },
 {ROAR_HT_HAVAL,       "HAVAL",      -1   },
 {ROAR_HT_SHA256,      "SHA256",      32  },
 {ROAR_HT_SHA384,      "SHA384",      48  },
 {ROAR_HT_SHA512,      "SHA512",      64  },
 {ROAR_HT_SHA224,      "SHA224",      28  },
 {ROAR_HT_MD4,         "MD4",         16  },
 {ROAR_HT_CRC32,       "CRC32",       4   },
 {ROAR_HT_RFC1510,     "RFC1510",     4   },
 {ROAR_HT_RFC2440,     "RFC2440",     3   },
 {ROAR_HT_WHIRLPOOL,   "WHIRLPOOL",   64  },
 {ROAR_HT_UUID,        "UUID",        16  },
 {ROAR_HT_GTN8,        "GTN8",        1   },
 {ROAR_HT_GTN16,       "GTN16",       2   },
 {ROAR_HT_GTN32,       "GTN32",       4   },
 {ROAR_HT_GTN64,       "GTN64",       8   },
 {ROAR_HT_CLIENTID,    "CLIENTID",   -1   },
 {ROAR_HT_STREAMID,    "STREAMID",   -1   },
 {ROAR_HT_SOURCEID,    "SOURCEID",   -1   },
 {ROAR_HT_SAMPLEID,    "SAMPLEID",   -1   },
 {ROAR_HT_MIXERID,     "MIXERID",    -1   },
 {ROAR_HT_BRIDGEID,    "BRIDGEID",   -1   },
 {ROAR_HT_LISTENID,    "LISTENID",   -1   },
 {ROAR_HT_ACTIONID,    "ACTIONID",   -1   },
 {ROAR_HT_MSGQUEUEID,  "MSGQUEUEID", -1   },
 {ROAR_HT_MSGBUSID,    "MSGBUSID",   -1   },
 {ROAR_HT_GTIN8,       "GTIN8",       4   },
 {ROAR_HT_GTIN13,      "GTIN13",      8   },
 {ROAR_HT_ISBN10,      "ISBN10",      8   },
 {ROAR_HT_ISBN13,      "ISBN13",      8   },
 {ROAR_HT_ADLER32,     "ADLER32",     4   },
 {-1, NULL, -1}
};

int roar_hash_sha1_init(void * state);
int roar_hash_sha1_uninit(void * state);
int roar_hash_sha1_digest(void * state, void * digest, size_t * len);
int roar_hash_sha1_proc_block(void * state, const void * block);
int roar_hash_sha1_proc(void * state, const void * data, size_t len);

static struct roar_hash_cmds _libroar_hash_cmds[] = {
 {ROAR_HT_TIGER, sizeof(struct roar_hash_tiger), 512,
  (int (*)(void *))roar_hash_tiger_init, (int (*)(void *))roar_hash_tiger_uninit,
  (int (*)(void *, void *, size_t *))roar_hash_tiger_get_digest,
  (int (*)(void *, const void *))roar_hash_tiger_proc_block,
  (int (*)(void *, const void *, size_t))roar_hash_tiger_proc
 },
 {ROAR_HT_RFC2440, sizeof(uint32_t), -1,
  roar_hash_crc24_init, NULL,
  roar_hash_crc24_digest, NULL, roar_hash_crc24_proc
 },
 {ROAR_HT_ADLER32, sizeof(uint32_t), -1,
  roar_hash_adler32_init, NULL,
  roar_hash_adler32_digest, NULL, roar_hash_adler32_proc
 },
 {ROAR_HT_SHA1, sizeof(struct roar_hash_sha1), 64,
  roar_hash_sha1_init, roar_hash_sha1_uninit,
  roar_hash_sha1_digest, roar_hash_sha1_proc_block, roar_hash_sha1_proc},
 {-1, -1, -1, NULL, NULL, NULL, NULL, NULL}
};

static struct roar_hash_cmds * roar_ht2cmds(const roar_hash_t ht) {
 size_t i;

 for(i = 0; _libroar_hash_cmds[i].algo != (roar_hash_t)-1; i++)
  if ( _libroar_hash_cmds[i].algo == ht )
   return &(_libroar_hash_cmds[i]);

 roar_err_set(ROAR_ERROR_NOENT);
 return NULL;
}

static inline int roar_ht2gcrypt_tested (const roar_hash_t ht) {
#ifdef ROAR_HAVE_LIBGCRYPT
 const char * name;

 if ( ht > 512 )
  return -1;

 // test the algo:
 name = gcry_md_algo_name(ht);

 if ( name == NULL )
  return -1;

 if ( *name == 0 )
  return -1;

 return ht;
#else
 return -1;
#endif
}

const char * roar_ht2str (const roar_hash_t  ht) {
 int i;

 for (i = 0; _libroar_hashes[i].id != (roar_hash_t)-1; i++)
  if ( _libroar_hashes[i].id == ht )
   return _libroar_hashes[i].name;

 roar_err_set(ROAR_ERROR_NOENT);
 return NULL;
}

roar_hash_t      roar_str2ht (const char * ht) {
 int i;

 for (i = 0; _libroar_hashes[i].id != (roar_hash_t)-1; i++)
  if ( !strcasecmp(_libroar_hashes[i].name, ht) )
   return _libroar_hashes[i].id;

 roar_err_set(ROAR_ERROR_NOENT);
 return -1;
}

ssize_t      roar_ht_digestlen (const roar_hash_t   ht) {
 int i;

 for (i = 0; _libroar_hashes[i].id != (uint_least32_t)-1; i++)
  if ( _libroar_hashes[i].id == ht )
   return _libroar_hashes[i].dlen;

 roar_err_set(ROAR_ERROR_NOENT);
 return -1;
}

static void _bin2hex(char * out, char * in, size_t inlen, int uppercase) {
 const char * tab = uppercase ? "0123456789ABCDEF" : "0123456789abcdef";
 unsigned char c;
 int nib0, nib1;

 for (; inlen; inlen--) {
  c = *(in++);
  nib0 = (c & 0x0F) >> 0;
  nib1 = (c & 0xF0) >> 4;
//  printf("inlen=%zu, c=%u, nibs={%i, %i}\n", inlen, (unsigned)c, nib1, nib0);
  *(out++) = tab[nib1];
  *(out++) = tab[nib0];
 }
 *out = 0;
}

ssize_t      roar_hash_digest2str(char * out, size_t outlen, void * digest, size_t digestlen, roar_hash_t ht) {
 ssize_t slen = roar_ht_digestlen(ht);
 union {
  long long int llsi;
 } tmp;
 enum {
  HEX,
  TIGER,
//  BASE64,
  UUID,
  INT,
  GTIN
 } type = HEX;

 if ( out == NULL || digest == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( !(slen == -1 || (ssize_t)digestlen == slen) ) {
  roar_err_set(ROAR_ERROR_INVAL);
  return -1;
 }

 switch (ht) {
  case ROAR_HT_TIGER: type = TIGER; break;
  case ROAR_HT_UUID:  type = UUID;  break;
  case ROAR_HT_CLIENTID:
  case ROAR_HT_STREAMID:
  case ROAR_HT_SOURCEID:
  case ROAR_HT_SAMPLEID:
  case ROAR_HT_MIXERID:
  case ROAR_HT_BRIDGEID:
  case ROAR_HT_LISTENID:
  case ROAR_HT_ACTIONID:
  case ROAR_HT_MSGQUEUEID:
  case ROAR_HT_MSGBUSID:
    type = INT;
   break;
  case ROAR_HT_GTIN8:
  case ROAR_HT_GTIN13:
  case ROAR_HT_ISBN10:
    type = GTIN;
   break;
 }

 switch (type) {
  case HEX:
    if ( outlen < (digestlen*2 + 1) ) {
     roar_err_set(ROAR_ERROR_FAULT);
     return -1;
    }
    _bin2hex(out, digest, digestlen, 0);
   break;
  case UUID:
    if ( outlen < 37 || digestlen != 16 ) {
     roar_err_set(ROAR_ERROR_FAULT);
     return -1;
    }
    _bin2hex(out+ 0, digest+ 0, 4, 0);
    out[ 8] = '-';
    _bin2hex(out+ 9, digest+ 4, 2, 0);
    out[13] = '-';
    _bin2hex(out+14, digest+ 6, 2, 0);
    out[18] = '-';
    _bin2hex(out+19, digest+ 8, 2, 0);
    out[23] = '-';
    _bin2hex(out+24, digest+10, 6, 0);
//    printf("%s\n", out+24);
   break;
  case TIGER:
    if ( outlen < 51 || digestlen != 24 ) {
     roar_err_set(ROAR_ERROR_FAULT);
     return -1;
    }
    _bin2hex(out+ 0, digest+ 0, 8, 1);
    out[16] = ' ';
    _bin2hex(out+17, digest+ 8, 8, 1);
    out[33] = ' ';
    _bin2hex(out+34, digest+16, 8, 1);
   break;
  case INT:
    switch (digestlen) {
     case 1:
       tmp.llsi = *(_LIBROAR_GOOD_CAST char*)digest;
      break;
     case 2:
       tmp.llsi = ROAR_NET2HOST16(*(int16_t*)digest);
      break;
     case 4:
       tmp.llsi = ROAR_NET2HOST32(*(int32_t*)digest);
      break;
     default:
       roar_err_set(ROAR_ERROR_NOSYS);
       return -1;
      break;
    }
    snprintf(out, outlen-1, "%" LIBROAR__ll "i", (LIBROAR__longlong int)tmp.llsi);
   break;
  default:
    roar_err_set(ROAR_ERROR_NOSYS);
    return -1;
 }

 out[outlen-1] = 0;
 return roar_mm_strlen(out);
}

int          roar_ht_is_supported(const roar_hash_t ht) {
 roar_crypto_init();

 if ( roar_ht2cmds(ht) != NULL )
  return 1;

#ifdef ROAR_HAVE_LIBGCRYPT
 if ( roar_ht2gcrypt_tested(ht) == -1 )
  return 0;

 return 1;
#else
 return 0;
#endif
}

struct roar_hash_state * roar_hash_new(roar_hash_t algo) {
 struct roar_hash_cmds  * cmds = roar_ht2cmds(algo);
 struct roar_hash_state * self;

 if ( cmds == NULL )
  return NULL;

 self = roar_mm_malloc(sizeof(struct roar_hash_state));

 if ( self == NULL )
  return NULL;

 memset(self, 0, sizeof(struct roar_hash_state));

 self->cmds = cmds;

 self->state = roar_mm_malloc(cmds->statelen);

 if ( self->state == NULL ) {
  roar_mm_free(self);
  return NULL;
 }

 memset(self->state, 0, cmds->statelen);

 if ( cmds->init != NULL ) {
  if ( cmds->init(self->state) == -1 ) {
   roar_mm_free(self->state);
   roar_mm_free(self);
   return NULL;
  }
 }

 return self;
}

int roar_hash_free(struct roar_hash_state * state) {
 int ret = 0;

 if ( state == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( state->cmds->uninit != NULL )
  ret = state->cmds->uninit(state->state);

 // clear crypto data.
 memset(state->state, 0, state->cmds->statelen);
 roar_mm_free(state->state);

 memset(state, 0, sizeof(struct roar_hash_state));
 roar_mm_free(state);

 return ret;
}

int roar_hash_digest(struct roar_hash_state * state, void * digest, size_t * len) {
 if ( state == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( state->cmds->digest == NULL ) {
  roar_err_set(ROAR_ERROR_NOSYS);
  return -1;
 }

 return state->cmds->digest(state->state, digest, len);
}

int roar_hash_proc(struct roar_hash_state * state, const void * data, size_t len) {
 if ( state == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( state->cmds->proc == NULL ) {
  roar_err_set(ROAR_ERROR_NOSYS);
  return -1;
 }

 return state->cmds->proc(state->state, data, len);
}

int roar_hash_buffer(void * digest, const void * data, size_t datalen, roar_hash_t algo) {
 roar_crypto_init();

 return roar_hash_salted_buffer(digest, data, datalen, algo, NULL, 0);
}

#ifdef ROAR_HAVE_LIBGCRYPT
static inline int roar_hash_salted_buffer_gcrypt(void * digest, const void * data, size_t datalen, roar_hash_t algo, const void * salt, size_t saltlen) {
 gcry_md_hd_t hdl;

 roar_crypto_init();

 algo = roar_ht2gcrypt_tested(algo);
 if ( algo == (roar_hash_t)-1 )
  return -1;


 if ( salt == NULL ) {
  // optimized for unsalted:
  gcry_md_hash_buffer(algo, digest, data, datalen);
  return 0;
 } else {
  if ( gcry_md_open(&hdl, algo, 0) != 0 )
   return -1;

  gcry_md_write(hdl, data, datalen);
  gcry_md_write(hdl, salt, saltlen);

  memcpy(digest, gcry_md_read(hdl, algo), gcry_md_get_algo_dlen(algo));

  gcry_md_close(hdl);
 }

 return 0;
}
#endif

int roar_hash_salted_buffer(void * digest, const void * data, size_t datalen, roar_hash_t algo, const void * salt, size_t saltlen) {
 struct roar_hash_state * state;
 size_t len;
 int ret = 0;

 ROAR_DBG("roar_hash_salted_buffer(digest=%p, data=%p, datalen=%llu, algo=%i, salt=%p, saltlen=%llu) = ?", digest, data, (unsigned long long int)datalen, algo, salt, (unsigned long long int)saltlen);

 if ( digest == NULL || data == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 len = roar_ht_digestlen(algo);
 if ( len == (size_t)-1 ) /* this is a hack to avoid len to be of type ssize_t which breaks following code */
  return -1;

 if ( (state = roar_hash_new(algo)) != NULL ) {
  ROAR_DBG("roar_hash_salted_buffer(*): ret=%i", ret);

  if ( roar_hash_proc(state, data, datalen) == -1 )
   ret = -1;

  ROAR_DBG("roar_hash_salted_buffer(*): ret=%i", ret);

  if ( saltlen != 0 )
   if ( roar_hash_proc(state, salt, saltlen) == -1 )
    ret = -1;

  ROAR_DBG("roar_hash_salted_buffer(*): ret=%i", ret);

  if ( roar_hash_digest(state, digest, &len) == -1 )
   ret = -1;

  ROAR_DBG("roar_hash_salted_buffer(*): ret=%i", ret);

  if ( roar_hash_free(state) == -1 )
   ret = -1;

  ROAR_DBG("roar_hash_salted_buffer(*): ret=%i", ret);

  return ret;
 }

 ROAR_DBG("roar_hash_salted_buffer(*): state=%p", state);

#ifdef ROAR_HAVE_LIBGCRYPT
 return roar_hash_salted_buffer_gcrypt(digest, data, datalen, algo, salt, saltlen);
#else
 roar_err_set(ROAR_ERROR_NOTSUP);
 return -1;
#endif
}

//ll
