//hashh_sha1.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2012-2013
 *      Copyright (C) Steve Reid <steve@edmweb.com>
 *
 *  This file is part of libroar a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  libroar is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 *  NOTE for everyone want's to change something and send patches:
 *  read README and HACKING! There a addition information on
 *  the license of this document you need to read before you send
 *  any patches.
 *
 *  NOTE for uses of non-GPL (LGPL,...) software using libesd, libartsc
 *  or libpulse*:
 *  The libs libroaresd, libroararts and libroarpulse link this lib
 *  and are therefore GPL. Because of this it may be illigal to use
 *  them with any software that uses libesd, libartsc or libpulse*.
 */

#include "libroar.h"

#define SHA1_BLOCK_LENGTH               64
#define SHA1_DIGEST_LENGTH              20
#define SHA1_DIGEST_STRING_LENGTH       (SHA1_DIGEST_LENGTH * 2 + 1)

union uint864uint3216 {
 uint8_t c[64];
 uint32_t l[16];
};

int roar_hash_sha1_proc(void * state, const void * data, size_t len);


#define rol(value, bits) (((value) << (bits)) | ((value) >> (32 - (bits))))

/*
 * blk0() and blk() perform the initial expand.
 * I got the idea of expanding during the round function from SSLeay
 */
#if BYTE_ORDER == LITTLE_ENDIAN
# define blk0(i) (compblock->l[i] = (rol(compblock->l[i],24)&0xFF00FF00) \
    |(rol(compblock->l[i],8)&0x00FF00FF))
#else
# define blk0(i) compblock->l[i]
#endif
#define blk(i) (compblock->l[i&15] = rol(compblock->l[(i+13)&15]^compblock->l[(i+8)&15] \
    ^compblock->l[(i+2)&15]^compblock->l[i&15],1))

/*
 * (R0+R1), R2, R3, R4 are the different operations (rounds) used in SHA1
 */
#define R0(v,w,x,y,z,i) z+=((w&(x^y))^y)+blk0(i)+0x5A827999+rol(v,5);w=rol(w,30);
#define R1(v,w,x,y,z,i) z+=((w&(x^y))^y)+blk(i)+0x5A827999+rol(v,5);w=rol(w,30);
#define R2(v,w,x,y,z,i) z+=(w^x^y)+blk(i)+0x6ED9EBA1+rol(v,5);w=rol(w,30);
#define R3(v,w,x,y,z,i) z+=(((w|x)&y)|(w&x))+blk(i)+0x8F1BBCDC+rol(v,5);w=rol(w,30);
#define R4(v,w,x,y,z,i) z+=(w^x^y)+blk(i)+0xCA62C1D6+rol(v,5);w=rol(w,30);

int roar_hash_sha1_init(void * state) {
 struct roar_hash_sha1 * context = state;

 if ( state == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 memset(context, 0, sizeof(struct roar_hash_sha1));

 context->state[0] = 0x67452301;
 context->state[1] = 0xEFCDAB89;
 context->state[2] = 0x98BADCFE;
 context->state[3] = 0x10325476;
 context->state[4] = 0xC3D2E1F0;

 return 0;
}

int roar_hash_sha1_uninit(void * state) {
 (void)state;
 return 0;
}

int roar_hash_sha1_digest(void * state, void * digest, size_t * len) {
 struct roar_hash_sha1 * context = state;
 uint64_t count;
 int i;

 if ( state == NULL || digest == NULL || len == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( context->is_final ) {
  roar_err_set(ROAR_ERROR_BUSY);
  return -1;
 }

 if ( *len < SHA1_DIGEST_LENGTH ) {
  roar_err_set(ROAR_ERROR_NOSPC);
  return -1;
 }

 *len = SHA1_DIGEST_LENGTH;

 context->count *= 8;
 count = ROAR_HOST2NET64(context->count);

 roar_hash_sha1_proc(state, "\200", 1);

 // TODO: make this calc the correct length directly.
 // this is inefficent.
 while (context->in_buffer != (SHA1_BLOCK_LENGTH - 8)) {
  ROAR_DBG("roar_hash_sha1_digest(state=%p, digest=%p, len=%p): context->in_buffer = %i", state, digest, len, (int)context->in_buffer);
  roar_hash_sha1_proc(state, "\0", 1);
 }

 roar_hash_sha1_proc(state, &count, 8);


 for (i = 0; i < SHA1_DIGEST_LENGTH; i++) {
  ((uint8_t *)digest)[i] = (uint8_t)((context->state[i>>2] >> ((3-(i & 3)) * 8) ) & 255);
 }

 memset(context, 0, sizeof(struct roar_hash_sha1));
 context->is_final = 1;

 return 0;
}

int roar_hash_sha1_proc_block(void * state, const void * block) {
 struct roar_hash_sha1 * context = state;
 uint32_t a, b, c, d, e;
 uint8_t workspace[SHA1_BLOCK_LENGTH];
 union uint864uint3216 * compblock = (union uint864uint3216 *)(void*)workspace;

 if ( state == NULL || block == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( context->is_final ) {
  roar_err_set(ROAR_ERROR_BUSY);
  return -1;
 }

 if ( context->in_buffer ) {
  return roar_hash_sha1_proc(state, block, SHA1_BLOCK_LENGTH);
 }

 memcpy(compblock, block, SHA1_BLOCK_LENGTH);

 /* Copy context->state[] to working vars */
 a = context->state[0];
 b = context->state[1];
 c = context->state[2];
 d = context->state[3];
 e = context->state[4];

 /* 4 rounds of 20 operations each. Loop unrolled. */
 R0(a,b,c,d,e, 0); R0(e,a,b,c,d, 1); R0(d,e,a,b,c, 2); R0(c,d,e,a,b, 3);
 R0(b,c,d,e,a, 4); R0(a,b,c,d,e, 5); R0(e,a,b,c,d, 6); R0(d,e,a,b,c, 7);
 R0(c,d,e,a,b, 8); R0(b,c,d,e,a, 9); R0(a,b,c,d,e,10); R0(e,a,b,c,d,11);
 R0(d,e,a,b,c,12); R0(c,d,e,a,b,13); R0(b,c,d,e,a,14); R0(a,b,c,d,e,15);
 R1(e,a,b,c,d,16); R1(d,e,a,b,c,17); R1(c,d,e,a,b,18); R1(b,c,d,e,a,19);
 R2(a,b,c,d,e,20); R2(e,a,b,c,d,21); R2(d,e,a,b,c,22); R2(c,d,e,a,b,23);
 R2(b,c,d,e,a,24); R2(a,b,c,d,e,25); R2(e,a,b,c,d,26); R2(d,e,a,b,c,27);
 R2(c,d,e,a,b,28); R2(b,c,d,e,a,29); R2(a,b,c,d,e,30); R2(e,a,b,c,d,31);
 R2(d,e,a,b,c,32); R2(c,d,e,a,b,33); R2(b,c,d,e,a,34); R2(a,b,c,d,e,35);
 R2(e,a,b,c,d,36); R2(d,e,a,b,c,37); R2(c,d,e,a,b,38); R2(b,c,d,e,a,39);
 R3(a,b,c,d,e,40); R3(e,a,b,c,d,41); R3(d,e,a,b,c,42); R3(c,d,e,a,b,43);
 R3(b,c,d,e,a,44); R3(a,b,c,d,e,45); R3(e,a,b,c,d,46); R3(d,e,a,b,c,47);
 R3(c,d,e,a,b,48); R3(b,c,d,e,a,49); R3(a,b,c,d,e,50); R3(e,a,b,c,d,51);
 R3(d,e,a,b,c,52); R3(c,d,e,a,b,53); R3(b,c,d,e,a,54); R3(a,b,c,d,e,55);
 R3(e,a,b,c,d,56); R3(d,e,a,b,c,57); R3(c,d,e,a,b,58); R3(b,c,d,e,a,59);
 R4(a,b,c,d,e,60); R4(e,a,b,c,d,61); R4(d,e,a,b,c,62); R4(c,d,e,a,b,63);
 R4(b,c,d,e,a,64); R4(a,b,c,d,e,65); R4(e,a,b,c,d,66); R4(d,e,a,b,c,67);
 R4(c,d,e,a,b,68); R4(b,c,d,e,a,69); R4(a,b,c,d,e,70); R4(e,a,b,c,d,71);
 R4(d,e,a,b,c,72); R4(c,d,e,a,b,73); R4(b,c,d,e,a,74); R4(a,b,c,d,e,75);
 R4(e,a,b,c,d,76); R4(d,e,a,b,c,77); R4(c,d,e,a,b,78); R4(b,c,d,e,a,79);

 /* Add the working vars back into context.state[] */
 context->state[0] += a;
 context->state[1] += b;
 context->state[2] += c;
 context->state[3] += d;
 context->state[4] += e;

 /* Wipe variables */
 a = b = c = d = e = 0;

 return 0;
}

int roar_hash_sha1_proc(void * state, const void * data, size_t len) {
 struct roar_hash_sha1 * context = state;
 size_t buflen;

 ROAR_DBG("roar_hash_sha1_proc(state=%p, data=%p, len=%llu) = ?", state, data, (long long unsigned int)len);

 if ( state == NULL || data == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( context->is_final ) {
  roar_err_set(ROAR_ERROR_BUSY);
  return -1;
 }

 ROAR_DBG("roar_hash_sha1_proc(state=%p, data=%p, len=%llu) = ?", state, data, (long long unsigned int)len);

 context->count += len;

 if ( context->in_buffer ) {
  ROAR_DBG("roar_hash_sha1_proc(state=%p, data=%p, len=%llu) = ?", state, data, (long long unsigned int)len);
  buflen = len > (SHA1_BLOCK_LENGTH - context->in_buffer) ? SHA1_BLOCK_LENGTH - context->in_buffer : len;
  memcpy(context->buffer + context->in_buffer, data, buflen);
  context->in_buffer += buflen;
  if ( context->in_buffer == SHA1_BLOCK_LENGTH ) {
   ROAR_DBG("roar_hash_sha1_proc(state=%p, data=%p, len=%llu) = ?", state, data, (long long unsigned int)len);
   context->in_buffer = 0;
   roar_hash_sha1_proc_block(state, context->buffer);
  } else {
   ROAR_DBG("roar_hash_sha1_proc(state=%p, data=%p, len=%llu) = 0", state, data, (long long unsigned int)len);
   return 0;
  }

  len  -= buflen;
  data += buflen;
 }

 ROAR_DBG("roar_hash_sha1_proc(state=%p, data=%p, len=%llu) = ?", state, data, (long long unsigned int)len);

 while (len >= SHA1_BLOCK_LENGTH) {
  ROAR_DBG("roar_hash_sha1_proc(state=%p, data=%p, len=%llu) = ?", state, data, (long long unsigned int)len);
  roar_hash_sha1_proc_block(state, data);
  len  -= SHA1_BLOCK_LENGTH;
  data += SHA1_BLOCK_LENGTH;
 }

 ROAR_DBG("roar_hash_sha1_proc(state=%p, data=%p, len=%llu) = ?", state, data, (long long unsigned int)len);

 if ( len ) {
  memcpy(context->buffer, data, len);
  context->in_buffer = len;
 }

 ROAR_DBG("roar_hash_sha1_proc(state=%p, data=%p, len=%llu) = 0", state, data, (long long unsigned int)len);
 return 0;
}

//ll
