/*

ChangeLog

* Sat Dec 09 2000 Sourav K. Mandal <Sourav.Mandal@ikaran.com>
- Implementation of "Consoledump" module; see corresponding header
  file for general info.
- CAVEAT:  Since tty dev names are hardcoded int, as of now
  the module probably only works on Linux.  I don't know how 
  to implement this functionality for other OSs, if it matters.

*/

#include "consoledump.h"
#include "../../module_registry.h"

#include <errno.h>
#include <cstring>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include <cstdio>
#include <sys/time.h>

using namespace std;

DataSetMap *ConsoleDump::moduleInfo = 0;

Module* consoledump_constructor() {
        return new ConsoleDump();
}

const DataSetMap& ConsoleDump::get_module_info_instance() {
        if (!moduleInfo) {
                moduleInfo = new DataSetMap();
                moduleInfo->set("description", 
                                DataSet("Dump messages from /dev/console"));
                moduleInfo->set("supported_vars", DataSet());
                moduleInfo->set("supported_var_types", DataSet());
                moduleInfo->set("supported_var_defaults", DataSet());
                moduleInfo->set("overridable", DataSet());
                moduleInfo->set("output_variables", DataSet("textline"));
        }
        return *moduleInfo;
}
extern "C" int consoledump_plugin_startup()
{
        ModuleRegistry::instance()->add_module(
                "Consoledump", 
                &consoledump_constructor, 
                ConsoleDump::get_module_info_instance()
                );
        return 0;
}

extern "C" void consoledump_plugin_shutdown()
{
        ModuleRegistry::instance()->remove_module("Consoledump");
        ConsoleDump::destroy_module_info();
}

ConsoleDump::ConsoleDump() {
        rpdbgmsg(getName() << ": Starting to set up");
        /* Open files */
        console = 0;
        slave = 0;
        if(access("/dev/console", R_OK|W_OK) == -1)
                errnomsg("Trying to access /dev/console");
        /* Use "O_NDELAY" aka "O_NONBLOCK" or reads will sit until
           terminal is flushed */
        console = open("/dev/ptyp0", O_RDWR | O_NDELAY);
        if(console == -1) {
                errnomsg("Trying to open /dev/ptyp0");
                return;
        }
        
	slave = open("/dev/ttyp0", O_RDWR);
        if(slave == -1) {
                errnomsg("Trying to open /dev/ttyp0");
                return;
        }
        
	if(ioctl(slave, TIOCCONS)) {
                errnomsg("Trying to use ioctl on /dev/ttyp0");
                return;
        }
        
	rpdbgmsg(getName() << ": Opened console");

        /* Create buffer */
        buf = (char *) malloc(BUFSIZ);
  
        /* Set up "select" parameters */
        FD_ZERO(&rdfs);
        FD_SET(console, &rdfs);
  
        rpdbgmsg(getName() << ": Done setting up");

}

ConsoleDump::~ConsoleDump() {
        rpdbgmsg(getName() << ": Starting clean up");
        free(buf);
        if(close(slave) == -1)
                errnomsg("Trouble closing /dev/ttyp0");
        if(close(console) == -1)
                errnomsg("Trouble closing /dev/ptyp0");
        rpdbgmsg(getName() << ": Done with clean up");
}

void ConsoleDump::service() {
        // rpdbgmsg(getName() << ": Starting service");
        Module::service();
        if (console <= 0) return;
        int retval = 0;
        int len = 0;
        struct timeval tv;
        /* 0 seconds + 0 microseconds -- immediate */
        tv.tv_sec = 0;
        tv.tv_usec = 0;
        retval = select(console + 1, &rdfs, NULL, NULL, &tv);
	if(retval != -1) {
                len = read(console, buf, BUFSIZ);
                if(len == -1) {
                        if(errno != EAGAIN) errnomsg("Trying to read from console");
                }
                else {
                        if(len == 0) {}
                        else {
                                /* 0 = '\0' string delimiter */
                                if(len >= (BUFSIZ - 1)) buf[BUFSIZ - 1] = 0;
                                else buf[len] = 0; 
                                DataSet textline;
                                textline.addString(buf);
                                updateParent("textline", textline);
                        }
                }
        }
        // rpdbgmsg(getName() << ": Ending service");
}

void ConsoleDump::updated(const string& keyName, const DataSet& data) {}

void ConsoleDump::errnomsg(const char *desc) {
        DataSet msg;
        msg.addString("Error: ");
        msg.addString(getName());
        msg.addString(": ");
        msg.addString(desc);
        msg.addString(": ");
        msg.addString(strerror(errno));
        return_message(msg);
        cerr << msg << endl;
}

