
/*
 * Copyright (C) 1999-2003, Ian Main <imain@stemwinder.org> and
 * Jim Meier <fatjim@home.com>
 *
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#ifndef __RTHERML_H__
#define __RTHERML_H__

#include <roy.h>
#include <stdlib.h>
#include <ctype.h>


typedef struct _RThermlAttribute RThermlAttribute;
typedef struct _RThermlNode RThermlNode;
typedef struct _RThermlParser RThermlParser;


typedef void (*RThermlErrorHandler) (RThermlParser *parser,
                                     int linenum, int colnum,
                                     const char *description);

typedef int (*RThermlAttribValueHandler) (RThermlParser *parser, RBuf *attrib,
                                          RBuf *value);

typedef int (*RThermlNodeStartHandler) (RThermlParser *parser, RBuf *type);


typedef int (*RThermlNodeEndHandler) (RThermlParser *parser, RBuf *type,
                                      RBuf *data);



struct _RThermlAttribute {
    RBHASH_HEADER;
    RBuf *value;
};


struct _RThermlNode {
    /* element type for this node */
    RBuf *type;

    /* line and column where this node was opened */
    int linenum;
    int colnum;

    /* Attributes stored in an rbuf hash */
    RBHash *attribs;
    
    /* TRUE or FALSE - node has non-whitespace data */
    unsigned int has_data;

    /* If this node has data, it will be stored in here */
    RBuf *data;
   
    /* Previous entry in linked list (used as stack) */
    RThermlNode *prev;
};



struct _RThermlParser {
    
    int state;
    int linenum;
    int colnum;

    /* A linked list (stack) of nodes */
    RThermlNode *nodes;

    /* The important bit of the last parsed chunk */
    RBuf *prev_chunk;

    /* Storage for the current attribute and value. */
    RBuf *current_attrib;
    RBuf *current_value;

    /* If a value has a length specifier, keep track of it here */
    unsigned int value_length;
    
    /* Current iterator location for a length specified value. */
    unsigned int i;

    /* Keep track of the quote char if applicable */
    char quote_char;

    /* The ending quote string if used with eg attrib=(]]>)"value]]> */
    RBuf *end_quote;

    /* user callbacks */
    RThermlErrorHandler error_handler;
    RThermlAttribValueHandler attrib_value_handler;
    RThermlNodeStartHandler node_start_handler;
    RThermlNodeEndHandler node_end_handler;
   
    /* User data */
    void *user_data;
};


/* Interface */

/* Parse a chunk of therml.  Returns 0 on success, -1 on error */
int 
rtherml_parse_chunk (RThermlParser *parser, char *chunk, int len,
		    int is_last_chunk);


/* Set a function to be called on a parse error. */
void
rtherml_set_error_handler (RThermlParser *parser,
                          RThermlErrorHandler handler);

/* Set a function to be called when a new attribute/value pair is found.
 * Return -1 from this callback if you want to signal an error. */
void
rtherml_set_attrib_value_handler (RThermlParser *parser,
                                 RThermlAttribValueHandler handler);

/* Set a function to be called when a new node is found.
 * Return -1 from this callback if you want to signal an error. */
void
rtherml_set_node_start_handler (RThermlParser *parser,
                               RThermlNodeStartHandler handler);

/* Set function to be called when a node is closed. 
 * Return -1 from this callback if you want to signal an error. */
void 
rtherml_set_node_end_handler (RThermlParser *parser,
                             RThermlNodeEndHandler handler);

/* Free the attributes in the callback.  Only use if 
 * you don't use the hash directly. */
void
rtherml_attribs_free (RBHash *attribs);

/* Set/Get user data */
void
rtherml_set_user_data (RThermlParser *parser, 
                      void *userdata);

void *
rtherml_get_user_data (RThermlParser *parser);

RThermlParser *
rtherml_parser_new (void);

void
rtherml_parser_free (RThermlParser *ptr);


#endif /* __RTHERML_H__ */



