
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#include <roy.h>

#define RXP_ENCODE_RINT32(data,value) \
    (data)[0] = (value & 0xFF000000) >> 24; \
    (data)[1] = (value & 0x00FF0000) >> 16; \
    (data)[2] = (value & 0x0000FF00) >> 8; \
    (data)[3] = (value & 0x000000FF) >> 0; 

#define RXP_DECODE_RINT32(data,value) \
    value = ((data)[0] & 0xFF) << 24; \
    value |= ((data)[1] & 0xFF) << 16; \
    value |= ((data)[2] & 0xFF) << 8; \
    value |= ((data)[3] & 0xFF) << 0; 


#define RXP_ENCODER_MAYBE_EXPAND(encoder) \
    if (encoder->length > encoder->alloced) { \
        unsigned int new_size; \
        unsigned int offset; \
        offset = encoder->cur_pos - encoder->data; \
        new_size = encoder->alloced * 2; \
        encoder->data = rmem_realloc (encoder->data, new_size); \
        encoder->cur_pos = encoder->data + offset; \
        encoder->alloced = new_size; \
    }

#ifdef WORDS_BIGENDIAN

/** 
 * Byte swaps a 32-bit unsigned value
 *
   @param v[in] unsigned 32-bit input value to swap
   @returns a byte swapped version of v
 */
static ruint32
rxp_swap_u32 (ruint32 v) 
{
    return  ((v << 24) & 0xff000000 ) |
            ((v <<  8) & 0x00ff0000 ) |
            ((v >>  8) & 0x0000ff00 ) |
            ((v >> 24) & 0x000000ff );
}
#endif /* WORDS_BIGENDIAN */

/** 
 * Extracts raw little-endian bits from a 32-bit floating point value
 *
   @ingroup FloatingPoint
   @param f[in] floating point value
   @returns a little-endian bit representation of f
 */
static ruint32
rxp_float_to_le_bits (float f)
{
    union
    {
        float f32;
        ruint32 u32;
    } u;

    u.f32 = f;

#ifdef WORDS_BIGENDIAN
    return rxp_swap_u32 (u.u32);
#else
    return u.u32;
#endif
}


/** 
 * Creates a floating point number from little endian bits
 *
   @ingroup FloatingPoint
   @param   bits[in] raw floating point bits in little-endian form
   @returns a floating point number based on the given bit representation
   @remarks No error checking is performed, so there are no guarantees that the 
            result is a valid number.  BE CAREFUL USING THIS.
 */
static float       
rxp_float_from_le_bits (ruint32 bits)
{
    union
    {
        float f32;
        ruint32 u32;
    } u;

    u.u32 = bits;
#ifdef WORDS_BIGENDIAN
    u.u32 = rxp_swap_u32 (u.u32);
#endif

    return u.f32;
}


/** 
 * Extracts raw, little-endian bit representation from a 64-bit double.
 *
   @param d[in] 64-bit double precision value
   @param dst[out] 8-byte storage buffer
   @ingroup FloatingPoint
   @returns the raw bits used to represent the value 'd', in the form dst[0]=LSB
 */
static void
rxp_double_to_bits (double d, ruint8 dst[8])
{
    union
    {
        double d64;
        ruint8 bytes[8];
    } u;

    u.d64 = d;

#ifdef WORDS_BIGENDIAN
    dst [0] = u.bytes [7];
    dst [1] = u.bytes [6];
    dst [2] = u.bytes [5];
    dst [3] = u.bytes [4];
    dst [4] = u.bytes [3];
    dst [5] = u.bytes [2];
    dst [6] = u.bytes [1];
    dst [7] = u.bytes [0];
#else
    dst [0] = u.bytes [0];
    dst [1] = u.bytes [1];
    dst [2] = u.bytes [2];
    dst [3] = u.bytes [3];
    dst [4] = u.bytes [4];
    dst [5] = u.bytes [5];
    dst [6] = u.bytes [6];
    dst [7] = u.bytes [7];
#endif
}


/** 
 * Creates a double-precision, 64-bit floating point value from a set if raw, 
 * little-endian bits

   @ingroup FloatingPoint
   @param src[in] little-endian byte representation of 64-bit double precision 
                  floating point value
   @returns double precision floating point representation of the raw bits
   @remarks No error checking is performed, so there are no guarantees that the 
            result is a valid number, nor is there any check to ensure that src is 
            non-NULL.  BE CAREFUL USING THIS.
 */
static double
rxp_double_from_bits (const ruint8 src[8])
{
    union
    {
        double d64;
        ruint8 bytes[8];
    } u;

#ifdef WORDS_BIGENDIAN
    u.bytes [0] = src [7];
    u.bytes [1] = src [6];
    u.bytes [2] = src [5];
    u.bytes [3] = src [4];
    u.bytes [4] = src [3];
    u.bytes [5] = src [2];
    u.bytes [6] = src [1];
    u.bytes [7] = src [0];
#else
    u.bytes [0] = src [0];
    u.bytes [1] = src [1];
    u.bytes [2] = src [2];
    u.bytes [3] = src [3];
    u.bytes [4] = src [4];
    u.bytes [5] = src [5];
    u.bytes [6] = src [6];
    u.bytes [7] = src [7];
#endif

   return u.d64;
}


RXpEncoder *
rxp_encoder_new (void)
{
    RXpEncoder *encoder;

    encoder = rchunk_alloc (sizeof (RXpEncoder));
    encoder->data = rmem_alloc (4096);
    encoder->cur_pos = encoder->data;
    encoder->alloced = 4096;
    encoder->length = 4;

    /* Encode the length of the block */
    RXP_ENCODE_RINT32 (encoder->data, 4);
    encoder->cur_pos += 4;
    
    return (encoder);
}

char *
rxp_encoder_get_buffer (RXpEncoder *encoder, unsigned int *length)
{
    if (!encoder)
        return NULL;

    *length = encoder->length;
    return encoder->data;
}

void
rxp_encoder_reset (RXpEncoder *encoder)
{
    encoder->cur_pos = encoder->data;
    encoder->length = 4;
    
    RXP_ENCODE_RINT32 (encoder->data, 4);
    encoder->cur_pos += 4;
}

void
rxp_encode_msg_end (RXpEncoder *encoder)
{
    /* Update length. */
    encoder->length += 4;

    /* Insure we have enough space */
    RXP_ENCODER_MAYBE_EXPAND (encoder);
    
    /* Update total length at start of block */
    RXP_ENCODE_RINT32 (encoder->data, encoder->length);

    /* Encode the data type */
    RXP_ENCODE_RINT32 (encoder->cur_pos, RXP_TYPE_MSG_END);
    encoder->cur_pos += 4;
}

void
rxp_encode_type (RXpEncoder *encoder, RXpDataType type)
{
    /* Update length. */
    encoder->length += 4;

    /* Insure we have enough space */
    RXP_ENCODER_MAYBE_EXPAND (encoder);
    
    /* Update total length at start of block */
    RXP_ENCODE_RINT32 (encoder->data, encoder->length);

    /* Encode the data type */
    RXP_ENCODE_RINT32 (encoder->cur_pos, type);
    encoder->cur_pos += 4;
}

void
rxp_encode_ruint32 (RXpEncoder *encoder, ruint32 value)
{
    /* Update length */
    encoder->length += 8;
    
    /* Insure we have enough space */
    RXP_ENCODER_MAYBE_EXPAND (encoder);
    
    /* Update total length at start of block */
    RXP_ENCODE_RINT32 (encoder->data, encoder->length);

    /* Encode the data type */
    RXP_ENCODE_RINT32 (encoder->cur_pos, RXP_TYPE_UINT32);
    encoder->cur_pos += 4;

    /* Encode integer */
    RXP_ENCODE_RINT32 (encoder->cur_pos, value);
    encoder->cur_pos += 4;
}

void
rxp_encode_rint32 (RXpEncoder *encoder, rint32 value)
{
    /* Update length */
    encoder->length += 8;
    
    /* Insure we have enough space */
    RXP_ENCODER_MAYBE_EXPAND (encoder);
    
    /* Update total length at start of block */
    RXP_ENCODE_RINT32 (encoder->data, encoder->length);

    /* Encode the data type */
    RXP_ENCODE_RINT32 (encoder->cur_pos, RXP_TYPE_INT32);
    encoder->cur_pos += 4;

    /* Encode integer */
    RXP_ENCODE_RINT32 (encoder->cur_pos, value);
    encoder->cur_pos += 4;
}

void
rxp_encode_float (RXpEncoder *encoder, float value)
{
    ruint32 u32;


    /* Update length */
    encoder->length += 4;
    encoder->length += sizeof (ruint32);
    
    /* Insure we have enough space */
    RXP_ENCODER_MAYBE_EXPAND (encoder);
    
    /* Update total length at start of block */
    RXP_ENCODE_RINT32 (encoder->data, encoder->length);

    /* Encode the data type */
    RXP_ENCODE_RINT32 (encoder->cur_pos, RXP_TYPE_FLOAT);
    encoder->cur_pos += 4;
    
    u32 = rxp_float_to_le_bits (value);
    memcpy (encoder->cur_pos, &u32, sizeof (ruint32));

    encoder->cur_pos += sizeof (ruint32);
}

void
rxp_encode_double (RXpEncoder *encoder, double value)
{
    ruint8 bytes[8];


    /* Update length */
    encoder->length += 12;
    
    /* Insure we have enough space */
    RXP_ENCODER_MAYBE_EXPAND (encoder);
    
    /* Update total length at start of block */
    RXP_ENCODE_RINT32 (encoder->data, encoder->length);

    /* Encode the data type */
    RXP_ENCODE_RINT32 (encoder->cur_pos, RXP_TYPE_DOUBLE);
    encoder->cur_pos += 4;
    
    rxp_double_to_bits (value, bytes);
    memcpy (encoder->cur_pos, bytes, 8 * sizeof (ruint8));

    encoder->cur_pos += (8 * sizeof (ruint8));
}


void
rxp_encode_rbuf (RXpEncoder *encoder, RBuf *value)
{
    /* NULL is special cased */
    if (value == NULL) {
        encoder->length += 4;
        /* Insure we have enough space */
        RXP_ENCODER_MAYBE_EXPAND (encoder);
        /* Update total length at start of block */
        RXP_ENCODE_RINT32 (encoder->data, encoder->length);
        /* Set data type to NULL_RBUF */
        RXP_ENCODE_RINT32 (encoder->cur_pos, RXP_TYPE_NULL_RBUF);
        encoder->cur_pos += 4;
        
        return;
    }
    
    /* Non-NULL rbuf - Update length */
    encoder->length += rbuf_len (value);
    encoder->length += 8;
    
    /* Insure we have enough space */
    RXP_ENCODER_MAYBE_EXPAND (encoder);
    /* Update total length at start of block */
    RXP_ENCODE_RINT32 (encoder->data, encoder->length);
    /* Set data type to RBUF */
    RXP_ENCODE_RINT32 (encoder->cur_pos, RXP_TYPE_RBUF);
    encoder->cur_pos += 4;
    /* Save length of rbuf */
    RXP_ENCODE_RINT32 (encoder->cur_pos, rbuf_len (value));
    encoder->cur_pos += 4;
    /* Save rbuf itself */
    memcpy (encoder->cur_pos, rbuf_str (value), rbuf_len (value));
    encoder->cur_pos += rbuf_len (value);
}


void
rxp_encode_str (RXpEncoder *encoder, char *str)
{
    int len;


    len = strlen (str);

    /* Update length */
    encoder->length += 8;
    encoder->length += len;
    
    RXP_ENCODER_MAYBE_EXPAND (encoder);
    
    /* Update total length at start of block */
    RXP_ENCODE_RINT32 (encoder->data, encoder->length);

    /* Encode the data type */
    RXP_ENCODE_RINT32 (encoder->cur_pos, RXP_TYPE_STR);
    encoder->cur_pos += 4;

    /* Length of the string. */
    RXP_ENCODE_RINT32 (encoder->cur_pos, len);
    encoder->cur_pos += 4;

    strncpy (encoder->cur_pos, str, len);
    encoder->cur_pos += len;
}


void
rxp_encode_data (RXpEncoder *encoder, void *data, int len)
{
    /* Update length */
    encoder->length += 8;
    encoder->length += len;
    
    RXP_ENCODER_MAYBE_EXPAND (encoder);
    
    /* Update total length at start of block */
    RXP_ENCODE_RINT32 (encoder->data, encoder->length);

    /* Encode the data type */
    RXP_ENCODE_RINT32 (encoder->cur_pos, RXP_TYPE_DATA);
    encoder->cur_pos += 4;

    /* Length of the data. */
    RXP_ENCODE_RINT32 (encoder->cur_pos, len);
    encoder->cur_pos += 4;

    memcpy (encoder->cur_pos, data, len);
    encoder->cur_pos += len;
}


void
rxp_encoder_free (RXpEncoder *encoder)
{
    rmem_free (encoder->data);
    rchunk_free (encoder, sizeof (RXpEncoder));
}


RXpDecoder *
rxp_decoder_new (char *data, int buflen)
{
    RXpDecoder *decoder;


    decoder = rchunk_alloc (sizeof (RXpDecoder));
    decoder->data = data;
    decoder->cur_pos = decoder->data;
    decoder->buflen = buflen;

    RXP_DECODE_RINT32 (data, decoder->length);
    decoder->cur_pos += 4;

    return (decoder);
}

void
rxp_decoder_reset (RXpDecoder *decoder, char *data, int length)
{
    decoder->data = data;
    decoder->cur_pos = decoder->data;
    decoder->buflen = length;

    RXP_DECODE_RINT32 (data, decoder->length);
    decoder->cur_pos += 4;
}


void
rxp_decoder_free (RXpDecoder *decoder)
{
    rchunk_free (decoder, sizeof (RXpDecoder));
}


unsigned int
rxp_decoder_msg_length (RXpDecoder *decoder)
{
    return (decoder->length);
}


RXpDataType
rxp_decode_peek_next_type (RXpDecoder *decoder)
{
    RXpDataType type;
    

    if (!decoder)
        return 0;

    if (decoder->cur_pos - decoder->data + 4 > decoder->length)
        return 0;

    RXP_DECODE_RINT32 (decoder->cur_pos, type);

    return type;
}

int 
rxp_decode_type (RXpDecoder *decoder, RXpDataType *type)
{
    RXpDataType tmptype;


    if (!decoder)
        return RXP_ERROR_NULL;

    if (decoder->cur_pos - decoder->data + 4 > decoder->length)
        return RXP_ERROR_LENGTH;

    RXP_DECODE_RINT32 (decoder->cur_pos, tmptype);
    /* No type checking on getting the type. :) */
    *type = tmptype;

    decoder->cur_pos += 4;

    return 0;
}

int 
rxp_decode_rint32 (RXpDecoder *decoder, rint32 *value)
{
    RXpDataType type;


    if (!decoder)
        return RXP_ERROR_NULL;

    if (decoder->cur_pos - decoder->data + 4 > decoder->length)
        return RXP_ERROR_LENGTH;

    RXP_DECODE_RINT32 (decoder->cur_pos, type);
    if (type != RXP_TYPE_INT32)
        return RXP_ERROR_TYPE;

    if (decoder->cur_pos - decoder->data + 8 > decoder->length)
        return RXP_ERROR_LENGTH;
    decoder->cur_pos += 4;

    RXP_DECODE_RINT32 (decoder->cur_pos, *value);
    decoder->cur_pos += 4;

    return 0;
}

int 
rxp_decode_ruint32 (RXpDecoder *decoder, ruint32 *value)
{
    RXpDataType type;


    if (!decoder)
        return RXP_ERROR_NULL;

    if (decoder->cur_pos - decoder->data + 4 > decoder->length)
        return RXP_ERROR_LENGTH;

    RXP_DECODE_RINT32 (decoder->cur_pos, type);
    if (type != RXP_TYPE_UINT32)
        return RXP_ERROR_TYPE;

    if (decoder->cur_pos - decoder->data + 8 > decoder->length)
        return RXP_ERROR_LENGTH;

    decoder->cur_pos += 4;

    RXP_DECODE_RINT32 (decoder->cur_pos, *value);
    decoder->cur_pos += 4;

    return 0;
}

int 
rxp_decode_float (RXpDecoder *decoder, float *value)
{
    RXpDataType type;
    ruint32 u32;

    if (!decoder)
        return RXP_ERROR_NULL;

    if (decoder->cur_pos - decoder->data + 4 > decoder->length)
        return RXP_ERROR_LENGTH;

    RXP_DECODE_RINT32 (decoder->cur_pos, type);
    if (type != RXP_TYPE_FLOAT)
        return RXP_ERROR_TYPE;

    if (decoder->cur_pos - decoder->data + 8 > decoder->length)
        return RXP_ERROR_LENGTH;

    decoder->cur_pos += 4;

    memcpy (&u32, decoder->cur_pos, sizeof (ruint32));
    *value = rxp_float_from_le_bits (u32);
    decoder->cur_pos += sizeof (ruint32);

    return 0;
}


int 
rxp_decode_double (RXpDecoder *decoder, double *value)
{
    RXpDataType type;
    ruint8 bytes[8];


    if (!decoder)
        return RXP_ERROR_NULL;

    if (decoder->cur_pos - decoder->data + 4 > decoder->length)
        return RXP_ERROR_LENGTH;

    RXP_DECODE_RINT32 (decoder->cur_pos, type);
    if (type != RXP_TYPE_DOUBLE)
        return RXP_ERROR_TYPE;

    if (decoder->cur_pos - decoder->data + 4 + (8 * sizeof (ruint8)) 
        > decoder->length)
        return RXP_ERROR_LENGTH;

    decoder->cur_pos += 4;

    memcpy (bytes, decoder->cur_pos, 8 * sizeof (ruint8));
    *value = rxp_double_from_bits (bytes);

    decoder->cur_pos += 8 * sizeof (ruint8);

    return 0;
}


int 
rxp_decode_rbuf (RXpDecoder *decoder, RBuf **value)
{
    RXpDataType type;
    unsigned int buflen;


    if (!decoder)
        return RXP_ERROR_NULL;

    if (decoder->cur_pos - decoder->data + 4 > decoder->length)
        return RXP_ERROR_LENGTH;

    RXP_DECODE_RINT32 (decoder->cur_pos, type);
    decoder->cur_pos += 4;

    /* The NULL RBuf case. */
    if (type == RXP_TYPE_NULL_RBUF) {
        *value = NULL;
        return 0;
    }

    if (type != RXP_TYPE_RBUF)
        return RXP_ERROR_TYPE;

    /* Space for rbuf_len */
    if (decoder->cur_pos - decoder->data + 4 > decoder->length)
        return RXP_ERROR_LENGTH;

    RXP_DECODE_RINT32 (decoder->cur_pos, buflen);
    decoder->cur_pos += 4;

    if (decoder->cur_pos - decoder->data + buflen > decoder->length) {
        return RXP_ERROR_LENGTH;
    }

    *value = rbuf_new_with_data (decoder->cur_pos, buflen);
    decoder->cur_pos += buflen;

    return 0;
}


int 
rxp_decode_str (RXpDecoder *decoder, char *buf, int len)
{
    RXpDataType type;
    int decodelen;


    if (!decoder)
        return RXP_ERROR_NULL;

    if (decoder->cur_pos - decoder->data + 4 > decoder->length)
        return RXP_ERROR_LENGTH;

    RXP_DECODE_RINT32 (decoder->cur_pos, type);
    if (type != RXP_TYPE_STR)
        return RXP_ERROR_TYPE;

    if (decoder->cur_pos - decoder->data + 8 > decoder->length)
        return RXP_ERROR_LENGTH;

    RXP_DECODE_RINT32 (decoder->cur_pos + 4, decodelen);

    /* Do they have enough space? */
    if (decodelen > len + 1)
        return RXP_ERROR_LENGTH;

    /* Do WE have enough space? */
    if (decoder->cur_pos - decoder->data + 8 + decodelen > decoder->length)
        return RXP_ERROR_LENGTH;

    decoder->cur_pos += 8;
  
    memcpy (buf, decoder->cur_pos, decodelen);
    buf [decodelen] = '\0';

    decoder->cur_pos += decodelen;
  
    return 0;
}


int 
rxp_decode_data (RXpDecoder *decoder, void *buf, int buflen, int *len)
{
    RXpDataType type;
    int decodelen;


    if (!decoder)
        return RXP_ERROR_NULL;

    if (decoder->cur_pos - decoder->data + 4 > decoder->length)
        return RXP_ERROR_LENGTH;

    RXP_DECODE_RINT32 (decoder->cur_pos, type);
    if (type != RXP_TYPE_DATA)
        return RXP_ERROR_TYPE;

    if (decoder->cur_pos - decoder->data + 8 > decoder->length)
        return RXP_ERROR_LENGTH;

    RXP_DECODE_RINT32 (decoder->cur_pos + 4, decodelen);

    /* Do they have enough space? */
    if (decodelen > buflen)
        return RXP_ERROR_LENGTH;

    /* Do WE have enough space? */
    if (decoder->cur_pos - decoder->data + 8 + decodelen > decoder->length)
        return RXP_ERROR_LENGTH;

    decoder->cur_pos += 8;
  
    memcpy (buf, decoder->cur_pos, decodelen);
    *len = decodelen;
  
    decoder->cur_pos += decodelen;
  
    return 0;
}

