/*Simple daemon logger for RSBAC
 ***************************************************
  You must change capabilities list for this program 
  to grand operations with RSBAC log
 ***************************************************
  rklog (c) - Stanislav I. Ievlev, 2000
  RSBAC (c) - Amon Ott., 1995-2000
  Some part of code from klogd - (c) 1995  Martin Schulze <Martin.Schulze@Linux.DE>
  Some changes made by Amon Ott, 2000
*/

/* Includes. */
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <sys/fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <time.h>
#include <stdarg.h>
#include <paths.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netdb.h>

#include <rsbac/syscalls.h>
#include <pwd.h>

#include "pidfile.h"

#ifndef _PATH_RSBAC_LOG
#define _PATH_RSBAC_LOG "/proc/rsbac-info/rmsg"
#endif

#ifndef _PATH_RSBAC_OUT
#define _PATH_RSBAC_OUT "/security-out"
#endif

#ifndef _PATH_PIDFILE
#define _PATH_PIDFILE "/var/run/rklogd.pid"
#endif

#ifndef _SECOFFUID
#define _SECOFFUID 400
#endif

#define LOG_LINE_SIZE 4096
#define LOG_LINE_LENGTH 1000
/*port for server mode*/
#define PORTNUM  1500  

/*define for testing*/
//#define TEST 
#define MAX_NAME 200

int	rsbac_msg=NULL  ; /*message file*/
FILE	*rsbac_out=NULL ; /*log file*/
FILE	*rsbac_out2=NULL ; /*second log file for network*/

int    use_proc=1;	/*use proc entry by default*/
int    netlisten=0;	/*don't listten network by default*/
int    network=0;	/*don't send messages to network by default*/
int    use_sound=0;     /*don't make sound*/

/*network part*/
char   *hostname;	/*name of the remote host*/
int    s,cl_s;		/*socket ID for network listening and writing*/
struct sockaddr_in serv_addr, clnt_addr; /*client and server data structures*/
extern int errno;

/*forward declarations*/
void stop_work(int);
void write_message_log();

/*init client*/
void init_client(char *servername)
{
struct hostent *hp;

/*fill remove server data structures*/
if((hp=gethostbyname(servername))==0)
{
perror("gethostbyname()");
exit(1);
}

bzero(&serv_addr, sizeof(serv_addr));
bcopy(hp->h_addr,&serv_addr.sin_addr,hp->h_length);

serv_addr.sin_family=hp->h_addrtype;
serv_addr.sin_port=htons(PORTNUM);

#ifdef TEST
fprintf(stderr,"Server Adress: %s\n",inet_ntoa(serv_addr.sin_addr));
#endif

/*create client socket*/
if((cl_s=socket(AF_INET, SOCK_DGRAM, 0))==-1){
perror("socket()");
exit(1);
}

/*fill client data*/
bzero(&clnt_addr, sizeof(clnt_addr));
clnt_addr.sin_family=AF_INET;
clnt_addr.sin_addr.s_addr=INADDR_ANY;

if ( bind(cl_s, (struct sockaddr_in *)&clnt_addr, sizeof(clnt_addr))==-1){
perror("bind()");
exit(1);
}
#ifdef TEST
fprintf(stderr,"Client Adress: %s\n ",inet_ntoa(clnt_addr.sin_addr));
fprintf(stderr,"Client ready for sending messages...\n");
#endif
}

/*write message to remote log-server*/
void write_message_network(char *tmpstring)
{
if(sendto(cl_s,tmpstring,LOG_LINE_SIZE,0,
   (struct sockaddr*)&serv_addr,sizeof(serv_addr))!=LOG_LINE_SIZE)
   {
#ifdef TEST
/*   fprintf(stderr,"Error sending message\n");*/
   perror("sendto()");
#endif
   syslog(LOG_INFO, "rklogd: Can't send message to network.");
   }
#ifdef TEST
   fprintf(stderr,"Sending data to network...\n");
#endif
}

/*init log-server*/
void init_server()
{
/*define UDP socket */
if((s=socket(AF_INET, SOCK_DGRAM, 0))==-1){
  perror("socket()");
  exit(1);
 }

/*create server data*/
bzero(&serv_addr, sizeof(serv_addr));
serv_addr.sin_family=AF_INET;
serv_addr.sin_addr.s_addr=INADDR_ANY;
serv_addr.sin_port=htons(PORTNUM);
/*and bind it together*/
if( bind(s, (struct sockaddr_in *)&serv_addr, sizeof(serv_addr))==-1){
  perror("bind()");
  exit(1);
 }

#ifdef TEST
fprintf(stderr,"server ready:%s\n",inet_ntoa(serv_addr.sin_addr));
#endif

}

/*read message from remote RSBAC host*/
char *read_message_server()
{
int maxaddrlen=sizeof(clnt_addr);
int nbytes;
char *tmpstring=malloc(LOG_LINE_SIZE);

memset(tmpstring,'\0',LOG_LINE_SIZE);
nbytes=recvfrom(s,tmpstring,LOG_LINE_SIZE-1,0,
                (struct sockaddr *)&clnt_addr,&maxaddrlen);
if (nbytes<0){
      printf("Receive error\n");
      exit(1);
  }

#ifdef TEST
printf("Receive data from client: %s: %s\n",
        inet_ntoa(clnt_addr.sin_addr),tmpstring);
#endif
return tmpstring;
}

/*init log files*/
void init_files(char *log_filename)
{
char *tmpstring=NULL;
	/*open message file*/
    if (use_proc) /*use proc*/{
	if  ((rsbac_msg = open(_PATH_RSBAC_LOG, O_RDONLY)) < 0 )
	{
		fprintf(stderr, "rklogd: Cannot open proc file system , " \
			"%d - %s.\n", errno, strerror(errno));
			
		exit(1);
	};
    }else{ /*use kernel*/
	if (rsbac_log(1,tmpstring,LOG_LINE_SIZE) <0)
	{
		fprintf(stderr, "rklogd: Cannot open kernel log buffer , " \
			"%d - %s.\n", errno, strerror(errno));
	               
	    exit(1);
	};
    }
    
	
	/*open log file*/
	if ( (rsbac_out = fopen(log_filename, "a")) < 0 )
	{
		fprintf(stderr, "rklogd: Cannot open log file , " \
			"%d - %s.\n", errno, strerror(errno));
			
		exit(1);
	}
	chmod(log_filename,0666);
	
	if (netlisten){
	    strncat(log_filename,"-fromnet",8);
	    if ( (rsbac_out2 = fopen(log_filename, "a")) < 0 )
		{
			fprintf(stderr, "rklogd: Cannot open log file , " \
		    	"%d - %s.\n", errno, strerror(errno));
			exit(1);
		}
	    chmod(log_filename,0666);
	}
	
        
}

/*read next message from kernel*/
char *read_message_kernel()
{
auto int rdcnt;
char *tmpstring=malloc(LOG_LINE_SIZE);

	/*
	 * Zero-fill the log buffer.  This should cure a multitude of
	 * problems with klogd logging the tail end of the message buffer
	 * which will contain old messages.  Then read the kernel log
	 * messages into this fresh buffer.
	 */
	memset(tmpstring, '\0', LOG_LINE_SIZE);
	if ( (rdcnt = rsbac_log(2, tmpstring, LOG_LINE_SIZE-1)) < 0 )
	{
		if ( errno == EINTR )
			return (char *)0;
#ifdef TEST
	fprintf(stderr, "rklogd: Cannot read from kernel , " \
                 			"%d - %s.\n", errno, strerror(errno));
#endif
	syslog(LOG_INFO, "rklogd: Can't read from kernel.");
	}
	
#ifdef TEST	
        fprintf(stderr,"Receive string from kernel: %s",tmpstring);
#endif

	return tmpstring;

}
/*read next message from kernel using /proc */
char *read_message_proc()
{
auto int rdcnt;
char *tmpstring=malloc(LOG_LINE_SIZE);


/*
* Zero-fill the log buffer.  This should cure a multitude of
* problems with klogd logging the tail end of the message buffer
* which will contain old messages.  Then read the kernel messages
* from the message pseudo-file into this fresh buffer.
*/


	memset(tmpstring, '\0', LOG_LINE_SIZE);
	if ( (rdcnt = read(rsbac_msg, tmpstring, LOG_LINE_SIZE-1)) < 0 )
	 switch ( errno) {
	  case  EINTR : 
	                return (char *)0; /*error*/
          default:
#ifdef TEST
                        fprintf(stderr, "rklogd: Cannot read from proc , %d - %s.\n", errno, strerror(errno));
#endif
                        syslog(LOG_INFO, "rklogd: Can't read from proc file system.");
	                break;
	   }/*switch&if*/

#ifdef TEST	
        fprintf(stderr,"Receive string from proc: %s",tmpstring);
#endif

  return tmpstring; /*OK I read message*/
}

/*stop all works*/
void stop_work(int sig)
{
char *tmpstring=NULL;
        syslog(LOG_INFO,"security kernel logger stopped");
	
	if (rsbac_out!=NULL) fclose(rsbac_out);
	if (use_proc)
          if (rsbac_msg!=NULL) close(rsbac_msg); /*close proc*/
	else
	  rsbac_log(0, tmpstring, LOG_LINE_SIZE ); /*close kernel log*/
	/*close system log*/
	closelog();
#ifdef TEST
        fprintf(stderr,"exiting...");
#endif

        remove_pid(_PATH_PIDFILE);
	
	exit (0);
}

/*write message to log-file*/
void write_message_log(char *tmpstring,int net)
{
char tmpstr[LOG_LINE_SIZE+20];
time_t now;
char timestamp[20];
char *workstr=tmpstring,*workstr1=NULL;
int length,new_length;
/*make time stamp*/
now=time(0);
strncpy(timestamp,ctime(&now),19);

if (net) {
	   fprintf(rsbac_out2,"%s: %s: %s\n",inet_ntoa(clnt_addr.sin_addr),timestamp,tmpstring);
	 }else{
	 
	    /*split message into separate lines*/

	    length=strlen(workstr);
	    while (length){
	    /*point to future line*/
	        workstr1=strchr(workstr,'\n');
		if (workstr1!=NULL) {
		    workstr1++;
	    	    new_length=strlen(workstr1);
	    	    strncpy(tmpstr,workstr,(length-new_length-1));
	    	    tmpstr[length-new_length-1]='\0';
	    
	    	    length=new_length;
	    	    workstr=workstr1;

	    	    fprintf(rsbac_out,"%s : %s\n",timestamp,tmpstr);
		    }else{
		    fprintf(rsbac_out,"%s : %s\n",timestamp,workstr);
		    length=0;
		    }
		if (use_sound) 
		if (strstr(tmpstr,"NOT_GRANTED")!=NULL)
		{
		    fprintf(stderr,"\a");
		    fflush(stderr);
		    usleep(200000);
		    
		    fprintf(stderr,"\a");
		    fflush(stderr);
		    usleep(200000);
		    
		    fprintf(stderr,"\a");
		    fflush(stderr);
		}
		if (network) write_message_network(tmpstr);
	        }/*while*/
	 }/*if*/
/*flush file stream*/
if (net){ 
    fflush(rsbac_out2);
    }else{
    fflush(rsbac_out);
    }/*if*/
    free(tmpstring);
}

/*smart log name selector*/
char *getlogname(uid_t secoffuid)
{
char *tmpstring=malloc(MAX_NAME);
struct passwd *mypasswd;
mypasswd=getpwuid(secoffuid);
strncpy(tmpstring,mypasswd->pw_dir,MAX_NAME);
strncat(tmpstring,"/log/",MAX_NAME);
strncat(tmpstring,_PATH_RSBAC_OUT,MAX_NAME);

return tmpstring;
}

/*main function*/
int main(int argc,char *argv[])
{
int i,ch;
uid_t secoffuid = _SECOFFUID; /*UID of security officer*/
char *logname=NULL;
char *tmpstring=malloc(LOG_LINE_SIZE);

/*open central system log for bug errors*/
openlog("rklogd",LOG_PID,LOG_AUTH);

while ((ch = getopt(argc, argv, "aspf:u:ln:")) != EOF)
    switch((char)ch)
	{

	    case 'a':		/* Make sound on failures*/
		use_sound = 1;
		break;

	    case 's':		/* Use kernel syscall*/
		use_proc = 0;
		break;
	    case 'p':		/* Use proc */
		use_proc = 1;
		break;
	    case 'f':		/* Define an output file. */
		logname = optarg;
		break;
	    case 'u':		/* Define an output file. */
		secoffuid = strtol(optarg,0,0);
		break;
	    case 'l':           /*Listen network for messages*/
	        if (network) {
		fprintf(stderr,"You cannot use l and n options together.\n");
		exit(1);
		}
	        netlisten = 1;
		break;
	    case 'n':           /*Use network for sending messages*/
	        if (netlisten) {
		fprintf(stderr,"You cannot use l and n options together.\n");
		exit(1);
		}
	        network = 1;
		hostname=optarg;
		break;
	}
	
	/*analize options*/
	if (logname == NULL) 
	          logname = getlogname(secoffuid);
	 

#ifdef TEST
fprintf(stderr,"modes:\n");

if (use_proc)
 fprintf(stderr,"read from proc\n");
else
 fprintf(stderr,"read from kernel\n");
 
if (netlisten)
 fprintf(stderr,"listen for network connections\n");
 
if (network)
 fprintf(stderr,"coping messages to network:%s\n",hostname);

 
 fprintf(stderr,"log file name:%s\n",logname);
#endif
/*signals handlers*/
#ifndef TEST
for (i = 1; i < NSIG; i++)	signal(i, SIG_IGN); //ignore all signals.

signal(SIGINT, stop_work); 
signal(SIGKILL, stop_work);
signal(SIGTERM, stop_work);

/*make a daemon*/
if ( fork() == 0 )
    {
	/*we need stay in root for create pid file*/	 
	if (!check_pid(_PATH_PIDFILE))
	{
		if (!write_pid(_PATH_PIDFILE))
			stop_work(9);
	}
	else
	{
		fputs("rklogd: Already running.\n", stderr);
		stop_work(9);
	}

      /*change uid to secoff*/
      if (setuid(secoffuid)<0){
      fprintf(stderr,"rklogd:Cannot change UID to %u\n", secoffuid);
      exit(1);
      }
      /* This is the child closing its file descriptors. */
      
      /*we need secoff for do it*/
      init_files(logname);
      if (netlisten) init_server(); 
      if (network) init_client(hostname); 
	
      fprintf(stderr,"\a");
      
      fclose(stdout);
      //fclose(stderr);
      fclose(stdin);

      setsid();
    }else /*parent process*/
    exit(0);
#endif

#ifdef TEST
/*change uid to secoff*/
if (setuid(secoffuid)<0){
  fprintf(stderr,"rklogd:Cannot change UID to %u\n", secoffuid);
  exit(1);
}
init_files(logname);
if (netlisten) init_server();
if (network) init_client(hostname); 
#endif

/*report to system log*/
syslog(LOG_INFO,"security kernel logger started.");

/*I have to make fork:
 First part to reading from proc/kernel
 Second part to reading from network
*/
if  ( fork() == 0 ) { 
#ifdef TEST
 fprintf(stderr,"Second process started...\n");
#endif
/*second process: read from network and write to other file*/
 if (!(netlisten)) {
#ifdef TEST
   fprintf(stderr,"No netlisten support...exited\n");
#endif
   exit(0); /*if not return*/
 }
 /*read data from network*/
 while(1)
 {
 /*read next line from net*/
 tmpstring=read_message_server();
 write_message_log(tmpstring,1);
 }
}else{
#ifdef TEST
 fprintf(stderr,"Main process started...\n");
#endif
/*first process: read from proc/kernel and write to file*/
 /*go to infinity loop*/
  while (1) {
  /* read next line from kernel/proc */
    switch (use_proc){
     case 0:    tmpstring=read_message_kernel();
                break;
     case 1:    tmpstring=read_message_proc();
                break;
     default:    pause();
                break;
     }/* switch() */
   write_message_log(tmpstring,0);
   }/*while*/
   syslog(LOG_INFO,"strange bug.");
   closelog();

   return 0;

}
return 0;
}

