/************************************* */
/* Rule Set Based Access Control       */
/* Author and (c) 1999-2005:           */
/*   Amon Ott <ao@rsbac.org>           */
/* Helper functions for all parts      */
/* Last modified: 13/Jun/2005          */
/************************************* */

#ifndef __KERNEL__
#include <stdlib.h>
#endif
#include <rsbac/types.h>
#include <rsbac/error.h>
#include <rsbac/helpers.h>
#include <rsbac/rc_types.h>
#include <rsbac/getname.h>
#include <rsbac/cap_getname.h>

#ifdef __KERNEL__
#include <linux/kernel.h>
#include <linux/module.h>
#include <asm/uaccess.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <rsbac/aci.h>
#include <rsbac/rkmem.h>
#include <rsbac/debug.h>
#ifdef CONFIG_RSBAC_RC
#include <rsbac/rc_getname.h>
#endif
#endif
#ifndef __KERNEL__
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <pwd.h>
#include <grp.h>
#endif

#ifndef __KERNEL__
int rsbac_u32_compare(__u32 * a, __u32 * b)
  {
    if(*a < *b)
     return -1;
    if(*a > *b)
      return 1;
    return 0;
  }

int rsbac_u32_void_compare(const void *a, const void *b)
{
	return rsbac_u32_compare((__u32 *) a, (__u32 *) b);
}

int rsbac_user_compare(const void * a, const void * b)
  {
    return rsbac_u32_compare((__u32 *) a, (__u32 *) b);
  }

int rsbac_group_compare(const void * a, const void * b)
  {
    return rsbac_u32_compare((__u32 *) a, (__u32 *) b);
  }

int rsbac_nettemp_id_compare(const void * a, const void * b)
  {
    return rsbac_u32_compare((__u32 *) a, (__u32 *) b);
  }

int rsbac_dev_compare(const void * desc1,
                      const void * desc2)
  {
    const struct rsbac_dev_desc_t * i_desc1 = desc1;
    const struct rsbac_dev_desc_t * i_desc2 = desc2;
    int result;
    
    result = memcmp(&i_desc1->type,
                    &i_desc2->type,
                    sizeof(i_desc1->type));
    if(result)
      return result;
    result = memcmp(&i_desc1->major,
                    &i_desc2->major,
                    sizeof(i_desc1->major));
    if(result)
      return result;
    return memcmp(&i_desc1->minor,
                  &i_desc2->minor,
                  sizeof(i_desc1->minor));
  }
#endif

char * inttostr(char * str, int i)
  {
    int j = 0;
    
    if(!str)
      return(NULL);
      
    if (i<0)
      {
        str[j] = '-';
        j++;
        i = -i;
      }
    if (i>=10000)
      {
        str[j] = '0' + (i / 10000);
        j++;
      }
    if (i>=1000)
      {
        str[j] = '0' + ((i % 10000) / 1000);
        j++;
      }
    if (i>=100)
      {
        str[j] = '0' + ((i % 1000) / 100);
        j++;
      }
    if (i>=10)
      {
        str[j] = '0' + ((i % 100) / 10);
        j++;
      }
    str[j] = '0' + (i % 10);
    j++;
    str[j] = 0;
    return (str);
  };

char * ulongtostr(char * str, u_long i)
  {
    int    j = 0;
    u_long k = 1000000000;
    
    if(!str)
      return(NULL);

    if (i>=k)
      {
        str[j] = '0' + ((i / k) % 100);
        j++;
      }
    k /= 10;

    while (k>1)
      {
        if (i>=k)
          {
            str[j] = '0' + ((i % (k*10)) / k);
            j++;
          }
        k /= 10;
      };

    str[j] = '0' + (i % 10);
    j++;
    str[j] = 0;
    return (str);
  };

char * longtostr(char * str, long i)
  {
    int    j = 0;
    u_long k = 1000000000;
    
    if(!str)
      return(NULL);

    if (i<0)
      {
        str[0] = '-';
        j = 1;
        i = -i;
      }
    if (i>=k)
      {
        str[j] = '0' + ((i / k) % 100);
        j++;
      }
    k /= 10;

    while (k>1)
      {
        if (i>=k)
          {
            str[j] = '0' + ((i % (k*10)) / k);
            j++;
          }
        k /= 10;
      };

    str[j] = '0' + (i % 10);
    j++;
    str[j] = 0;
    return (str);
  };

char * u64tostrmac(char * str, __u64 i)
  {
    int    j = 0;
    __u64  k;

    if(!str)
      return(NULL);

    k = 1;
    for(j = RSBAC_MAC_MAX_CAT;j >= 0;j--)
      {
        if (i & k)
          str[j] = '1';
        else
          str[j] = '0';
        k<<=1;
      };

    str[RSBAC_MAC_NR_CATS] = 0;
    return (str);
  };

#ifndef __KERNEL__

int rsbac_lib_version(void)
  {
    return RSBAC_VERSION_NR;
  }

void error_exit(int error)
  {
    char tmp1[80];

    if(error<0)
      {
        get_error_name(tmp1,error);
        fprintf(stderr, "Error: %s\n", tmp1);
        exit(1);
      }
  }

void show_error(int error)
  {
    char tmp1[80];

    if(error<0)
      {
        get_error_name(tmp1,error);
        fprintf(stderr, "Error: %s\n", tmp1);
      }
  }

int rsbac_get_uid_name(rsbac_uid_t * uid, char * name, char * sourcename)
  {
    struct passwd * user_info_p;
    rsbac_uid_t uid_i;

    if(!(user_info_p = getpwnam(sourcename)))
      {
        uid_i = strtoul(sourcename,0,10);
        if(   !uid_i
           && strcmp("0", sourcename)
          )
          {
            return -RSBAC_EINVALIDVALUE;
          }
        if(name)
          {
            if((user_info_p = getpwuid(uid_i)))
              strcpy(name, user_info_p->pw_name);
            else
              sprintf(name, "%u", uid_i);
          }
      }
    else
      {
        uid_i = user_info_p->pw_uid;
        if(name)
          strcpy(name, user_info_p->pw_name);
      }
    if(uid)
      *uid = uid_i;
    return 0;
  }

int rsbac_get_fullname(char * fullname, rsbac_uid_t uid)
  {
    struct passwd * user_info_p;
    rsbac_uid_t uid_i;

    if(!fullname)
      return -RSBAC_EINVALIDPOINTER;
    if(!(user_info_p = getpwuid(uid)))
      {
        sprintf(fullname, "%u", uid);
      }
    else
      {
        strcpy(fullname, user_info_p->pw_gecos);
      }
    return 0;
  }

char * get_user_name(rsbac_uid_t user, char * name)
  {
    struct passwd * user_info_p;

    if((user_info_p = getpwuid(user)))
      {
        strcpy(name, user_info_p->pw_name);
      }
    else
      {
        sprintf(name, "%u", user);
      }
    return name;
  }

char * get_group_name(rsbac_gid_t group, char * name)
  {
    struct group * group_info_p;

    if((group_info_p = getgrgid(group)))
      {
        strcpy(name, group_info_p->gr_name);
      }
    else
      {
        sprintf(name, "%u", group);
      }
    return name;
  }

int rsbac_get_gid_name(rsbac_gid_t * gid, char * name, char * sourcename)
  {
    struct group * group_info_p;
    rsbac_gid_t gid_i;

    if(!(group_info_p = getgrnam(sourcename)))
      {
        gid_i = strtoul(sourcename,0,10);
        if(   !gid_i
           && strcmp("0", sourcename)
          )
          {
            return -RSBAC_EINVALIDVALUE;
          }
        if(name)
          {
            if((group_info_p = getgrgid(gid_i)))
              strcpy(name, group_info_p->gr_name);
            else
              sprintf(name, "%u", gid_i);
          }
      }
    else
      {
        gid_i = group_info_p->gr_gid;
        if(name)
          strcpy(name, group_info_p->gr_name);
      }
    if(gid)
      *gid = gid_i;
    return 0;
  }


char * u64tostrlog(char * str, __u64 i)
  {
    int    j = 0;
    __u64  k;

    if(!str)
      return(NULL);

    k = 1;
    for(j = R_NONE - 1;j >= 0;j--)
      {
        if (i & k)
          str[j] = '1';
        else
          str[j] = '0';
        k<<=1;
      };

    str[R_NONE] = 0;
    return (str);
  };

__u64 strtou64log(char * str, __u64 * i_p)
  {
    int    j;
    __u64  k = 1, res=0;
    
    if(!str)
      return(0);

    if (strlen(str) < R_NONE)
      return(-1);
    for(j=R_NONE-1;j>=0;j--)
      {
        if(str[j] != '0')
          {
            res |= k;
          }
        k <<= 1;
      }
    for(j=R_NONE;j<64;j++)
      {
        res |= k;
        k <<= 1;
      }
    *i_p = res;
    return(res);
  };

char * u64tostrrc(char * str, __u64 i)
  {
    int    j = 0;
    __u64  k;

    if(!str)
      return(NULL);

    k = 1;
    for(j = 63;j >= 0;j--)
      {
        if (i & k)
          str[j] = '1';
        else
          str[j] = '0';
        k<<=1;
      };

    str[64] = 0;
    return (str);
  };

__u64 strtou64rc(char * str, __u64 * i_p)
  {
    int    j;
    __u64  k = 1, res=0;
    
    if(!str)
      return(0);

    if (strlen(str) < 64)
      return(-1);
    for(j=63;j>=0;j--)
      {
        if(str[j] != '0')
          {
            res |= k;
          }
        k <<= 1;
      }
    *i_p = res;
    return(res);
  };

char * u64tostrrcr(char * str, __u64 i)
  {
    int    j = 0;
    __u64  k;

    if(!str)
      return(NULL);

    k = 1;
    for(j = RCR_NONE - 1;j >= 0;j--)
      {
        if (i & k)
          str[j] = '1';
        else
          str[j] = '0';
        k<<=1;
      };

    str[RCR_NONE] = 0;
    return (str);
  };

__u64 strtou64rcr(char * str, __u64 * i_p)
  {
    int    j;
    __u64  k = 1, res=0;
    
    if(!str)
      return(0);

    if (strlen(str) < RCR_NONE)
      return(-1);
    for(j=RCR_NONE-1;j>=0;j--)
      {
        if(str[j] != '0')
          {
            res |= k;
          }
        k <<= 1;
      }
    for(j=RCR_NONE;j<64;j++)
      {
        res |= k;
        k <<= 1;
      }
    *i_p = res;
    return(res);
  };

__u64 strtou64mac(char * str, __u64 * i_p)
  {
    int    j;
    __u64  k = 1, res=0;
    
    if(!str)
      return(0);

    if (strlen(str) < RSBAC_MAC_NR_CATS)
      return(-1);
    for(j=RSBAC_MAC_MAX_CAT;j>=0;j--)
      {
        if(str[j] != '0')
          {
            res |= k;
          }
        k <<= 1;
      }
    for(j=RSBAC_MAC_NR_CATS;j<64;j++)
      {
        res |= k;
        k <<= 1;
      }
    *i_p = res;
    return(res);
  };

__u64 strtou64acl(char * str, __u64 * i_p)
  {
    int    j;
    __u64  k = 1, res=0;
    
    if(!str)
      return(0);

    if (strlen(str) < (ACLR_NONE - 1))
      return(-1);
    for(j=ACLR_NONE-1;j>=0;j--)
      {
        if(str[j] != '0')
          {
            res |= k;
          }
        k <<= 1;
      }
    for(j=ACLR_NONE-1;j<64;j++)
      {
        res |= k;
        k <<= 1;
      }
    *i_p = res;
    return(res);
  }

int strtodevdesc(char * str, struct rsbac_dev_desc_t * dev_p)
  {
    char * p;
    char * c;

    if(!str)
      return -RSBAC_EINVALIDVALUE;
    if(!strcmp(str, ":DEFAULT:"))
      {
        *dev_p = RSBAC_ZERO_DEV_DESC;
        return 0;
      }
    p = str;
    c = strchr(p,':');
    switch(*p)
      {
        case 'b':
        case 'B':
          if(c)
            dev_p->type = D_block;
          else
            dev_p->type = D_block_major;
          break;
        case 'c':
        case 'C':
          if(c)
            dev_p->type = D_char;
          else
            dev_p->type = D_char_major;
          break;
        default:
          return -RSBAC_EINVALIDTARGET;
      }
    p++;
    dev_p->major = strtoul(p,0,0);
    if(c)
      {
        c++;
        dev_p->minor = strtoul(c,0,0);
      }
    else
      dev_p->minor = 0;
    return 0;
  }

char * devdesctostr(char * str, struct rsbac_dev_desc_t dev)
  {
    if(RSBAC_IS_ZERO_DEV_DESC(dev))
      {
        sprintf(str, ":DEFAULT:");
        return str;
      }
    switch(dev.type)
      {
        case D_block:
        case D_char:
          sprintf(str, "%c%u:%u", 'b' + dev.type, dev.major, dev.minor);
          break;
        case D_block_major:
        case D_char_major:
          sprintf(str, "%c%u",
                  'b' + dev.type - (D_block_major - D_block),
                  dev.major);
          break;
        default:
          sprintf(str, "invalid!");
      }
    return str;
  }
#endif /* ifndef __KERNEL__ */

char * u64tostracl(char * str, __u64 i)
  {
    int    j = 0;
    __u64  k;

    if(!str)
      return(NULL);

    k = 1;
    for(j = ACLR_NONE - 1;j >= 0;j--)
      {
        if (i & k)
          str[j] = '1';
        else
          str[j] = '0';
        k<<=1;
      };

    str[ACLR_NONE] = 0;
    return (str);
  };

char * u32tostrcap(char * str, __u32 i)
  {
    int    j = 0;
    __u32  k;

    if(!str)
      return(NULL);

    k = 1;
    for(j = CAP_NONE - 1;j >= 0;j--)
      {
        if (i & k)
          str[j] = '1';
        else
          str[j] = '0';
        k<<=1;
      };

    str[CAP_NONE] = 0;
    return (str);
  };

__u32 strtou32cap(char * str, __u32 * i_p)
  {
    int    j;
    __u32  k = 1, res=0;
    
    if(!str)
      return(0);

    if (strlen(str) < CAP_NONE)
      return(-1);
    for(j=CAP_NONE-1;j>=0;j--)
      {
        if(str[j] != '0')
          {
            res |= k;
          }
        k <<= 1;
      }
    for(j=CAP_NONE;j<32;j++)
      {
        res |= k;
        k <<= 1;
      }
    *i_p = res;
    return(res);
  };


#ifdef __KERNEL__

/* find the current owner of this process */
int rsbac_get_owner(rsbac_uid_t * user_p)
  {
    *user_p = current->uid;
    return(0);
  }

void rsbac_ds_get_error(char * function, enum rsbac_attribute_t attr)
  {
    if(!function)
      return;
    if(attr != A_none)
      {
        char tmp[80];

        get_attribute_name(tmp, attr);
        rsbac_printk(KERN_WARNING
                     "%s: rsbac_get_attr() for %s returned error!\n",
                     function, tmp);
      }
    else
      {
        rsbac_printk(KERN_WARNING
                     "%s: rsbac_get_attr() returned error!\n",
                     function);
      }
  }

void rsbac_ds_get_error_num(char * function, enum rsbac_attribute_t attr, int err)
  {
    if(!function)
      return;
    if(attr != A_none)
      {
        char tmp[80];
        char tmp2[80];

        get_attribute_name(tmp, attr);
        get_error_name(tmp2, err);
        rsbac_printk(KERN_WARNING
                     "%s: rsbac_get_attr() for %s returned error %s!\n",
                     function, tmp, tmp2);
      }
    else
      {
        rsbac_printk(KERN_WARNING
                     "%s: rsbac_get_attr() returned error!\n",
                     function);
      }
  }

void rsbac_ds_set_error(char * function, enum rsbac_attribute_t attr)
  {
    if(!function)
      return;
    if(attr != A_none)
      {
        char tmp[80];

        get_attribute_name(tmp, attr);
        rsbac_printk(KERN_WARNING
                     "%s: rsbac_set_attr() for %s returned error!\n",
                     function, tmp);
      }
    else
      {
        rsbac_printk(KERN_WARNING
                     "%s: rsbac_set_attr() returned error!\n",
                     function);
      }
  }

void rsbac_ds_set_error_num(char * function, enum rsbac_attribute_t attr, int err)
  {
    if(!function)
      return;
    if(attr != A_none)
      {
        char tmp[80];
        char tmp2[80];

        get_attribute_name(tmp, attr);
        get_error_name(tmp2, err);
        rsbac_printk(KERN_WARNING
                     "%s: rsbac_set_attr() for %s returned error %s!\n",
                     function, tmp, tmp2);
      }
    else
      {
        rsbac_printk(KERN_WARNING
                     "%s: rsbac_set_attr() returned error!\n",
                     function);
      }
  }

#ifdef CONFIG_RSBAC_RC
void rsbac_rc_ds_get_error(char * function, enum rsbac_rc_item_t item)
  {
    if(!function)
      return;
    if(item != RI_none)
      {
        char tmp[80];

        get_rc_item_name(tmp, item);
        rsbac_printk(KERN_WARNING
                     "%s: rsbac_rc_get_item() for %s returned error!\n",
                     function, tmp);
      }
    else
      {
        rsbac_printk(KERN_WARNING
                     "%s: rsbac_rc_get_item() returned error!\n",
                     function);
      }
  }

void rsbac_rc_ds_set_error(char * function, enum rsbac_rc_item_t item)
  {
    if(!function)
      return;
    if(item != RI_none)
      {
        char tmp[80];

        get_rc_item_name(tmp, item);
        rsbac_printk(KERN_WARNING
                     "%s: rsbac_rc_set_item() for %s returned error!\n",
                     function, tmp);
      }
    else
      {
        rsbac_printk(KERN_WARNING
                     "%s: rsbac_rc_set_item() returned error!\n",
                     function);
      }
  }
#endif


/****************************************************************/
/* Access to user data space                                    */


#if defined(CONFIG_RSBAC_REG) || defined(CONFIG_RSBAC_REG_MAINT)
EXPORT_SYMBOL(rsbac_get_user);
#endif
int rsbac_get_user(unsigned char * kern_p, unsigned char * user_p, int size)
  {
    if(kern_p && user_p && (size > 0))
      {
        return copy_from_user(kern_p, user_p, size);
      }
    return(0);
  }


#if defined(CONFIG_RSBAC_REG) || defined(CONFIG_RSBAC_REG_MAINT)
EXPORT_SYMBOL(rsbac_put_user);
#endif
int rsbac_put_user(unsigned char * kern_p, unsigned char * user_p, int size)
  {
    if(kern_p && user_p && (size > 0))
      {
        return copy_to_user(user_p,kern_p,size);
      }
    return(0);
  };

#if defined(CONFIG_RSBAC_REG) || defined(CONFIG_RSBAC_REG_MAINT)
EXPORT_SYMBOL(rsbac_getname);
#endif
char * rsbac_getname(const char * name)
  {
    return getname(name);
  };

#if defined(CONFIG_RSBAC_REG) || defined(CONFIG_RSBAC_REG_MAINT)
EXPORT_SYMBOL(rsbac_putname);
#endif
void rsbac_putname(const char * name)
  {
    putname(name);
  }

inline int clear_user_buf(char * ubuf, int len)
  {
    return clear_user(ubuf,len);
  }

#endif /* __KERNEL__ */
