/*************************************************** */
/* Rule Set Based Access Control                     */
/*                                                   */
/* Author and (c) 1999-2005: Amon Ott <ao@rsbac.org> */
/*                                                   */
/* Last modified: 30/Aug/2005                        */
/*************************************************** */

#include <stdio.h>
#include <stdlib.h>
#include <termios.h>
#include <unistd.h>
#include <sys/mman.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <rsbac/types.h>
#include <rsbac/syscalls.h>
#include <rsbac/error.h>
#include "nls.h"
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

int main(int argc, char ** argv)
{
  int res = 0;
  char * progname;
  rsbac_uid_t user = RSBAC_NO_USER;
  int verbose = 0;
  int noold = 0;
  int err;
  char * old_pass;
  char * new_pass;
  char * new_pass2;
  struct termios old_term;
  struct termios tmp_term;

  locale_init();
  progname = argv[0];

  while((argc > 1) && (argv[1][0] == '-'))
    {
      char * pos = argv[1];
      pos++;
      while(*pos)
        {
          switch(*pos)
            {
              case 'v':
                verbose++;
                break;

              case 'n':
                noold = 1;
                break;

              case 'h':
                printf(gettext("%s (RSBAC %s)\n***\n"), argv[0], VERSION);
                printf(gettext("Use: %s [flags] [username]\n"), progname);
                printf(gettext("  -v = verbose,\n"));
                printf(gettext("  -n = do not ask for old password\n"));
                exit(0);

              default:
                fprintf(stderr, gettext("%s: unknown parameter %c\n"), progname, *pos);
                exit(1);
            }
          pos++;
        }
      argv++;
      argc--;
    }

  if (argc > 1)
    {
      if(rsbac_um_get_uid(0, argv[1], &user))
        {
          user = strtoul(argv[1],0,0);
          if(!user && strcmp(argv[1],"0"))
            {
              fprintf(stderr, gettext("%s: Unknown user %s\n"), progname, argv[1]);
              exit(1);
            }
        }
    }
  else
    user=getuid();

  if(isatty(STDIN_FILENO))
    {
      res = tcgetattr(STDIN_FILENO, &old_term);
      error_exit(res);
      memcpy(&tmp_term, &old_term, sizeof(old_term));
      tmp_term.c_lflag &= ~(ECHO);
    }
  if(noold)
    old_pass = NULL;
  else
    {
     old_pass = malloc(RSBAC_MAXNAMELEN);
     res = mlock(old_pass, RSBAC_MAXNAMELEN);
     if (res) {
	     fprintf(stderr, gettext("Unable to lock old password into physical memory!\n"));
     }
     if(isatty(STDIN_FILENO))
        {
          res = tcsetattr(STDIN_FILENO, TCSAFLUSH, &tmp_term);
          error_exit(res);
        }
      if(argc > 1)
        printf("Old RSBAC password for user %s (uid %u): ", argv[1], user);
      else
        printf("Old RSBAC password for user %u: ", user);
      res = scanf("%254s", old_pass);
      if(isatty(STDIN_FILENO))
        tcsetattr(STDIN_FILENO, TCSAFLUSH, &old_term);
      printf("\n");
      if(res <= 0)
        {
          fprintf(stderr, gettext("%s: invalid old password!\n"), progname);
          exit(1);
        }
    }
  new_pass = malloc(RSBAC_MAXNAMELEN);
  res = mlock(new_pass, RSBAC_MAXNAMELEN);
  if (res) {
	  fprintf(stderr, gettext("Unable to lock new password into physical memory!\n"));
  }
  if(noold)
    {
      if(argc > 1)
        printf("New RSBAC password for user %s (uid %u): ", argv[1], user);
      else
        printf("New RSBAC password for user %u: ", user);
    }
  else
    printf("New password: ");
  if(isatty(STDIN_FILENO))
    {
      res = tcsetattr(STDIN_FILENO, TCSAFLUSH, &tmp_term);
      error_exit(res);
    }
  res = scanf("%254s", new_pass);
  if(isatty(STDIN_FILENO))
    tcsetattr(STDIN_FILENO, TCSAFLUSH, &old_term);
  printf("\n");
  if(res <= 0)
    {
      fprintf(stderr, gettext("%s: invalid new password!\n"), progname);
      exit(1);
    }
  new_pass2 = malloc(RSBAC_MAXNAMELEN);
  res = mlock(new_pass2, RSBAC_MAXNAMELEN);
  if (res) {
	  fprintf(stderr, gettext("Unable to lock retyped new password into physical memory!\n"));
  }
  printf("Repeat new password: ");
  if(isatty(STDIN_FILENO))
    tcsetattr(STDIN_FILENO, TCSAFLUSH, &tmp_term);
  res = scanf("%254s", new_pass2);
  if(isatty(STDIN_FILENO))
    tcsetattr(STDIN_FILENO, TCSAFLUSH, &old_term);
  printf("\n");
  if(res <= 0)
    {
      fprintf(stderr, gettext("%s: invalid repeated new password!\n"), progname);
      exit(1);
    }
  if(strcmp(new_pass, new_pass2))
    {
      fprintf(stderr, gettext("%s: new passwords do not match!\n"), progname);
      exit(1);
    }

  res = rsbac_um_set_pass(user, old_pass, new_pass);
  error_exit(res);
  exit(0);
  return (res);
}
