/* $Id: threadsafety.c 2076 2008-12-28 15:52:34Z dreibh $
 * --------------------------------------------------------------------------
 *
 *              //===//   //=====   //===//   //       //   //===//
 *             //    //  //        //    //  //       //   //    //
 *            //===//   //=====   //===//   //       //   //===<<
 *           //   \\         //  //        //       //   //    //
 *          //     \\  =====//  //        //=====  //   //===//    Version II
 *
 * ------------- An Efficient RSerPool Prototype Implementation -------------
 *
 * Copyright (C) 2002-2009 by Thomas Dreibholz
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: dreibh@iem.uni-due.de
 */

#include "tdtypes.h"
#include "loglevel.h"
#include "threadsafety.h"

#include <unistd.h>


static unsigned long long gMutexCounter = 0;


/* ###### Constructor #################################################### */
void threadSafetyNew(struct ThreadSafety* threadSafety,
                     const char*          name)
{
#if !defined(__APPLE__) && !defined(__FreeBSD__)
   pthread_mutexattr_t attr;
   pthread_mutexattr_init(&attr);
   pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_RECURSIVE_NP);
   pthread_mutex_init(&threadSafety->Mutex, &attr);
   pthread_mutexattr_destroy(&attr);
#else
   threadSafety->MutexOwner          = 0;
   threadSafety->MutexRecursionLevel = 0;
   pthread_mutex_init(&threadSafety->Mutex, NULL);
#endif
   snprintf((char*)&threadSafety->Name, sizeof(threadSafety->Name), "%llu-%s",
            gMutexCounter++, name);
   if(threadSafety != &gLogMutex) {
      LOG_MUTEX
      fprintf(stdlog, "Created mutex \"%s\"\n", threadSafety->Name);
      LOG_END
   }
}


/* ###### Destructor ##################################################### */
void threadSafetyDelete(struct ThreadSafety* threadSafety)
{
   if(threadSafety != &gLogMutex) {
      LOG_MUTEX
      fprintf(stdlog, "Destroying mutex \"%s\"...\n", threadSafety->Name);
      LOG_END
   }
   pthread_mutex_destroy(&threadSafety->Mutex);
   if(threadSafety != &gLogMutex) {
      LOG_MUTEX
      fprintf(stdlog, "Destroyed mutex \"%s\"\n", threadSafety->Name);
      LOG_END
   }
}


/* ###### Lock ########################################################### */
void threadSafetyLock(struct ThreadSafety* threadSafety)
{
   if(threadSafety != &gLogMutex) {
      LOG_MUTEX
      fprintf(stdlog, "Locking mutex \"%s\"...\n", threadSafety->Name);
      LOG_END
   }
#ifndef __APPLE__
   pthread_mutex_lock(&threadSafety->Mutex);
#else
   if(!pthread_equal(threadSafety->MutexOwner, pthread_self())) {
      pthread_mutex_lock(&threadSafety->Mutex);
      threadSafety->MutexOwner = pthread_self();
   }
   threadSafety->MutexRecursionLevel++;
#endif
   if(threadSafety != &gLogMutex) {
      LOG_MUTEX
#ifndef __APPLE__
      fprintf(stdlog, "Locked mutex \"%s\"\n", threadSafety->Name);
#else
      fprintf(stdlog, "Locked mutex \"%s\", recursion level %d\n",
              threadSafety->Name, threadSafety->MutexRecursionLevel);
#endif
      LOG_END
   }
}


/* ###### Unlock ######################################################### */
void threadSafetyUnlock(struct ThreadSafety* threadSafety)
{
   if(threadSafety != &gLogMutex) {
      LOG_MUTEX
#ifndef __APPLE__
      fprintf(stdlog, "Unlocking mutex \"%s\"...\n", threadSafety->Name);
#else
      fprintf(stdlog, "Unlocking mutex \"%s\", recursion level %d...\n",
              threadSafety->Name, threadSafety->MutexRecursionLevel);
#endif
      LOG_END
   }
#ifndef __APPLE__
   pthread_mutex_unlock(&threadSafety->Mutex);
#else
   if(threadSafety->MutexRecursionLevel == 0) {
      LOG_ERROR
      fputs("Mutex is already unlocked!\n",stdlog);
      LOG_END
      exit(1);
   }
   if(pthread_equal(threadSafety->MutexOwner, pthread_self())) {
      threadSafety->MutexRecursionLevel--;
      if(threadSafety->MutexRecursionLevel == 0) {
         threadSafety->MutexOwner = 0;
         pthread_mutex_unlock(&threadSafety->Mutex);
      }
   }
   else {
      LOG_ERROR
      fputs("Mutex is not owned!\n",stdlog);
      LOG_END
   }
#endif
   if(threadSafety != &gLogMutex) {
      LOG_MUTEX
      fprintf(stdlog, "Unlocked mutex \"%s\"\n", threadSafety->Name);
      LOG_END
   }
}
