/***************************************************************************
 * RT2400 SourceForge Project - http://rt2400.sourceforge.net              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 *                                                                         *
 *   Licensed under the GNU GPL                                            *
 *   Original code supplied under license from RaLink Inc, 2003.           *
 ***************************************************************************/

 /***************************************************************************
 *      Module Name: sanity.c
 *
 *      Abstract:
 *
 *      Revision History:
 *      Who             When            What
 *      --------        -----------     -----------------------------
 *      MarkW           9th  Feb 04     Baseline of code
 ***************************************************************************/


#include "rt_config.h"

/*
    ==========================================================================
    Description:
        MLME message sanity check
    Return:
        TRUE if all parameters are OK, FALSE otherwise
    ==========================================================================
 */
BOOLEAN ScanReqParmSanity(
    IN PRTMP_ADAPTER pAd,
    IN VOID *Msg,
    IN ULONG MsgLen,
    OUT UCHAR *BssType,
    OUT CHAR Ssid[],
    OUT UCHAR *SsidLen,
    OUT UCHAR *ScanType)
{
    MLME_SCAN_REQ_STRUCT *Info;

    Info = (MLME_SCAN_REQ_STRUCT *)(Msg);
    *BssType = Info->BssType;
    *SsidLen = Info->SsidLen;
    memcpy(Ssid, Info->Ssid, *SsidLen);
    *ScanType = Info->ScanType;

    if ((*BssType == BSS_INFRA || *BssType == BSS_INDEP || *BssType == BSS_ANY) &&
       (*ScanType == SCAN_ACTIVE || *ScanType == SCAN_PASSIVE))
        return TRUE;
    else
    {
        DBGPRINT(RT_DEBUG_TRACE, "ScanReqParmSanity fail - wrong BssType or ScanType\n");
        return FALSE;
    }
}

/*
    ==========================================================================
    Description:
        MLME message sanity check
    Return:
        TRUE if all parameters are OK, FALSE otherwise
    ==========================================================================
 */
BOOLEAN StartParmSanity(
    IN PRTMP_ADAPTER pAd,
    IN VOID *Msg,
    IN ULONG MsgLen,
    OUT CHAR Ssid[],
    OUT UCHAR *SsidLen)
{
    MLME_START_REQ_STRUCT *Info;

    Info = (MLME_START_REQ_STRUCT *)(Msg);

    if (Info->SsidLen > MAX_LEN_OF_SSID)
    {
        DBGPRINT(RT_DEBUG_TRACE, "StartParmSanity fail - wrong SSID length\n");
        return FALSE;
    }

    *SsidLen = Info->SsidLen;
    memcpy(Ssid, Info->Ssid, *SsidLen);

    return TRUE;
}

/*
    ==========================================================================
    Description:
        MLME message sanity check
    Return:
        TRUE if all parameters are OK, FALSE otherwise
    ==========================================================================
 */
BOOLEAN MlmeAssocReqSanity(
    IN PRTMP_ADAPTER pAd,
    IN VOID *Msg,
    IN ULONG MsgLen,
    OUT MACADDR *ApAddr,
    OUT USHORT *CapabilityInfo,
    OUT ULONG *Timeout,
    OUT USHORT *ListenIntv)
{
    MLME_ASSOC_REQ_STRUCT *Info;

    Info = (MLME_ASSOC_REQ_STRUCT *)Msg;
    *Timeout = Info->Timeout;                             // timeout
    COPY_MAC_ADDR(ApAddr, &Info->Addr);                   // AP address
    *CapabilityInfo = Info->CapabilityInfo;               // capability info
    *ListenIntv = Info->ListenIntv;

    return TRUE;
}

/*
    ==========================================================================
    Description:
        MLME message sanity check
    Return:
        TRUE if all parameters are OK, FALSE otherwise
    ==========================================================================
 */
BOOLEAN AuthReqParmSanity(
    IN PRTMP_ADAPTER pAd,
    IN VOID *Msg,
    IN ULONG MsgLen,
    OUT MACADDR *Addr,
    OUT ULONG *Timeout,
    OUT USHORT *Alg)
{
    MLME_AUTH_REQ_STRUCT *Info;

    Info  = (MLME_AUTH_REQ_STRUCT *)Msg;
    COPY_MAC_ADDR(Addr, &Info->Addr);
    *Timeout = Info->Timeout;
    *Alg = Info->Alg;

    if ((*Alg == AUTH_MODE_KEY || *Alg == AUTH_MODE_OPEN) && !MAC_ADDR_IS_GROUP(*Addr))
    {
        return TRUE;
    }
    else
    {
        DBGPRINT(RT_DEBUG_TRACE, "AuthReqParmSanity fail - wrong algorithm\n");
        return FALSE;
    }
}

/*
    ==========================================================================
    Description:
        MLME message sanity check
    Return:
        TRUE if all parameters are OK, FALSE otherwise
    ==========================================================================
 */
#if 0
BOOLEAN JoinParmSanity(
    IN PRTMP_ADAPTER pAd,
    IN VOID *Msg,
    IN ULONG MsgLen,
    OUT ULONG *BssIdx,
    OUT UCHAR SupportedRates[],
    OUT UCHAR *SupportedRatesLen)
{

    PBSS_ENTRY   pBss;
    MLME_JOIN_REQ_STRUCT *Info = (MLME_JOIN_REQ_STRUCT *)Msg;

    *BssIdx = Info->BssIdx;
    pBss = &pAd->Mlme.CntlAux.SsidBssTab.BssEntry[Info->BssIdx];

    *SupportedRatesLen = pBss->RatesLen;
    if (*SupportedRatesLen > MAX_LEN_OF_SUPPORTED_RATES)
    {
        DBGPRINT(RT_DEBUG_TRACE, ("JoinParmSanity fail - wrong SupportedRates IE\n"));
        return FALSE;
    }

    memcpy(SupportedRates, pBss->Rates, pBss->RatesLen);
    return TRUE;
}
#endif
/*
    ==========================================================================
    Description:
        MLME message sanity check
    Return:
        TRUE if all parameters are OK, FALSE otherwise
    ==========================================================================
 */
BOOLEAN PeerAssocParmSanity(
    IN PRTMP_ADAPTER pAd,
    IN VOID *Msg,
    IN ULONG MsgLen,
    OUT MACADDR *Addr2,
    OUT USHORT *CapabilityInfo,
    OUT USHORT *Status,
    OUT USHORT *Aid,
    OUT UCHAR Rates[],
    OUT UCHAR *RatesLen)
{
    CHAR          IeType, *Ptr;
    MACFRAME     *Fr = (MACFRAME *)Msg;

    COPY_MAC_ADDR(Addr2, &Fr->Hdr.Addr2);
    Ptr = Fr->Octet;

    memcpy(CapabilityInfo, &Fr->Octet[0], 2);
    memcpy(Status,         &Fr->Octet[2], 2);

    if (*Status == MLME_SUCCESS)
    {
        memcpy(Aid, &Fr->Octet[4], 2);
        *Aid = (*Aid) & 0x3fff; // AID is low 14-bit

        // -- get supported rates from payload and advance the pointer
        IeType = Fr->Octet[6];
        *RatesLen = Fr->Octet[7];
        if (IeType != IE_SUPP_RATES)
        {
            DBGPRINT(RT_DEBUG_TRACE, "PeerAssocParmSanity fail - wrong SupportedRates IE\n");
            return FALSE;
        }
        else
        {
            int		index;
            UCHAR	tmp_rate, i;

            i = 0;
			for (index = 0; index < *RatesLen; index++)
           	{
            	// Mask out basic rate set bit
            	tmp_rate = Fr->Octet[8 + index] & 0x7f;
            	if ((tmp_rate == 0x02) || (tmp_rate == 0x04) || (tmp_rate == 0x0B) || (tmp_rate == 0x16))
            		Rates[i++] = Fr->Octet[8 + index];	// Save rate with basic rate set bit if exists
            }
            *RatesLen = i;
        }
    }

    return TRUE;
}

/*
    ==========================================================================
    Description:
        MLME message sanity check
    Return:
        TRUE if all parameters are OK, FALSE otherwise
    ==========================================================================
 */
BOOLEAN PeerDisassocSanity(
    IN PRTMP_ADAPTER pAd,
    IN VOID *Msg,
    IN ULONG MsgLen,
    OUT MACADDR *Addr2,
    OUT USHORT *Reason)
{
    MACFRAME *Fr = (MACFRAME *)Msg;

    COPY_MAC_ADDR(Addr2, &Fr->Hdr.Addr2);
    memcpy(Reason, &Fr->Octet[0], 2);

    return TRUE;
}

/*
    ==========================================================================
    Description:
        MLME message sanity check
    Return:
        TRUE if all parameters are OK, FALSE otherwise
    ==========================================================================
 */
BOOLEAN PeerDeauthSanity(
    IN PRTMP_ADAPTER pAd,
    IN VOID *Msg,
    IN ULONG MsgLen,
    OUT MACADDR *Addr2,
    OUT USHORT *Reason)
{
    MACFRAME *Fr = (MACFRAME *)Msg;

    COPY_MAC_ADDR(Addr2, &Fr->Hdr.Addr2);
    memcpy(Reason, &Fr->Octet[0], 2);

    return TRUE;
}

/*
    ==========================================================================
    Description:
        MLME message sanity check
    Return:
        TRUE if all parameters are OK, FALSE otherwise
    ==========================================================================
 */
BOOLEAN PeerAuthSanity(
    IN PRTMP_ADAPTER pAd,
    IN VOID *Msg,
    IN ULONG MsgLen,
    OUT MACADDR *Addr,
    OUT USHORT *Alg,
    OUT USHORT *Seq,
    OUT USHORT *Status,
    CHAR *ChlgText)
{
    MACFRAME     *Fr = (MACFRAME *)Msg;

    COPY_MAC_ADDR(Addr,    &Fr->Hdr.Addr2);
    memcpy(Alg,    &Fr->Octet[0], 2);
    memcpy(Seq,    &Fr->Octet[2], 2);
    memcpy(Status, &Fr->Octet[4], 2);

    if (*Alg == AUTH_MODE_OPEN)
    {
        if (*Seq == 1 || *Seq == 2)
        {
            return TRUE;
        }
        else
        {
            DBGPRINT(RT_DEBUG_TRACE, "PeerAuthSanity fail - wrong Seg#\n");
            return FALSE;
        }
    }
    else if (*Alg == AUTH_MODE_KEY)
    {
        if (*Seq == 1 || *Seq == 4)
        {
            return TRUE;
        }
        else if (*Seq == 2 || *Seq == 3)
        {
            memcpy(ChlgText, &Fr->Octet[8], CIPHER_TEXT_LEN);
            return TRUE;
        }
        else
        {
            DBGPRINT(RT_DEBUG_TRACE, "PeerAuthSanity fail - wrong Seg#\n");
            return FALSE;
        }
    }
    else
    {
        DBGPRINT(RT_DEBUG_TRACE, "PeerAuthSanity fail - wrong algorithm\n");
        return FALSE;
    }
}

/*
    ==========================================================================
    Description:
        MLME message sanity check
    Return:
        TRUE if all parameters are OK, FALSE otherwise
    ==========================================================================
 */
BOOLEAN PeerProbeReqSanity(
    IN PRTMP_ADAPTER pAd,
    IN VOID *Msg,
    IN ULONG MsgLen,
    OUT MACADDR *Addr2,
    OUT CHAR Ssid[],
    OUT UCHAR *SsidLen,
    OUT UCHAR Rates[],
    OUT UCHAR *RatesLen)
{
    UCHAR Idx;
    CHAR          IeType;
    MACFRAME *Fr = (MACFRAME *)Msg;

    COPY_MAC_ADDR(Addr2, &Fr->Hdr.Addr2);

    if (Fr->Octet[0] != IE_SSID || Fr->Octet[1] > MAX_LEN_OF_SSID)
    {
        DBGPRINT(RT_DEBUG_TRACE, "PeerProbeReqSanity fail - wrong SSID IE\n");
        return FALSE;
    }

    *SsidLen = Fr->Octet[1];
    memcpy(Ssid, &Fr->Octet[2], *SsidLen);

    Idx = *SsidLen + 2;

    // -- get supported rates from payload and advance the pointer
    IeType = Fr->Octet[Idx];
    *RatesLen = Fr->Octet[Idx + 1];
    if (IeType != IE_SUPP_RATES)
    {
    	DBGPRINT(RT_DEBUG_TRACE, "PeerAssocParmSanity fail - wrong SupportedRates IE\n");
    	return FALSE;
    }
    else
    {
    	int 	index;
    	UCHAR	tmp_rate, i;

    	i = 0;
    	Idx += 2;
    	for (index = 0; index < *RatesLen; index++)
    	{
    		// Mask out basic rate set bit
    		tmp_rate = Fr->Octet[Idx + index] & 0x7f;
    		if ((tmp_rate == 0x02) || (tmp_rate == 0x04) || (tmp_rate == 0x0B) || (tmp_rate == 0x16))
    			Rates[i++] = Fr->Octet[Idx + index];	// Save rate with basic rate set bit if exists
    	}
    	*RatesLen = i;
	}

    return TRUE;
}

/*
    ==========================================================================
    Description:
        MLME message sanity check
    Return:
        TRUE if all parameters are OK, FALSE otherwise
    ==========================================================================
 */
BOOLEAN BeaconAndProbeRspSanity(
    IN PRTMP_ADAPTER pAd,
    IN VOID *Msg,
    IN ULONG MsgLen,
    OUT MACADDR *Addr2,
    OUT MACADDR *Bssid,
    OUT CHAR Ssid[],
    OUT UCHAR *SsidLen,
    OUT UCHAR *BssType,
    OUT USHORT *BeaconPeriod,
    OUT UCHAR *Channel,
    OUT LARGE_INTEGER *Timestamp,
    OUT BOOLEAN *CfExist,
    OUT CF_PARM *CfParm,
    OUT USHORT *AtimWin,
    OUT USHORT *CapabilityInfo,
    OUT UCHAR Rate[],
    OUT UCHAR *RateLen,
    OUT UCHAR *DtimCount,
    OUT UCHAR *DtimPeriod,
    OUT UCHAR *BcastFlag,
    OUT UCHAR *MessageToMe,
    OUT UCHAR *Legacy)
{
    CHAR				*Ptr, TimLen;
    MACFRAME			*Fr;
    PBEACON_EID_STRUCT	eid_ptr;
    UCHAR				SubType,CheckIdx=0;

    Fr = (MACFRAME *)Msg;

    // get subtype from header
    SubType = (UCHAR)Fr->Hdr.SubType;

    // get Addr2 and BSSID from header
    COPY_MAC_ADDR(Addr2, &Fr->Hdr.Addr2);
    COPY_MAC_ADDR(Bssid, &Fr->Hdr.Addr3);

    Ptr = Fr->Octet;

    // get timestamp from payload and advance the pointer
    memcpy(Timestamp, Ptr, TIMESTAMP_LEN);
    Ptr += TIMESTAMP_LEN;

    // get beacon interval from payload and advance the pointer
    memcpy(BeaconPeriod, Ptr, 2);
    Ptr += 2;

    // get capability info from payload and advance the pointer
    memcpy(CapabilityInfo, Ptr, 2);
    Ptr += 2;
    if (CAP_IS_ESS_ON(*CapabilityInfo))
    {
        *BssType = BSS_INFRA;
    }
    else
    {
        *BssType = BSS_INDEP;
    }

    eid_ptr = (PBEACON_EID_STRUCT) Ptr;

    // get variable fields from payload and advance the pointer
    while(((UCHAR*)eid_ptr + eid_ptr->Len + 1) < ((UCHAR*)Fr + MsgLen))
    {
        switch(eid_ptr->Eid)
        {
            case IE_SSID:
                if(eid_ptr->Len <= MAX_LEN_OF_SSID)
                {
                    CheckIdx++;
                    memcpy(Ssid, eid_ptr->Octet, eid_ptr->Len);
                    *SsidLen = eid_ptr->Len;
                }
                else
                {
                    DBGPRINT(RT_DEBUG_TRACE, "BeaconAndProbeRspSanity(IE_SSID) \n");
                    return FALSE;
                }
                break;

            case IE_SUPP_RATES:
            	// Add special care for b / g mixed mode AP which will send with more than 8 supported rates.
            	// We have to parse this rate one by one and make sure its 802.11 b supported.
            	{
            		int		index;
            		UCHAR	tmp_rate, i;
            		PUCHAR	eid_rate;

            		i = 0;
            		CheckIdx++;
            		eid_rate = eid_ptr->Octet;
            		for (index = 0; index < eid_ptr->Len; index++)
            		{
            			// Mask out basic rate set bit
            			tmp_rate = eid_rate[index] & 0x7f;
            			if ((tmp_rate == 0x02) || (tmp_rate == 0x04) || (tmp_rate == 0x0B) || (tmp_rate == 0x16))
            				Rate[i++] = eid_rate[index];	// Save rate with basic rate set bit if exists
            		}
            		*RateLen = i;
            	}
            	/*
                if(eid_ptr->Len <= MAX_LEN_OF_SUPPORTED_RATES)
                {
                    CheckIdx++;
                    memcpy(Rate, eid_ptr->Octet, eid_ptr->Len);
                    *RateLen = eid_ptr->Len;
                }
                else
                {
                    DBGPRINT(RT_DEBUG_TRACE, ("BeaconAndProbeRspSanity(IE_SUPP_RATES) \n"));
                    return FALSE;
                }
                */
                break;

            case IE_FH_PARM:
                DBGPRINT(RT_DEBUG_TRACE, "BeaconAndProbeRspSanity(IE_FH_PARM) \n");
                break;

            case IE_DS_PARM:
                if(eid_ptr->Len == 1)
                {
                    CheckIdx++;
                    *Channel = *eid_ptr->Octet;
                    if (ChannelSanity(pAd, *Channel) == 0)
                    {
                    	DBGPRINT(RT_DEBUG_TRACE, "BeaconAndProbeRspSanity(IE_DS_PARM)\n");
                    	return FALSE;
                    }
                }
                else
                {
                    DBGPRINT(RT_DEBUG_TRACE, "BeaconAndProbeRspSanity(IE_DS_PARM)\n");
                    return FALSE;
                }
                break;

            case IE_CF_PARM:
                if(eid_ptr->Len == 6)
                {
                    *CfExist = TRUE;
                    memcpy(CfParm, eid_ptr->Octet, eid_ptr->Len);
                }
                else
                {
                    DBGPRINT(RT_DEBUG_TRACE, "BeaconAndProbeRspSanity(IE_CF_PARM)\n");
                    return FALSE;
                }
                break;

            case IE_IBSS_PARM:
                if(eid_ptr->Len == 2)
                {
                    memcpy(AtimWin, eid_ptr->Octet, eid_ptr->Len);
                }
                else
                {
                    DBGPRINT(RT_DEBUG_TRACE, "BeaconAndProbeRspSanity(IE_IBSS_PARM)\n");
                    return FALSE;
                }
                break;

            case IE_TIM:
                if(INFRA_ON(pAd) && SubType == SUBTYPE_BEACON)
                {
                    GetTimBit((PUCHAR)eid_ptr, pAd->PortCfg.Aid, &TimLen, BcastFlag, DtimCount, DtimPeriod, MessageToMe);
                }
                break;

            default:
                DBGPRINT(RT_DEBUG_INFO, "BeaconAndProbeRspSanity Unrecognized EID = %d in Beacon or ProbeRsp\n", eid_ptr->Eid);
                break;
        }

        eid_ptr = (PBEACON_EID_STRUCT)((UCHAR*)eid_ptr + 2 + eid_ptr->Len);
    }

    if(CheckIdx < 3)
    {
        DBGPRINT(RT_DEBUG_WARN, "BeaconAndProbeRspSanity(Check mandatory field failed)\n");
        return FALSE;
    }
    else
    {
        return TRUE;
    }

}

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
BOOLEAN GetTimBit(
    IN CHAR *Ptr,
    IN USHORT Aid,
    OUT UCHAR *TimLen,
    OUT UCHAR *BcastFlag,
    OUT UCHAR *DtimCount,
    OUT UCHAR *DtimPeriod,
    OUT UCHAR *MessageToMe)
{
    UCHAR          BitCntl, N1, N2, MyByte, MyBit;
    CHAR          *IdxPtr;

    IdxPtr = Ptr;

    IdxPtr ++;
    *TimLen = *IdxPtr;

    // get DTIM Count from TIM element
    IdxPtr ++;
    *DtimCount = *IdxPtr;

    // get DTIM Period from TIM element
    IdxPtr++;
    *DtimPeriod = *IdxPtr;

    // get Bitmap Control from TIM element
    IdxPtr++;
    BitCntl = *IdxPtr;

    if ((*DtimCount == 0) && (BitCntl & 0x01))
        *BcastFlag = TRUE;
    else
        *BcastFlag = FALSE;

#if 1
    // Parse Partial Virtual Bitmap from TIM element
    N1 = BitCntl & 0xfe;    // N1 is the first bitmap byte#
    N2 = *TimLen - 4 + N1;  // N2 is the last bitmap byte#

    if ((Aid < (N1 << 3)) || (Aid >= ((N2 + 1) << 3)))
        *MessageToMe = FALSE;
    else
    {
        MyByte = (Aid >> 3) - N1;                       // my byte position in the bitmap byte-stream
        MyBit = Aid % 16 - ((MyByte & 0x01)? 8:0);

        IdxPtr += (MyByte + 1);

        //if (*IdxPtr)
        //    DBGPRINT(RT_DEBUG_WARN, ("TIM bitmap = 0x%02x\n", *IdxPtr));

        if (*IdxPtr & (0x01 << MyBit))
            *MessageToMe = TRUE;
        else
            *MessageToMe = FALSE;
    }
#endif

    return TRUE;
}


/*!
 *  \brief Get legacy bit, right now for 11b it is always 0
 *  \param
 *  \return TRUE if the parameters are OK, FALSE otherwise. Always return TRUE
 *  \pre
 *  \post
 */
BOOLEAN GetLegacy(
    IN CHAR *Ptr,
    OUT UCHAR *Legacy)
{
    *Legacy = 0;
    return TRUE;
}

UCHAR ChannelSanity(
    IN PRTMP_ADAPTER pAd,
    IN UCHAR channel)
{
	switch (pAd->PortCfg.CountryRegion)
	{
		case REGION_FCC:	// 1 - 11
			if ((channel > 0) && (channel < 12))
				return 1;
			break;

		case REGION_IC:		// 1 -11
			if ((channel > 0) && (channel < 12))
				return 1;
			break;

		case REGION_ETSI:	// 1 - 13
			if ((channel > 0) && (channel < 14))
				return 1;
			break;

		case REGION_SPAIN:	// 10 - 11
			if ((channel > 9) && (channel < 12))
				return 1;
			break;

		case REGION_FRANCE:	// 10 -13
			if ((channel > 9) && (channel < 14))
				return 1;
			break;

		case REGION_MKK:	// 14
			if (channel == 14)
				return 1;
			break;

		case REGION_MKK1:	// 1 - 14
			if ((channel > 0) && (channel < 15))
				return 1;
			break;

		case REGION_ISRAEL:	// 3 - 9
			if ((channel > 2) && (channel < 10))
				return 1;
			break;

		default:			// Error
			return 0;
	}
    return (0);
}
