/*
 * RT2X00 Compatability fixes for specific kernels.
 */
#ifndef RT2X00_COMPAT_H
#define RT2X00_COMPAT_H

/*
 * First include the 2 config headers.
 * The rt2x00_config.h should overrule
 * the kernel configuration.
 */
#include <linux/autoconf.h>
#include "rt2x00_config.h"

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>

/*
 * Check minimal requirements.
 */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 17)
#error Invalid kernel version!
#error Please refer to README file for minimal requirements.
#endif

#if !defined(CONFIG_NET_RADIO)
#error Wireless radio support not enabled in kernel!
#error Please refer to README file for minimal requirements.
#endif

#if (!defined(CONFIG_CRYPTO) && !defined(CONFIG_CRYPTO_MODULE))
#error Crypto support not enabled in kernel!
#error Please refer to README file for minimal requirements.
#endif

#if (!defined(CONFIG_CRYPTO_AES) && !defined(CONFIG_CRYPTO_AES_MODULE))
#if (!defined(CONFIG_CRYPTO_AES_586) && !defined(CONFIG_CRYPTO_AES_586_MODULE))
#if (!defined(CONFIG_CRYPTO_AES_X86_64) && !defined(CONFIG_CRYPTO_AES_X86_64_MODULE))
#error Crypto AES support not enabled in kernel!
#error Please refer to README file for minimal requirements.
#endif
#endif
#endif

#if (!defined(CONFIG_CRYPTO_ARC4) && !defined(CONFIG_CRYPTO_ARC4_MODULE))
#error Crypto ARC4 support not enabled in kernel!
#error Please refer to README file for minimal requirements.
#endif

#if (!defined(CONFIG_D80211) && !defined(CONFIG_D80211_MODULE))
#error d80211 support not enabled in kernel!
#error Please refer to README file for minimal requirements.
#endif

#if (defined(CONFIG_RT2400PCI) || defined(CONFIG_RT2500PCI) || defined(CONFIG_RT61PCI))
#if (!defined(CONFIG_PCI) && !defined(CONFIG_PCI_MODULE))
#error PCI has been disabled in your kernel!
#error Please refer to README file for minimal requirements.
#endif
#if (!defined(CONFIG_EEPROM_93CX6) && !defined(CONFIG_EEPROM_93CX6_MODULE))
#error EEPROM_93CX6 has been disabled in your kernel!
#error Please refer to README file for minimal requirements.
#endif
#endif

#if (defined(CONFIG_RT2500USB) || defined(CONFIG_RT73USB))
#if (!defined(CONFIG_USB) && !defined(CONFIG_USB_MODULE))
#error USB has been disabled in your kernel!
#error Please refer to README file for minimal requirements.
#endif
#endif

#if (defined(CONFIG_RT61PCI) || defined(CONFIG_RT73USB))
#if (!defined(CONFIG_FW_LOADER) && !defined(CONFIG_FW_LOADER_MODULE))
#error Firmware loading has been disabled in your kernel!
#error Please refer to README file for minimal requirements.
#endif
#if (!defined(CONFIG_CRC_ITU_T) && !defined(CONFIG_CRC_ITU_T_MODULE))
#error CRC_ITU_T loading has been disabled in your kernel!
#error Please refer to README file for minimal requirements.
#endif
#endif

#if (defined(CONFIG_RFKILL) || defined (CONFIG_RFKILL_MODULE))
#if (!defined(CONFIG_INPUT) && !defined(CONFIG_INPUT_MODULE))
#error Input device support has been disabled in your kernel!
#error Please refer to README file for minimal requirements.
#endif
#endif

#if (defined(CONFIG_RT2400PCI_BUTTON) || defined(CONFIG_RT2500PCI_BUTTON) || defined(CONFIG_RT61PCI_BUTTON))
#if (!defined(CONFIG_RFKILL) && !defined (CONFIG_RFKILL_MODULE))
#error RFKILL has been disabled in your kernel!
#error Please refer to README file for minimal requirements.
#endif
#endif

/*
 * Backward compatibility fixes for all kernels starting from 2.6.17
 */
#if LINUX_VERSION_CODE == KERNEL_VERSION(2, 6, 17)
/*
 * From 2.6.18 and above all access to the xmit_lock should
 * go through the netif_tx_lock_bh and netif_tx_unlock_bh functions.
 */
#define netif_tx_lock_bh(__dev)					\
	do {							\
		spin_lock_bh(&(__dev)->xmit_lock);		\
		(__dev)->xmit_lock_owner = smp_processor_id();	\
	} while (0)

#define netif_tx_unlock_bh(__dev)			\
	do {						\
		(__dev)->xmit_lock_owner = -1;		\
		spin_unlock_bh(&(__dev)->xmit_lock);	\
	} while (0)

/*
 * SA_SHIRQ has been deprecated in kernel 2.6.18
 */
#define IRQF_SHARED	SA_SHIRQ

/*
 * Simplified asprintf.
 * Please this in seperate define to prevent it being compiled
 * in when we are not compiling for rfkill.
 */
#if defined(CONFIG_RFKILL)
__attribute__((unused)) static char *kasprintf(gfp_t gfp, const char *fmt, ...)
{
	va_list ap;
	unsigned int len;
	char *p;

	va_start(ap, fmt);
	len = vsnprintf(NULL, 0, fmt, ap);
	va_end(ap);

	p = kmalloc(len+1, gfp);
	if (!p)
		return NULL;
	va_start(ap, fmt);
	vsnprintf(p, len+1, fmt, ap);
	va_end(ap);
	return p;
}
#endif
#endif /* LINUX_VERSION_CODE == KERNEL_VERSION(2, 6, 17) */

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2, 6, 18)
/*
 * Add backwards compatability checks for the new crypto api that arrived in kernel 2.6.19
 */
#include <linux/crypto.h>
#include <linux/scatterlist.h>
#include <linux/err.h>

#define crypto_blkcipher	crypto_tfm
#define crypto_cipher		crypto_tfm

#define AES_BLOCK_LEN 16
#define CRYPTO_ALG_ASYNC 0

struct blkcipher_desc {
	struct crypto_tfm *tfm;
};

static inline struct crypto_tfm *crypto_alloc_cipher(const char *alg_name, u32 tfm_flags, u32 new_flag)
{
	struct crypto_tfm *tfm;

	tfm = crypto_alloc_tfm(alg_name, 0);
	if (!tfm)
		return ERR_PTR(-ENOMEM);
	return tfm;
}

static inline struct crypto_tfm *crypto_alloc_blkcipher(const char *alg_name, u32 tfm_flags, u32 new_flag)
{
	struct crypto_tfm *tfm;
	
	if (strcmp("ecb(arc4)", alg_name) == 0)
		tfm = crypto_alloc_tfm("arc4", 0);
	else
		tfm = crypto_alloc_tfm(alg_name, 0);
	if (!tfm)
		return ERR_PTR(-ENOMEM);
	return tfm;
}

static inline void crypto_free_cipher(struct crypto_tfm *tfm)
{
	crypto_free_tfm(tfm);
}

static inline void crypto_free_blkcipher(struct crypto_tfm *tfm)
{
	crypto_free_tfm(tfm);
}

static inline void crypto_cipher_encrypt_one(struct crypto_tfm *tfm, u8 ct[16], const u8 pt[16])
{
	struct scatterlist src, dst;

	src.page = virt_to_page(pt);
	src.offset = offset_in_page(pt);
	src.length = AES_BLOCK_LEN;

	dst.page = virt_to_page(ct);
	dst.offset = offset_in_page(ct);
	dst.length = AES_BLOCK_LEN;

	crypto_cipher_encrypt(tfm, &dst, &src, AES_BLOCK_LEN);
}

static inline int crypto_blkcipher_setkey(struct crypto_tfm *tfm,
					  const u8 *rc4key, unsigned int klen)
{
	return crypto_cipher_setkey(tfm, rc4key, klen);
}

static inline int crypto_blkcipher_encrypt(struct blkcipher_desc *desc,
                                        struct scatterlist *dst,
                                        struct scatterlist *src,
                                        unsigned int nbytes)
{
	return crypto_cipher_encrypt(desc->tfm, dst, src, dst->length);
}

static inline int crypto_blkcipher_decrypt(struct blkcipher_desc *desc,
                                        struct scatterlist *dst,
                                        struct scatterlist *src,
                                        unsigned int nbytes)
{
	return crypto_cipher_decrypt(desc->tfm, dst, src, dst->length);
}

#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2, 6, 18) */

/*
 * The net_device ieee80211_ptr field is not available in any kernel
 * except for the dscape and wireless-dev git trees.
 * We are going to use the net_device base_addr field instead.
 */
#define get_80211_ptr(__dev)		( (void*)(__dev)->base_addr )
#define set_80211_ptr(__dev, __ptr)	( (__dev)->base_addr = (unsigned long)(__ptr) )

/*
 * For rfkill driver, this driver is not yet in the kernel,
 * and neither is this KEY define which should go into linux/input.h
 */
#define KEY_RFKILL		0x1a0

#endif /* RT2X00_COMPAT_H */
