/*
	Copyright (C) 2004 - 2006 rt2x00 SourceForge Project
	<http://rt2x00.serialmonkey.com>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
	Module: rt73usb
	Abstract: rt73usb device specific routines.
	Supported chipsets: RT2573.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/usb.h>
#include <linux/delay.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/wireless.h>
#include <linux/firmware.h>
#include <linux/ethtool.h>
#include <linux/crc-itu-t.h>

#include <net/iw_handler.h>
#include <net/d80211.h>

/*
 * Set enviroment defines for rt2x00.h
 */
#define DRV_NAME "rt73usb"

#ifdef CONFIG_RT73USB_DEBUG
#define CONFIG_RT2X00_DEBUG
#endif /* CONFIG_RT73USB_DEBUG */

#include "rt2x00.h"
#include "rt2x00usb.h"
#include "rt73usb.h"

/*
 * Register access.
 * All access to the CSR registers will go through the methods
 * rt2x00_register_read and rt2x00_register_write.
 * BBP and RF register require indirect register access,
 * and use the CSR registers BBPCSR and RFCSR to achieve this.
 * These indirect registers work with busy bits,
 * and we will try maximal REGISTER_BUSY_COUNT times to access
 * the register while taking a REGISTER_BUSY_DELAY us delay
 * between each attampt. When the busy bit is still set at that time,
 * the access attempt is considered to have failed,
 * and we will print an error.
 */
static int rt2x00_vendor_request(const struct rt2x00_dev *rt2x00dev,
	const u8 request, const u8 type, const u16 offset,
	u32 value, void *buffer, const u16 buffer_length, const u16 timeout)
{
	struct usb_device *usb_dev = interface_to_usbdev(
		rt2x00dev_usb(rt2x00dev));
	int status;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		status = usb_control_msg(
			usb_dev,
			(type == USB_VENDOR_REQUEST_IN) ?
				usb_rcvctrlpipe(usb_dev, 0) :
				usb_sndctrlpipe(usb_dev, 0),
			request, type, value, offset, buffer, buffer_length,
			timeout);
		if (status >= 0)
			return 0;
	}

	ERROR("vendor request error. Request 0x%02x failed "
		"for offset 0x%04x with error %d.\n", request, offset, status);

	return status;
}

static inline void rt2x00_register_read(
	const struct rt2x00_dev *rt2x00dev,
	const u16 offset, u32 *value)
{
	__le32 reg;
	rt2x00_vendor_request(
		rt2x00dev, USB_MULTI_READ, USB_VENDOR_REQUEST_IN,
		offset, 0x00, &reg, 4, REGISTER_TIMEOUT);
	*value = le32_to_cpu(reg);
}

static inline void rt2x00_register_multiread(
	const struct rt2x00_dev *rt2x00dev,
	const u16 offset, u32 *value, const u32 length)
{
	rt2x00_vendor_request(
		rt2x00dev, USB_MULTI_READ, USB_VENDOR_REQUEST_IN,
		offset, 0x00, value, length, REGISTER_TIMEOUT);
}

static inline void rt2x00_register_write(
	const struct rt2x00_dev *rt2x00dev,
	const u16 offset, u32 value)
{
	__le32 reg = cpu_to_le32(value);
	rt2x00_vendor_request(
		rt2x00dev, USB_MULTI_WRITE, USB_VENDOR_REQUEST_OUT,
		offset, 0x00, &reg, 4, REGISTER_TIMEOUT);
}

static inline void rt2x00_register_multiwrite(
	const struct rt2x00_dev *rt2x00dev,
	const u16 offset, u32 *value, const u32 length)
{
	rt2x00_vendor_request(
		rt2x00dev, USB_MULTI_WRITE, USB_VENDOR_REQUEST_OUT,
		offset, 0x00, value, length, REGISTER_TIMEOUT);
}

static u32 rt2x00_bbp_check(const struct rt2x00_dev *rt2x00dev)
{
	u32 reg;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00dev, PHY_CSR3, &reg);
		if (!rt2x00_get_field32(reg, PHY_CSR3_BUSY))
			return reg;
		udelay(REGISTER_BUSY_DELAY);
	}

	return 0xffff;
}

static void rt2x00_bbp_write(const struct rt2x00_dev *rt2x00dev,
	const u8 reg_id, const u8 value)
{
	u32 reg;

	/*
	 *  Wait untill the BBP becomes ready.
	 */
	if (rt2x00_bbp_check(rt2x00dev) == 0xffff) {
		ERROR("PHY_CSR3 register busy. Write failed.\n");
		return;
	}

	/*
	 * Write the data into the BBP.
	 */
	reg = 0;
	rt2x00_set_field32(&reg, PHY_CSR3_VALUE, value);
	rt2x00_set_field32(&reg, PHY_CSR3_REGNUM, reg_id);
	rt2x00_set_field32(&reg, PHY_CSR3_BUSY, 1);
	rt2x00_set_field32(&reg, PHY_CSR3_READ_CONTROL, 0);

	rt2x00_register_write(rt2x00dev, PHY_CSR3, reg);
}

static void rt2x00_bbp_read(const struct rt2x00_dev *rt2x00dev,
	const u8 reg_id, u8 *value)
{
	u32 reg;

	/*
	 *  Wait untill the BBP becomes ready.
	 */
	if (rt2x00_bbp_check(rt2x00dev) == 0xffff) {
		ERROR("PHY_CSR3 register busy. Read failed.\n");
		return;
	}

	/*
	 * Write the request into the BBP.
	 */
	reg =0;
	rt2x00_set_field32(&reg, PHY_CSR3_REGNUM, reg_id);
	rt2x00_set_field32(&reg, PHY_CSR3_BUSY, 1);
	rt2x00_set_field32(&reg, PHY_CSR3_READ_CONTROL, 1);

	rt2x00_register_write(rt2x00dev, PHY_CSR3, reg);

	/*
	 *  Wait untill the BBP becomes ready.
	 */
	reg = rt2x00_bbp_check(rt2x00dev);
	if (reg == 0xffff)
		ERROR("PHY_CSR3 register busy. Read failed.\n");

	*value = rt2x00_get_field32(reg, PHY_CSR3_VALUE);
}

static void rt2x00_rf_write(const struct rt2x00_dev *rt2x00dev,
	const u32 value)
{
	u32 reg;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00dev, PHY_CSR4, &reg);
		if (!rt2x00_get_field32(reg, PHY_CSR4_BUSY))
			goto rf_write;
		udelay(REGISTER_BUSY_DELAY);
	}

	ERROR("PHY_CSR4 register busy. Write failed.\n");
	return;

rf_write:
	reg = 0;
	rt2x00_set_field32(&reg, PHY_CSR4_VALUE, value);
	rt2x00_set_field32(&reg, PHY_CSR4_NUMBER_OF_BITS, 20);
	rt2x00_set_field32(&reg, PHY_CSR4_IF_SELECT, 0);
	rt2x00_set_field32(&reg, PHY_CSR4_BUSY, 1);

	rt2x00_register_write(rt2x00dev, PHY_CSR4, reg);
}

/*
 * EEPROM access.
 * The EEPROM is being accessed by word index.
 */
static void* rt2x00_eeprom_addr(const struct rt2x00_dev *rt2x00dev,
	const u8 word)
{
	return (void*)&rt2x00dev->eeprom[word];
}

static void rt2x00_eeprom_read(const struct rt2x00_dev *rt2x00dev,
	const u8 word, u16 *data)
{
	*data = le16_to_cpu(rt2x00dev->eeprom[word]);
}

static void rt2x00_eeprom_write(const struct rt2x00_dev *rt2x00dev,
	const u8 word, u16 data)
{
	rt2x00dev->eeprom[word] = cpu_to_le16(data);
}

/*
 * TX/RX Descriptor access functions.
 */
static inline struct urb* rt2x00_urb(struct data_entry *entry)
{
	return (struct urb*)entry->priv;
}

static inline void* rt2x00_data_addr(struct data_entry *entry)
{
	return entry->data_addr + entry->ring->desc_size;
}

static inline struct data_desc* rt2x00_desc_addr(struct data_entry *entry)
{
	return entry->data_addr;
}

static inline void rt2x00_desc_read(struct data_desc *desc,
	const u8 word, u32 *value)
{
	*value = le32_to_cpu(desc->word[word]);
}

static inline void rt2x00_desc_write(struct data_desc *desc,
	const u8 word, const u32 value)
{
	desc->word[word] = cpu_to_le32(value);
}

/*
 * Ethtool handlers.
 */
static void rt73usb_get_drvinfo(struct net_device *net_dev,
	struct ethtool_drvinfo *drvinfo)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_get_hw(net_dev)->priv;

	strcpy(drvinfo->driver, DRV_NAME);
	strcpy(drvinfo->version, DRV_VERSION);
	strcpy(drvinfo->fw_version, rt2x00_fw(&rt2x00dev->chip));
	strcpy(drvinfo->bus_info, rt2x00dev_usb(rt2x00dev)->dev.bus_id);
	drvinfo->eedump_len = EEPROM_SIZE;
	drvinfo->regdump_len = CSR_REG_SIZE;
}

static int rt73usb_get_regs_len(struct net_device *net_dev)
{
	return CSR_REG_SIZE;
}

static void rt73usb_get_regs(struct net_device *net_dev,
	struct ethtool_regs *regs, void *data)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_get_hw(net_dev)->priv;

	rt2x00_register_multiread(rt2x00dev, CSR_REG_BASE, data, CSR_REG_SIZE);
	regs->version = rt2x00_rev(&rt2x00dev->chip);
	regs->len = CSR_REG_SIZE;
}

#ifdef CONFIG_RT73USB_DEBUG
static u32 rt73usb_get_msglevel(struct net_device *net_dev)
{
	return rt2x00_debug_level;
}

static void rt73usb_set_msglevel(struct net_device *net_dev, u32 msg)
{
	rt2x00_debug_level = !!msg;
}
#endif /* CONFIG_RT73USB_DEBUG */

static int rt73usb_get_eeprom_len(struct net_device *net_dev)
{
	return EEPROM_SIZE;
}

static int rt73usb_get_eeprom(struct net_device *net_dev,
	struct ethtool_eeprom *eeprom, u8 *data)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_get_hw(net_dev)->priv;

	eeprom->magic = 0xbad;

	memcpy(data,
		rt2x00_eeprom_addr(rt2x00dev, eeprom->offset / sizeof(u16)),
		eeprom->len);

	return 0;
}

static int rt73usb_set_eeprom(struct net_device *net_dev,
	struct ethtool_eeprom *eeprom, u8 *data)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_get_hw(net_dev)->priv;
	unsigned int i;
	unsigned int words;

	if (eeprom->magic != 0xbad)
		return -EINVAL;

	words = eeprom->len / sizeof(u16);
	for (i = eeprom->offset / sizeof(u16); i < words; i++)
		rt2x00_eeprom_write(rt2x00dev, i, ((u16*)data)[i]);

	return 0;
}

static struct ethtool_ops rt73usb_ethtool_ops = {
	.get_drvinfo	= rt73usb_get_drvinfo,
	.get_regs_len	= rt73usb_get_regs_len,
	.get_regs	= rt73usb_get_regs,
#ifdef CONFIG_RT73USB_DEBUG
	.get_msglevel	= rt73usb_get_msglevel,
	.set_msglevel	= rt73usb_set_msglevel,
#endif /* CONFIG_RT2500USB_DEBUG */
	.get_link	= ethtool_op_get_link,
	.get_eeprom_len	= rt73usb_get_eeprom_len,
	.get_eeprom	= rt73usb_get_eeprom,
	.set_eeprom	= rt73usb_set_eeprom,
	.get_perm_addr	= ethtool_op_get_perm_addr,
};

/*
 * Configuration handlers.
 */
static void rt73usb_config_bssid(struct rt2x00_dev *rt2x00dev, u8 *bssid)
{
	u32 reg[2] = { 0, 0 };

	/*
	 * The BSSID is passed to us as an array of bytes,
	 * that array is little endian, so no need for byte ordering.
	 * We only need to set the BSS ID MASK at the correct offset.
	 */
	memcpy(&reg, bssid, ETH_ALEN);
	rt2x00_set_field32(&reg[1], MAC_CSR5_BSS_ID_MASK, 3);
	rt2x00_register_multiwrite(rt2x00dev, MAC_CSR4, &reg[0], sizeof(reg));
}

static void rt73usb_config_promisc(struct rt2x00_dev *rt2x00dev, int promisc)
{
	u32 reg;

	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);

	if (promisc) {
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_NOT_TO_ME, 0);
		SET_FLAG(rt2x00dev, INTERFACE_ENABLED_PROMISC);
	} else {
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_NOT_TO_ME, 1);
		CLEAR_FLAG(rt2x00dev, INTERFACE_ENABLED_PROMISC);
	}

	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);
}

static void rt73usb_config_type(struct rt2x00_dev *rt2x00dev, int type)
{
	u32 reg;

	/*
	 * Only continue when there is something to be done.
	 */
	if (!(GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED) ^
	      GET_FLAG(rt2x00dev, INTERFACE_ENABLED)) &&
	    !(GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) ^
	      GET_FLAG(rt2x00dev, INTERFACE_ENABLED_MONITOR)))
		return;

	rt2x00_register_write(rt2x00dev, TXRX_CSR9, 0);

	/*
	 * Apply hardware packet filter.
	 */
	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);

	if (!GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) &&
	    (type == IEEE80211_IF_TYPE_IBSS || type == IEEE80211_IF_TYPE_STA))
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_TO_DS, 1);
	else
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_TO_DS, 0);

	rt2x00_set_field32(&reg, TXRX_CSR0_DROP_CRC, 1);
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR)) {
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_PHYSICAL, 0);
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_CONTROL, 0);
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_VERSION_ERROR, 0);
	} else {
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_PHYSICAL, 1);
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_CONTROL, 1);
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_VERSION_ERROR, 1);
	}

	rt2x00_set_field32(&reg, TXRX_CSR0_DROP_MULTICAST, 0);
	rt2x00_set_field32(&reg, TXRX_CSR0_DROP_BORADCAST, 0);

	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);

	/*
	 * Enable promisc mode when in monitor mode.
	 */
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR))
		rt73usb_config_promisc(rt2x00dev, 1);

	/*
	 * Enable synchronisation.
	 */
	rt2x00_register_read(rt2x00dev, TXRX_CSR9, &reg);
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED)) {
		rt2x00_set_field32(&reg, TXRX_CSR9_BEACON_INTERVAL, 100 * 16);
		rt2x00_set_field32(&reg, TXRX_CSR9_TSF_TICKING, 1);
		rt2x00_set_field32(&reg, TXRX_CSR9_TBTT_ENABLE, 1);
	}

	if (type == IEEE80211_IF_TYPE_IBSS) {
		rt2x00_set_field32(&reg, TXRX_CSR9_TSF_SYNC_MODE, 2);
		rt2x00_set_field32(&reg, TXRX_CSR9_BEACON_GEN, 1);
	} else if (type == IEEE80211_IF_TYPE_STA) {
		rt2x00_set_field32(&reg, TXRX_CSR9_TSF_SYNC_MODE, 1);
		rt2x00_set_field32(&reg, TXRX_CSR9_BEACON_GEN, 0);
	} else if (type == IEEE80211_IF_TYPE_AP) {
		rt2x00_set_field32(&reg, TXRX_CSR9_TSF_SYNC_MODE, 0);
		rt2x00_set_field32(&reg, TXRX_CSR9_BEACON_GEN, 1);
	} else if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) &&
		   !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED)) {
		rt2x00_set_field32(&reg, TXRX_CSR9_TSF_SYNC_MODE, 0);
		rt2x00_set_field32(&reg, TXRX_CSR9_BEACON_GEN, 0);
	}
	rt2x00_register_write(rt2x00dev, TXRX_CSR9, reg);

	/*
	 * Change flags of enabled interfaces.
	 */
	if (type != IEEE80211_IF_TYPE_MNTR) {
		if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
			SET_FLAG(rt2x00dev, INTERFACE_ENABLED);
		else
			CLEAR_FLAG(rt2x00dev, INTERFACE_ENABLED);
	} else {
		if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR))
			SET_FLAG(rt2x00dev, INTERFACE_ENABLED_MONITOR);
		else
			CLEAR_FLAG(rt2x00dev, INTERFACE_ENABLED_MONITOR);
	}
}

static void rt73usb_config_channel(struct rt2x00_dev *rt2x00dev,
	int rf2, int channel, int freq, int txpower)
{
	u8 reg = 0;
	u32 rf1 = rt2x00dev->rf1;
	u32 rf3 = rt2x00dev->rf3;
	u32 rf4 = 0;

	/*
	 * Only continue when there is something to be done.
	 */
	if (channel == rt2x00dev->rx_params.channel)
		return;

	if (txpower == 0xff)
		txpower = rt2x00dev->tx_power;
	txpower = TXPOWER_TO_DEV(txpower);

	if (rt2x00_rf(&rt2x00dev->chip, RF5225)) {
		if (channel <= 14)
			rf3 = 0x00068455;
		else if (channel >= 36 && channel <= 48)
			rf3 = 0x0009be55;
		else if (channel >= 52 && channel <= 64)
			rf3 = 0x0009ae55;
		else if (channel >= 100 && channel <= 112)
			rf3 = 0x000bae55;
		else
			rf3 = 0x000bbe55;
	}

	if (channel < 14) {
		if (channel & 0x01)
			rf4 = 0x000fea0b;
		else
			rf4 = 0x000fea1f;
	} else if (channel == 14) {
		rf4 = 0x000fea13;
	} else {
		switch (channel) {
			case 36:
			case 56:
			case 116:
			case 136:
				rf4 = 0x000fea23;
				break;
			case 40:
			case 60:
			case 100:
			case 120:
			case 140:
				rf4 = 0x000fea03;
				break;
			case 44:
			case 64:
			case 104:
			case 124:
				rf4 = 0x000fea0b;
				break;
			case 48:
			case 108:
			case 128:
				rf4 = 0x000fea13;
				break;
			case 52:
			case 112:
			case 132:
				rf4 = 0x000fea1b;
				break;
			case 149:
				rf4 = 0x000fea1f;
				break;
			case 153:
				rf4 = 0x000fea27;
				break;
			case 157:
				rf4 = 0x000fea07;
				break;
			case 161:
				rf4 = 0x000fea0f;
				break;
			case 165:
				rf4 = 0x000fea17;
				break;
		}
	}

	if (rt2x00_rf(&rt2x00dev->chip, RF2527) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5225))
		rf4 |= 0x00010000;

	/*
	 * Set TXpower.
	 */
	rt2x00_set_field32(&rf3, RF3_TXPOWER, txpower);

	INFO("Switching channel. RF1: 0x%08x, RF2: 0x%08x, RF3: 0x%08x, "
		"RF4: 0x%08x.\n", rf1, rf2, rf3, rf4);

	/*
	 * Set Frequency offset.
	 */
	rt2x00_set_field32(&rf4, RF4_FREQ_OFFSET, rt2x00dev->freq_offset);

	rt2x00_bbp_read(rt2x00dev, 3, &reg);
	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF2527))
		reg &= ~0x01;
	else
		reg |= 0x01;
	rt2x00_bbp_write(rt2x00dev, 3, reg);

	rt2x00_rf_write(rt2x00dev, rf1);
	rt2x00_rf_write(rt2x00dev, rf2);
	rt2x00_rf_write(rt2x00dev, rf3 & ~0x00000004);
	rt2x00_rf_write(rt2x00dev, rf4);

	rt2x00_rf_write(rt2x00dev, rf1);
	rt2x00_rf_write(rt2x00dev, rf2);
	rt2x00_rf_write(rt2x00dev, rf3 | 0x00000004);
	rt2x00_rf_write(rt2x00dev, rf4);

	rt2x00_rf_write(rt2x00dev, rf1);
	rt2x00_rf_write(rt2x00dev, rf2);
	rt2x00_rf_write(rt2x00dev, rf3 & ~0x00000004);
	rt2x00_rf_write(rt2x00dev, rf4);

	msleep(1);

	/*
	 * Update active info for RX.
	 */
	rt2x00dev->rx_params.freq = freq;
	rt2x00dev->rx_params.channel = channel;

	rt2x00dev->tx_power = txpower;

	/*
	 * Update rf fields
	 */
	rt2x00dev->rf1 = rf1;
	rt2x00dev->rf2 = rf2;
	rt2x00dev->rf3 = rf3;
	rt2x00dev->rf4 = rf4;

	rt2x00dev->tx_power = txpower;
}

static void rt73usb_config_txpower(struct rt2x00_dev *rt2x00dev, int txpower)
{
	txpower = TXPOWER_TO_DEV(txpower);

	/*
	 * Only continue when there is something to be done.
	 */
	if (txpower == rt2x00dev->tx_power)
		return;

	rt2x00_set_field32(&rt2x00dev->rf3, RF3_TXPOWER, txpower);

	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf1);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf2);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf3 & ~0x00000004);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf4);

	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf1);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf2);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf3 | 0x00000004);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf4);

	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf1);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf2);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf3 & ~0x00000004);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf4);

	rt2x00dev->tx_power = txpower;
}

static void rt73usb_config_antenna(struct rt2x00_dev *rt2x00dev,
	int antenna, int phymode)
{
	u32 reg;
	u8 reg_r3;
	u8 reg_r4;
	u8 reg_r77;
	u8 frame_type;

	/*
	 * Only continue when there is something to be done.
	 */
	if (rt2x00dev->rx_params.antenna == antenna)
		return;

	rt2x00_register_read(rt2x00dev, PHY_CSR0, &reg);

	if (phymode == MODE_IEEE80211A) {
		if (GET_FLAG(rt2x00dev, CONFIG_EXTERNAL_LNA)) {
			rt2x00_bbp_write(rt2x00dev, 17, 0x38);
			rt2x00_bbp_write(rt2x00dev, 96, 0x78);
			rt2x00_bbp_write(rt2x00dev, 104, 0x48);
			rt2x00_bbp_write(rt2x00dev, 75, 0x80);
			rt2x00_bbp_write(rt2x00dev, 86, 0x80);
			rt2x00_bbp_write(rt2x00dev, 88, 0x80);
		} else {
			rt2x00_bbp_write(rt2x00dev, 17, 0x28);
			rt2x00_bbp_write(rt2x00dev, 96, 0x58);
			rt2x00_bbp_write(rt2x00dev, 104, 0x38);
			rt2x00_bbp_write(rt2x00dev, 75, 0xfe);
			rt2x00_bbp_write(rt2x00dev, 86, 0xfe);
			rt2x00_bbp_write(rt2x00dev, 88, 0xfe);
		}
		rt2x00_bbp_write(rt2x00dev, 35, 0x60);
		rt2x00_bbp_write(rt2x00dev, 97, 0x58);
		rt2x00_bbp_write(rt2x00dev, 98, 0x58);

		rt2x00_set_field32(&reg, PHY_CSR0_PA_PE_BG, 0);
		rt2x00_set_field32(&reg, PHY_CSR0_PA_PE_A, 1);
	} else {
		if (GET_FLAG(rt2x00dev, CONFIG_EXTERNAL_LNA)) {
			rt2x00_bbp_write(rt2x00dev, 17, 0x30);
			rt2x00_bbp_write(rt2x00dev, 96, 0x68);
			rt2x00_bbp_write(rt2x00dev, 104, 0x3c);
			rt2x00_bbp_write(rt2x00dev, 75, 0x80);
			rt2x00_bbp_write(rt2x00dev, 86, 0x80);
			rt2x00_bbp_write(rt2x00dev, 88, 0x80);
		} else {
			rt2x00_bbp_write(rt2x00dev, 17, 0x20);
			rt2x00_bbp_write(rt2x00dev, 96, 0x48);
			rt2x00_bbp_write(rt2x00dev, 104, 0x2c);
			rt2x00_bbp_write(rt2x00dev, 75, 0xfe);
			rt2x00_bbp_write(rt2x00dev, 86, 0xfe);
			rt2x00_bbp_write(rt2x00dev, 88, 0xfe);
		}
		rt2x00_bbp_write(rt2x00dev, 35, 0x50);
		rt2x00_bbp_write(rt2x00dev, 97, 0x48);
		rt2x00_bbp_write(rt2x00dev, 98, 0x48);

		rt2x00_set_field32(&reg, PHY_CSR0_PA_PE_BG, 1);
		rt2x00_set_field32(&reg, PHY_CSR0_PA_PE_A, 0);
	}

	rt2x00_register_write(rt2x00dev, PHY_CSR0, reg);

	rt2x00_bbp_read(rt2x00dev, 3, &reg_r3);
	rt2x00_bbp_read(rt2x00dev, 4, &reg_r4);
	rt2x00_bbp_read(rt2x00dev, 77, &reg_r77);

	reg_r3 &= ~0x01;
	reg_r4 &= ~0x23;
	frame_type = ~(GET_FLAG(rt2x00dev, CONFIG_FRAME_TYPE) << 5);

	if (rt2x00_rf(&rt2x00dev->chip, RF5226) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5225)) {
		if (antenna == 0) {
			/* Diversity. */
			reg_r4 |= 0x02;
			if (phymode != MODE_IEEE80211A)
				reg_r4 |= 0x20;
			reg_r4 &= frame_type;
		} else if (antenna == 1) {
			/* RX: Antenna B */
			/* TX: Antenna A */
			reg_r4 |= 0x01;
			reg_r4 &= frame_type;
			if (phymode == MODE_IEEE80211A)
				reg_r77 |= 0x03;
			else
				reg_r77 &= ~0x03;
			rt2x00_bbp_write(rt2x00dev, 77, reg_r77);
		} else if (antenna == 2) {
			/* RX: Antenna A */
			/* TX: Antenna B */
			reg_r4 |= 0x01;
			reg_r4 &= frame_type;
			if (phymode == MODE_IEEE80211A)
				reg_r77 &= ~0x03;
			else
				reg_r77 |= 0x03;
			rt2x00_bbp_write(rt2x00dev, 77, reg_r77);
		}
	} else if (rt2x00_rf(&rt2x00dev->chip, RF2528) ||
		   rt2x00_rf(&rt2x00dev->chip, RF2527)) {
		if (antenna == 0) {
			/* Diversity. */
			reg_r4 |= 0x22;
			reg_r4 &= frame_type;
		} else if (antenna == 1) {
			/* RX: Antenna B */
			/* TX: Antenna A */
			reg_r4 |= 0x21;
			reg_r4 &= frame_type;
			reg_r77 &= ~0x03;
			rt2x00_bbp_write(rt2x00dev, 77, reg_r77);
		} else if (antenna == 2) {
			/* RX: Antenna A */
			/* TX: Antenna B */
			reg_r4 |= 0x21;
			reg_r4 &= frame_type;
			reg_r77 |= 0x03;
			rt2x00_bbp_write(rt2x00dev, 77, reg_r77);
		}
	}

	rt2x00_bbp_write(rt2x00dev, 3, reg_r3);
	rt2x00_bbp_write(rt2x00dev, 4, reg_r4);

	/*
	 * Update active info for RX.
	 */
	rt2x00dev->rx_params.antenna = antenna;
}

static void rt73usb_config_duration(struct rt2x00_dev *rt2x00dev,
	int short_slot_time)
{
	u32 reg;

	short_slot_time = short_slot_time ? SHORT_SLOT_TIME : SLOT_TIME;

	rt2x00_register_read(rt2x00dev, MAC_CSR9, &reg);
	rt2x00_set_field32(&reg, MAC_CSR9_SLOT_TIME, short_slot_time);
	rt2x00_register_write(rt2x00dev, MAC_CSR9, reg);

	rt2x00_register_read(rt2x00dev, MAC_CSR8, &reg);
	rt2x00_set_field32(&reg, MAC_CSR8_SIFS, SIFS);
	rt2x00_set_field32(&reg, MAC_CSR8_SIFS_AFTER_RX_OFDM, 3);
	rt2x00_set_field32(&reg, MAC_CSR8_EIFS, EIFS);
	rt2x00_register_write(rt2x00dev, MAC_CSR8, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR0_TSF_OFFSET, IEEE80211_HEADER);
	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR4, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR4_AUTORESPOND_ENABLE, 1);
	rt2x00_register_write(rt2x00dev, TXRX_CSR4, reg);
}

static void rt73usb_config_rate(struct rt2x00_dev *rt2x00dev, const int rate)
{
	struct ieee80211_conf *conf = &rt2x00dev->hw->conf;
	u32 reg;
	u32 value;
	u32 preamble;

	preamble = DEVICE_GET_RATE_FIELD(rate, PREAMBLE)
		? SHORT_PREAMBLE : PREAMBLE;

	reg = DEVICE_GET_RATE_FIELD(rate, RATEMASK) & DEV_BASIC_RATE;

	rt2x00_register_write(rt2x00dev, TXRX_CSR5, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
	value = ((conf->flags & IEEE80211_CONF_SHORT_SLOT_TIME) ?
		 SHORT_DIFS :  DIFS) +
		PLCP + preamble + get_duration(ACK_SIZE, 10);
	rt2x00_set_field32(&reg, TXRX_CSR0_RX_ACK_TIMEOUT, value);
	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR4, &reg);
	if (preamble == SHORT_PREAMBLE)
		rt2x00_set_field32(&reg, TXRX_CSR4_AUTORESPOND_PREAMBLE, 1);
	else
		rt2x00_set_field32(&reg, TXRX_CSR4_AUTORESPOND_PREAMBLE, 0);
	rt2x00_register_write(rt2x00dev, TXRX_CSR4, reg);
}

static void rt73usb_config_phymode(struct rt2x00_dev *rt2x00dev,
	const int phymode)
{
	struct ieee80211_rate *rate;

	/*
	 * Only continue when there is something to be done.
	 */
	if (rt2x00dev->rx_params.phymode == phymode)
		return;

	if (phymode == MODE_IEEE80211A &&
	    (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5226)))
		rate = &rt2x00dev->hw->modes[2].rates[
			rt2x00dev->hw->modes[2].num_rates - 1];
	else if (phymode == MODE_IEEE80211B)
		rate = &rt2x00dev->hw->modes[1].rates[
			rt2x00dev->hw->modes[1].num_rates - 1];
	else
		rate = &rt2x00dev->hw->modes[0].rates[
			rt2x00dev->hw->modes[0].num_rates - 1];

	rt73usb_config_rate(rt2x00dev, rate->val2);

	/*
	 * Update physical mode for rx ring.
	 */
	rt2x00dev->rx_params.phymode = phymode;
}

static void rt73usb_config_mac_address(struct rt2x00_dev *rt2x00dev,
	void *addr)
{
	u32 reg[2] = { 0, 0 };

	/*
	 * The MAC address is passed to us as an array of bytes,
	 * that array is little endian, so no need for byte ordering.
	 * We only need to set the MAC_CSR3_UNICAST_TO_ME_MASK
	 * at the correct offset.
	 */
	memcpy(&reg, addr, ETH_ALEN);
	rt2x00_set_field32(&reg[1], MAC_CSR3_UNICAST_TO_ME_MASK, 0xff);
	rt2x00_register_multiwrite(rt2x00dev, MAC_CSR2, &reg[0], sizeof(reg));
}

/*
 * Link tuning
 */
static void rt73usb_link_tuner(void *data)
{
	struct rt2x00_dev *rt2x00dev = data;
	u32 reg;
	u32 rssi;
	u8 reg_r17;
	u8 up_bound;
	u8 low_bound;

	/*
	 * Retreive link quality.
	 */
	rssi = rt2x00_get_link_rssi(&rt2x00dev->link);
	if (!rssi)
		goto exit;

	/*
	 * Update LED.
	 */
	rt73usb_activity_led(rt2x00dev, rssi);

	/*
	 * Determine upper and lower limit for BBP17 register.
	 */
	if (rt2x00dev->rx_params.phymode == MODE_IEEE80211A) {
		up_bound = 0x48;
		low_bound = 0x28;
	} else {
		if (rssi > 38) {
			up_bound = 0x40;
			low_bound = 0x1c;
		} else if (rssi > 36) {
			up_bound = 0x20;
			low_bound = 0x1c;
		} else {
			up_bound = 0x1c;
			low_bound = 0x1c;
		}

		if (GET_FLAG(rt2x00dev, CONFIG_EXTERNAL_LNA)) {
			up_bound += 0x10;
			low_bound += 0x14;
		}
	}

	rt2x00_bbp_read(rt2x00dev, 17, &reg_r17);

	if (rssi >= 85) {
		if (reg_r17 != 0x60)
			rt2x00_bbp_write(rt2x00dev, 17, 0x60);
		goto exit;
	} else if (rssi >= 62) {
		if (reg_r17 != up_bound)
			rt2x00_bbp_write(rt2x00dev, 17, up_bound);
		goto exit;
	} else if (rssi >= 54) {
		low_bound += 0x10;
		if (reg_r17 != low_bound)
			rt2x00_bbp_write(rt2x00dev, 17, low_bound);
		goto exit;
	} else if (rssi >= 46) {
		low_bound += 0x08;
		if (reg_r17 != low_bound)
			rt2x00_bbp_write(rt2x00dev, 17, low_bound);
		goto exit;
	} else {
		up_bound -= 2 * (46 - rssi);
		if (up_bound < low_bound)
			up_bound =  low_bound;

		if (reg_r17 > up_bound) {
			rt2x00_bbp_write(rt2x00dev, 17, up_bound);
			goto exit;
		}
	}

	rt2x00_register_read(rt2x00dev, STA_CSR1, &reg);
	reg = rt2x00_get_field32(reg, STA_CSR1_FALSE_CCA_ERROR);

	if (reg > 512 && reg_r17 < up_bound)
		rt2x00_bbp_write(rt2x00dev, 17, ++reg_r17);
	else if (reg < 100 && reg_r17 > low_bound)
		rt2x00_bbp_write(rt2x00dev, 17, --reg_r17);

exit:
	if (reg_r17)
		rt2x00_update_link_noise(&rt2x00dev->link, reg_r17);

	queue_delayed_work(rt2x00dev->workqueue, &rt2x00dev->link.work,
		LINK_TUNE_INTERVAL);
}

/*
 * LED functions.
 */
static void rt73usb_enable_led(struct rt2x00_dev *rt2x00dev)
{
	u32 reg;

	rt2x00_register_read(rt2x00dev, MAC_CSR14, &reg);
	rt2x00_set_field32(&reg, MAC_CSR14_ON_PERIOD, 70);
	rt2x00_set_field32(&reg, MAC_CSR14_OFF_PERIOD, 30);
	rt2x00_register_write(rt2x00dev, MAC_CSR14, reg);

	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_RADIO_STATUS, 1);
	if (rt2x00dev->rx_params.phymode == MODE_IEEE80211A)
		rt2x00_set_field16(
			&rt2x00dev->led_reg, MCU_LEDCS_LINK_A_STATUS, 1);
	else
		rt2x00_set_field16(
			&rt2x00dev->led_reg, MCU_LEDCS_LINK_BG_STATUS, 1);

	rt2x00_vendor_request(
		rt2x00dev, USB_LED_CONTROL, USB_VENDOR_REQUEST_OUT,
		0x00, rt2x00dev->led_reg, NULL, 0, REGISTER_TIMEOUT);
}

static void rt73usb_disable_led(struct rt2x00_dev *rt2x00dev)
{
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_RADIO_STATUS, 0);
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_LINK_BG_STATUS, 0);
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_LINK_A_STATUS, 0);

	rt2x00_vendor_request(
		rt2x00dev, USB_LED_CONTROL, USB_VENDOR_REQUEST_OUT,
		0x00, rt2x00dev->led_reg, NULL, 0, REGISTER_TIMEOUT);
}

static void rt73usb_activity_led(struct rt2x00_dev *rt2x00dev, char rssi)
{
	u32 led;

	if (rt2x00dev->led_mode != LED_MODE_SIGNAL_STRENGTH)
		return;

	if (rssi <= 30)
		led = 0;
	else if (rssi <= 39)
		led = 1;
	else if (rssi <= 49)
		led = 2;
	else if (rssi <= 53)
		led = 3;
	else if (rssi <= 63)
		led = 4;
	else
		led = 5;

	rt2x00_vendor_request(
		rt2x00dev, USB_LED_CONTROL, USB_VENDOR_REQUEST_OUT,
		led, rt2x00dev->led_reg, NULL, 0, REGISTER_TIMEOUT);
}

/*
 * Device state switch.
 * This will put the device to sleep, or awake it.
 */
static int rt73usb_set_state(struct rt2x00_dev *rt2x00dev,
	enum dev_state state)
{
	u32 reg;
	unsigned int i;
	char put_to_sleep;
	char current_state;

	put_to_sleep = (state != STATE_AWAKE);

	if (!put_to_sleep)
		rt2x00_vendor_request(rt2x00dev,
			USB_DEVICE_MODE, USB_VENDOR_REQUEST_OUT,
			0x00, USB_MODE_WAKEUP, NULL, 0, REGISTER_TIMEOUT);

	rt2x00_register_read(rt2x00dev, MAC_CSR12, &reg);
	rt2x00_set_field32(&reg, MAC_CSR12_FORCE_WAKEUP, !put_to_sleep);
	rt2x00_set_field32(&reg, MAC_CSR12_PUT_TO_SLEEP, put_to_sleep);
	rt2x00_register_write(rt2x00dev, MAC_CSR12, reg);

	if (put_to_sleep)
		rt2x00_vendor_request(rt2x00dev,
			USB_DEVICE_MODE, USB_VENDOR_REQUEST_OUT,
			0x00, USB_MODE_SLEEP, NULL, 0, REGISTER_TIMEOUT);

	/*
	 * Device is not guarenteed to be in the requested state yet.
	 * We must wait untill the register indicates that the
	 * device has entered the correct state.
	 */
	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00dev, MAC_CSR12, &reg);
		current_state = rt2x00_get_field32(reg,
			MAC_CSR12_BBP_CURRENT_STATE);
		if (current_state == !put_to_sleep)
			return 0;
		msleep(10);
	}

	NOTICE("Device failed to enter state %d, "
		"current device state %d.\n", !put_to_sleep, current_state);

	return -EBUSY;
}

/*
 * Initialization functions.
 */
static int rt73usb_init_firmware_wait(struct rt2x00_dev *rt2x00dev)
{
	unsigned int i;

	for (i = 0; i < 150; i++) {
		if (GET_FLAG(rt2x00dev, FIRMWARE_FAILED))
			return -EIO;
		if (GET_FLAG(rt2x00dev, FIRMWARE_LOADED))
			return 0;
		msleep(20);
	}

	ERROR("Firmware loading timed out.\n");
	return -EIO;
}

static void rt73usb_init_firmware_cont(const struct firmware *fw,
	void *context)
{
	struct rt2x00_dev *rt2x00dev = context;
	unsigned int i;
	int status;
	u32 reg;
	u16 crc;

	if (!fw || !fw->size || !fw->data) {
		ERROR("Failed to load Firmware.\n");
		goto exit;
	}

	/*
	 * Wait for stable hardware.
	 */
	for (i = 0; i < 100; i++) {
		rt2x00_register_read(rt2x00dev, MAC_CSR0, &reg);
		if (reg)
			break;
		msleep(1);
	}

	if (!reg) {
		ERROR("Unstable hardware.\n");
		goto exit;
	}

	/*
	 * Validate the firmware using 16 bit CRC.
	 * The last 2 bytes of the firmware are the CRC
	 * so substract those 2 bytes from the CRC checksum,
	 * and set those 2 bytes to 0 when calculating CRC.
	 */
	reg = 0;
	crc = crc_itu_t(0, fw->data, fw->size - 2);
	crc = crc_itu_t(crc, (u8*)&reg, 2);

	if (crc != (fw->data[fw->size - 2] << 8 | fw->data[fw->size - 1])) {
		ERROR("Firmware CRC error.\n");
		goto exit;
	}

	rt2x00_set_chip_fw(&rt2x00dev->chip,
		fw->data[fw->size - 4], fw->data[fw->size - 3]);

	/*
	 * Write firmware to device.
	 */
	for (i = 0; i < fw->size; i += sizeof(u32))
		rt2x00_register_write(rt2x00dev, FIRMWARE_IMAGE_BASE + i,
			*((u32*)(fw->data + i)));

	/*
	 * Send firmware request to device to load firmware,
	 * we need to specify a long timeout time.
	 */
	status = rt2x00_vendor_request(rt2x00dev, USB_DEVICE_MODE,
		USB_VENDOR_REQUEST_OUT, 0x00, USB_MODE_FIRMWARE,
		NULL, 0, REGISTER_TIMEOUT_FIRMWARE);
	if (status  < 0) {
		ERROR("Failed to load Firmware error %d.\n", status);
		goto exit;
	}

	rt73usb_disable_led(rt2x00dev);

	SET_FLAG(rt2x00dev, FIRMWARE_LOADED);

	return;

exit:
	SET_FLAG(rt2x00dev, FIRMWARE_FAILED);
}

static int rt73usb_init_firmware(struct rt2x00_dev *rt2x00dev)
{
	/*
	 * Read correct firmware from harddisk.
	 */
	return request_firmware_nowait(THIS_MODULE, 1,
		"rt73.bin", &rt2x00dev_usb(rt2x00dev)->dev, rt2x00dev,
		rt73usb_init_firmware_cont);
}

static int rt73usb_alloc_dma_ring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type, void (*handler)(void *),
	const u16 max_entries, const u16 data_size, const u16 desc_size)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	unsigned int i;
	int status;

	/*
	 * Initialize work structure for deferred work.
	 */
	INIT_WORK(&ring->irq_work, handler, ring);

	ring->stats.limit = max_entries;
	ring->data_size = data_size;
	ring->desc_size = desc_size;

	/*
	 * Allocate all ring entries.
	 */
	ring->entry = kzalloc(ring->stats.limit * sizeof(struct data_entry),
		GFP_KERNEL);
	if (!ring->entry)
		return -ENOMEM;

	/*
	 * Allocate DMA memory for descriptor and buffer.
	 */
	ring->data_addr = usb_buffer_alloc(
		interface_to_usbdev(rt2x00dev_usb(rt2x00dev)),
		rt2x00_get_ring_size(ring), GFP_KERNEL, &ring->data_dma);
	if (!ring->data_addr) {
		kfree(ring->entry);
		return -ENOMEM;
	}

	/*
	 * Initialize all ring entries to contain valid
	 * addresses.
	 */
	status = 0;
	for (i = 0; i < ring->stats.limit; i++) {
		ring->entry[i].flags = 0;
		ring->entry[i].ring = ring;
		ring->entry[i].priv =
			(!status) ? usb_alloc_urb(0, GFP_KERNEL) :  NULL;
		if (!ring->entry[i].priv)
			status = -ENOMEM;
		ring->entry[i].skb = NULL;
		ring->entry[i].data_addr = ring->data_addr
			+ (i * ring->desc_size)
			+ (i * ring->data_size);
		ring->entry[i].data_dma = ring->data_dma
			+ (i * ring->desc_size)
			+ (i * ring->data_size);
	}

	return status;
}

static void rt73usb_free_ring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	unsigned int i;

	if (!ring->entry)
		goto exit;

	for (i = 0; i < ring->stats.limit; i++) {
		usb_kill_urb(rt2x00_urb(&ring->entry[i]));
		usb_free_urb(rt2x00_urb(&ring->entry[i]));
	}

	kfree(ring->entry);
	ring->entry = NULL;

exit:
	if (ring->data_addr)
		usb_buffer_free(
			interface_to_usbdev(rt2x00dev_usb(rt2x00dev)),
			rt2x00_get_ring_size(ring), ring->data_addr,
			ring->data_dma);
	ring->data_addr = NULL;
}

static int rt73usb_allocate_dma_rings(struct rt2x00_dev *rt2x00dev)
{
	if (rt73usb_alloc_dma_ring(rt2x00dev, RING_RX,
		rt73usb_rxdone, RX_ENTRIES, DATA_FRAME_SIZE,
		RXD_DESC_SIZE) ||
	    rt73usb_alloc_dma_ring(rt2x00dev, RING_AC_VO,
		rt73usb_txdone, TX_ENTRIES, DATA_FRAME_SIZE,
		TXD_DESC_SIZE) ||
	    rt73usb_alloc_dma_ring(rt2x00dev, RING_AC_VI,
		rt73usb_txdone, TX_ENTRIES, DATA_FRAME_SIZE,
		TXD_DESC_SIZE) ||
	    rt73usb_alloc_dma_ring(rt2x00dev, RING_AC_BE,
		rt73usb_txdone, TX_ENTRIES, DATA_FRAME_SIZE,
		TXD_DESC_SIZE) ||
	    rt73usb_alloc_dma_ring(rt2x00dev, RING_AC_BK,
		rt73usb_txdone, TX_ENTRIES, DATA_FRAME_SIZE,
		TXD_DESC_SIZE) ||
	    rt73usb_alloc_dma_ring(rt2x00dev, RING_PRIO,
		rt73usb_txdone, TX_ENTRIES, DATA_FRAME_SIZE,
		TXD_DESC_SIZE) ||
	    rt73usb_alloc_dma_ring(rt2x00dev, RING_BEACON,
		rt73usb_beacondone, BEACON_ENTRIES, MGMT_FRAME_SIZE,
		TXD_DESC_SIZE) ) {
		return -ENOMEM;
	}

	return 0;
}

static void rt73usb_free_rings(struct rt2x00_dev *rt2x00dev)
{
	rt73usb_free_ring(rt2x00dev, RING_RX);
	rt73usb_free_ring(rt2x00dev, RING_AC_VO);
	rt73usb_free_ring(rt2x00dev, RING_AC_VI);
	rt73usb_free_ring(rt2x00dev, RING_AC_BE);
	rt73usb_free_ring(rt2x00dev, RING_AC_BK);
	rt73usb_free_ring(rt2x00dev, RING_PRIO);
	rt73usb_free_ring(rt2x00dev, RING_BEACON);
}

static void rt73usb_init_rxring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	struct usb_device *usb_dev =
		interface_to_usbdev(rt2x00dev_usb(rt2x00dev));
	struct urb *urb;
	unsigned int i;

	ring->type = ring_type;

	for (i = 0; i < ring->stats.limit; i++) {
		urb = rt2x00_urb(&ring->entry[i]);

		urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;
		urb->transfer_dma = ring->entry[i].data_dma;
		usb_fill_bulk_urb(
			urb,
			usb_dev,
			usb_rcvbulkpipe(usb_dev, 1),
			ring->entry[i].data_addr,
			ring->data_size + ring->desc_size,
			rt73usb_interrupt,
			&ring->entry[i]);
	}

	rt2x00_ring_index_clear(ring);
}

static void rt73usb_init_txring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	struct urb *urb;
	unsigned int i;

	ring->type = ring_type;

	for (i = 0; i < ring->stats.limit; i++) {
		urb = rt2x00_urb(&ring->entry[i]);

		urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;
		urb->transfer_dma = ring->entry[i].data_dma;
		CLEAR_FLAGS(&ring->entry[i]);
	}

	rt2x00_ring_index_clear(ring);
}

static int rt73usb_init_rings(struct rt2x00_dev *rt2x00dev)
{
	rt73usb_init_rxring(rt2x00dev, RING_RX);
	rt73usb_init_txring(rt2x00dev, RING_AC_VO);
	rt73usb_init_txring(rt2x00dev, RING_AC_VI);
	rt73usb_init_txring(rt2x00dev, RING_AC_BE);
	rt73usb_init_txring(rt2x00dev, RING_AC_BK);
	rt73usb_init_txring(rt2x00dev, RING_PRIO);
	rt73usb_init_txring(rt2x00dev, RING_BEACON);

	return 0;
}

static int rt73usb_init_registers(struct rt2x00_dev *rt2x00dev)
{
	u32 reg;

	if (rt73usb_set_state(rt2x00dev, STATE_AWAKE))
		return -EBUSY;

	rt2x00_register_write(rt2x00dev, MAC_CSR10, 0x00000718);

	rt2x00_register_write(rt2x00dev, TXRX_CSR0, 0x025eb032);

	rt2x00_register_write(rt2x00dev, TXRX_CSR1, 0x9eaa9eaf);
	rt2x00_register_write(rt2x00dev, TXRX_CSR2, 0x8a8b8c8d);
	rt2x00_register_write(rt2x00dev, TXRX_CSR3, 0x00858687);

	rt2x00_register_write(rt2x00dev, TXRX_CSR7, 0x2e31353b);
	rt2x00_register_write(rt2x00dev, TXRX_CSR8, 0x2a2a2a2c);

	rt2x00_register_write(rt2x00dev, TXRX_CSR15, 0x0000000f);

	rt2x00_register_write(rt2x00dev, MAC_CSR6, 0x00000fff);

	rt2x00_register_write(rt2x00dev, MAC_CSR13, 0x00007f00);

	rt2x00_register_write(rt2x00dev, SEC_CSR0, 0x00000000);
	rt2x00_register_write(rt2x00dev, SEC_CSR1, 0x00000000);
	rt2x00_register_write(rt2x00dev, SEC_CSR5, 0x00000000);

	rt2x00_register_write(rt2x00dev, PHY_CSR1, 0x000023b0);
	rt2x00_register_write(rt2x00dev, PHY_CSR5, 0x00040a06);
	rt2x00_register_write(rt2x00dev, PHY_CSR6, 0x00080606);
	rt2x00_register_write(rt2x00dev, PHY_CSR7, 0x00000408);

	rt2x00_register_read(rt2x00dev, AC_TXOP_CSR0, &reg);
	rt2x00_set_field32(&reg, AC_TXOP_CSR0_AC0_TX_OP, 0);
	rt2x00_set_field32(&reg, AC_TXOP_CSR0_AC1_TX_OP, 0);
	rt2x00_register_write(rt2x00dev, AC_TXOP_CSR0, reg);

	rt2x00_register_read(rt2x00dev, AC_TXOP_CSR1, &reg);
	rt2x00_set_field32(&reg, AC_TXOP_CSR1_AC2_TX_OP, 192);
	rt2x00_set_field32(&reg, AC_TXOP_CSR1_AC3_TX_OP, 48);
	rt2x00_register_write(rt2x00dev, AC_TXOP_CSR1, reg);

	rt2x00_register_read(rt2x00dev, MAC_CSR9, &reg);
	rt2x00_set_field32(&reg, MAC_CSR9_CW_SELECT, 0);
	rt2x00_register_write(rt2x00dev, MAC_CSR9, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR0_AUTO_TX_SEQ, 1);
	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);

	/*
	 * We must clear the error counters.
	 * These registers are cleared on read,
	 * so we may pass a useless variable to store the value.
	 */
	rt2x00_register_read(rt2x00dev, STA_CSR0, &reg);
	rt2x00_register_read(rt2x00dev, STA_CSR1, &reg);
	rt2x00_register_read(rt2x00dev, STA_CSR2, &reg);

	/*
	 * Reset MAC and BBP registers.
	 */
	reg = 0;
	rt2x00_set_field32(&reg, MAC_CSR1_SOFT_RESET, 1);
	rt2x00_set_field32(&reg, MAC_CSR1_BBP_RESET, 1);
	rt2x00_register_write(rt2x00dev, MAC_CSR1, reg);

	rt2x00_register_read(rt2x00dev, MAC_CSR1, &reg);
	rt2x00_set_field32(&reg, MAC_CSR1_SOFT_RESET, 0);
	rt2x00_set_field32(&reg, MAC_CSR1_BBP_RESET, 0);
	rt2x00_register_write(rt2x00dev, MAC_CSR1, reg);

	rt2x00_register_read(rt2x00dev, MAC_CSR1, &reg);
	rt2x00_set_field32(&reg, MAC_CSR1_HOST_READY, 1);
	rt2x00_register_write(rt2x00dev, MAC_CSR1, reg);

	return 0;
}

static int rt73usb_init_bbp(struct rt2x00_dev *rt2x00dev)
{
	unsigned int i;
	u16 eeprom;
	u8 reg_id;
	u8 value;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_bbp_read(rt2x00dev, 0, &value);
		if ((value != 0xff) && (value != 0x00))
			goto continue_csr_init;
		NOTICE("Waiting for BBP register.\n");
	}

	ERROR("BBP register access failed, aborting.\n");
	return -EACCES;

continue_csr_init:
	rt2x00_bbp_write(rt2x00dev, 3, 0x80);
	rt2x00_bbp_write(rt2x00dev, 15, 0x30);
	rt2x00_bbp_write(rt2x00dev, 17, 0x20);
	rt2x00_bbp_write(rt2x00dev, 21, 0xc8);
	rt2x00_bbp_write(rt2x00dev, 22, 0x38);
	rt2x00_bbp_write(rt2x00dev, 23, 0x06);
	rt2x00_bbp_write(rt2x00dev, 24, 0xfe);
	rt2x00_bbp_write(rt2x00dev, 25, 0x0a);
	rt2x00_bbp_write(rt2x00dev, 26, 0x0d);
	rt2x00_bbp_write(rt2x00dev, 32, 0x0b);
	rt2x00_bbp_write(rt2x00dev, 34, 0x12);
	rt2x00_bbp_write(rt2x00dev, 37, 0x07);
	rt2x00_bbp_write(rt2x00dev, 39, 0xf8);
	rt2x00_bbp_write(rt2x00dev, 41, 0x60);
	rt2x00_bbp_write(rt2x00dev, 53, 0x10);
	rt2x00_bbp_write(rt2x00dev, 54, 0x18);
	rt2x00_bbp_write(rt2x00dev, 60, 0x10);
	rt2x00_bbp_write(rt2x00dev, 61, 0x04);
	rt2x00_bbp_write(rt2x00dev, 62, 0x04);
	rt2x00_bbp_write(rt2x00dev, 75, 0xfe);
	rt2x00_bbp_write(rt2x00dev, 86, 0xfe);
	rt2x00_bbp_write(rt2x00dev, 88, 0xfe);
	rt2x00_bbp_write(rt2x00dev, 90, 0x0f);
	rt2x00_bbp_write(rt2x00dev, 99, 0x00);
	rt2x00_bbp_write(rt2x00dev, 102, 0x16);
	rt2x00_bbp_write(rt2x00dev, 107, 0x04);

	DEBUG("Start initialization from EEPROM...\n");
	for (i = 0; i < EEPROM_BBP_SIZE; i++) {
		rt2x00_eeprom_read(rt2x00dev, EEPROM_BBP_START + i, &eeprom);

		if (eeprom != 0xffff && eeprom != 0x0000) {
			reg_id = rt2x00_get_field16(eeprom, EEPROM_BBP_REG_ID);
			value = rt2x00_get_field16(eeprom, EEPROM_BBP_VALUE);
			DEBUG("BBP: 0x%02x, value: 0x%02x.\n", reg_id, value);
			rt2x00_bbp_write(rt2x00dev, reg_id, value);
		}
	}
	DEBUG("...End initialization from EEPROM.\n");

	return 0;
}

static int rt73usb_init_channel_time(struct rt2x00_dev *rt2x00dev)
{
	unsigned long jiffies_start;
	unsigned long jiffies_end;

	/*
	 * Only initialize the channel_change_time
	 * if it has not been set previously.
	 */
	if (rt2x00dev->hw->channel_change_time)
		return 0;

	/*
	 * Invalidate the rx_params.channel value to make sure
	 * the config channel will be correctly executed.
	 */
	rt2x00dev->rx_params.channel = 0;

	/*
	 * Determine channel_change_time
	 * by measuring the time it takes
	 * to switch the channel.
	 */
	jiffies_start = jiffies;
	rt73usb_config_channel(rt2x00dev,
		rt2x00dev->hw->modes[0].channels[0].val,
		rt2x00dev->hw->modes[0].channels[0].chan,
		rt2x00dev->hw->modes[0].channels[0].freq,
		rt2x00dev->hw->modes[0].channels[0].power_level);
	jiffies_end = jiffies;

	rt2x00dev->hw->channel_change_time =
		jiffies_to_usecs((long)jiffies_end - (long)jiffies_start);

	NOTICE("Channel change time has been set to %d.\n",
		rt2x00dev->hw->channel_change_time);

	return 0;
}

/*
 * Device initialization functions.
 */
static int rt73usb_initialize(struct rt2x00_dev *rt2x00dev)
{
	if (GET_FLAG(rt2x00dev, DEVICE_INITIALIZED))
		return 0;

	/*
	 * We must wait on the firmware before
	 * we can safely continue.
	 */
	if (rt73usb_init_firmware_wait(rt2x00dev))
		return -ENODEV;

	/*
	 * Allocate all data rings.
	 */
	if (rt73usb_allocate_dma_rings(rt2x00dev)) {
		ERROR("DMA allocation failed.\n");
		goto exit_fail;
	}

	/*
	 * Reset the channel_change_time value
	 * to make sure it will be correctly initialized
	 * after the radio has been enabled.
	 */
	rt2x00dev->hw->channel_change_time = 0;

	SET_FLAG(rt2x00dev, DEVICE_INITIALIZED);

	return 0;

exit_fail:
	rt73usb_free_rings(rt2x00dev);

	return -EIO;
}

static void rt73usb_uninitialize(struct rt2x00_dev *rt2x00dev)
{
	if (!GET_FLAG(rt2x00dev, DEVICE_INITIALIZED))
		return;

	/*
	 * Cancel scanning.
	 */
	if (rt2x00dev->scan)
		rt2x00_signal_scan(rt2x00dev->scan, SCANNING_CANCELLED);

	/*
	 * Flush out all pending work.
	 */
	flush_workqueue(rt2x00dev->workqueue);

	/*
	 * Free DMA rings.
	 */
	rt73usb_free_rings(rt2x00dev);

	CLEAR_FLAG(rt2x00dev, DEVICE_INITIALIZED);
}

/*
 * Radio control functions.
 */
static int rt73usb_enable_radio(struct rt2x00_dev *rt2x00dev)
{
	struct data_ring *ring;
	u32 reg;
	unsigned int i;

	/*
	 * Don't enable the radio twice.
	 */
	if (GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO))
		return 0;

	/*
	 * Check if the hardware has been initialized,
	 * if not then do it now.
	 */
	if (!GET_FLAG(rt2x00dev, DEVICE_INITIALIZED))
		if (rt73usb_initialize(rt2x00dev))
			return -EIO;

	/*
	 * Initialize all registers.
	 */
	if (rt73usb_init_rings(rt2x00dev) ||
	    rt73usb_init_registers(rt2x00dev) ||
	    rt73usb_init_bbp(rt2x00dev)) {
		ERROR("Register initialization failed.\n");
		goto exit_fail;
	}

	/*
	 * Determine channel change time.
	 */
	if (rt73usb_init_channel_time(rt2x00dev))
		goto exit_fail;

	SET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO);

	/*
	 * Enable RX.
	 */
	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR0_DISABLE_RX, 0);
	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);

	ring = &rt2x00dev->ring[RING_RX];
	for (i = 0; i < ring->stats.limit; i++) {
		SET_FLAG(&ring->entry[i], ENTRY_OWNER_NIC);
		usb_submit_urb(rt2x00_urb(&ring->entry[i]), GFP_ATOMIC);
	}

	/*
	 * Enable LED
	 */
	rt73usb_enable_led(rt2x00dev);

	ieee80211_start_queues(rt2x00dev->hw);
	ieee80211_netif_oper(rt2x00dev->hw, NETIF_WAKE);

	return 0;

exit_fail:
	rt73usb_uninitialize(rt2x00dev);
	return -EIO;
}

static void rt73usb_disable_radio(struct rt2x00_dev *rt2x00dev)
{
	struct data_ring *ring;
	u32 reg;
	unsigned int i;

	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO))
		return;

	ieee80211_netif_oper(rt2x00dev->hw, NETIF_STOP);
	ieee80211_stop_queues(rt2x00dev->hw);

	/*
	 * Disable LED
	 */
	rt73usb_disable_led(rt2x00dev);

	CLEAR_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO);

	rt2x00_register_write(rt2x00dev, MAC_CSR10, 0x00001818);

	/*
	 * Disable synchronisation.
	 */
	rt2x00_register_write(rt2x00dev, TXRX_CSR9, 0);

	/*
	 * Cancel RX and TX.
	 */
	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR0_DISABLE_RX, 1);
	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);

	rt2x00_vendor_request(rt2x00dev, USB_RX_CONTROL,
		USB_VENDOR_REQUEST_OUT, 0x00, 0x00, NULL, 0, REGISTER_TIMEOUT);

	ring = &rt2x00dev->ring[RING_RX];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00_urb(&ring->entry[i]));

	ring = &rt2x00dev->ring[RING_AC_VO];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00_urb(&ring->entry[i]));

	ring = &rt2x00dev->ring[RING_AC_VI];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00_urb(&ring->entry[i]));

	ring = &rt2x00dev->ring[RING_AC_BE];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00_urb(&ring->entry[i]));

	ring = &rt2x00dev->ring[RING_AC_BK];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00_urb(&ring->entry[i]));

	ring = &rt2x00dev->ring[RING_PRIO];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00_urb(&ring->entry[i]));

	ring = &rt2x00dev->ring[RING_BEACON];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00_urb(&ring->entry[i]));
}

/*
 * RTS frame creation.
 */
static struct sk_buff* rt73usb_create_rts(struct rt2x00_dev *rt2x00dev,
	struct ieee80211_hdr *hdr, unsigned short duration)
{
	struct ieee80211_hdr *ieee80211hdr;
	struct sk_buff *skb;
	u16 frame_control;

	skb = dev_alloc_skb(IEEE80211_HEADER);
	if (!skb)
		return NULL;

	/*
	 * Copy the entire header over to RTS frame.
	 */
	memcpy(skb_put(skb, IEEE80211_HEADER), hdr, IEEE80211_HEADER);
	ieee80211hdr = (struct ieee80211_hdr*)skb->data;

	frame_control = IEEE80211_FTYPE_CTL | IEEE80211_STYPE_RTS;
	ieee80211hdr->frame_control = cpu_to_le16(frame_control);

	ieee80211hdr->duration_id = cpu_to_le16(duration);

	ieee80211hdr->seq_ctrl = 0;

	return skb;
}

/*
 * TX descriptor initialization
 */
static void rt73usb_write_tx_desc(struct rt2x00_dev *rt2x00dev,
	struct data_desc *txd, struct sk_buff *skb,
	struct ieee80211_tx_control *control)
{
	struct ieee80211_hdr *ieee80211hdr = (struct ieee80211_hdr*)skb->data;
	struct data_ring *ring;
	int tx_rate;
	u32 word;
	u32 length;
	u32 residual;
	u16 length_high;
	u16 length_low;
	u16 frame_control;
	u16 seq_ctrl;
	char rts_frame;
	char ofdm_rate;
	char req_timestamp;
	char more_frag;
	char req_seq;
	char ifs;
	char queue;
	u8 signal;
	u8 service;
	u8 bitrate;

	/*
	 * We require the ring structure this packet is being send to.
	 */
	ring = rt2x00_get_ring(rt2x00dev, control->queue);
	if (unlikely(!ring))
		return;

	/*
	 * Read required fields from ieee80211 header.
	 */
	frame_control = le16_to_cpu(ieee80211hdr->frame_control);
	seq_ctrl = le16_to_cpu(ieee80211hdr->seq_ctrl);

	/*
	 * Check if this frame is a RTS frame.
	 */
	rts_frame = is_rts_frame(frame_control);

	/*
	 * Check which rate should be used for this frame.
	 */
	if (rts_frame && control->rts_cts_rate)
		tx_rate = control->rts_cts_rate;
	else
		tx_rate = control->tx_rate;

	/*
	 * Are we working with OFDM rates.
	 */
	ofdm_rate = !!(DEVICE_GET_RATE_FIELD(tx_rate, RATEMASK) &
		       DEV_OFDM_RATE);

	/*
	 * Check if more fragments will follow this frame.
	 */
	more_frag = !!(ieee80211_get_morefrag(ieee80211hdr));

	/*
	 * Check if we require to enable the hw sequence counter.
	 */
	req_seq = !!(rt2x00_require_sequence(ieee80211hdr));

	/*
	 * Beacons and probe responses require the tsf timestamp
	 * to be inserted into the frame.
	 */
	req_timestamp = !!(control->queue == IEEE80211_TX_QUEUE_BEACON ||
		  	   control->pkt_type == PKT_PROBE_RESP);

	/*
	 * Determine with what IFS priority this frame should be send.
	 * Set ifs to IFS_SIFS when the this is not the first fragment,
	 * or this fragment came after RTS/CTS.
	 */
	if (((seq_ctrl & IEEE80211_SCTL_FRAG) > 0) || rts_frame)
		ifs = IFS_SIFS;
	else
		ifs = IFS_BACKOFF;

	/*
	 * Determine queue identification number.
	 */
	if (control->queue < rt2x00dev->hw->queues)
		queue = control->queue;
	else
		queue = 15;

	/*
	 * Add 4 bytes for FCS.
	 */
	length = skb->len + FCS_LEN;

	/*
	 * How the length should be processed depends
	 * on if we are working with OFDM rates or not.
	 */
	if (ofdm_rate) {
		residual = 0;
		length_high = (length >> 6) & 0x3f;
		length_low = (length & 0x3f);

	} else {
		bitrate = DEVICE_GET_RATE_FIELD(tx_rate, RATE);

		/*
		 * Convert length to microseconds.
		 */
		residual = get_duration_res(length, bitrate);
		length = get_duration(length, bitrate);

		if (residual != 0)
			length++;

		length_high = length >> 8;
		length_low = length & 0xff;
	}

	/*
	 * Create the signal and service values.
	 */
	signal = DEVICE_GET_RATE_FIELD(tx_rate, PLCP);
	if (DEVICE_GET_RATE_FIELD(tx_rate, PREAMBLE))
		signal |= 0x08;

	service = 0x04;
	if (residual <= (8 % 11))
		service |= 0x80;

	/*
	 * Start writing the descriptor words.
	 */
	rt2x00_desc_read(txd, 1, &word);
	rt2x00_set_field32(&word, TXD_W1_HOST_Q_ID, queue);
	rt2x00_set_field32(&word, TXD_W1_AIFSN, ring->tx_params.aifs);
	rt2x00_set_field32(&word, TXD_W1_CWMIN, ring->tx_params.cw_min);
	rt2x00_set_field32(&word, TXD_W1_CWMAX, ring->tx_params.cw_max);
	rt2x00_set_field32(&word, TXD_W1_IV_OFFSET, IEEE80211_HEADER);
	rt2x00_set_field32(&word, TXD_W1_HW_SEQUENCE, req_seq);
	rt2x00_desc_write(txd, 1, word);

	rt2x00_desc_read(txd, 2, &word);
	rt2x00_set_field32(&word, TXD_W2_PLCP_SIGNAL, signal);
	rt2x00_set_field32(&word, TXD_W2_PLCP_SERVICE, service);
	rt2x00_set_field32(&word, TXD_W2_PLCP_LENGTH_LOW, length_low);
	rt2x00_set_field32(&word, TXD_W2_PLCP_LENGTH_HIGH, length_high);
	rt2x00_desc_write(txd, 2, word);

	rt2x00_desc_read(txd, 5, &word);
	rt2x00_set_field32(&word, TXD_W5_TX_POWER,
		TXPOWER_TO_DEV(control->power_level));
	rt2x00_set_field32(&word, TXD_W5_WAITING_DMA_DONE_INT, 1);
	rt2x00_desc_write(txd, 5, word);

	rt2x00_desc_read(txd, 0, &word);
	rt2x00_set_field32(&word, TXD_W0_VALID, 1);
	rt2x00_set_field32(&word, TXD_W0_MORE_FRAG, more_frag);
	rt2x00_set_field32(&word, TXD_W0_ACK,
		!(control->flags & IEEE80211_TXCTL_NO_ACK));
	rt2x00_set_field32(&word, TXD_W0_TIMESTAMP, req_timestamp);
	rt2x00_set_field32(&word, TXD_W0_OFDM, ofdm_rate);
	rt2x00_set_field32(&word, TXD_W0_IFS, ifs);
	rt2x00_set_field32(&word, TXD_W0_RETRY_MODE, 0);
	rt2x00_set_field32(&word, TXD_W0_TKIP_MIC, 0);
	rt2x00_set_field32(&word, TXD_W0_DATABYTE_COUNT, skb->len);
	rt2x00_set_field32(&word, TXD_W0_CIPHER_ALG, CIPHER_NONE);
	rt2x00_desc_write(txd, 0, word);
}

/*
 * Interrupt functions.
 */
static void rt73usb_beacondone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_dev *rt2x00dev = ring->rt2x00dev;
	struct data_entry *entry = rt2x00_get_data_entry(
		&rt2x00dev->ring[RING_BEACON]);
	struct sk_buff *skb;

	skb = ieee80211_beacon_get(rt2x00dev->hw,
		rt2x00dev->interface.id, &entry->tx_status.control);
	if (!skb)
		return;

	rt73usb_beacon_update(rt2x00dev->hw, skb, &entry->tx_status.control);

	dev_kfree_skb_any(skb);
}

static void rt73usb_rxdone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_dev *rt2x00dev = ring->rt2x00dev;
	struct data_entry *entry;
	struct sk_buff *skb;
	struct data_desc *rxd;
	struct urb *urb;
	u32 word0;
	u32 word1;
	u16 size;

	while (1) {
		entry = rt2x00_get_data_entry(ring);
		rxd = rt2x00_desc_addr(entry);
		urb = rt2x00_urb(entry);
		rt2x00_desc_read(rxd, 0, &word0);
		rt2x00_desc_read(rxd, 1, &word1);

		if (GET_FLAG(entry, ENTRY_OWNER_NIC))
			break;

		/*
		 * There has been a problem. Ignore packet.
		 */
		if (urb->status) {
			SET_FLAG(entry, ENTRY_OWNER_NIC);
			usb_submit_urb(urb, GFP_ATOMIC);
			rt2x00_ring_index_inc(ring);
			continue;
		}

		size = rt2x00_get_field32(word0, RXD_W0_DATABYTE_COUNT);

		/*
		 * TODO: Don't we need to keep statistics
		 * updated about events like CRC and physical errors?
		 */
		if (!rt2x00_get_field32(word0, RXD_W0_CRC) &&
		    !rt2x00_get_field32(word0, RXD_W0_CIPHER_ERROR)) {
			skb = dev_alloc_skb(size + NET_IP_ALIGN);
			if (!skb)
				break;

			skb_reserve(skb, NET_IP_ALIGN);

			memcpy(skb_put(skb, size),
				rt2x00_data_addr(entry), size);

			rt2x00dev->rx_params.rate = device_signal_to_rate(
				&rt2x00dev->hw->modes[1],
				rt2x00_get_field32(word1, RXD_W1_SIGNAL),
				rt2x00_get_field32(word0, RXD_W0_OFDM));

			rt2x00dev->rx_params.ssi =
				rt2x00_get_field32(word1, RXD_W1_RSSI);

			/*
			 * The following set of assignments supply
			 * additional data for wireless statistics. The
			 * necessary quantities are (1) the "noise" value
			 * in dBm, and (2) the "signal" value in dBm.
			 */
			rt2x00dev->rx_params.noise =
				rt2x00_get_link_noise(&rt2x00dev->link);
			rt2x00dev->rx_params.signal = rt2x00dev->rx_params.ssi
				- rt2x00dev->hw->maxssi;

			__ieee80211_rx(rt2x00dev->hw,
				skb, &rt2x00dev->rx_params);

			/*
			 * Update link statistics
			 */
			rt2x00_update_link_rssi(&rt2x00dev->link,
				rt2x00dev->rx_params.ssi);
		}

		SET_FLAG(entry, ENTRY_OWNER_NIC);
		usb_submit_urb(urb, GFP_ATOMIC);

		rt2x00_ring_index_inc(ring);
	}
}

static void rt73usb_txdone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_dev *rt2x00dev = ring->rt2x00dev;
	struct data_entry *entry;
	struct data_desc *txd;
	struct urb *urb;
	u32 word;
	int tx_status;
	int ack;
	int rts;

	while (!rt2x00_ring_empty(ring)) {
		entry = rt2x00_get_data_entry_done(ring);
		txd = rt2x00_desc_addr(entry);
		urb = rt2x00_urb(entry);
		rt2x00_desc_read(txd, 0, &word);

		if (GET_FLAG(entry, ENTRY_OWNER_NIC))
			break;

		entry->tx_status.flags = 0;
		entry->tx_status.queue_length = entry->ring->stats.limit;
		entry->tx_status.queue_number = entry->tx_status.control.queue;

		/*
		 * Check if we have received an
		 * ACK response when ACK was requested and status
		 * was succesfull.
		 */
		ack = rt2x00_get_field32(word, TXD_W0_ACK);
		rts = GET_FLAG(entry, ENTRY_RTS_FRAME);
		tx_status = !urb->status ? TX_SUCCESS : TX_FAIL_RETRY;
		rt2x00_update_tx_stats(rt2x00dev, &entry->tx_status, tx_status,
				       ack, rts);

		rt2x00_bbp_read(rt2x00dev, 32,
			(u8*)&entry->tx_status.ack_signal);

		/*
		 * If this is not an RTS frame send the tx_status to d80211,
		 * that method also cleans up the skb structure. When this
		 * is a RTS frame, that it is our job to clean this structure up.
		 */
		if (!rts)
			ieee80211_tx_status(rt2x00dev->hw,
				entry->skb, &entry->tx_status);
		else
			dev_kfree_skb(entry->skb);

		CLEAR_FLAG(entry, ENTRY_RTS_FRAME);
		entry->skb = NULL;

		rt2x00_ring_index_done_inc(entry->ring);
	}

	/*
	 * Check if we are waiting on an empty queue
	 * to start scanning.
	 */
	if (rt2x00dev->scan &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_AC_VO]) &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_AC_VI]) &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_AC_BE]) &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_AC_BK]) &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_PRIO]))
		rt2x00_signal_scan(rt2x00dev->scan, SCANNING_READY);

	/*
	 * If the data ring was full before the txdone handler
	 * we must make sure the packet queue in the d80211 stack
	 * is reenabled when the txdone handler has finished.
	 */
	entry = ring->entry;
	if (!rt2x00_ring_full(ring))
		ieee80211_wake_queue(rt2x00dev->hw,
			entry->tx_status.control.queue);
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 19)
static void rt73usb_interrupt(struct urb *urb)
#else
static void rt73usb_interrupt(struct urb *urb, struct pt_regs *regs)
#endif
{
	struct data_entry *entry = (struct data_entry*)urb->context;
	struct rt2x00_dev *rt2x00dev = entry->ring->rt2x00dev;

	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO))
		return;

	CLEAR_FLAG(entry, ENTRY_OWNER_NIC);

	if (urb->status)
		return;

	queue_work(rt2x00dev->workqueue, &entry->ring->irq_work);
}

/*
 * IEEE80211 stack callback functions.
 */
static int rt73usb_tx(struct ieee80211_hw *hw,
	struct sk_buff *skb, struct ieee80211_tx_control *control)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	struct ieee80211_hdr *ieee80211hdr = (struct ieee80211_hdr*)skb->data;
	struct usb_device *usb_dev =
		interface_to_usbdev(rt2x00dev_usb(rt2x00dev));
	struct data_ring *ring;
	struct data_entry *entry;
	struct data_desc *txd;
	struct sk_buff *skb_rts;
	u16 frame_control;
	int res;
	int length;

	/*
	 * Determine which ring to put packet on.
	 */
	ring = rt2x00_get_ring(rt2x00dev, control->queue);
	if (unlikely(!ring)) {
		ERROR("Attempt to send packet over invalid queue %d.\n"
			"Please file bug report to %s.\n",
			control->queue, DRV_PROJECT);
		dev_kfree_skb_any(skb);
		return NETDEV_TX_OK;
	}

	if (rt2x00_ring_full(ring)) {
		ieee80211_stop_queue(hw, control->queue);
		return NETDEV_TX_BUSY;
	}

	/*
	 * If RTS is required. and this frame is not RTS,
	 * create and queue that frame first.
	 */
	frame_control = le16_to_cpu(ieee80211hdr->frame_control);
	if ((control->flags & IEEE80211_TXCTL_USE_RTS_CTS) &&
	    !is_rts_frame(frame_control)) {
		skb_rts = rt73usb_create_rts(rt2x00dev,
				ieee80211hdr, control->rts_cts_duration);
		if (!skb_rts) {
			WARNING("Failed to create RTS frame.\n");
			return NETDEV_TX_BUSY;
		}

		res = rt73usb_tx(hw, skb_rts, control);
		if (res) {
			WARNING("Failed to send RTS frame.\n");
			return res;
		}
	}

	entry = rt2x00_get_data_entry(ring);
	txd = rt2x00_desc_addr(entry);

	if (GET_FLAG(entry, ENTRY_OWNER_NIC)) {
		ERROR("Arrived at non-free entry in the non-full queue %d.\n"
			"Please file bug report to %s.\n",
			control->queue, DRV_PROJECT);
		ieee80211_stop_queue(hw, control->queue);
		return NETDEV_TX_BUSY;
	}

	if (entry->flags)
		ERROR("Arrived at broken entry (flags 0x%08x) in queue %d.\n"
			"Please file bug report to %s.\n",
			control->queue, entry->flags, DRV_PROJECT);

	memcpy(rt2x00_data_addr(entry), skb->data, skb->len);
	rt73usb_write_tx_desc(rt2x00dev, txd, skb, control);
	memcpy(&entry->tx_status.control, control, sizeof(*control));
	if (is_rts_frame(frame_control))
		SET_FLAG(entry, ENTRY_RTS_FRAME);
	entry->skb = skb;

	/*
	 * Length passed to usb_fill_urb cannot be an odd number,
	 * so add 1 byte to make it even.
	 */
	length = skb->len + ring->desc_size;
	if (length % 2)
		length++;

	SET_FLAG(entry, ENTRY_OWNER_NIC);
	usb_fill_bulk_urb(
		rt2x00_urb(entry),
		usb_dev,
		usb_sndbulkpipe(usb_dev, 1),
		entry->data_addr,
		length,
		rt73usb_interrupt,
		entry);
	usb_submit_urb(rt2x00_urb(entry), GFP_ATOMIC);

	rt2x00_ring_index_inc(ring);

	if (rt2x00_ring_full(ring))
		ieee80211_stop_queue(hw, control->queue);

	return NETDEV_TX_OK;
}

static int rt73usb_reset(struct ieee80211_hw *hw)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	rt73usb_disable_radio(rt2x00dev);
	return rt73usb_enable_radio(rt2x00dev);
}

static int rt73usb_add_interface(struct ieee80211_hw *hw,
	struct ieee80211_if_init_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	int status;

	/*
	 * We only support 1 non-monitor interface.
	 */
	if (conf->type != IEEE80211_IF_TYPE_MNTR &&
	    GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
		return -ENOBUFS;

	/*
	 * Add the new interface.
	 */
	rt2x00_add_interface(&rt2x00dev->interface, conf);

	/*
	 * Initialize interface, and enable the radio when this
	 * is the first interface that is brought up.
	 */
	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO)) {
		/*
		 * Before doing anything else, the MAC address
		 * of this device should be initialized correctly.
		 */
		rt73usb_config_mac_address(rt2x00dev, conf->mac_addr);

		/*
		 * Initialize the device.
		 */
		status = rt73usb_initialize(rt2x00dev);
		if (status)
			return status;

		/*
		 * Enable radio.
		 */
		status = rt73usb_enable_radio(rt2x00dev);
		if (status)
			return status;

		/*
		 * Enable periodic link tuning if this is a non-monitor
		 * interface. Also set the INTERFACE_INITIALIZED FLAG
		 * to prevent new non-monitor interfaces to be added.
		 */
		if (conf->type != IEEE80211_IF_TYPE_MNTR) {
			queue_delayed_work(rt2x00dev->workqueue,
				&rt2x00dev->link.work, LINK_TUNE_INTERVAL);
			SET_FLAG(rt2x00dev, INTERFACE_INITIALIZED);
		} else
			SET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR);
	}

	return 0;
}

static void rt73usb_remove_interface(struct ieee80211_hw *hw,
	struct ieee80211_if_init_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	/*
	 * We only support 1 non-monitor interface.
	 */
	if (conf->type != IEEE80211_IF_TYPE_MNTR &&
	    !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
		return;

	/*
	 * Remove the interface.
	 */
	rt2x00_remove_interface(&rt2x00dev->interface, conf);

	/*
	 * When this is a non-monitor mode,
	 * stop the periodic link tuning,
	 * and clear the INTERFACE_INITIALIZED FLAG to allow
	 * new non-monitor interfaces to be added.
	 */
	if (conf->type != IEEE80211_IF_TYPE_MNTR) {
		cancel_rearming_delayed_workqueue(rt2x00dev->workqueue,
			&rt2x00dev->link.work);
		CLEAR_FLAG(rt2x00dev, INTERFACE_INITIALIZED);
	}

	/*
	 * Disable radio if this was the last interface
	 * that was working with this device.
	 */
	if (!GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) &&
	    !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
		rt73usb_disable_radio(rt2x00dev);

	/*
	 * Check if we still have 1 non-monitor or a monitor
	 * interface enabled. In that case we should update the
	 * registers.
	 */
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) ^
	    GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED)) {
		if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
			rt73usb_config_type(rt2x00dev,
				rt2x00dev->interface.type);
		else
			rt73usb_config_type(rt2x00dev,
				IEEE80211_IF_TYPE_MNTR);
	}
}

static int rt73usb_config(struct ieee80211_hw *hw,
	struct ieee80211_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	u32 reg;

	/*
	 * Check if we need to disable the radio,
	 * if this is not the case, at least the RX must be disabled.
	 */
	if (GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO)) {
		if (!conf->radio_enabled)
			rt73usb_disable_radio(rt2x00dev);
		else {
			rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
			rt2x00_set_field32(&reg, TXRX_CSR0_DISABLE_RX, 1);
			rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);
		}
	}

	rt73usb_config_channel(rt2x00dev,
		conf->channel_val, conf->channel, conf->freq,
		conf->power_level);
	rt73usb_config_txpower(rt2x00dev, conf->power_level);
	rt73usb_config_antenna(rt2x00dev, conf->antenna_sel, conf->phymode);
	rt73usb_config_duration(rt2x00dev,
		(conf->flags & IEEE80211_CONF_SHORT_SLOT_TIME));
	rt73usb_config_phymode(rt2x00dev, conf->phymode);

	/*
	 * Reenable RX only if the radio should be on.
	 */
	if (GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO)) {
		rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
		rt2x00_set_field32(&reg, TXRX_CSR0_DISABLE_RX, 0);
		rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);
	} else if (conf->radio_enabled)
		return rt73usb_enable_radio(rt2x00dev);

	return 0;
}

static int rt73usb_config_interface(struct ieee80211_hw *hw, int if_id,
	struct ieee80211_if_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	/*
	 * Monitor mode does not need configuring.
	 * If the given type does not match the configured type,
	 * there has been a problem.
	 */
	if (conf->type == IEEE80211_IF_TYPE_MNTR)
		return 0;
	else if (conf->type != rt2x00dev->interface.type)
		return -EINVAL;

	/*
	 * If the interface does not work in master mode,
	 * then the bssid value in the interface structure
	 * should now be set.
	 */
	if (conf->type != IEEE80211_IF_TYPE_AP)
		memcpy(&rt2x00dev->interface.bssid, conf->bssid, ETH_ALEN);

	/*
	 * Enable configuration.
	 */
	rt73usb_config_type(rt2x00dev, rt2x00dev->interface.type);
	rt73usb_config_bssid(rt2x00dev, &rt2x00dev->interface.bssid[0]);

	return 0;
}

static void rt73usb_set_multicast_list(struct ieee80211_hw *hw,
	unsigned short flags, int mc_count)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	int update = 0;

	if (GET_FLAG(rt2x00dev, INTERFACE_ENABLED_PROMISC)) {
		if (!(flags & IFF_PROMISC)) {
			rt2x00dev->interface.promisc = 0;
			update = 1;
		}
	} else {
		if (flags & IFF_PROMISC) {
			rt2x00dev->interface.promisc = 1;
			update = 1;
		}
	}

	/*
	 * Monitor mode works with PROMISC mode forced on,
	 * so there is nothing to be done here.
	 */
	if (update && !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR)) {
		NOTICE("For the moment promisc mode is ignored");
	/*	queue_work(rt2x00dev->workqueue,
			&rt2x00dev->interface.work);*/
	}
}

static void rt73usb_scan(void *data)
{
	struct rt2x00_dev *rt2x00dev = data;

	if (unlikely(!rt2x00dev->scan))
		return;

	/*
	 * Before we can start switch the channel for scanning
	 * we need to wait untill all TX rings are empty to
	 * guarentee that all frames are send on the correct channel.
	 */
	if (rt2x00_wait_scan(rt2x00dev->scan))
		goto exit;

	/*
	 * Switch channel and update active info for RX.
	 */
	if (rt2x00dev->scan->state == IEEE80211_SCAN_START) {
		rt73usb_config_phymode(rt2x00dev,
			rt2x00dev->scan->conf.scan_phymode);

		rt73usb_config_channel(rt2x00dev,
			rt2x00dev->scan->conf.scan_channel_val,
			rt2x00dev->scan->conf.scan_channel,
			rt2x00dev->scan->conf.scan_freq,
			rt2x00dev->scan->conf.scan_power_level);
	} else {
		rt73usb_config_phymode(rt2x00dev,
			rt2x00dev->scan->conf.running_phymode);

		rt73usb_config_channel(rt2x00dev,
			rt2x00dev->scan->conf.running_channel_val,
			rt2x00dev->scan->conf.running_channel,
			rt2x00dev->scan->conf.running_freq,
			rt2x00dev->scan->conf.scan_power_level);
	}

exit:
	kfree(rt2x00dev->scan);
	rt2x00dev->scan = NULL;
}

static int rt73usb_passive_scan(struct ieee80211_hw *hw,
	int state, struct ieee80211_scan_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	/*
	 * Check if we are not busy with the previous
	 * passive scan request.
	 */
	if (rt2x00dev->scan)
		return -EBUSY;

	/*
	 * Check if the radio is enabled.
	 */
	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO))
		return -EIO;

	/*
	 * Allocate scanning structure to store scanning info.
	 */
	rt2x00dev->scan = kzalloc(sizeof(struct scanning), GFP_ATOMIC);
	if (!rt2x00dev->scan)
		return -ENOMEM;

	/*
	 * Check if we have to send a packet before the
	 * channel switch.
	 */
	if (conf->skb) {
		if (rt73usb_tx(hw, conf->skb, conf->tx_control))
			goto exit;
	}

	/*
	 * Initialize Scanning structure.
	 */
	rt2x00_start_scan(rt2x00dev->scan, conf, state);

	/*
	 * Queue work.
	 */
	INIT_WORK(&rt2x00dev->scan->work, rt73usb_scan, rt2x00dev);
	if (!queue_work(rt2x00dev->workqueue, &rt2x00dev->scan->work))
		goto exit;

	return 0;

exit:
	kfree(rt2x00dev->scan);
	rt2x00dev->scan = NULL;

	return -EIO;
}

static int rt73usb_get_stats(struct ieee80211_hw *hw,
	struct ieee80211_low_level_stats *stats)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	u32 reg;

	/*
	 * Update FCS error count from register.
	 * The dot11ACKFailureCount, dot11RTSFailureCount and
	 * dot11RTSSuccessCount are updated in interrupt time.
	 */
	rt2x00_register_read(rt2x00dev, STA_CSR0, &reg);
	rt2x00dev->low_level_stats.dot11FCSErrorCount +=
		rt2x00_get_field32(reg, STA_CSR0_FCS_ERROR);

	memcpy(stats, &rt2x00dev->low_level_stats, sizeof(*stats));

	return 0;
}

static int rt73usb_set_retry_limit(struct ieee80211_hw *hw,
	u32 short_retry, u32 long_retry)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	u32 reg;

	rt2x00_register_read(rt2x00dev, TXRX_CSR4, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR4_LONG_RETRY_LIMIT, long_retry);
	rt2x00_set_field32(&reg, TXRX_CSR4_SHORT_RETRY_LIMIT, short_retry);
	rt2x00_register_write(rt2x00dev, TXRX_CSR4, reg);

	return 0;
}

static int rt73usb_conf_tx(struct ieee80211_hw *hw,
	int queue, const struct ieee80211_tx_queue_params *params)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	struct data_ring *ring;

	ring = rt2x00_get_ring(rt2x00dev, queue);
	if (unlikely(!ring))
		return -EINVAL;

	memcpy(&ring->tx_params, params, sizeof(*params));

	/*
	 * The passed variables are stored as real value ((2^n)-1).
	 * RT73 registers require to know the bit number 'n'.
	 */
	if (params->cw_min)
		ring->tx_params.cw_min = HIGHEST_BIT16(params->cw_min) + 1;
	else
		ring->tx_params.cw_min = 4; /* cw_min: 2^4 = 16. */

	if (params->cw_max)
		ring->tx_params.cw_max = HIGHEST_BIT16(params->cw_max) + 1;
	else
		ring->tx_params.cw_max = 10; /* cw_min: 2^10 = 1024. */

	if (!params->aifs)
		ring->tx_params.aifs = 2;

	INFO("Configured TX ring %d - CWmin: %d, CWmax: %d, Aifs: %d.\n",
		queue, ring->tx_params.cw_min, ring->tx_params.cw_max,
		ring->tx_params.aifs);

	return 0;
}

static int rt73usb_get_tx_stats(struct ieee80211_hw *hw,
	struct ieee80211_tx_queue_stats *stats)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA0],
		&rt2x00dev->ring[RING_AC_VO].stats,
		sizeof(rt2x00dev->ring[RING_AC_VO].stats));
	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA1],
		&rt2x00dev->ring[RING_AC_VI].stats,
		sizeof(rt2x00dev->ring[RING_AC_VI].stats));
	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA2],
		&rt2x00dev->ring[RING_AC_BE].stats,
		sizeof(rt2x00dev->ring[RING_AC_BE].stats));
	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA3],
		&rt2x00dev->ring[RING_AC_BK].stats,
		sizeof(rt2x00dev->ring[RING_AC_BK].stats));
	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA4],
		&rt2x00dev->ring[RING_PRIO].stats,
		sizeof(rt2x00dev->ring[RING_PRIO].stats));

	return 0;
}

static u64 rt73usb_get_tsf(struct ieee80211_hw *hw)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	u64 tsf;
	u32 reg;

	rt2x00_register_read(rt2x00dev, TXRX_CSR13, &reg);
	tsf = (u64)rt2x00_get_field32(reg, TXRX_CSR13_HIGH_TSFTIMER) << 32;
	rt2x00_register_read(rt2x00dev, TXRX_CSR12, &reg);
	tsf |= rt2x00_get_field32(reg, TXRX_CSR12_LOW_TSFTIMER);

	return tsf;
}

static void rt73usb_reset_tsf(struct ieee80211_hw *hw)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	rt2x00_register_write(rt2x00dev, TXRX_CSR12, 0);
	rt2x00_register_write(rt2x00dev, TXRX_CSR13, 0);
}

static int rt73usb_beacon_update(struct ieee80211_hw *hw,
	struct sk_buff *skb, struct ieee80211_tx_control *control)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	struct usb_device *usb_dev =
		interface_to_usbdev(rt2x00dev_usb(rt2x00dev));
	struct data_entry *entry;

	entry = rt2x00_get_data_entry(&rt2x00dev->ring[RING_BEACON]);

	/*
	 * Just in case the ieee80211 doesn't set this,
	 * but we need this queue set for the descriptor
	 * initialization.
	 */
	control->queue = IEEE80211_TX_QUEUE_BEACON;

	/*
	 * Update the beacon entry.
	 */
	memcpy(rt2x00_data_addr(entry), skb->data, skb->len);
	rt73usb_write_tx_desc(rt2x00dev,
		rt2x00_desc_addr(entry), skb, control);

	SET_FLAG(entry, ENTRY_OWNER_NIC);
	usb_fill_bulk_urb(
		rt2x00_urb(entry),
		usb_dev,
		usb_sndbulkpipe(usb_dev, 1),
		entry->data_addr,
		skb->len + rt2x00dev->ring[RING_BEACON].desc_size,
		rt73usb_interrupt,
		entry);
	usb_submit_urb(rt2x00_urb(entry), GFP_ATOMIC);

	return 0;
}

static struct ieee80211_ops rt73usb_d80211_ops = {
	.tx			= rt73usb_tx,
	.reset			= rt73usb_reset,
	.add_interface		= rt73usb_add_interface,
	.remove_interface	= rt73usb_remove_interface,
	.config			= rt73usb_config,
	.config_interface	= rt73usb_config_interface,
	.set_multicast_list	= rt73usb_set_multicast_list,
	.passive_scan		= rt73usb_passive_scan,
	.get_stats		= rt73usb_get_stats,
	.set_retry_limit	= rt73usb_set_retry_limit,
	.conf_tx		= rt73usb_conf_tx,
	.get_tx_stats		= rt73usb_get_tx_stats,
	.get_tsf		= rt73usb_get_tsf,
	.reset_tsf		= rt73usb_reset_tsf,
	.beacon_update		= rt73usb_beacon_update,
	.ethtool		= &rt73usb_ethtool_ops,
};

/*
 * Device initialization functions.
 */
static int rt73usb_alloc_eeprom(struct rt2x00_dev *rt2x00dev)
{
	/*
	 * Allocate the eeprom memory, check the eeprom width
	 * and copy the entire eeprom into this allocated memory.
	 */
	rt2x00dev->eeprom = kzalloc(EEPROM_SIZE, GFP_KERNEL);
	if (!rt2x00dev->eeprom)
		return -ENOMEM;

	rt2x00_vendor_request(
		rt2x00dev, USB_EEPROM_READ, USB_VENDOR_REQUEST_IN,
		EEPROM_BASE * sizeof(u16), 0x00, rt2x00dev->eeprom,
		EEPROM_SIZE * sizeof(u16), REGISTER_TIMEOUT);

	return 0;
}

static int rt73usb_alloc_rings(struct rt2x00_dev *rt2x00dev)
{
	unsigned int i;

	rt2x00dev->ring = kzalloc(
		sizeof(struct data_ring) * RING_NUM, GFP_KERNEL);
	if (!rt2x00dev->ring) {
		ERROR("Ring allocation failed.\n");
		return -ENOMEM;
	}

	for (i = 0; i < RING_NUM; i++) {
		rt2x00dev->ring[i].rt2x00dev = rt2x00dev;

		/*
		 * Initialize ring parameters.
		 * cw_min: 2^5 = 32.
		 * cw_max: 2^10 = 1024.
		 */
		rt2x00dev->ring[i].tx_params.aifs = 2;
		rt2x00dev->ring[i].tx_params.cw_min = 5;
		rt2x00dev->ring[i].tx_params.cw_max = 10;
	}

	return 0;
}

static int rt73usb_init_eeprom(struct rt2x00_dev *rt2x00dev)
{
	u32 reg;
	u16 value;
	u16 eeprom;

	/*
	 * Read EEPROM word for configuration.
	 */
	rt2x00_eeprom_read(rt2x00dev, EEPROM_ANTENNA, &eeprom);

	/*
	 * Identify RF chipset.
	 */
	value = rt2x00_get_field16(eeprom, EEPROM_ANTENNA_RF_TYPE);
	rt2x00_register_read(rt2x00dev, MAC_CSR0, &reg);
	rt2x00_set_chip(&rt2x00dev->chip, RT73, value, reg);

	if (!rt2x00_rf(&rt2x00dev->chip, RF5226) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF2528) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF5225) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF2527))
		return -ENODEV;

	/*
	 * Identify default antenna configuration.
	 * Ralink devices have have antenna options for both TX as RX.
	 * The ieee80211 stack currently only provide the user to set
	 * 1 antenna, by default this is considered to be the TX antenna.
	 */
	rt2x00dev->hw->conf.antenna_sel = rt2x00_get_field16(eeprom,
		EEPROM_ANTENNA_TX_DEFAULT);

	/*
	 * Read the Frame type.
	 */
	if (rt2x00_get_field16(eeprom, EEPROM_ANTENNA_FRAME_TYPE))
		SET_FLAG(rt2x00dev, CONFIG_FRAME_TYPE);

	/*
	 * Read frequency offset.
	 */
	rt2x00_eeprom_read(rt2x00dev, EEPROM_FREQ, &eeprom);
	rt2x00dev->freq_offset = rt2x00_get_field16(eeprom,
					EEPROM_FREQ_OFFSET);
	if (rt2x00dev->freq_offset == 0xff)
		rt2x00dev->freq_offset = 0;

	/*
	 * Read external LNA informations.
	 */
	rt2x00_eeprom_read(rt2x00dev, EEPROM_NIC, &eeprom);
	if (eeprom == 0xffff)
		eeprom = 0;
	if (rt2x00_get_field16(eeprom, EEPROM_NIC_EXTERNAL_LNA))
		SET_FLAG(rt2x00dev, CONFIG_EXTERNAL_LNA);

	/*
	 * Store led settings, for correct led behaviour.
	 */
	rt2x00_eeprom_read(rt2x00dev, EEPROM_LED, &eeprom);

	/*
	 * If the eeprom value is invalid,
	 * switch to default led mode, and clear all bits.
	 */
	if (eeprom == 0xffff) {
		rt2x00dev->led_mode = LED_MODE_DEFAULT;
		eeprom = 0x0000;
	} else
		rt2x00dev->led_mode = rt2x00_get_field16(eeprom,
			EEPROM_LED_LED_MODE);

	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_LED_MODE,
		rt2x00dev->led_mode);
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_GPIO_0,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_0));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_GPIO_1,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_1));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_GPIO_2,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_2));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_GPIO_3,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_3));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_GPIO_4,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_4));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_ACT,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_ACT));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_READY_BG,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_RDY_G));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_READY_A,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_RDY_A));

	return 0;
}

static int rt73usb_init_hw_mac(struct rt2x00_dev *rt2x00dev)
{
	u8 *addr;

	/*
	 * Get the pointer to the MAC address in the EEPROM.
	 */
	addr = rt2x00_eeprom_addr(rt2x00dev, EEPROM_MAC_ADDR_0);

	/*
	 * Check if a valid MAC address is present.
	 */
	if (!is_valid_ether_addr(addr)) {
		ERROR("Invalid MAC address: " MAC_FMT ".\n", MAC_ARG(addr));
		return -EINVAL;
	}

	/*
	 * Write MAC address to register.
	 */
	rt73usb_config_mac_address(rt2x00dev, addr);

	/*
	 * Copy MAC address to the hw structure.
	 */
	memcpy(&rt2x00dev->hw->perm_addr, addr, ETH_ALEN);

	return 0;
}

static void rt73usb_init_hw_channels(struct rt2x00_dev *rt2x00dev,
	struct ieee80211_channel *channels)
{
	unsigned int i;
	u32 rf2_base;
	u16 eeprom;
	static const struct {
		unsigned int chip;
		u32 val[3];
	} rf[] = {
		{ RF5226,	{ 0x00002c0c, 0x00068255 } },
		{ RF2528,	{ 0x00002c0c, 0x00068255 } },
		{ RF5225,	{ 0x00002ccc, 0x00000000 } },
		{ RF2527,	{ 0x00002ccc, 0x00068455 } },
	};
	static const u32 vals[] = {
		0x00000786, 0x00000786, 0x0000078a, 0x0000078a,
		0x0000078e, 0x0000078e, 0x00000792, 0x00000792,
		0x00000796, 0x00000796, 0x0000079a, 0x0000079a,
		0x0000079e, 0x000007a2
	};

	/*
	 * Channel initialization.
	 * First we set the basic variables.
	 */
	for (i = 0; i < 13; i++) {
		channels[i].chan = i + 1;
		channels[i].freq = 2407 + ((i + 1) * 5);
		channels[i].flag = IEEE80211_CHAN_W_IBSS |
			IEEE80211_CHAN_W_ACTIVE_SCAN | IEEE80211_CHAN_W_SCAN;
		channels[i].antenna_max = 0xff;
	}

	channels[13].chan = 14;
	channels[13].freq = 2484;
	channels[13].flag = IEEE80211_CHAN_W_IBSS |
		IEEE80211_CHAN_W_ACTIVE_SCAN | IEEE80211_CHAN_W_SCAN;
	channels[13].antenna_max = 0xff;

	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5226)) {
		for (i = 14; i < 38; i++) {
			if (i < 22)
				channels[i].chan = 36;
			else if (i < 33)
				channels[i].chan = 100;
			else
				channels[i].chan = 149;
			channels[i].chan += ((i - 14) * 4);
			channels[i].freq = ((i - 13) + 1000) * 5;
			channels[i].flag = IEEE80211_CHAN_W_IBSS |
				IEEE80211_CHAN_W_ACTIVE_SCAN |
				IEEE80211_CHAN_W_SCAN;
			channels[i].antenna_max = 0xff;
		}
	}

	/*
	 * Set device specific value.
	 */
	rf2_base = 0;
	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF2527))
		rf2_base = 0x00004000;

	for (i = 0; i < ARRAY_SIZE(vals); i++)
		channels[i].val = vals[i] | rf2_base;

	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5226)) {
		static const u32 vals[] = {
			0x0000099a, 0x000009a2, 0x000009a6, 0x000009aa,
			0x000009ae, 0x000009b2, 0x000009ba, 0x000009be,
			0x00000a2a, 0x00000a2e, 0x00000a32, 0x00000a36,
			0x00000a3a, 0x00000a82, 0x00000a86, 0x00000a8a,
			0x00000a8e, 0x00000a92, 0x00000a9a, 0x00000aa2,
			0x00000aa6, 0x00000aae, 0x00000ab2, 0x00000ab6
		};

		struct ieee80211_channel *chan = channels + 14;

		for (i = 0; i < ARRAY_SIZE(vals); i++)
			(chan++)->val = vals[i];
	}

	/*
	 * Set TX power, each EEPROM TXpower entry
	 * contains the TXpower value for 2 channels.
	 */
	for (i = 0; i < EEPROM_TXPOWER_G_SIZE; i++) {
		rt2x00_eeprom_read(rt2x00dev,
			EEPROM_TXPOWER_G_START + i, &eeprom);

		channels[(i * 2)].power_level = TXPOWER_FROM_DEV(
			rt2x00_get_field16(eeprom, EEPROM_TXPOWER_G_1));

		channels[(i * 2) + 1].power_level = TXPOWER_FROM_DEV(
			rt2x00_get_field16(eeprom, EEPROM_TXPOWER_G_2));
	}

	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5226)) {
		for (; i < EEPROM_TXPOWER_A_SIZE; i++) {
			rt2x00_eeprom_read(rt2x00dev,
				EEPROM_TXPOWER_G_START + i, &eeprom);

			channels[(i * 2)].power_level = TXPOWER_FROM_DEV(
				rt2x00_get_field16(eeprom,
					EEPROM_TXPOWER_A_1));

			channels[(i * 2) + 1].power_level = TXPOWER_FROM_DEV(
				rt2x00_get_field16(eeprom,
					EEPROM_TXPOWER_A_2));
		}
	}

	/*
	 * Set device specific, but channel independent RF values.
	 */
	for (i = 0; i < ARRAY_SIZE(rf); i++) {
		if (rt2x00_rf(&rt2x00dev->chip, rf[i].chip)) {
			rt2x00dev->rf1 = rf[i].val[0];
			rt2x00dev->rf3 = rf[i].val[1];
		}
	}
}

static void rt73usb_init_hw_rates(struct rt2x00_dev *rt2x00dev,
	struct ieee80211_rate *rates)
{
	/*
	 * Rates initialization.
	 */
	device_rate_entry(&rates[0], 10, 0x001, 0x00, IEEE80211_RATE_CCK);
	device_rate_entry(&rates[1], 20, 0x003, 0x01, IEEE80211_RATE_CCK_2);
	device_rate_entry(&rates[2], 55, 0x007, 0x02, IEEE80211_RATE_CCK_2);
	device_rate_entry(&rates[3], 110, 0x00f, 0x03, IEEE80211_RATE_CCK_2);
	device_rate_entry(&rates[4], 60, 0x01f, 0x0b, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[5], 90, 0x03f, 0x0f, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[6], 120, 0x07f, 0x0a, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[7], 180, 0x0ff, 0x0e, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[8], 240, 0x1ff, 0x09, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[9], 360, 0x3ff, 0x0d, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[10], 480, 0x7ff, 0x08, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[11], 540, 0xfff, 0x0c, IEEE80211_RATE_OFDM);
}

static int rt73usb_init_hw_modes(struct rt2x00_dev *rt2x00dev)
{
	struct ieee80211_hw *hw = rt2x00dev->hw;
	int num_modes;
	int num_channels;

	/*
	 * RF2527 and RF2528 only supports 802.11b & 802.11g,
	 * so we should allocate 14 OFDM channels, 4 CCK rates
	 * and 8 OFDM rates.
	 * RF5225 and RF5226 also supports 802.11a, so allocate an
	 * additional 23 5.2GHz channels.
	 */
	num_modes = 2;
	num_channels = 14;
	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5226)) {
		num_modes = 3;
		num_channels = 38;
	}

	hw->num_modes = num_modes;
	hw->modes =
		kzalloc((sizeof(struct ieee80211_hw_modes) * num_modes),
		GFP_KERNEL);
	if (!hw->modes)
		goto exit;

	hw->modes->channels =
		kzalloc((sizeof(struct ieee80211_channel) * num_channels),
		GFP_KERNEL);
	if (!hw->modes->channels)
		goto exit_free_modes;

	hw->modes->rates =
		kzalloc((sizeof(struct ieee80211_rate) * 12),
		GFP_KERNEL);
	if (!hw->modes->rates)
		goto exit_free_channels;

	/*
	 * Intitialize 802.11g
	 * Rates: CCK, OFDM.
	 * Channels: OFDM.
	 */
	hw->modes[0].mode = MODE_IEEE80211G;
	hw->modes[0].num_channels = 14;
	hw->modes[0].num_rates = 12;

	/*
	 * Intitialize 802.11b
	 * Rates: CCK.
	 * Channels: OFDM.
	 */
	hw->modes[1].mode = MODE_IEEE80211B;
	hw->modes[1].num_channels = 14;
	hw->modes[1].num_rates = 4;
	hw->modes[1].channels = hw->modes[0].channels;
	hw->modes[1].rates = hw->modes[0].rates;

	/*
	 * Intitialize 802.11a
	 * Rates: OFDM.
	 * Channels: OFDM, UNII, HiperLAN2.
	 */
	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5226)) {
		hw->modes[2].mode = MODE_IEEE80211A;
		hw->modes[2].num_channels = 38;
		hw->modes[2].num_rates = 8;
		hw->modes[2].channels = &hw->modes[0].channels[14];
		hw->modes[2].rates = &hw->modes[0].rates[4];
	}

	rt73usb_init_hw_channels(rt2x00dev, hw->modes[0].channels);
	rt73usb_init_hw_rates(rt2x00dev, hw->modes[0].rates);

	return ieee80211_update_hw(hw);

exit_free_channels:
	kfree(hw->modes->channels);
	hw->modes->channels = NULL;

exit_free_modes:
	kfree(hw->modes);
	hw->modes = NULL;

exit:
	ERROR("Allocation ieee80211 modes failed.\n");
	return -ENOMEM;
}

static int rt73usb_init_hw(struct rt2x00_dev *rt2x00dev)
{
	int status;

	if (GET_FLAG(rt2x00dev, DEVICE_INITIALIZED_HW))
		return 0;

	rt2x00dev->hw->dev = &rt2x00dev_usb(rt2x00dev)->dev;

	/*
	 * Read MAC address from EEPROM.
	 */
	status = rt73usb_init_hw_mac(rt2x00dev);
	if (status)
		return status;

	/*
	 * Initialize all hw fields.
	 */
	rt2x00dev->hw->flags = IEEE80211_HW_HOST_GEN_BEACON |
		IEEE80211_HW_HOST_BROADCAST_PS_BUFFERING |
		IEEE80211_HW_WEP_INCLUDE_IV |
		IEEE80211_HW_DATA_NULLFUNC_ACK |
		IEEE80211_HW_NO_TKIP_WMM_HWACCEL |
		IEEE80211_HW_MONITOR_DURING_OPER;
	rt2x00dev->hw->extra_tx_headroom = 0;
	rt2x00dev->hw->maxssi = MAX_RX_SSI;
	rt2x00dev->hw->queues = RING_NUM_TX;

	status = rt73usb_init_hw_modes(rt2x00dev);
	if (status)
		return status;

	if (ieee80211_register_hw(rt2x00dev->hw))
		return -EIO;

	SET_FLAG(rt2x00dev, DEVICE_INITIALIZED_HW);

	return 0;
}

static void rt73usb_free_dev(struct rt2x00_dev *rt2x00dev)
{
	/*
	 * Free ring structures.
	 */
	kfree(rt2x00dev->ring);
	rt2x00dev->ring = NULL;

	/*
	 * Free workqueue.
	 */
	if (likely(rt2x00dev->workqueue)) {
		destroy_workqueue(rt2x00dev->workqueue);
		rt2x00dev->workqueue = NULL;
	}

	/*
	 * Free EEPROM memory.
	 */
	kfree(rt2x00dev->eeprom);

	/*
	 * Free ieee80211_hw memory.
	 */
	if (likely(rt2x00dev->hw->modes)) {
		kfree(rt2x00dev->hw->modes->channels);
		kfree(rt2x00dev->hw->modes->rates);
		kfree(rt2x00dev->hw->modes);
		rt2x00dev->hw->modes = NULL;
	}
}

static int rt73usb_alloc_dev(struct usb_interface *usb_intf,
	struct ieee80211_hw *hw)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	rt2x00dev->dev = usb_intf;
	rt2x00dev->hw = hw;

	/*
	 * Allocate eeprom data.
	 */
	if (rt73usb_alloc_eeprom(rt2x00dev))
		goto exit;

	/*
	 * Create workqueue.
	 */
	rt2x00dev->workqueue = create_singlethread_workqueue(DRV_NAME);
	if (!rt2x00dev->workqueue)
		return -ENODEV;

	/*
	 * Initialize configuration work.
	 */
	INIT_WORK(&rt2x00dev->link.work, rt73usb_link_tuner, rt2x00dev);

	/*
	 * Reset current working type.
	 */
	rt2x00dev->interface.type = -EINVAL;

	/*
	 * Intialize scanning attributes.
	 */
	rt2x00dev->scan = NULL;

	/*
	 * Allocate ring array.
	 */
	if (rt73usb_alloc_rings(rt2x00dev))
		goto exit;

	/*
	 * Initialize hardware.
	 */
	if (rt73usb_init_eeprom(rt2x00dev) ||
	    rt73usb_init_hw(rt2x00dev)) {
		ERROR("Failed to initialize device.\n");
		goto exit;
	}

	if (rt73usb_init_firmware(rt2x00dev)) {
		ERROR("Failed to load Firmware.\n");
		goto exit;
	}

	return 0;

exit:
	rt73usb_free_dev(rt2x00dev);

	return -ENODEV;
}

/*
 * USB driver handlers.
 */
static int rt73usb_probe(struct usb_interface *usb_intf,
	const struct usb_device_id *id)
{
	struct usb_device *usb_dev = interface_to_usbdev(usb_intf);
	struct ieee80211_hw *hw;
	int status;

	usb_dev = usb_get_dev(usb_dev);

	hw = ieee80211_alloc_hw(sizeof(struct rt2x00_dev),
		&rt73usb_d80211_ops);
	if (!hw) {
		ERROR("Failed to allocate hardware.\n");
		status = -ENOMEM;
		goto exit_put_device;
	}

	usb_set_intfdata(usb_intf, hw);

	status = rt73usb_alloc_dev(usb_intf, hw);
	if (status) {
		ERROR("Failed to allocate device.\n");
		goto exit_free_device;
	}

	ieee80211_netif_oper(hw, NETIF_ATTACH);

	return 0;

exit_free_device:
	ieee80211_free_hw(hw);

exit_put_device:
	usb_put_dev(usb_dev);

	return status;
}

static void rt73usb_disconnect(struct usb_interface *usb_intf)
{
	struct ieee80211_hw *hw = usb_get_intfdata(usb_intf);
	struct rt2x00_dev *rt2x00dev = hw->priv;

	ieee80211_netif_oper(hw, NETIF_DETACH);

	/*
	 * Uninitialize and free the rt73usb driver data.
	 */
	rt73usb_disable_radio(rt2x00dev);
	rt73usb_uninitialize(rt2x00dev);
	rt73usb_free_dev(rt2x00dev);

	/*
	 * Uninitialize and free the 80211 stack data.
	 */
	ieee80211_unregister_hw(hw);
	ieee80211_free_hw(hw);

	/*
	 * Free the USB device data.
	 */
	usb_set_intfdata(usb_intf, NULL);
	usb_put_dev(interface_to_usbdev(usb_intf));
}

#ifdef CONFIG_PM
static int rt73usb_suspend(struct usb_interface *usb_intf, pm_message_t state)
{
	struct ieee80211_hw *hw = usb_get_intfdata(usb_intf);
	struct rt2x00_dev *rt2x00dev = hw->priv;
	int status;

	NOTICE("Going to sleep.\n");

	ieee80211_netif_oper(hw, NETIF_DETACH);

	/*
	 * Disable the radio.
	 */
	rt73usb_disable_radio(rt2x00dev);

	/*
	 * Set device mode to sleep for power management.
	 */
	status = rt73usb_set_state(rt2x00dev, STATE_SLEEP);
	if (status)
		return status;

	/*
	 * Uninitialize device and hardware.
	 */
	rt73usb_uninitialize(rt2x00dev);
	rt73usb_free_dev(rt2x00dev);

	/*
	 * Decrease usbdev refcount.
	 */
	usb_put_dev(interface_to_usbdev(usb_intf));

	return 0;
}

static int rt73usb_resume(struct usb_interface *usb_intf)
{
	struct ieee80211_hw *hw = usb_get_intfdata(usb_intf);
	struct rt2x00_dev *rt2x00dev = hw->priv;
	int status;

	NOTICE("Waking up.\n");

	/*
	 * Increase usbdev refcount.
	 */
	usb_get_dev(interface_to_usbdev(usb_intf));

	/*
	 * Initialize hardware.
	 */
	status = rt73usb_alloc_dev(usb_intf, hw);
	if (status) {
		ERROR("Failed to allocate device.\n");
		return status;
	}

	/*
	 * Set device mode to awake for power management.
	 */
	status = rt73usb_set_state(rt2x00dev, STATE_AWAKE);
	if (status)
		return status;

	ieee80211_netif_oper(hw, NETIF_ATTACH);
	return 0;
}
#endif /* CONFIG_PM */

/*
 * rt73usb module information.
 */
static char version[] =
	DRV_NAME " - " DRV_VERSION " (" DRV_RELDATE ") by " DRV_PROJECT;

static struct usb_device_id rt73usb_device_table[] = {
	/* AboCom */
	{ USB_DEVICE(0x07b8, 0xb21d) },
	/* Askey */
	{ USB_DEVICE(0x1690, 0x0722) },
	/* ASUS */
	{ USB_DEVICE(0x0b05, 0x1723) },
	/* Belkin */
	{ USB_DEVICE(0x050d, 0x7050) },
	{ USB_DEVICE(0x050d, 0x705a) },
	/* Billionton */
	{ USB_DEVICE(0x1631, 0xc019) },
	/* CNet */
	{ USB_DEVICE(0x1371, 0x9022) },
	{ USB_DEVICE(0x1371, 0x9032) },
	/* Conceptronic */
	{ USB_DEVICE(0x14b2, 0x3c22) },
	/* D-Link */
	{ USB_DEVICE(0x07d1, 0x3c03) },
	{ USB_DEVICE(0x07d1, 0x3c04) },
	/* Gigabyte */
	{ USB_DEVICE(0x1044, 0x8008) },
	{ USB_DEVICE(0x1044, 0x800a) },
	/* Huawei-3Com */
	{ USB_DEVICE(0x1472, 0x0009) },
	/* Hercules */
	{ USB_DEVICE(0x06f8, 0xe010) },
	/* Linksys */
	{ USB_DEVICE(0x13b1, 0x0020) },
	{ USB_DEVICE(0x13b1, 0x0023) },
	/* MSI */
	{ USB_DEVICE(0x0db0, 0x6877) },
	{ USB_DEVICE(0x0db0, 0xa861) },
	{ USB_DEVICE(0x0db0, 0xa874) },
	/* Ralink */
	{ USB_DEVICE(0x148f, 0x2573) },
	{ USB_DEVICE(0x148f, 0x2671) },
	/* Qcom */
	{ USB_DEVICE(0x18e8, 0x6196) },
	{ USB_DEVICE(0x18e8, 0x6229) },
	/* Sitecom */
	{ USB_DEVICE(0x0df6, 0x9712) },
	{ USB_DEVICE(0x0df6, 0x90ac) },
	/* Surecom */
	{ USB_DEVICE(0x0769, 0x31f3) },
	{ 0, }
};

MODULE_AUTHOR(DRV_PROJECT);
MODULE_VERSION(DRV_VERSION);
MODULE_DESCRIPTION("Ralink RT73 USB Wireless LAN driver.");
MODULE_SUPPORTED_DEVICE("Ralink RT2573 USB chipset based cards");
MODULE_DEVICE_TABLE(usb, rt73usb_device_table);
MODULE_LICENSE("GPL");

#ifdef CONFIG_RT73USB_DEBUG
module_param_named(debug, rt2x00_debug_level, bool, S_IWUSR | S_IRUGO);
MODULE_PARM_DESC(debug, "Set this parameter to 1 to enable debug output.");
#endif /* CONFIG_RT73USB_DEBUG */

static struct usb_driver rt73usb_driver = {
	.name		= DRV_NAME,
	.id_table	= rt73usb_device_table,
	.probe		= rt73usb_probe,
	.disconnect	= rt73usb_disconnect,
#ifdef CONFIG_PM
	.suspend	= rt73usb_suspend,
	.resume		= rt73usb_resume,
#endif /* CONFIG_PM */
};

static int __init rt73usb_init(void)
{
	printk(KERN_INFO "Loading module: %s.\n", version);
	return usb_register(&rt73usb_driver);
}

static void __exit rt73usb_exit(void)
{
	printk(KERN_INFO "Unloading module: %s.\n", version);
	usb_deregister(&rt73usb_driver);
}

module_init(rt73usb_init);
module_exit(rt73usb_exit);
