/*
*  program name: Rubrica <http://digilander.iol.it/nfragale/download.html>
*  file: autosave.c
*  
*  Copyright (C) 2000-2001 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include <gnome.h>
#include <libxml/tree.h>
#include <libxml/parser.h>

#include "autosave.h"
#include "log.h"
#include "prefer.h"
#include "file.h"
#include "card.h"
#include "rubrica.h"

static gboolean have_timeout = FALSE;
static gint timeout_tag = 0;
static guint32 wait_time = 0;


void rubrica_autosave_start(guint32 time)
{
  if (time == 0)
    {
      wait_time = rubrica_preferences_get_time_rate();
      
      if (wait_time <= 0)
	wait_time = 15;  /* default time to wait is 15 minutes */
    }
  else
    wait_time = time;

  /*     wait_time store the minutes that rubrica must wait
	 wait_time *= 60000 (1 minute = 60 seconds * 1000 microseconds)
  */
  wait_time *= 60000;
  timeout_tag = gtk_timeout_add(wait_time, 
				(GtkFunction) rubrica_autosave_save_file, 
				NULL);
  
  have_timeout = TRUE;      
  rubrica_log_write_log("\nRubrica: autosave started.");
}


void rubrica_autosave_enable(guint32 time)
{
  rubrica_autosave_start(time);  
}


void rubrica_autosave_disable(void)
{
  if (timeout_tag != 0)
    gtk_timeout_remove(timeout_tag);
  else
    return;

  timeout_tag = 0;
  have_timeout = FALSE;
}


void rubrica_autosave_restart(guint32 time)
{
  rubrica_autosave_disable();
  rubrica_autosave_enable(time);
}


gint rubrica_autosave_save_file(void)
{
  Rubrica *rubrica;
  gint pages, i;

  pages = rubrica_card_get_pages();

  for (i = 0; i < pages; i++)
    {
      rubrica = rubrica_get_nth_rubrica(i);
  
      rubrica_autosave_write_file(rubrica, i);      
    }

  rubrica_log_write_log("\nRubrica: autosave performed");

  return TRUE; 
}


void rubrica_autosave_write_file(Rubrica *rubrica, gint page)
{
  xmlDocPtr doc;             /* puntatore al documento xml */
  gchar *filename = NULL;
  gchar *home;
  gint n, items; 
  
  filename = rubrica_get_tmpfile(rubrica);  
  if (!filename)
    {
      n = rand();
      home = g_strdup(g_get_home_dir());
      
      filename = g_strdup_printf("%s/.rubrica/rubrica%d.tmp", home, n);
      rubrica_set_tmpfile(rubrica, filename);     
    }

  if (g_file_exists(filename))
    remove(filename);
  
  /* creazione di un nuovo documento xml 
   */
  doc = xmlNewDoc("1.0");
  
  /* creo un nuovo nodo radice rubrica 
   */
  doc->children = xmlNewDocNode(doc, NULL, "Rubrica", NULL);
  xmlSetProp(doc->children, "version", VERSION);

  /* attraverso la rubrica e scrivo i dati 
     nell'albero xml
  */
  items = rubrica_get_items(rubrica);
  for (n = 0; n < items; n++) 
    {
      RubricaItem *item = rubrica_get_nth_item(rubrica, n);
      
      rubrica_file_build_xmlnode(doc->children, item);
    }  
  
  if (xmlSaveFormatFile(filename, doc, 1) != -1 )
    {
      gchar *str;

      str = g_strdup_printf("\nRubrica: save %s to temporary file %s", 
			    rubrica->name, filename);
      rubrica_log_write_log(str);

      g_free(str);
      g_free(filename);
    }
  else
    {
      rubrica_log_write_log("\nCan't save temporary file. "
			    "xmlSaveFormatFile return -1");
    }
  
  xmlFreeDoc(doc);
}


void rubrica_autosave_remove_tmp_files(void)
{
  gint pages, i;

  pages = rubrica_card_get_pages();
  
  for (i = 0; i < pages; i++)
    {
      gchar *tmpfile;

      tmpfile = rubrica_autosave_get_nth_tmpfile(i);
  
      if (tmpfile && g_file_exists(tmpfile))
	remove(tmpfile);      
    }
}

/*  return the nth temp file name
*/
gchar *rubrica_autosave_get_nth_tmpfile(gint i) 
{
  Rubrica *rubrica = NULL;

  rubrica = rubrica_get_nth_rubrica(i);

  g_return_val_if_fail(rubrica != NULL, NULL);

  return (gchar *) rubrica_get_tmpfile(rubrica);    
}
