/*
*  Rubrica
*  file: vcard.c
*
*  the vcard import engine v0.5 
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <glib.h>
#include <glib-object.h>

#include <string.h>
#include <stdlib.h>
#include <time.h>

#include "vcard.h"

#define BUFLEN    128
#define ARRAY_LEN 8

#define TOKEN_DELIM ':'
#define PARAM_DELIM ';'
#define VALUE_DELIM '='


RubricaVCardData *data;
RubricaVCardTokenType oldtoken;
RubricaVCardParamType oldparam[ARRAY_LEN];
RubricaVCardValueType oldvalue[ARRAY_LEN];

gchar* tmpbuf = NULL;
gboolean multiline_data = FALSE;


static RubricaVCardType tokens[] = { 
  {"N",   N},             {"FN",  FN},        {"TZ",  TZ},
  {"END", END},           {"ADR", ADR},       {"TEL", TEL},
  {"GEO", GEO},           {"ORG", ORG},       {"REV", REV},
  {"UID", UID},           {"URL", URL},       {"KEY", KEY},
  {"BDAY", BDAY},         {"ROLE", ROLE},     {"LOGO", LOGO}, 
  {"NOTE", NOTE},         {"BEGIN", BEGIN},   {"PHOTO", PHOTO},  
  {"LABEL", LABEL},       {"EMAIL", EMAIL},   {"TITLE", TITLE}, 
  {"AGENT", AGENT},       {"PROID", PROID},   {"SOUND", SOUND},  
  {"CLASS", CLASS},       {"MAILER", MAILER}, {"VERSION", VERSION}, 
  {"NICKNAME", NICKNAME}, {"CATEGORIES", CATEGORIES},

  /* kde extension.
  */
  {"X-KADDRESSBOOK-X-Profession",     KDE_EXT_Profession},
  {"X-KADDRESSBOOK-X-Department",     KDE_EXT_Department},
  {"X-KADDRESSBOOK-X-ManagersName",   KDE_EXT_ManagersName},
  {"X-KADDRESSBOOK-X-AssistantsName", KDE_EXT_AssistantsName},
  {"X-KADDRESSBOOK-X-Office",         KDE_EXT_Office},
  {"X-KADDRESSBOOK-X-SpousesName",    KDE_EXT_SpousesName},
  {"X-KADDRESSBOOK-X-Anniversary",    KDE_EXT_Anniversary},
  {"X-KADDRESSBOOK-X-CUSTOM",         KDE_EXT_CUSTOM},
  
  /* rubrica extension. 
     So rubrica can import a previously 
     exported rubrica's addressbook to vcard format
  */
  {"X-RUBRICA-X-Group",          RUBRICA_EXT_Group},
  {"X-RUBRICA-X-CompanyName",    RUBRICA_EXT_CompanyName},
  {"X-RUBRICA-X-Street",         RUBRICA_EXT_Street},
  {"X-RUBRICA-X-StreetNumber",   RUBRICA_EXT_StreetNumber},
  {"X-RUBRICA-X-ZipCode",        RUBRICA_EXT_ZipCode},
  {"X-RUBRICA-X-City",           RUBRICA_EXT_City},
  {"X-RUBRICA-X-Province",       RUBRICA_EXT_Province},
  {"X-RUBRICA-X-Country",        RUBRICA_EXT_Country},
  {"X-RUBRICA-X-Web",            RUBRICA_EXT_Web},
  {"X-RUBRICA-X-Email",          RUBRICA_EXT_Email},
  {"X-RUBRICA-X-Operator",       RUBRICA_EXT_Operator},
  {"X-RUBRICA-X-Fax",            RUBRICA_EXT_Fax},
  {"X-RUBRICA-X-Green",          RUBRICA_EXT_Green},
  {"X-RUBRICA-X-CustomerCare",   RUBRICA_EXT_CustomerCare},
  {"X-RUBRICA-X-Notes",          RUBRICA_EXT_Notes},
  {"X-RUBRICA-X-Department",     RUBRICA_EXT_Department},
  {"X-RUBRICA-X-SubDepartment",  RUBRICA_EXT_SubDepartment},
  {"X-RUBRICA-X-SecretaryName",  RUBRICA_EXT_SecretaryName},
  {"X-RUBRICA-X-SecretaryPhone", RUBRICA_EXT_SecretaryPhone},
  {"X-RUBRICA-X-SpouseName",     RUBRICA_EXT_SpouseName},
  {"X-RUBRICA-X-Child",          RUBRICA_EXT_Child},
  {"X-RUBRICA-X-Hobbies",        RUBRICA_EXT_Hobbies},

  /*     TODO: add evolution extensions
  */
  {NULL,  BAD_TOKEN}
};
	
	
static RubricaVCardType paramtokens[] = {
  {"type",     TYPE},
  {"value",    VALUE},   
  {"encoding", ENCODING},
  {NULL,       BAD_PARAM}
};		


static RubricaVCardType valuetokens[] = {
  {"home",     HOME},     {"work",       WORK},       {"cell",    CELL},     
  {"fax",      FAX},      {"postal",     POSTAL},     {"parcel",  PARCEL},  
  {"dom",      DOM},      {"intl",       INTL},       {"pref",    PREF},    
  {"voice",    VOICE},    {"msg",        MSG},        {"pager",   PAGER},   
  {"bbs",      BBS},      {"modem",      MODEM},      {"car",     CAR},     
  {"isdn",     ISDN},     {"video",      VIDEO},      {"aol",     AOL},     
  {"internet", INTERNET}, {"applelink",  APPLELINK},  {"attmail", ATTMAIL},
  {"cis",      CIS},      {"eworld",     EWORLD},     {"ibmmail", IBMMail},  
  {"mcimail",  MCIMAIL},  {"powershare", POWERSHARE}, {"prodigy", PRODIGY}, 
  {"tlx",      TLX},      {"x400",       X400},       {"vcard",   VCARD},
  {"pgp",      PGP},      {"x509",       X509},       {"b",       BINARY},   
  {"binary",   BINARY},   {"text",       TEXT},       {"uri",     URI},
  {"basic",    BASIC},    {"jpeg",       JPEG},       {"public",  PUBLIC},
  {"private",  PRIVATE},  {"confidential", CONFIDENTIAL}, 
  {"quoted-printable", QUOTED_PRINTABLE},  
  {NULL,       BAD_VALUE}
};


/*    signals enumeration 
 */
enum {
  VCARD_FILE_OPEN,               // vcard file was/wasn't opened
  VCARD_DECODE_CARD,             // vcard item was decoded
  VCARD_DECODE_ADDRESSBOOK,      // vcard file was decoded
  LAST_SIGNAL
};

static guint rubrica_vcard_signals[LAST_SIGNAL] = {0};


static void   rubrica_vcard_init        (RubricaVCard* obj);
static void   rubrica_vcard_class_init  (RubricaVCardClass* klass);

static void   rubrica_vcard_dispose     (RubricaVCard* obj);
static void   rubrica_vcard_finalize    (RubricaVCard* obj);


/* private functions
*/
static void   rubrica_vcard_free_string (gchar* string);
static void   rubrica_vcard_data_init   (RubricaVCardData* data);
static void   rubrica_vcard_data_clean  (RubricaVCardData* data);

static gchar* rubrica_vcard_append_data (gchar* buf, gchar* data);
// static time_t rubrica_vcard_decode_date (gchar *date, gboolean has_hour);

RubricaVCardTokenType rubrica_vcard_decode_token(gchar *string);
RubricaVCardValueType rubrica_vcard_decode_value(gchar *string);
RubricaVCardParamType rubrica_vcard_decode_param(gchar *string);

void   rubrica_vcard_decode_data_buffer(char *buf, gchar *dec[7]);
gchar* rubrica_vcard_extract_tokens (gchar* buffer,
				     RubricaVCardTokenType* token,
				     RubricaVCardParamType* param,
				     RubricaVCardValueType* value);

void   rubrica_vcard_route_tokens   (RubricaVCard* vcard,
				     gchar* buffer, gchar* data, 
				     RubricaVCardTokenType token, 
				     RubricaVCardParamType* parray, 
				     RubricaVCardValueType* varray);

void   rubrica_vcard_build_card     (RubricaVCard *vcard, 
				     RubricaVCardTokenType token,
				     RubricaVCardParamType *parray, 
				     RubricaVCardValueType *varray, 
				     gchar *data);

void   rubrica_vcard_optimize_card  (RubricaVCard* vcard);


void   rubrica_vcard_set_fn       (RubricaVCard* vcard, gchar* str); 
void   rubrica_vcard_set_n        (RubricaVCard* vcard, gchar* str);
void   rubrica_vcard_set_gn       (RubricaVCard* vcard, gchar* str);
void   rubrica_vcard_set_an       (RubricaVCard* vcard, gchar* str);
void   rubrica_vcard_set_hp       (RubricaVCard* vcard, gchar* str);
void   rubrica_vcard_set_hs       (RubricaVCard* vcard, gchar* str);

void   rubrica_vcard_set_email    (RubricaVCard* vcard, gchar* str);
void   rubrica_vcard_set_url      (RubricaVCard* vcard, gchar* str);
void   rubrica_vcard_set_bday     (RubricaVCard* vcard, gchar* str);
void   rubrica_vcard_set_title    (RubricaVCard* vcard, gchar* str);
void   rubrica_vcard_set_org      (RubricaVCard* vcard, gchar* str);
void   rubrica_vcard_set_role     (RubricaVCard* vcard, gchar* str);
void   rubrica_vcard_set_note     (RubricaVCard* vcard, gchar* str);
void   rubrica_vcard_set_key      (RubricaVCard* vcard, gchar* str);

void   rubrica_vcard_set_pbox     (RubricaVCardAddress* adr, gchar* str);
void   rubrica_vcard_set_extadd   (RubricaVCardAddress* adr, gchar* str);
void   rubrica_vcard_set_str      (RubricaVCardAddress* adr, gchar* str);
void   rubrica_vcard_set_loc      (RubricaVCardAddress* adr, gchar* str);
void   rubrica_vcard_set_reg      (RubricaVCardAddress* adr, gchar* str);
void   rubrica_vcard_set_code     (RubricaVCardAddress* adr, gchar* str);
void   rubrica_vcard_set_count    (RubricaVCardAddress* adr, gchar* str);
void   rubrica_vcard_set_adr_type (RubricaVCardAddress* adr, gchar* str);

void   rubrica_vcard_set_tel_num  (RubricaVCardTelephone* tel, gchar* str);
void   rubrica_vcard_set_tel_type (RubricaVCardTelephone* tel, gchar* str);

void   rubrica_vcard_set_adr      (RubricaVCard* vcard, gpointer adr);
void   rubrica_vcard_set_tel      (RubricaVCard* vcard, gpointer tel);


struct _RubricaVCardPrivate {
  gboolean dispose_has_run;
};




GType
rubrica_vcard_get_type()
{
  static GType rubrica_vcard_type = 0;
  
  if (!rubrica_vcard_type)
    {
      static const GTypeInfo rubrica_vcard_info =
	{
	  sizeof(RubricaVCardClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_vcard_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaVCard),
	  0,
	  (GInstanceInitFunc) rubrica_vcard_init
	};

      rubrica_vcard_type = g_type_register_static (G_TYPE_OBJECT,
						       "RubricaVCard",
						       &rubrica_vcard_info,
						       0);
    }
  
  return rubrica_vcard_type;
}


static void
rubrica_vcard_init(RubricaVCard* vcard)
{
  g_return_if_fail(RUBRICA_VCARD(vcard) != NULL);

  vcard->fp  = NULL;
  
  vcard->data = g_malloc(sizeof(RubricaVCardData));
  if (!vcard->data)
    g_error("\nRubricaVCard: Out of memory!");

  rubrica_vcard_data_init(vcard->data);

  vcard->private = g_new(RubricaVCardPrivate, 1);
  vcard->private->dispose_has_run = FALSE;
}


static void
rubrica_vcard_class_init(RubricaVCardClass* klass)
{
  GObjectClass *class;
  
  class = G_OBJECT_CLASS (klass);

  class->dispose  = (GObjectFinalizeFunc) rubrica_vcard_dispose;
  class->finalize = (GObjectFinalizeFunc) rubrica_vcard_finalize;

  rubrica_vcard_signals[VCARD_FILE_OPEN] = 
    g_signal_new("vcard_file_open", 
		 RUBRICA_VCARD_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RubricaVCardClass, vcard_file_opened),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__INT,
		 G_TYPE_NONE,         /* return type */
		 1,                   /* params      */
		 G_TYPE_INT);         /* params type */

  rubrica_vcard_signals[VCARD_DECODE_CARD] = 
    g_signal_new("vcard_decode_card", 
		 RUBRICA_VCARD_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RubricaVCardClass, vcard_decoded_item),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__VOID,
		 G_TYPE_NONE,
		 0,
		 G_TYPE_NONE);
  
  rubrica_vcard_signals[VCARD_DECODE_ADDRESSBOOK] =     
    g_signal_new("vcard_decode_addressbook", 
		 RUBRICA_VCARD_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RubricaVCardClass, vcard_decoded),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__VOID,
		 G_TYPE_NONE,
		 0,
		 G_TYPE_NONE);
}


static void 
rubrica_vcard_dispose (RubricaVCard* vcard)
{
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));

  if (vcard->private->dispose_has_run)
    return;

  vcard->private->dispose_has_run = TRUE;
}


static void 
rubrica_vcard_finalize (RubricaVCard* vcard)
{
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));
  
  rubrica_vcard_data_clean(vcard->data);

  g_free(vcard->data);
  g_free(vcard->private);
}


static void
rubrica_vcard_free_string(gchar* str)
{
  if (str) g_free(str);
}


static void
rubrica_vcard_data_init(RubricaVCardData* data)
{
  g_return_if_fail(data != NULL);

  data->fn      = data->n     = data->gn    =
    data->an    = data->hp    = data->hs    =
    data->nick  = data->photo = data->bday  =
    data->label = data->title = data->role  = 
    data->org   = data->note  = data->class = data->key  = NULL;
  
  data->email = data->url = data->adr = data->tel = data->cat = NULL;
}


void 
rubrica_vcard_init_adr(RubricaVCardAddress* adr)
{
  g_return_if_fail(adr != NULL);

  adr->pbox  = adr->extadd = NULL;
  adr->str   = adr->loc    = NULL;
  adr->reg   = adr->code   = NULL;
  adr->count = adr->type   = NULL;
}


static void 
rubrica_vcard_data_clean(RubricaVCardData* data)
{
  GList* l;

  g_return_if_fail(data != NULL);
  
  rubrica_vcard_free_string(data->fn);
  rubrica_vcard_free_string(data->n); 
  rubrica_vcard_free_string(data->gn);
  rubrica_vcard_free_string(data->an);
  rubrica_vcard_free_string(data->hp);
  rubrica_vcard_free_string(data->hs);
  rubrica_vcard_free_string(data->nick);
  rubrica_vcard_free_string(data->photo);
  rubrica_vcard_free_string(data->bday);
  rubrica_vcard_free_string(data->label);
  rubrica_vcard_free_string(data->title);
  rubrica_vcard_free_string(data->role);
  rubrica_vcard_free_string(data->org);
  rubrica_vcard_free_string(data->note);
  rubrica_vcard_free_string(data->class);
  rubrica_vcard_free_string(data->key);  

  l = data->email;
  for(; l; l = l->next)
    {
      gchar* email = l->data;
      
      if (email)
	g_free(email);
    } 
  g_list_free(data->email);

  l = data->url;
  for(; l; l = l->next)
    {
      gchar* url = l->data;
      
      if (url)
	g_free(url);      
    }
  g_list_free(data->url);
  
  l = data->adr;
  for (; l; l = l->next)
    {
      RubricaVCardAddress* adr = (RubricaVCardAddress*) l->data;
      
      rubrica_vcard_free_string(adr->pbox);
      rubrica_vcard_free_string(adr->extadd);
      rubrica_vcard_free_string(adr->str);
      rubrica_vcard_free_string(adr->loc);
      rubrica_vcard_free_string(adr->reg);
      rubrica_vcard_free_string(adr->code);
      rubrica_vcard_free_string(adr->count);
    }
  g_list_free(data->adr);
    
  l = data->tel;
  for (; l; l = l->next)
    {
      RubricaVCardTelephone* vctel = (RubricaVCardTelephone *) l->data;

      rubrica_vcard_free_string(vctel->num);
      rubrica_vcard_free_string(vctel->type);      
    }
  g_list_free(data->tel);

  l = data->cat;
  for (; l; l = l->next)
    {
      rubrica_vcard_free_string(l->data);
    }
  g_list_free(data->cat);

  rubrica_vcard_data_init(data);
}


static gchar* 
rubrica_vcard_append_data(gchar* buf, gchar* data)
{
  if (buf)
    {
      gchar* tmp;

      tmp =  g_strdup_printf("%s%s", buf, data);

      g_free(buf);
      buf = tmp;
    }
  else
    buf = g_strdup(data);

  return buf;
}


static time_t 
rubrica_vcard_decode_date (gchar *date, gboolean has_hour)
{

  /* date: 2001-10-21T15:53:29
           1973-03-18
  */

  struct tm tm;
  time_t t;
  gchar *str, *tok;

  if (!date) 
    return -1;

  str = g_strdup(date);
  tok = strtok(str, "-");
  tm.tm_year = atoi(tok) - 1900;
  tok = strtok(NULL, "-");
  tm.tm_mon = atoi(tok) - 1;
  tok = strtok(NULL, "T");
  tm.tm_mday = atoi(tok);
  

  if (has_hour)
    {
      tok = strtok(NULL, ":");
      tm.tm_hour = atoi(tok);
      tok = strtok(NULL, ":");
      tm.tm_min = atoi(tok);
      tok = strtok(NULL, "");
      tm.tm_sec = atoi(tok);
    }

  g_free(str);
  t = mktime(&tm);

  return t;
}


RubricaVCardTokenType 
rubrica_vcard_decode_token(gchar *string)
{
  gint i=0;
  
  g_return_val_if_fail(string != NULL, BAD_TOKEN);
    
  while (tokens[i].str)
    {
      if (string && ((strcasecmp(string, tokens[i].str)) == 0))
	return (RubricaVCardTokenType) tokens[i].tok;
      else
	i++;
    }

  return (RubricaVCardTokenType) BAD_TOKEN;
}


RubricaVCardParamType 
rubrica_vcard_decode_param(gchar *string)
{
  gint i=0;
  
  g_return_val_if_fail(string != NULL, BAD_PARAM);
    
  while (paramtokens[i].str)
    {
      if (string && ((strcasecmp(string, paramtokens[i].str)) == 0))
	return (RubricaVCardParamType) paramtokens[i].tok;
      else
	i++;
    }

  return (RubricaVCardParamType) BAD_PARAM;
}


RubricaVCardValueType 
rubrica_vcard_decode_value(gchar *string)
{
  gint i=0;
  
  g_return_val_if_fail(string != NULL, BAD_VALUE);
  
  while (valuetokens[i].str)
    {
      if (string && ((strcasecmp(string, valuetokens[i].str)) == 0))
	return (RubricaVCardValueType) valuetokens[i].tok;
      else
	i++;
    }

  return (RubricaVCardValueType) BAD_VALUE;
}


gchar*
rubrica_vcard_extract_tokens(gchar* buffer,
			     RubricaVCardTokenType* token,
			     RubricaVCardParamType* param,
			     RubricaVCardValueType* value)
{
  gchar tok[BUFLEN];
  gchar dt[BUFLEN];
  gchar buf[BUFLEN];
  gboolean build_param = FALSE;
  gboolean build_value = FALSE;
  gboolean build_data  = FALSE;
  gint i = 0, j = 0, k = 0;

  memset(tok, 0, BUFLEN);
  memset(dt,  0, BUFLEN);
  memset(buf, 0, BUFLEN);

  param[0] = BAD_PARAM;
  value[0] = BAD_VALUE;

  /* build token */
  j = 0;
  while(buffer[i] && (buffer[i] != TOKEN_DELIM) && (buffer[i] != PARAM_DELIM))
    {
      if (buffer[i] != '\\' && ((buffer[i+1] == 'n') || (buffer[i+1] == 'r')))
	{
	  tok[j] = '\n';
	  i++;
	}
      else
	tok[j] = buffer[i];

      j++;	
      i++;      
    }
  *token = rubrica_vcard_decode_token(tok);

  /* build param */
  build_param = TRUE;
  if (buffer[i] == PARAM_DELIM)
    i++;

  j = 0;
  while(buffer[i] && !build_data) 
    {
      if (buffer[i] == TOKEN_DELIM)   // : il buffer rimanente contiene dati
	{
	  build_data = TRUE;

	  if (build_value)
	    value[k] = rubrica_vcard_decode_value(buf);
	  else if (j > 0)
	    {
	      param[k] = TYPE;
	      value[k] = rubrica_vcard_decode_value(buf);
	    }
	}
      else
	{ 
	  if (build_param && buffer[i] == VALUE_DELIM)    // = 
	    {
	      build_value = TRUE;
	      build_param = FALSE;

	      param[k] = rubrica_vcard_decode_param(buf);
	      memset(buf, 0, BUFLEN);
	      i++;
	      j = 0;
	    }
	  
	  if (build_value && buffer[i] == PARAM_DELIM)    // ;
	    {
	      build_param = TRUE;
	      build_value = FALSE;
	      
	      value[k] = rubrica_vcard_decode_value(buf);
	      memset(buf, 0, BUFLEN);
	      i++;
	      j = 0;
	      k++;
	    }
	  
	  if (build_param && buffer[i] == PARAM_DELIM)    // ;
	    {
	      param[k] = TYPE;
	      value[k] = rubrica_vcard_decode_value(buf);
	      memset(buf, 0, BUFLEN);
	      i++;
	      j = 0;
	      k++;
	    }  
	  
	  buf[j] = buffer[i];
	  i++;
	  j++; 
	}
    }
 
  /* read data */
  i++;
  j = 0;
  while(buffer[i])
    {
      if (buffer[i] == '\\' && ((buffer[i+1] == 'n') || (buffer[i+1] == 'r')))
	{
	  dt[j] = '\n';
	  i++;
	}
      else
	dt[j] = buffer[i];
      
      j++;      
      i++;
    }

  if (strcmp(dt, "") == 0)
    return NULL;
  
  return g_strdup(dt);
}


void 
rubrica_vcard_route_tokens(RubricaVCard* vcard,
			   gchar* buffer, 
			   gchar* data, 
			   RubricaVCardTokenType token, 
			   RubricaVCardParamType* parray, 
			   RubricaVCardValueType* varray)
{  
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));

  if ((token != BAD_TOKEN) && 
      ((oldtoken == NOTE) || (oldtoken == ADR) || (oldtoken == LABEL)))
    {
      multiline_data = FALSE;
      
      rubrica_vcard_build_card(vcard, oldtoken, oldparam, oldvalue, tmpbuf);

      oldtoken = BAD_TOKEN;
      g_free(tmpbuf);
      tmpbuf = NULL;
    }
  
  if (token == NOTE)
    {
      oldtoken  = NOTE;
      *oldparam = BAD_PARAM;
      *oldvalue = BAD_VALUE;
      
      multiline_data = TRUE;
      if (tmpbuf)
	g_free(tmpbuf);

      tmpbuf = rubrica_vcard_append_data(NULL, data);
    }
  
  if (token == ADR)
    {
      memset(oldparam, 0, ARRAY_LEN);
      memset(oldvalue, 0, ARRAY_LEN);
      
      oldtoken  = ADR;
      *oldparam = *parray;
      *oldvalue = *varray;	      
      
      multiline_data = TRUE;
      if (tmpbuf)
	g_free(tmpbuf);
      
      tmpbuf = rubrica_vcard_append_data(NULL, data);      
    }
  
  if (token == LABEL)
    {
      memset(oldparam, 0, ARRAY_LEN);
      memset(oldvalue, 0, ARRAY_LEN);
      
      oldtoken  = LABEL;
      *oldparam = *parray;
      *oldvalue = *varray;	      
      
      multiline_data = TRUE;
      if (tmpbuf)
	g_free(tmpbuf);
      
      tmpbuf = rubrica_vcard_append_data(NULL, data); 
    }
  
  if (buffer && (token == BAD_TOKEN))
    tmpbuf = rubrica_vcard_append_data(tmpbuf, buffer);
  
  if (!multiline_data)
    rubrica_vcard_build_card(vcard, token, parray, varray, data);     
  
  if (data)
    g_free(data);	      
}



void 
rubrica_vcard_build_card(RubricaVCard *vcard, 
			 RubricaVCardTokenType token,
			 RubricaVCardParamType *parray, 
			 RubricaVCardValueType *varray, 
			 gchar *data)
{
  RubricaVCardAddress* adr;
  RubricaVCardTelephone* tt;

  gchar** decoded;
  gchar* tmp;
  
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));
  
  if (!data) return;

  switch(token)
    {
    case BEGIN:
      rubrica_vcard_data_clean(vcard->data);
      break;
      
    case END:
      break;

    case FN:
      /* card name */ 
      rubrica_vcard_set_fn(vcard, data);  
      break;
      
    case N:
      /* ord:: surname, name, second name, prefix, suffix 
	       cognome, nome, secondo nome, prefisso, suffisso 
      */
      decoded = g_strsplit(data, ";", 5);
      
      if (decoded[0]) 
	{
	  rubrica_vcard_set_n(vcard, decoded[0]);            // family name
	  
	  if (decoded[1]) 
	    {
	      rubrica_vcard_set_gn(vcard, decoded[1]);         // given name
	      
	      if (decoded[2]) 
		{
		  rubrica_vcard_set_an(vcard, decoded[2]);       // additional name
		  
		  if (decoded[3]) 
		    {
		      rubrica_vcard_set_hp(vcard, decoded[3]);     // honorific prefix
		      if (decoded[4])
			rubrica_vcard_set_hs(vcard, decoded[4]);   // honorific suffix

		    }
		}
	    }
	}

      g_strfreev(decoded);
      break;

    case TZ:
      break;

    case REV:
      break;

    case ADR:
      /* mail address + mail type 
	 indirizzo di posta + tipo 
      */
      adr = g_malloc(sizeof(RubricaVCardAddress));
      if (!adr)
	{
	  g_warning("\nmalloc adr");
	  break;
	}
      
      rubrica_vcard_init_adr(adr);

      decoded = g_strsplit(data, ";", 7);

      if (decoded[0]) 
	{
	  rubrica_vcard_set_pbox(adr, decoded[0]);
	  
	  if (decoded[1]) 
	    {
	      rubrica_vcard_set_extadd(adr, decoded[1]);
	      
	      if (decoded[2]) 
		{
		  rubrica_vcard_set_str(adr, decoded[2]);
		  
		  if (decoded[3]) 
		    {
		      rubrica_vcard_set_loc(adr, decoded[3]);
		      
		      if (decoded[4]) 
			{
			  rubrica_vcard_set_reg(adr, decoded[4]);
			  
			  if (decoded[5]) 
			    {
			      rubrica_vcard_set_code  (adr, decoded[5]);
			      
			      if (decoded[6])
				rubrica_vcard_set_count (adr, decoded[6]);
			    }
			}
		    }
		}
	    }
	}
    
      g_strfreev(decoded);
      
      switch(*varray)
	{
	case HOME:
	  tmp = "home";
	  break;
	case WORK:
	  tmp = "work";
	  break;
	case BAD_VALUE:
	  tmp = "unknown";
	  break;
	default:
	  tmp = "other";
	  break;	      
	} 

      rubrica_vcard_set_adr_type(adr, tmp);      
      rubrica_vcard_set_adr(vcard, adr);
      break;
      
    case TEL:
      /* telefono + tipo */   
      if (*varray)
	{
	  RubricaVCardValueType value;
	  gboolean unknown;
	  gchar* tmp;
	  
	  tt = (RubricaVCardTelephone*)g_malloc(sizeof(RubricaVCardTelephone));

	  if (!tt)
	    {
	      g_warning("\nmalloc telephone");
	      break;
	    }

	  rubrica_vcard_set_tel_num(tt, data);
      
	  do
	    {
	      unknown = FALSE;
	      value = (RubricaVCardValueType) *varray;

	      switch (value)
		{	  
		case HOME:
		  tmp = "home";
		  break;
		case WORK:
		  tmp = "work";
		  break;
		case FAX:
		  tmp = "fax";
		  break;
		case CELL:
		  tmp = "cellphone";
		  break;
		  
		case BAD_VALUE:
		  tmp = "unknown";
		  unknown = TRUE;
		  break;
		  
		  /* other telephone type */
		case PREF:
		case VOICE:
		case MSG:
		case PAGER:
		case BBS:
		case MODEM:
		case CAR:
		case ISDN:
		case VIDEO:
		default:
		  tmp = "other";
		  unknown = TRUE;				  
		  break;
		}
	      
	      varray++;
	    } while (*varray && unknown == TRUE);
	  
	  rubrica_vcard_set_tel_type(tt, tmp);
	  rubrica_vcard_set_tel(vcard, tt);
	}
      break;
      
    case EMAIL:  
      rubrica_vcard_set_email(vcard, data);
      break;

    case URL:      
      rubrica_vcard_set_url(vcard, data); 
     break;

    case GEO:
      break;

    case BDAY:
      rubrica_vcard_set_bday(vcard, data);
      break;
      
    case TITLE:
    case KDE_EXT_Profession:
      rubrica_vcard_set_title(vcard, data);
      break;

    case ORG:
      rubrica_vcard_set_org(vcard, data);
      break;

    case CATEGORIES:
    case RUBRICA_EXT_Group:
//      rubrica_vcard_set_group(vcard, data);
      break;

    case NOTE:
      rubrica_vcard_set_note(vcard, data);
      break;
      
    case KEY: 
      rubrica_vcard_set_key(vcard, data);
      break;

    case ROLE:
      rubrica_vcard_set_role(vcard, data);
      break;

      /*   TODO -- recuperare campi condivisi
      */
    case UID:
      break;

    case LOGO:
      break;

    case PHOTO:
      break;

    case LABEL:
      break;

    case AGENT:
      break;

    case PROID:
      break;

    case SOUND:
      break;

    case CLASS:
      break;

    case MAILER:
      break;

    case VERSION:
      break;

    case NICKNAME:
      break;
      
    case RUBRICA_EXT_CompanyName:
      break;

    case RUBRICA_EXT_Street:
      break;
      
    case RUBRICA_EXT_StreetNumber:
      break;
      
    case RUBRICA_EXT_ZipCode:
      break;
      
    case RUBRICA_EXT_City:
      break;
      
    case RUBRICA_EXT_Province:
      break;
      
    case RUBRICA_EXT_Country:
      break;
      
    case RUBRICA_EXT_Web:
      break;
      
    case RUBRICA_EXT_Email:
      break;
      
    case RUBRICA_EXT_Operator:
      break;

    case RUBRICA_EXT_Fax:
      break;
      
    case RUBRICA_EXT_Green:
      break;
      
    case RUBRICA_EXT_CustomerCare:
      break;

    case RUBRICA_EXT_Notes:
      break;

    case RUBRICA_EXT_Department:
    case KDE_EXT_Department:
      break;
      
    case RUBRICA_EXT_SubDepartment:
      break;

    case RUBRICA_EXT_SecretaryName:
    case KDE_EXT_AssistantsName:
      break;
      
    case RUBRICA_EXT_SecretaryPhone:
      break;
      
    case RUBRICA_EXT_SpouseName:
    case KDE_EXT_SpousesName:
      break;
      
    case RUBRICA_EXT_Child:
      break;
      
    case RUBRICA_EXT_Hobbies:
      break;

    case KDE_EXT_ManagersName:
      break;

    case KDE_EXT_Office:
      break;

    case KDE_EXT_Anniversary:
      break;

    case KDE_EXT_CUSTOM:  
      break;
        
    case BAD_TOKEN:      
    default:
      break;
    }
}


void 
rubrica_vcard_optimize_card(RubricaVCard* vcard)
{
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));

  if (!vcard->data->fn)
    {
      if (vcard->data->n && vcard->data->gn)
	vcard->data->fn = g_strconcat(vcard->data->n, " ", 
				      vcard->data->gn, NULL);
      else if (vcard->data->n)
	vcard->data->fn = g_strdup(vcard->data->n);
      else if (vcard->data->gn)
	vcard->data->fn = g_strdup(vcard->data->gn);
      else if (vcard->data->org)
	vcard->data->fn = g_strdup(vcard->data->org);
      else vcard->data->fn = g_strdup("no name card");
    }
}


void 
rubrica_vcard_set_fn(RubricaVCard* vcard, gchar* str)
{
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));
  g_return_if_fail(str != NULL);

  vcard->data->fn = g_strdup(str);
}


void 
rubrica_vcard_set_n(RubricaVCard* vcard, gchar* str)
{
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));
  g_return_if_fail(str != NULL);

  if (strcmp(str, "") == 0)
    return;

//    vcard->data->n = NULL;
//  else
  vcard->data->n = g_strdup(str);
}


void 
rubrica_vcard_set_gn(RubricaVCard* vcard, gchar* str)
{
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));
  g_return_if_fail(str != NULL);

  if (strcmp(str, "") == 0)
    return;
//    vcard->data->gn = NULL;
//  else
  vcard->data->gn = g_strdup(str);
}


void 
rubrica_vcard_set_an(RubricaVCard* vcard, gchar* str)
{
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));
  g_return_if_fail(str != NULL);

  if (strcmp(str, "") == 0)
    return;
//    vcard->data->an = NULL;
//  else
  vcard->data->an = g_strdup(str);  
}


void 
rubrica_vcard_set_hp(RubricaVCard* vcard, gchar* str)
{
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));
  g_return_if_fail(str != NULL);
  
  if (strcmp(str, "") == 0)
    return;
//    vcard->data->hp = NULL;
//  else
    vcard->data->hp = g_strdup(str); 
}


void 
rubrica_vcard_set_hs(RubricaVCard* vcard, gchar* str)
{
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));
  g_return_if_fail(str != NULL);

  if (strcmp(str, "") == 0)
//    vcard->data->hs = NULL;
//  else
    return;

  vcard->data->hs = g_strdup(str);  
}


void 
rubrica_vcard_set_pbox(RubricaVCardAddress* adr, gchar* str)
{
  g_return_if_fail(adr != NULL);
  g_return_if_fail(str != NULL);

  if (strcmp(str, "") == 0)
//    adr->pbox = NULL;
//  else
    return;

  adr->pbox = g_strdup(str);
}


void 
rubrica_vcard_set_extadd(RubricaVCardAddress* adr, gchar* str)
{
  g_return_if_fail(adr != NULL);
  g_return_if_fail(str != NULL);

  if (strcmp(str, "") == 0)
//    adr->extadd = NULL;
//  else
    return;
    
  adr->extadd = g_strdup(str);
}


void 
rubrica_vcard_set_str(RubricaVCardAddress* adr, gchar* str)
{
  g_return_if_fail(adr != NULL);
  g_return_if_fail(str != NULL);

  if (strcmp(str, "") == 0)
//    adr->str = NULL;
//  else
    return;

  adr->str = g_strdup(str);
}


void 
rubrica_vcard_set_loc(RubricaVCardAddress* adr, gchar* str)
{
  g_return_if_fail(adr != NULL);
  g_return_if_fail(str != NULL);

  if (strcmp(str, "") == 0)
//    adr->loc = NULL;
//  else
    return;

  adr->loc = g_strdup(str);
}


void 
rubrica_vcard_set_reg(RubricaVCardAddress* adr, gchar* str)
{
  g_return_if_fail(adr != NULL);
  g_return_if_fail(str != NULL);

  if (strcmp(str, "") == 0)
//    adr->reg = NULL;
//  else
    return;

  adr->reg = g_strdup(str);
}


void 
rubrica_vcard_set_code(RubricaVCardAddress* adr, gchar* str)
{
  g_return_if_fail(adr != NULL);
  g_return_if_fail(str != NULL);

  if (strcmp(str, "") == 0)
//    adr->code = NULL;
//  else
    return;

  adr->code = g_strdup(str);
}


void 
rubrica_vcard_set_count(RubricaVCardAddress* adr, gchar* str)
{
  g_return_if_fail(adr != NULL);
  g_return_if_fail(str != NULL);
  
  if (strcmp(str, "") == 0)
//    adr->count = NULL;
//  else
    return;
  
  adr->count = g_strdup(str);
}


void 
rubrica_vcard_set_adr_type(RubricaVCardAddress* adr, gchar* str)
{
  g_return_if_fail(adr != NULL);
  g_return_if_fail(str != NULL);
  
  if (strcmp(str, "") == 0)
  //  adr->type = NULL;
    //  else
    return;

  adr->type = g_strdup(str);
}


void 
rubrica_vcard_set_adr(RubricaVCard* vcard, gpointer adr)
{
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));
  g_return_if_fail(adr != NULL);
      
  vcard->data->adr = g_list_append(vcard->data->adr, adr);
}


void
rubrica_vcard_set_tel_num(RubricaVCardTelephone* tel, gchar* str)
{
  g_return_if_fail(tel != NULL);
  g_return_if_fail(str != NULL);

  tel->num = g_strdup(str);
}


void 
rubrica_vcard_set_tel_type(RubricaVCardTelephone* tel, gchar* str)
{
  g_return_if_fail(tel != NULL);
  g_return_if_fail(str != NULL);

  tel->type = g_strdup(str);
}


void 
rubrica_vcard_set_tel(RubricaVCard* vcard, gpointer tel)
{
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));
  g_return_if_fail(tel != NULL);

  vcard->data->tel = g_list_append(vcard->data->tel, tel);
}


void 
rubrica_vcard_set_email(RubricaVCard* vcard, gchar* str)
{
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));
  g_return_if_fail(str != NULL);

  vcard->data->email = g_list_append(vcard->data->email, g_strdup(str));
}


void 
rubrica_vcard_set_url(RubricaVCard* vcard, gchar* str)
{
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));
  g_return_if_fail(str != NULL);

  vcard->data->url = g_list_append(vcard->data->url, g_strdup(str));  
}


void 
rubrica_vcard_set_bday(RubricaVCard* vcard, gchar* str)
{
  gint t;

  g_return_if_fail(IS_RUBRICA_VCARD(vcard));
  g_return_if_fail(str != NULL);
  
  t = rubrica_vcard_decode_date(str, FALSE);
  vcard->data->bday = g_strdup_printf("%d", t);
}


void 
rubrica_vcard_set_title(RubricaVCard* vcard, gchar* str)
{
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));
  g_return_if_fail(str != NULL);

  vcard->data->title = g_strdup(str);  
}


void 
rubrica_vcard_set_org(RubricaVCard* vcard, gchar* str)
{
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));
  g_return_if_fail(str != NULL);

  vcard->data->org = g_strdup(str);  
}


void 
rubrica_vcard_set_role(RubricaVCard* vcard, gchar* str)
{
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));
  g_return_if_fail(str != NULL);

  vcard->data->role = g_strdup(str);  
}


void 
rubrica_vcard_set_note(RubricaVCard* vcard, gchar* str)
{
//  gchar* tmp = NULL;

  g_return_if_fail(IS_RUBRICA_VCARD(vcard));
  g_return_if_fail(str != NULL);

  if (vcard->data->note)
    {
//      tmp = g_strdup(vcard->data->note);
//      g_free(vcard->data->note);
      gchar *tmp = vcard->data->note;
      vcard->data->note = g_strconcat(tmp, str, NULL);  
      g_free(tmp);
    }
//  else tmp = "";

//  vcard->data->note = g_strconcat(tmp, str, NULL);  
  else 
    vcard->data->note = g_strdup(str);
}


void 
rubrica_vcard_set_key(RubricaVCard* vcard, gchar* str)
{
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));
  g_return_if_fail(str != NULL);

  vcard->data->key = g_strdup(str);  
}



/* Public functions
 */

RubricaVCard* 
rubrica_vcard_new(void)
{
  RubricaVCard *vcard;

  vcard = g_object_new(rubrica_vcard_get_type(), NULL);

  return (RubricaVCard*) vcard;
}


void 
rubrica_vcard_free_vcard(RubricaVCard *vcard)
{
  g_return_if_fail(IS_RUBRICA_VCARD(vcard));

  g_object_unref(vcard);
}



void rubrica_vcard_clean_array(RubricaVCardParamType parray[],
			       RubricaVCardValueType varray[])
{
  gint i=0;

  for (; i < ARRAY_LEN; i++)
    parray[i] = varray[i] = 0;
}


gint          
rubrica_vcard_open_file (RubricaVCard* vcard, gchar* filename)
{
  gchar buffer[BUFLEN]; 
  gunichar ch;
  gchar* data;
  GString *string;

  RubricaVCardTokenType token;
  RubricaVCardParamType parray[ARRAY_LEN];
  RubricaVCardValueType varray[ARRAY_LEN];

  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), -1);
  g_return_val_if_fail(filename != NULL, -1);

  vcard->fp = fopen(filename, "r");
  if (!vcard->fp)
    {
      g_signal_emit_by_name(vcard, "vcard_file_open", 
			     RUBRICA_VCARD_FILE_OPEN_FAIL, G_TYPE_INT);

      return RUBRICA_VCARD_FILE_OPEN_FAIL;
    }
  else
    g_signal_emit_by_name(vcard, "vcard_file_open",
			   RUBRICA_VCARD_FILE_OPEN_SUCCESS, G_TYPE_INT);

  memset(buffer, 0, BUFLEN);
  memset(parray, 0, ARRAY_LEN);
  memset(varray, 0, ARRAY_LEN);
  string = g_string_new(NULL);

  do
    {
      ch = fgetc(vcard->fp);
      
      if ((ch != feof(vcard->fp)) && (ch != '\n') && (ch != '\r'))
	string = g_string_append_unichar(string, ch); 
      else if ((ch != feof(vcard->fp)))
	{
	  if (g_utf8_validate(string->str, -1, NULL))
	    g_utf8_strncpy(buffer, string->str, g_utf8_strlen(string->str,-1));
	  
	  if (buffer)
	    {
	      data = rubrica_vcard_extract_tokens(buffer, &token, 
						  parray, varray);
	      
	      rubrica_vcard_route_tokens(vcard, buffer, data,
					 token, parray, varray);
	      
	      if (token == END)
		{
		  rubrica_vcard_optimize_card(vcard);
		  g_signal_emit_by_name(vcard, "vcard_decode_card", 
					vcard, G_TYPE_NONE);
		}
	      
	    }

	  g_string_free(string, TRUE);
	  memset(buffer, 0, BUFLEN);
	  rubrica_vcard_clean_array(parray, varray);

/*
	  memset(parray, 0, ARRAY_LEN);
	  memset(varray, 0, ARRAY_LEN);
*/
	  string = g_string_new(NULL);	  
	}
    }
  while(!feof(vcard->fp));

  g_signal_emit_by_name(vcard, "vcard_decode_addressbook", NULL, G_TYPE_NONE);

  fclose(vcard->fp);
  g_string_free(string, TRUE);
  
  return RUBRICA_VCARD_FILE_IMPORT_SUCCESS;
}


gchar*   
rubrica_vcard_get_fn (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");
  
  return (gchar*) vcard->data->fn;
}


gchar*   
rubrica_vcard_get_n (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");
  
  return (gchar*) vcard->data->n;
}


gchar*   
rubrica_vcard_get_gn (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");
  
  return (gchar*) vcard->data->gn;
}


gchar*   
rubrica_vcard_get_an (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");
  
  return (gchar*) vcard->data->an;
}


gchar*   
rubrica_vcard_get_hp (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");
  
  return (gchar*) vcard->data->hp;
}


gchar*   
rubrica_vcard_get_hs (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");
  
  return (gchar*) vcard->data->hs;
}


RubricaVCardAddress*  
rubrica_vcard_get_adr (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), NULL);
  g_return_val_if_fail(vcard->data != NULL, NULL);
  
  return (gpointer) vcard->data->adr;
}


RubricaVCardTelephone* 
rubrica_vcard_get_tel (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), NULL);
  g_return_val_if_fail(vcard->data != NULL, NULL);
  
  return (gpointer) vcard->data->tel;
}


gchar*   
rubrica_vcard_get_email (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");
  
  if (!vcard->data->email)
    return NULL;
  
  return (gchar*) vcard->data->email->data;
}


gchar*   
rubrica_vcard_get_url (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");
  
  if (!vcard->data->url)
    return NULL;
  
  return (gchar*) vcard->data->url->data;
}


gchar*   
rubrica_vcard_get_bday (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");
  
  return (gchar*) vcard->data->bday;
}


gchar*   
rubrica_vcard_get_title (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");
  
  return (gchar*) vcard->data->title;
}


gchar*   
rubrica_vcard_get_org (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");
  
  return (gchar*) vcard->data->org;
}


gchar*   
rubrica_vcard_get_role (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");
  
  return (gchar*) vcard->data->role;
}


gchar*   
rubrica_vcard_get_note (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");
  
  return (gchar*) vcard->data->note;
}


gchar*   
rubrica_vcard_get_key (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");
  
  return (gchar*) vcard->data->key;
}


gchar*   
rubrica_vcard_get_pbox (RubricaVCard* vcard)
{
  RubricaVCardAddress* adr;

  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");
  
  if (!vcard->data->adr)
    return NULL;

  if (!vcard->data->adr->data)
    return NULL;

  adr = vcard->data->adr->data;
  
  return (gchar*) adr->pbox;
}


gchar*   
rubrica_vcard_get_extadd (RubricaVCard* vcard)
{
  RubricaVCardAddress* adr;

  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");

  if (!vcard->data->adr)
    return NULL;

  if (!vcard->data->adr->data)
    return NULL;

  adr = vcard->data->adr->data;

  return (gchar*) adr->extadd;
}
  

gchar*   
rubrica_vcard_get_str (RubricaVCard* vcard)
{
  RubricaVCardAddress* adr;

  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");

  if (!vcard->data->adr)
    return NULL;

  if (!vcard->data->adr->data)
    return NULL;

  adr = vcard->data->adr->data;  

  return (gchar*) adr->str;
}
  

gchar*   
rubrica_vcard_get_loc (RubricaVCard* vcard)
{
  RubricaVCardAddress* adr;

  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");

  if (!vcard->data->adr)
    return NULL;

  if (!vcard->data->adr->data)
    return NULL;

  adr = vcard->data->adr->data;

  return (gchar*) adr->loc;
}

  
gchar*   
rubrica_vcard_get_reg (RubricaVCard* vcard)
{
  RubricaVCardAddress* adr;

  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");

  if (!vcard->data->adr)
    return NULL;

  if (!vcard->data->adr->data)
    return NULL;

  adr = vcard->data->adr->data;  

  return (gchar*) adr->reg;
}


gchar*   
rubrica_vcard_get_code (RubricaVCard* vcard)
{
  RubricaVCardAddress* adr;

  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");

  if (!vcard->data->adr)
    return NULL;

  if (!vcard->data->adr->data)
    return NULL;

  adr = vcard->data->adr->data;

  return (gchar*) adr->code;
}
  

gchar*   
rubrica_vcard_get_count (RubricaVCard* vcard)
{
  RubricaVCardAddress* adr;

  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");

  if (!vcard->data->adr)
    return NULL;

  if (!vcard->data->adr->data)
    return NULL;

  adr = vcard->data->adr->data;

  return (gchar*) adr->count;
}


gchar*   
rubrica_vcard_get_adr_type (RubricaVCard* vcard)
{
  RubricaVCardAddress* adr;

  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");

  if (!vcard->data->adr)
    return NULL;

  if (!vcard->data->adr->data)
    return NULL;

  adr = vcard->data->adr->data;

  return (gchar*) adr->type;
}


gchar*   
rubrica_vcard_get_tel_num (RubricaVCard* vcard)
{
  RubricaVCardTelephone* tt;

  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");

  if (!vcard->data->tel)
    return NULL;

  if (!vcard->data->tel->data)
    return NULL;

  tt = vcard->data->tel->data;

  return (gchar*) tt->num;
}
  

gchar*   
rubrica_vcard_get_tel_type (RubricaVCard* vcard)
{
  RubricaVCardTelephone* tt;

  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), "");
  g_return_val_if_fail(vcard->data != NULL, "");

  if (!vcard->data->tel)
    return NULL;

  if (!vcard->data->tel->data)
    return NULL;

  tt = vcard->data->tel->data;  

  return (gchar*) tt->type;
}


gboolean 
rubrica_vcard_next_email (RubricaVCard* vcard)
{ 
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), FALSE);

  vcard->data->email = g_list_next(vcard->data->email);

  return (vcard->data->email != NULL);
}

gboolean 
rubrica_vcard_next_url (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), FALSE);

  vcard->data->url = g_list_next(vcard->data->url);

  return (vcard->data->url != NULL);
}

gboolean 
rubrica_vcard_next_adr (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), FALSE);

  vcard->data->adr = g_list_next(vcard->data->adr);

  return (vcard->data->adr != NULL);
}

gboolean 
rubrica_vcard_next_tel (RubricaVCard* vcard)
{
  g_return_val_if_fail(IS_RUBRICA_VCARD(vcard), FALSE);

  vcard->data->tel = g_list_next(vcard->data->tel);

  return (vcard->data->tel != NULL);
}
