/*
*  RAL -- Rubrica Addressbook Library
*  file: csv.c
*
*  the vcard import engine v0.5 
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>

#include "rubrica.h"
#include "io.h"
#include "load.h"
#include "save.h"
#include "../../abook.h"
#include "../../plugin.h"
#include "../../card.h"
#include "../../personal.h"
#include "../../company.h"
#include "../../contact.h"
#include "../../work.h"
#include "../../notes.h"
#include "../../address.h"
#include "../../net.h"
#include "../../telephone.h"
#include "../../ref.h"

// #define R_FILE_FORMAT 4


static RAbookClass* parent_class = NULL;


struct _RRubricaPrivate{
  FILE* fp;                        /* the rubrica file */

  gboolean dispose_has_run;
};


static void   r_rubrica_init        (RRubrica* obj);
static void   r_rubrica_class_init  (RRubricaClass* klass);

static void   r_rubrica_dispose     (GObject* obj);
static void   r_rubrica_finalize    (GObject* obj);



/* private functions
*/

static gboolean r_rubrica_open_file      (RAbook* abook, gchar* filename);
static gboolean r_rubrica_save_file      (RAbook* abook, gchar* filename, 
					  gint compression_rate);
static gboolean r_rubrica_overwrite_file (RAbook* abook, 
					  gint compression_rate);

static gboolean r_rubrica_parse_doc      (RAbook* abook, xmlDocPtr xmldoc);
static gboolean r_rubrica_parse_old_doc  (RAbook* abook, xmlDocPtr xmldoc);
static gboolean r_rubrica_write_doc      (RAbook* abook, gchar* name, 
					  gint compression_rate);


GType
r_rubrica_get_type()
{
  static GType r_rubrica_type = 0;
  
  if (!r_rubrica_type)
    {
      static const GTypeInfo r_rubrica_info =
	{
	  sizeof(RRubricaClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_rubrica_class_init,
	  NULL,
	  NULL,
	  sizeof(RRubrica),
	  0,
	  (GInstanceInitFunc) r_rubrica_init
	};

      r_rubrica_type = g_type_register_static (R_ABOOK_TYPE, "RRubrica",
					       &r_rubrica_info, 0);
    }
  
  return r_rubrica_type;
}


static void
r_rubrica_class_init(RRubricaClass* klass)
{
  GObjectClass *object_class;
  
  object_class = G_OBJECT_CLASS (klass);
  parent_class = R_ABOOK_CLASS (klass);

  object_class->dispose  = (GObjectFinalizeFunc) r_rubrica_dispose;
  object_class->finalize = (GObjectFinalizeFunc) r_rubrica_finalize;

  parent_class->abook_open_file = (gpointer) r_rubrica_open_file;
  parent_class->abook_save_file = (gpointer) r_rubrica_save_file;
}


static void
r_rubrica_init(RRubrica* self)
{
  g_return_if_fail(IS_R_RUBRICA(self));
 
  self->private = g_new(RRubricaPrivate, 1);
  if (!self->private)
    g_error("\nOut of memory");

  self->private->fp  = NULL;
 
  self->private->dispose_has_run = FALSE;
}


static void 
r_rubrica_dispose (GObject* obj)
{
  RRubrica* self;
 
  self = R_RUBRICA(obj);
  g_return_if_fail(IS_R_RUBRICA(self));

  if (self->private->dispose_has_run)
    return;

  self->private->dispose_has_run = TRUE;
}


static void 
r_rubrica_finalize (GObject* obj)
{
  RRubrica* self;

  self = R_RUBRICA(obj);
  g_return_if_fail(IS_R_RUBRICA(self));
  
  g_free(self->private);
  if (self->private->fp)
    fclose(self->private->fp);
}


static gboolean 
r_rubrica_open_file (RAbook* abook, gchar* filename) 
{ 
  xmlDocPtr xmldoc = NULL;
  RError err; 
  gboolean loaded = FALSE; 
  gint fileformat  = 0;

  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  
  if (!filename)
    {
      g_signal_emit_by_name(R_ABOOK(abook), "open_fail", 
			    NO_FILENAME, G_TYPE_INT);

      return FALSE;
    }

  if (!g_file_test(filename, G_FILE_TEST_EXISTS))
    {
      g_signal_emit_by_name(R_ABOOK(abook), "open_fail", 
			    FILE_NOT_EXIST, G_TYPE_INT);

      return FALSE;
    }

  if((xmldoc = r_open_doc(filename, &err)) == NULL)
    return FALSE;
  
  if(!r_is_rubrica_doc(xmldoc, &err))
    {
      //      g_signal_emit_by_name(R_ABOOK(abook), "open_fail", err, G_TYPE_INT);
      
      xmlFreeDoc(xmldoc);
      return FALSE;
    }

  /* get the file format
   */
  fileformat = r_get_fileformat(xmldoc, &err);
  if (fileformat < atoi(RUBRICA_FILE_FORMAT)) /* this is an old rubrica file */
    {
      if (fileformat == 3)
	loaded = r_rubrica_parse_old_doc(abook, xmldoc);
    }
  else
    loaded = r_rubrica_parse_doc(abook, xmldoc);

  if (!loaded)
    {
      g_signal_emit_by_name(R_ABOOK(abook), "open_fail",
			    FILE_NOT_OPENED, G_TYPE_INT);
      
      xmlFreeDoc(xmldoc);
      
      return FALSE;
    }
  else
    g_signal_emit_by_name(R_ABOOK(abook), "addressbook_read", 
			  NULL, G_TYPE_NONE);

  xmlFreeDoc(xmldoc);

  g_object_set(R_ABOOK(abook),
	       "addressbook-name", g_path_get_basename(filename),
	       "addressbook-path", g_path_get_dirname(filename),
	       NULL);

  return TRUE;
}


static gboolean 
r_rubrica_save_file (RAbook* abook, gchar* filename, gint compression_rate)
{
  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
    
  if (!filename)
    {
      g_signal_emit_by_name(abook, "save_fail", NO_FILENAME, G_TYPE_INT);

      return FALSE;
    }

  if (g_file_test(filename, G_FILE_TEST_EXISTS))
    {
      g_signal_emit_by_name(abook, "save_fail", FILE_EXIST, G_TYPE_INT);

      return FALSE;
    }

  if (!r_rubrica_write_doc(abook, filename, compression_rate))
    {
      g_signal_emit_by_name(abook, "save_fail", WRITING_XML, G_TYPE_INT);
      
      return FALSE;
    }
    
  return TRUE;
}


static gboolean 
r_rubrica_overwrite_file(RAbook* abook, gint compression_rate)
{
  gchar* file;
  gchar* path;
  gchar* filename;
  
  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  
  g_object_get(abook, "addressbook-path", &path, 
	       "addressbook-name", &file, NULL);
  filename = g_strdup_printf("%s%s%s", path, G_DIR_SEPARATOR_S, file);
  
  if (g_file_test(filename, G_FILE_TEST_EXISTS))
    g_remove(filename);

  if (!r_rubrica_write_doc(abook, filename, compression_rate))
    {
      g_signal_emit_by_name(abook, "save_fail", OVERWRITING, G_TYPE_INT);
      
      g_free(filename);
      return FALSE;
    }
    
  g_free(filename);
  g_signal_emit_by_name(abook, "addressbook_saved", NULL, G_TYPE_NONE);  
  
  return TRUE;
}


static gboolean 
r_rubrica_parse_old_doc(RAbook* abook, xmlDocPtr xmldoc)
{
  xmlNodePtr xmlabook;
     
  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  g_return_val_if_fail(xmldoc != NULL, FALSE);
  
  xmlabook = xmldoc->children;
  if (xmlIsBlankNode(xmlabook))
    xmlabook = xmlabook->next;
  
  if (xmlabook && xmlabook->name && xmlStrcmp(xmlabook->name,
					      (xmlChar*) "Rubrica") == 0) 
    {
      xmlabook = xmlabook->children;
      
      if (xmlIsBlankNode(xmlabook))
	xmlabook = xmlabook->next;

      while (xmlabook)
	{
	  xmlNodePtr xmlcard = NULL;

	  xmlcard = r_io_get_node(xmlabook, (xmlChar*) "Card");
	  
	  if (xmlcard)
	    {
	      RPersonalCard *card;  
	      
	      card = r_read_old_personal_xmlcard(xmlcard);  
	      if (card)
		{
		  r_abook_add_loaded_card(abook, (gpointer) card); 
		  g_signal_emit_by_name(abook, "card_read", card, 
					G_TYPE_POINTER);
		}
	    }  

	   /* parse next Card */
	   xmlabook = xmlabook->next; 
	   if(xmlIsBlankNode(xmlabook))
	     xmlabook = xmlabook->next;
	}
    }
  else
    {
      g_signal_emit_by_name(abook, "open_fail", NO_RUBRICA_DOC, G_TYPE_INT);
      
      return FALSE;
    }
  
  g_signal_emit_by_name(abook, "addressbook_read", NULL, G_TYPE_NONE);
  
  return TRUE;
}



static gboolean 
r_rubrica_parse_doc(RAbook* abook, xmlDocPtr xmldoc)
{  
  xmlNodePtr xmlabook;
  RError err;
     
  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  g_return_val_if_fail(xmldoc != NULL, FALSE);
  
  xmlabook = xmldoc->children;
  if (xmlIsBlankNode(xmlabook))
    xmlabook = xmlabook->next;
  
  if (xmlabook && xmlabook->name && xmlStrcmp(xmlabook->name,
					      (xmlChar*) "Rubrica") == 0) 
    {
      xmlabook = xmlabook->children;
      
      if (xmlIsBlankNode(xmlabook))
	xmlabook = xmlabook->next;

      while (xmlabook)
	{
	  xmlNodePtr xmlcard = NULL;

	  xmlcard = r_io_get_node(xmlabook, (xmlChar*) "Card");
	  
	  if (xmlcard)
	    {
	      gchar* tmp = NULL;
	      
	      tmp = r_io_get_prop(xmlcard, "type", &err);
	      if (!tmp || (g_ascii_strcasecmp("personal", tmp) == 0))
		{
		  RPersonalCard *card;  

		  card = r_read_personal_xmlcard(xmlcard);  
		  if (card)
		    {
		      r_abook_add_loaded_card(abook, (gpointer) card); 
		      g_signal_emit_by_name(abook, "card_read", card, 
					    G_TYPE_POINTER);
		    }
		  
		}
	      else if (g_ascii_strcasecmp("company", tmp) == 0)
		{
		  RCompanyCard* card;

		  card = r_read_company_xmlcard(xmlcard);

		  if (card)
		    {
		      r_abook_add_loaded_card(abook, (gpointer) card);
		      
		      g_signal_emit_by_name(abook, "card_read", card, 
					    G_TYPE_POINTER);
		    }
		}
	      
	      if (tmp) g_free(tmp);	   	      	      	     
	    }  

	   /* parse next Card */
	   xmlabook = xmlabook->next; 
	   if(xmlIsBlankNode(xmlabook))
	     xmlabook = xmlabook->next;
	}
    }
  else
    return FALSE;
  
  return TRUE;
}



static gboolean
r_rubrica_write_doc(RAbook* abook, gchar* name, gint compression_rate)
{
  xmlDocPtr doc;        /* puntatore al documento xml */
  gpointer card;
  gchar *path = NULL, *fname;
  gchar *fileformat;

  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  g_return_val_if_fail(name != NULL, FALSE);

  path  = g_dirname(name);
  fname = g_strdup(name);

  /* creazione di un nuovo documento xml 
   */
  doc = xmlNewDoc((xmlChar*) "1.0"); 
  xmlSetDocCompressMode (doc, compression_rate);

  /* creo un nuovo nodo radice rubrica 
   */
  fileformat = g_strdup_printf("%d", atoi(RUBRICA_FILE_FORMAT));

  doc->children = xmlNewDocNode(doc, NULL, (xmlChar*) "Rubrica", NULL);
  xmlSetProp(doc->children, (xmlChar*) "version", (xmlChar*) RUBRICA_VERSION);
  xmlSetProp(doc->children, (xmlChar*) "fileformat", (xmlChar*) fileformat); 
  xmlSetProp(doc->children, (xmlChar*) "doctype", (xmlChar*) "AddressBook"); 

  g_free(fileformat);

  /* attraverso la rubrica e scrivo i dati 
     nell'albero xml
  */
  r_abook_reset_book(abook);
  card = r_abook_get_card(abook);
  for ( ; card; card = r_abook_get_next_card(abook))
    {
      xmlNodePtr cardxml;     /* card node */
      gboolean destroyed;
      
      g_object_get(card, "card-destroyed", &destroyed, NULL);
      
      if (!destroyed)
	{
	  cardxml = xmlNewChild(doc->children, NULL, 
				(xmlChar*) "Card", 
				(xmlChar*) NULL);
	  
	  r_write_card(R_CARD(card), cardxml);
	}
    }     
  
  if (xmlSaveFormatFile(fname, doc, 1) == -1) 
    return FALSE;

  /* if we are here the addressbook is write 
   */  
  xmlFreeDoc(doc);    
  g_free(fname);

  return TRUE;
}



/* *****************************  Public functions  ***********************
 */

/**
 * r_rubrica_new
 *
 * create a new #RRubrica
 *
 * returns: a #RRubrica*
 */
RRubrica* 
r_rubrica_new(void)
{
  return (RRubrica*) g_object_new(r_rubrica_get_type(), NULL);
}


/**
 * r_rubrica_free
 * @rubrica: a #RRubrica
 * 
 * free the #RRubrica object
 */
void  
r_rubrica_free(RRubrica *rubrica) 
{ 
  g_return_if_fail(IS_R_RUBRICA(rubrica)); 

  g_object_unref(rubrica); 
} 



G_MODULE_EXPORT void
plugin_init (RPlugin* plugin, gchar* file)
{
  RFilter* filter;
  RPluginAction* action;
  
  g_return_if_fail(plugin != NULL);
   
  r_plugin_set_obj(plugin, r_rubrica_new());
  g_object_set(plugin, 
	       "plugin-name", "rubrica", 	       
	       "plugin-filename", file,
	       "plugin-info", "This plugin manages the rubrica's file format",
	       "plugin-configurable", FALSE, NULL);
  
  filter = r_filter_new();
  g_object_set(filter, 
	       "filter-name", "rubrica", 
	       "filter-extension", "rub",
	       "filter-mime", "application/x-rubrica", NULL);
  r_filter_add_pattern(filter, "rub");
  r_filter_add_pattern(filter, "*.rub");
  r_plugin_add_filter (plugin, filter);
 
  action = g_malloc(sizeof(RPluginAction));
  action->name   = g_strdup("read");
  action->handle = (gpointer) r_rubrica_open_file;
  r_plugin_add_action(plugin, action);  
    
  action = g_malloc(sizeof(RPluginAction));
  action->name   = g_strdup("write");
  action->handle = (gpointer) r_rubrica_save_file;
  r_plugin_add_action(plugin, action);  

  action = g_malloc(sizeof(RPluginAction));
  action->name   = g_strdup("overwrite");
  action->handle = (gpointer) r_rubrica_overwrite_file;
  r_plugin_add_action(plugin, action);   
}


G_MODULE_EXPORT void     
plugin_fini (void)
{
  // TODO: liberare la memoria e rilasciare l'handler al plugin
}


