# frozen_string_literal: true

require 'scientist'
require 'active_support/callbacks'
require 'active_support/core_ext/object/blank'
require 'active_support/core_ext/string/inflections'

require 'gitlab/experiment/caching'
require 'gitlab/experiment/callbacks'
require 'gitlab/experiment/configuration'
require 'gitlab/experiment/cookies'
require 'gitlab/experiment/context'
require 'gitlab/experiment/dsl'
require 'gitlab/experiment/variant'
require 'gitlab/experiment/version'
require 'gitlab/experiment/engine' if defined?(Rails::Engine)

module Gitlab
  class Experiment
    include Scientist::Experiment
    include Caching
    include Callbacks

    class << self
      def configure
        yield Configuration
      end

      def run(name = nil, variant_name = nil, **context, &block)
        raise ArgumentError, 'name is required' if name.nil? && base?

        instance = constantize(name).new(name, variant_name, **context, &block)
        return instance unless block

        instance.context.frozen? ? instance.run : instance.tap(&:run)
      end

      def experiment_name(name = nil, suffix: true, suffix_word: 'experiment')
        name = (name.presence || self.name).to_s.underscore.sub(%r{(?<char>[_/]|)#{suffix_word}$}, '')
        name = "#{name}#{Regexp.last_match(:char) || '_'}#{suffix_word}"
        suffix ? name : name.sub(/_#{suffix_word}$/, '')
      end

      def base?
        self == Gitlab::Experiment || name == Configuration.base_class
      end

      private

      def constantize(name = nil)
        return self if name.nil?

        experiment_name(name).classify.safe_constantize || Configuration.base_class.constantize
      end
    end

    def initialize(name = nil, variant_name = nil, **context)
      raise ArgumentError, 'name is required' if name.blank? && self.class.base?

      @name = self.class.experiment_name(name, suffix: false)
      @context = Context.new(self, **context)
      @variant_name = cache_variant(variant_name) { nil } if variant_name.present?

      compare { false }

      yield self if block_given?
    end

    def inspect
      "#<#{self.class.name || 'AnonymousClass'}:#{format('0x%016X', __id__)} @name=#{name} @signature=#{signature}>"
    end

    def context(value = nil)
      return @context if value.blank?

      @context.value(value)
      @context
    end

    def variant(value = nil)
      if value.blank? && @variant_name || @resolving_variant
        return Variant.new(name: (@variant_name || :unresolved).to_s)
      end

      @variant_name = value unless value.blank?

      if enabled?
        @variant_name ||= :control if excluded?

        @resolving_variant = true
        if (result = cache_variant(@variant_name) { resolve_variant_name }).present?
          @variant_name = result.to_sym
        end
      end

      Variant.new(name: (@variant_name || :control).to_s)
    ensure
      @resolving_variant = false
    end

    def run(variant_name = nil)
      @result ||= begin
        variant_name = variant(variant_name).name
        run_callbacks(run_with_segmenting? ? :segmented_run : :unsegmented_run) do
          super(@variant_name ||= variant_name)
        end
      end
    end

    def publish(result)
      instance_exec(result, &Configuration.publishing_behavior)
    end

    def track(action, **event_args)
      return unless should_track?

      instance_exec(action, event_args, &Configuration.tracking_behavior)
    end

    def name
      [Configuration.name_prefix, @name].compact.join('_')
    end

    def variant_names
      @variant_names ||= behaviors.keys.map(&:to_sym) - [:control]
    end

    def behaviors
      @behaviors ||= public_methods.each_with_object(super) do |name, behaviors|
        next unless name.end_with?('_behavior')

        behavior_name = name.to_s.sub(/_behavior$/, '')
        behaviors[behavior_name] ||= -> { send(name) } # rubocop:disable GitlabSecurity/PublicSend
      end
    end

    def try(name = nil, &block)
      name = (name || 'candidate').to_s
      behaviors[name] = block
    end

    def signature
      { variant: variant.name, experiment: name }.merge(context.signature)
    end

    def id
      "#{name}:#{key_for(context.value)}"
    end
    alias_method :session_id, :id

    def flipper_id
      "Experiment;#{id}"
    end

    def enabled?
      true
    end

    def excluded?
      @excluded ||= !@context.trackable? || # adhere to DNT headers
        !run_callbacks(:exclusion_check) { :not_excluded } # didn't pass exclusion check
    end

    def should_track?
      enabled? && !excluded?
    end

    def key_for(hash)
      instance_exec(hash, &Configuration.context_hash_strategy)
    end

    protected

    def run_with_segmenting?
      !variant_assigned? && enabled? && !excluded?
    end

    def variant_assigned?
      !@variant_name.nil?
    end

    def resolve_variant_name
      instance_exec(@variant_name, &Configuration.variant_resolver)
    end

    def generate_result(variant_name)
      observation = Scientist::Observation.new(variant_name, self, &behaviors[variant_name])
      Scientist::Result.new(self, [observation], observation)
    end
  end
end
