# frozen_string_literal: true

require 'singleton'
require 'logger'
require 'digest'

module Gitlab
  class Experiment
    class Configuration
      include Singleton

      # Prefix all experiment names with a given value. Use `nil` for none.
      @name_prefix = nil

      # The logger is used to log various details of the experiments.
      @logger = Logger.new($stdout)

      # The base class that should be instantiated for basic experiments.
      @base_class = 'Gitlab::Experiment'

      # The caching layer is expected to respond to fetch, like Rails.cache.
      @cache = nil

      # The domain to use on cookies.
      @cookie_domain = :all

      # Logic this project uses to resolve a variant for a given experiment.
      # If no variant is determined, the control will be used.
      @variant_resolver = lambda do |requested_variant|
        requested_variant
      end

      # Tracking behavior can be implemented to link an event to an experiment.
      @tracking_behavior = lambda do |event, args|
        Configuration.logger.info "Gitlab::Experiment[#{name}] #{event}: #{args.merge(signature: signature)}"
      end

      # Called at the end of every experiment run, with the result.
      @publishing_behavior = lambda do |_result|
        track(:assignment)
      end

      # Algorithm that consistently generates a hash key for a given hash map.
      @context_hash_strategy = lambda do |hash_map|
        values = hash_map.values.map { |v| (v.respond_to?(:to_global_id) ? v.to_global_id : v).to_s }
        Digest::MD5.hexdigest(([name] + hash_map.keys + values).join('|'))
      end

      class << self
        attr_accessor(
          :name_prefix,
          :logger,
          :base_class,
          :cache,
          :cookie_domain,
          :variant_resolver,
          :tracking_behavior,
          :publishing_behavior,
          :context_hash_strategy
        )
      end
    end
  end
end
