/*
 * Copyright (C) 2006 Sjoerd Simons <sjoerd@luon.net>
 * Copyright (C) 2003, 2004 Laurent Sansonetti <lrz@gnome.org>
 *
 * This file is part of Ruby/GStreamer.
 *
 * Ruby/GStreamer is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * Ruby/GStreamer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Ruby/GStreamer; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include "rbgst.h"
#include <unistd.h>

/* Class: Gst::Element
 * Base class for all pipeline elements.
 */

struct thread_state_data {
  GstElement *element;
  GstState state;
  GstState pending;
  GstState ret;
  GstClockTime timeout;
  int writefd;
};

static void
thread_set_state_function(gpointer data, gpointer user_data) {
  struct thread_state_data *sdata = (struct thread_state_data *) data;

  sdata->ret = gst_element_set_state (sdata->element, sdata->state);
  write(sdata->writefd, "d", 1);
}

/* Threaded set_state function */
static VALUE
rb_gst_element_threaded_set_state(VALUE element, int state) {
  int comm[2];
  GstState ret;
  struct thread_state_data *data;
  GError *error = NULL;
  static GThreadPool *pool = NULL;

  if (G_UNLIKELY(pool == NULL)) {
    pool = g_thread_pool_new(thread_set_state_function, 
                             NULL, -1, FALSE, &error);
    if (pool == NULL) {
      rb_bug("Couldn't create rbgst thread: %s", error->message);
    }
  }

  if (pipe(comm) != 0) {
    rb_bug("Unable to create rbgst communication pipe");
  }

  data = g_slice_new(struct thread_state_data);
  data->element = RGST_ELEMENT(element);
  data->state = state;
  data->writefd = comm[1];

  g_thread_pool_push(pool, data, &error);
  if (error != NULL) {
    rb_bug("Couldn't create rbgst thread: %s", error->message);
  }
  rb_thread_wait_fd(comm[0]);
  ret = data->ret;

  g_slice_free(struct thread_state_data, data);
  close(comm[0]);
  close(comm[1]);

  return GENUM2RVAL (ret, GST_TYPE_STATE_CHANGE_RETURN);
}

static void
thread_get_state_function(gpointer data, gpointer user_data) {
  struct thread_state_data *sdata = (struct thread_state_data *) data;

  sdata->ret = gst_element_get_state (sdata->element, 
                                      &(sdata->state),
                                      &(sdata->pending),
                                      sdata->timeout);
  write(sdata->writefd, "d", 1);
}

/* Threaded get_state function */
static VALUE
rb_gst_element_threaded_get_state(VALUE element, GstState *state,
                                  GstState *pending,   GstClockTime timeout) {
  int comm[2];
  struct thread_state_data *data;
  GError *error = NULL;
  static GThreadPool *pool = NULL;
  GstState ret;

  if (G_UNLIKELY(pool == NULL)) {
    pool = g_thread_pool_new(thread_get_state_function, 
                             NULL, -1, FALSE, &error);
    if (pool == NULL) {
      rb_bug("Couldn't create rbgst thread: %s", error->message);
    }
  }

  if (pipe(comm) != 0) {
    rb_bug("Unable to create rbgst communication pipe");
  }

  data = g_slice_new(struct thread_state_data);
  data->element = RGST_ELEMENT(element);
  data->writefd = comm[1];

  g_thread_pool_push(pool, data, &error);
  if (error != NULL) {
    rb_bug("Couldn't create rbgst thread: %s", error->message);
  }
  rb_thread_wait_fd(comm[0]);

  *state = data->state;
  *pending = data->pending;
  ret = data->ret;

  g_slice_free(struct thread_state_data, data);
  close(comm[0]);
  close(comm[1]);

  return GENUM2RVAL (ret, GST_TYPE_STATE_CHANGE_RETURN);
}


/*
 * Method: set_state(state)
 * state: the state you want to set (see Gst::Element::State).
 *
 * Sets the state of the element. 
 *
 * This method will try to set the requested state by going through all 
 * the intermediary states and calling the class's state change function 
 * for each.
 *
 * Returns: a code (see Gst::Element::StateReturn).
 */
static VALUE
rb_gst_element_set_state (VALUE self, VALUE value)
{
    const int state = RVAL2GENUM (value, GST_TYPE_STATE);
    return rb_gst_element_threaded_set_state(self, state);
}

static VALUE
rb_gst_element_get_state (int argc, VALUE *argv, VALUE self)
{
  VALUE result, rstate, rpending, timeout;
  GstState state;
  GstState pending;
  rb_scan_args(argc, argv, "01", &timeout);

  result = rb_gst_element_threaded_get_state(self, &state, &pending,
                                      NIL_P(timeout) ? GST_CLOCK_TIME_NONE 
                                        : NUM2ULL(timeout)); 
  rstate = GENUM2RVAL(state, GST_TYPE_STATE);
  rpending = GENUM2RVAL(state, GST_TYPE_STATE);
  return rb_ary_new3(3, result, rstate, rpending);
}

/*
 * Method: stop
 *
 * This method calls Gst::Element#set_state with Gst::Element::STATE_NULL.
 *
 * Returns: a code (see Gst::Element::StateReturn).
 */
static VALUE
rb_gst_element_stop (VALUE self)
{
    return rb_gst_element_threaded_set_state(self, GST_STATE_NULL);
}

/*
 * Method: ready
 *
 * This method calls Gst::Element#set_state with Gst::Element::STATE_READY.
 *
 * Returns: a code (see Gst::Element::StateReturn).
 */
static VALUE
rb_gst_element_ready (VALUE self)
{
  return rb_gst_element_threaded_set_state(self, GST_STATE_READY);
}

/*
 * Method: pause
 *
 * This method calls Gst::Element#set_state with Gst::Element::STATE_PAUSED.
 *
 * Returns: a code (see Gst::Element::StateReturn).
 */
static VALUE
rb_gst_element_pause (VALUE self)
{
  return rb_gst_element_threaded_set_state(self, GST_STATE_PAUSED);
}

/*
 * Method: play
 *
 * This method calls Gst::Element#set_state with Gst::Element::STATE_PLAYING.
 *
 * Returns: a code (see Gst::Element::StateReturn).
 */
static VALUE
rb_gst_element_play (VALUE self)
{
  return rb_gst_element_threaded_set_state(self, GST_STATE_PLAYING);
}

/* Method: stopped?
 * Returns: true if the current state is set to Gst::Element::STATE_NULL,
 * false otherwise.
 */
/* FIXME
static VALUE
rb_gst_element_is_stopped (VALUE self)
{
    return CBOOL2RVAL (gst_element_get_state (RGST_ELEMENT (self)) ==
                       GST_STATE_NULL);
}
*/

/* Method: ready?
 * Returns: true if the current state equals Gst::Element::STATE_READY,
 * false otherwise.
 */
/* FIXME 
static VALUE
rb_gst_element_is_ready (VALUE self)
{
    return CBOOL2RVAL (gst_element_get_state (RGST_ELEMENT (self)) ==
                       GST_STATE_READY);
}
*/

/* Method: paused?
 * Returns: true if the current state equals Gst::Element::STATE_PAUSED,
 * false otherwise.
 */
/* FIXME
static VALUE
rb_gst_element_is_paused (VALUE self)
{
    return CBOOL2RVAL (gst_element_get_state (RGST_ELEMENT (self)) ==
                       GST_STATE_PAUSED);
}
*/

/* Method: playing?
 * Returns: true if the current state equals Gst::Element::STATE_PLAYING,
 * false otherwise.
 */
/* FIXME
static VALUE
rb_gst_element_is_playing (VALUE self)
{
    return CBOOL2RVAL (gst_element_get_state (RGST_ELEMENT (self)) ==
                       GST_STATE_PLAYING);
}
*/

/*
 * Method: wait_state_change
 *
 * Waits and blocks until the element changed its state.
 *
 * Returns: self.
 */
/* FIXME
static VALUE
rb_gst_element_wait_state_change (VALUE self)
{
    gst_element_wait_state_change (RGST_ELEMENT (self));
    return self;
}
*/

/*
 * Method: link(element)
 * element: a Gst::Element object.
 *
 * Links this element (source) to the provided element (destination). 
 *
 * The method looks for existing pads and request pads that 
 * aren't linked yet. If multiple links are possible, only one 
 * is established.
 *
 * Returns: the destination element, or nil if the link failed.
 */
static VALUE
rb_gst_element_link (VALUE self, VALUE other_element)
{
    GstElement *element1, *element2;

    element1 = RGST_ELEMENT (self);
    element2 = RGST_ELEMENT (other_element);
    return gst_element_link (element1, element2) == TRUE ? other_element : Qnil;
}

/*
 * Method: unlink(element)
 * element: a Gst::Element object.
 *
 * Unlinks all source pads of this element with all sink pads of element to
 * which they are linked.
 *
 * Returns: self
 */
static VALUE
rb_gst_element_unlink (VALUE self, VALUE other_element)
{
    GstElement *element1, *element2;

    element1 = RGST_ELEMENT (self);
    element2 = RGST_ELEMENT (other_element);
    gst_element_unlink (element1, element2);
    return self;
}

/*
 * Method: link_filtered(element, caps)
 * element: a Gst::Element object.
 * caps: a Gst::Caps object.
 *
 * Links this element (source) to the provided element (destination), 
 * filtered by the given caps.
 *
 * The method looks for existing pads and request pads that 
 * aren't linked yet. If multiple links are possible, only one 
 * is established.
 *
 * Returns: the destination element, or nil if the link failed.
 */
static VALUE
rb_gst_element_link_filtered (VALUE self, VALUE other_element, VALUE rcaps)
{
    GstElement *element1, *element2;
    GstCaps *caps;
    
    element1 = RGST_ELEMENT (self);
    element2 = RGST_ELEMENT (other_element);
    caps = RGST_CAPS (rcaps);
    return gst_element_link_filtered (element1, element2, caps)
        ? other_element 
        : Qnil;
}

/* Method: requires_clock?
 * Returns: true if the element requires a clock, false otherwise.
 */
static VALUE
rb_gst_element_requires_clock (VALUE self)
{
    return CBOOL2RVAL (gst_element_requires_clock (RGST_ELEMENT (self)));
}

/* Method: provides_clock?
 * Returns: true if the element provides a clock, false otherwise.
 */
static VALUE
rb_gst_element_provides_clock (VALUE self)
{
    return CBOOL2RVAL (gst_element_provides_clock (RGST_ELEMENT (self)));
}

/* Method: clock
 * Returns: the clock of the element (as a Gst::Clock object), or nil
 * if this element does not provide a clock.
 */
static VALUE
rb_gst_element_get_clock (VALUE self)
{
    GstClock *clock;

    clock = gst_element_get_clock (RGST_ELEMENT (self));
    return clock != NULL ? RGST_CLOCK_NEW (clock)
        : Qnil;
}

/*
 * Method: set_clock(clock)
 * clock: the Gst::Clock to set for the element.
 *
 * Sets the clock for the element.
 *
 * Returns: self.
 */
static VALUE
rb_gst_element_set_clock (VALUE self, VALUE clock)
{
    gst_element_set_clock (RGST_ELEMENT (self), RGST_CLOCK (clock));
    return self;
}

/*
 * Method: num_pads
 *
 * Returns: the number of pads of an element
 */
static VALUE
rb_gst_element_num_pads (VALUE self) 
{
  return INT2NUM(RGST_ELEMENT(self)->numpads);
}

/*
 * Method: each_pad { |pad| ... }
 *
 * Calls the block for each pad associated with the element, passing a 
 * reference to the Gst::Pad as parameter.
 *
 * Returns: always nil.
 */
static VALUE
rb_gst_element_each_pad (VALUE self)
{
  GstIterator* it = gst_element_iterate_pads(RGST_ELEMENT(self));
  GstPad *pad;
  gboolean done = FALSE;

  while (!done) {
    switch (gst_iterator_next (it, (gpointer)&pad)) {
      case GST_ITERATOR_OK:
        rb_yield(RGST_PAD_NEW(pad));
        gst_object_unref (pad);
        break;
      case GST_ITERATOR_RESYNC:
        gst_iterator_resync (it);
        break;
      case GST_ITERATOR_ERROR:
      case GST_ITERATOR_DONE:
        done = TRUE;
        break;
    }
  }
  gst_iterator_free (it);

  return Qnil;
}

/*
 * Method: get_pad(name)
 * name: the name of a pad.
 *
 * Retrieves a Gst::Pad object from the element by name.
 *
 * Returns: a Gst::Pad object, or nil if the pad cannot be found.
 */
static VALUE
rb_gst_element_get_pad (VALUE self, VALUE pad_name)
{
    GstPad *pad = gst_element_get_pad (RGST_ELEMENT (self),
                                       RVAL2CSTR (pad_name));

    return pad != NULL ? RGST_PAD_NEW (pad)
        : Qnil;
}

/*
 * Method: link_pads(element)
 * element: a Gst::Element.
 *
 * Links the "src" named pad of the current element to the
 * "sink" named pad of the destination element, returning
 * true on success.
 *
 * If you want to link specific named pads, you should use
 * the Gst::Pad#link method directly:
 *
 *	element1.link_pads(element2)
 *	
 * 	# This does the same
 *	element1.get_pad("src").link(element2.get_pad("sink"))
 *
 * Returns: true on success, false on failure.
 */
static VALUE
rb_gst_element_link_pads (VALUE self, VALUE other_element)
{
    return CBOOL2RVAL (gst_element_link_pads (RGST_ELEMENT (self),
                                              "src",
                                              RGST_ELEMENT (other_element),
                                              "sink"));
}

/*
 * Method: unlink_pads(element)
 * element: a Gst::Element.
 *
 * Unlinks the "src" named pad of the current element from the
 * "sink" named pad of the destination element.
 *
 * If you want to unlink specific named pads, you should use
 * the Gst::Pad#unlink method directly:
 *
 * 	element1.unlink_pads(element2)
 *	
 *	# This does the same
 *	element1.get_pad("src").unlink(element2.get_pad("sink"))
 *
 * Returns: self.
 */
static VALUE
rb_gst_element_unlink_pads (self, other_element)
        VALUE self, other_element;
{
    rb_warn("element#unlink_pads is deperated, please use element#unlink");
    rb_gst_element_unlink(self, other_element);
    return self;
}

/* Method: indexable?
 * Returns: true if the element can be indexed, false otherwise.
 */
static VALUE
rb_gst_element_is_indexable (VALUE self)
{
    return CBOOL2RVAL (gst_element_is_indexable (RGST_ELEMENT (self)));
}

/*
 * Method: query(query_type, format=Gst::Format::DEFAULT)
 * query_type: a query type (see Gst::QueryType::Types).
 * format: a format (see Gst::Format::Types).
 *
 * Performs a query on the element.
 *
 * Returns: a Fixnum value returned by the query, or nil if the query
 * could not be performed.
 */
/* FIXME
static VALUE
rb_gst_element_query (int argc, VALUE * argv, VALUE self)
{
    VALUE query_type, format;
    GstFormat gstformat;
    gint64 value;

    rb_scan_args (argc, argv, "11", &query_type, &format);
    gstformat =
        NIL_P (format) ? GST_FORMAT_DEFAULT : RVAL2GENUM (format,
                                                          GST_TYPE_FORMAT);

    if (gst_element_query (RGST_ELEMENT (self),
                           RVAL2GENUM (query_type, GST_TYPE_QUERY_TYPE),
                           &gstformat, &value)) {
        format = INT2FIX (gstformat);
        return ULL2NUM (value);
    }
    return Qnil;
}
*/

/*
 * Method: send_event(event)
 * event: a Gst::Event object.
 *
 * Sends an event to an element, through a Gst::Event object. 
 * If the element doesn't implement an event handler, the event will be 
 * forwarded to a random sink pad.
 *
 * Returns: true if the request event was successfully handled, false
 * otherwise.
 */
static VALUE
rb_gst_element_send_event (VALUE self, VALUE event)
{
    return CBOOL2RVAL (gst_element_send_event (RGST_ELEMENT (self),
                                               RGST_EVENT (event)));
}

/*
 * Method: base_time
 *
 * Queries the element's base time.
 *
 * Returns:  the current stream time (in nanoseconds) in Gst::Element::STATE_PLAYING, the 
 * element base time in Gst::Element::STATE_PAUSED, or -1 otherwise.
 */
static VALUE
rb_gst_element_get_base_time (VALUE self)
{
    return ULL2NUM (gst_element_get_base_time (RGST_ELEMENT (self)));
}

/* 
 * Method: index
 *
 * Gets the index from the element. 
 *
 * Returns: a Gst::Index or nil when no index was set on the element.
 */
static VALUE
rb_gst_element_get_index (VALUE self)
{
    GstIndex *index = gst_element_get_index (RGST_ELEMENT (self));

    return index != NULL ? RGST_INDEX_NEW (index)
        : Qnil;
}

/*
 * Method: set_index(index)
 * index: the index to set, as a Gst::Index.
 * 
 * Sets the specified index on the element.
 *
 * Returns: self. 
 */
static VALUE
rb_gst_element_set_index (VALUE self, VALUE index)
{
    gst_element_set_index (RGST_ELEMENT (self), RGST_INDEX (index));
    return self;
}

/*
 * Method: get_static_pad(name)
 * name: the name of the static Gst::Pad to retrieve.
 *
 * Retrieves a pad from the element by name.  This version only retrieves
 * already existing (i.e. 'static') pads.
 *
 * Returns: the requested Gst::Pad if found, otherwise nil.
 */
static VALUE
rb_gst_element_get_static_pad (VALUE self, VALUE name)
{
    GstPad *pad =
        gst_element_get_static_pad (RGST_ELEMENT (self), RVAL2CSTR (name));
    return pad != NULL ? RGST_PAD_NEW (pad) : Qnil;
}

/*
 * Method: get_request_pad(name)
 * name: the name of the request Gst::Pad to retrieve.
 *
 * Retrieves a pad from the element by name.  This version only retrieves
 * request pads.
 *
 * Returns: the requested Gst::Pad if found, otherwise nil.
 */
static VALUE
rb_gst_element_get_request_pad (VALUE self, VALUE name)
{
    GstPad *pad =
        gst_element_get_request_pad (RGST_ELEMENT (self), RVAL2CSTR (name));
    return pad != NULL ? RGST_PAD_NEW (pad) : Qnil;
}

/*
 * Method: release_request_pad(pad)
 * pad: the Gst::Pad to release.
 *
 * Makes the element free the previously requested pad ass obtained with
 * Gst::Element#get_requested_pad.
 *
 * Returns: self.
 */
static VALUE
rb_gst_element_release_request_pad (VALUE self, VALUE pad)
{
    gst_element_release_request_pad (RGST_ELEMENT (self), RGST_PAD (pad));
    return self;
}

/*
 * Method: add_pad(pad)
 * pad: the Gst::Pad to add to the element.
 *
 * Adds a pad (link point) to the element.  Pads are automatically activated
 * when the element is in state Gst::Element::PLAYING.
 *
 * Returns: self.
 */
static VALUE
rb_gst_element_add_pad (VALUE self, VALUE pad)
{
    gst_object_ref(RVAL2GOBJ(pad));
    gst_element_add_pad (RGST_ELEMENT (self), RGST_PAD (pad));
    return self;
}

/*
 * Method: remove_pad(pad)
 * pad: the Gst::Pad to remove from the element.
 *
 * Removes the given pad from the element.
 *
 * Returns: self.
 */
static VALUE
rb_gst_element_remove_pad (VALUE self, VALUE pad)
{
    gst_element_remove_pad (RGST_ELEMENT (self), RGST_PAD (pad));
    return self;
}

/*
 * Method: get_compatible_pad(pad, caps=nil)
 * pad: the Gst::Pad to find a compatible one for.
 * caps: the Gst::Caps to use as a filter.
 *
 * Looks for an unlinked pad to which the given pad can link to.  It is not
 * guaranteed that linking the pads will work, though it should work in most
 * cases.  You can eventually pass a Gst::Caps that will act as a filter, but
 * this is not mandatory.
 *
 * Returns: the Gst::Pad to which a link can be made, or nil if one could not
 * be found.
 */
static VALUE
rb_gst_element_get_compatible_pad (int argc, VALUE * argv, VALUE self)
{
    VALUE pad, caps;
    GstPad *pad2;

    rb_scan_args (argc, argv, "11", &pad, &caps);

    pad2 = gst_element_get_compatible_pad(RGST_ELEMENT(self), RGST_PAD(pad), 
        NIL_P(caps) ? RGST_CAPS(caps) : NULL);

    return pad2 != NULL ? RGST_PAD_NEW (pad2) : Qnil;
}

/*
 * Method: get_pad_template(name)
 * name: the name of the Gst::PadTemplate to get.
 *
 * Retrieves a Gst::PadTemplate from this element with the given name.
 *
 * Returns: the Gst::PadTemplate with the given name, or nil if none was found.
 */
/* FIXME
static VALUE
rb_gst_element_get_pad_template (VALUE self, VALUE name)
{
    GstPadTemplate *pad =
        gst_element_get_pad_template (RGST_ELEMENT (self), RVAL2CSTR (name));
    return pad != NULL ? RGST_PAD_TEMPLATE_NEW (pad) : Qnil;
}
*/

/*
 * Method: pad_templates
 *
 * Retrieves a list of pad templates associated with the element.
 * 
 * Returns: an Array of Gst::PadTemplate objects.
 */
/*
static VALUE
rb_gst_element_get_pad_templates (VALUE self)
{
    GList *list;
    VALUE ary;

    ary = rb_ary_new ();

    for (list = gst_element_get_pad_template_list (RGST_ELEMENT (self));
         list != NULL; list = g_list_next (list))
        rb_ary_push (ary, RGST_PAD_TEMPLATE_NEW (list->data));

    g_list_free (list);
    return ary;
}
*/

/*
 * Method: each_pad_template { |pad_template| ... }
 *
 * Calls the block for each pad template associated with the element,
 * passing a reference to a Gst::PadTemplate object as parameter.
 *
 * Returns: always nil.
 */
/*
static VALUE
rb_gst_element_each_pad_template (VALUE self)
{
    return rb_ary_yield (rb_gst_element_get_pad_templates (self));
}
*/

/*
 * Method: get_compatible_pad_template(pad)
 * pad: the Gst::PadTemplate to find a compatible template for.
 *
 * Retrieves a pad template from the element that is compatible with the given
 * pad template.  Pads from compatible templates can be linked together.
 *
 * Returns: a compatible Gst::PadTemplate, or nil if none was found.
 */
static VALUE
rb_gst_element_get_compatible_pad_template (VALUE self, VALUE pad)
{
    GstPadTemplate *pad2 =
        gst_element_get_compatible_pad_template (RGST_ELEMENT (self),
                                                 RGST_PAD_TEMPLATE (pad));

    return pad2 != NULL ? RGST_PAD_TEMPLATE_NEW (pad2) : Qnil;
}

/*
 * FIXME
 * Method: seek(seek_type, offset)
 * seek_type: the method to use for seeking (see Gst::EventSeek::Type).
 * offset: the offset to seek to.
 *
 * Sends a seek event (Gst::EventSseek) to the element.
 * 
 * Returns: true if the event was handled.
 */
static VALUE
rb_gst_element_seek (VALUE self, VALUE rate, VALUE format, VALUE flags,
    VALUE cur_type, VALUE cur, VALUE stop_type, VALUE stop) {
    return
        CBOOL2RVAL (gst_element_seek (RGST_ELEMENT (self),
                     NUM2DBL (rate),
                     RVAL2GENUM (format, GST_TYPE_FORMAT),
                     NUM2INT(flags),
                     RVAL2GENUM(cur_type, GST_TYPE_SEEK_TYPE),
                     NUM2ULL(cur),
                     RVAL2GENUM(stop_type, GST_TYPE_SEEK_TYPE),
                     NUM2ULL(stop)));
}

/*
 * Method: locked_state?
 *
 * Checks if the state of an element is locked. If the state of an element is 
 * locked, state changes of the parent don't affect the element. ahis way you 
 * can leave currently unused elements inside bins. Just lock their state 
 * before changing the state from Gst::Element::STATE_NULL.
 *
 * Returns: true if the element's state is locked.
 */
static VALUE
rb_gst_element_is_locked_state (VALUE self)
{
    return CBOOL2RVAL (gst_element_is_locked_state (RGST_ELEMENT (self)));
}

/*
 * Method: set_locked_state(state)
 * state: whether to lock the element's state.
 *
 * Locks the state of an element, so state changes of the parent don't affect 
 * this element anymore.
 *
 * Returns: self.
 */
static VALUE
rb_gst_element_set_locked_state (VALUE self, VALUE state)
{
    gst_element_set_locked_state (RGST_ELEMENT (self), RVAL2CBOOL (state));
    return self;
}

/*
 * Method: sync_state_with_parent
 *
 * Tries to change the state of the element to the same as its parent. If this 
 * method returns false, the state of element is undefined.
 *
 * Returns: true if the element's state could be synced to the parent's state.
 */
static VALUE
rb_gst_element_sync_state_with_parent (VALUE self)
{
    return
        CBOOL2RVAL (gst_element_sync_state_with_parent (RGST_ELEMENT (self)));
}

/*
 * Method: no_more_pads
 *
 * Uses this method to signal that the element does not expect any more pads 
 * to show up in the current pipeline. This method should be called whenever 
 * pads have been added by the element itself. 
 * Elements with Gst::Pad::SOMETIMES pad templates use this in combination 
 * with autopluggers to figure out that the element is done initializing its 
 * pads.
 *
 * Returns: self.
 */
static VALUE
rb_gst_element_no_more_pads (VALUE self)
{
    gst_element_no_more_pads (RGST_ELEMENT (self));
    return self;
}

static VALUE 
rb_gst_element_query_pos_or_dur(int argc, VALUE *argv, VALUE self, 
    gboolean duration) {
  GstFormat format = GST_FORMAT_TIME;
  gint64 value = -1;

  if (duration) {
    gst_element_query_duration(RGST_ELEMENT(self), &format, &value);
  } else {
    gst_element_query_position(RGST_ELEMENT(self), &format, &value);
  }

  return rb_ary_new3(2, GENUM2RVAL(format, GST_TYPE_FORMAT), 
      LL2NUM(value));
  
}

static VALUE
rb_gst_element_query_position(int argc, VALUE *argv, VALUE self) {
  return rb_gst_element_query_pos_or_dur(argc, argv, self, FALSE);
}

static VALUE
rb_gst_element_query_duration(int argc, VALUE *argv, VALUE self) {
  return rb_gst_element_query_pos_or_dur(argc, argv, self, TRUE);
}

static VALUE 
rb_gst_element_found_tag_sig (guint num, const GValue *values)
{
    GstElement *element, *source;
    GstTagList *tag_list;

    element = g_value_get_object (&values[0]);
    source = g_value_get_object (&values[1]);
    tag_list = g_value_get_boxed (&values[2]);

    return rb_ary_new3 (3, RGST_ELEMENT_NEW (element), 
                           RGST_ELEMENT_NEW (source),
                           RGST_STRUCTURE_NEW (tag_list));
}

static VALUE rb_gst_element_get_basetime(VALUE self) {
  return ULL2NUM(gst_element_get_base_time(RGST_ELEMENT(self)));
}

static VALUE rb_gst_element_set_basetime(VALUE self, VALUE time) {
  gst_element_set_base_time(RGST_ELEMENT(self), NUM2ULL(time));
  return time;
}


void
Init_gst_element (void)
{
    VALUE c = G_DEF_CLASS (GST_TYPE_ELEMENT, "Element", mGst);

    rb_define_method (c, "set_state", rb_gst_element_set_state, 1);
    //FIMXE rb_define_method (c, "state", rb_gst_element_get_state, 0);
    rb_define_method (c, "stop", rb_gst_element_stop, 0);
    rb_define_method (c, "ready", rb_gst_element_ready, 0);
    rb_define_method (c, "pause", rb_gst_element_pause, 0);
    rb_define_method (c, "play", rb_gst_element_play, 0);

    rb_define_method(c, "get_state", rb_gst_element_get_state, -1);

    //rb_define_method (c, "stopped?", rb_gst_element_is_stopped, 0);
    //rb_define_method (c, "ready?", rb_gst_element_is_ready, 0);
    //rb_define_method (c, "paused?", rb_gst_element_is_paused, 0);
    //rb_define_method (c, "playing?", rb_gst_element_is_playing, 0);
    //rb_define_method (c, "wait_state_change",
    //                  rb_gst_element_wait_state_change, 0);
    //rb_define_method (c, "eos", rb_gst_element_set_eos, 0);
    rb_define_method (c, "link", rb_gst_element_link, 1);
    rb_define_alias (c, ">>", "link");
    rb_define_method (c, "link_filtered", rb_gst_element_link_filtered, 2);
    rb_define_method (c, "unlink", rb_gst_element_unlink, 1);
    rb_define_method (c, "provides_clock?", rb_gst_element_provides_clock, 0);
    rb_define_method (c, "requires_clock?", rb_gst_element_requires_clock, 0);
    rb_define_method (c, "clock", rb_gst_element_get_clock, 0);
    rb_define_method (c, "set_clock", rb_gst_element_set_clock, 1);
    rb_define_method (c, "base_time", rb_gst_element_get_base_time, 0);
    //rb_define_method (c, "set_time", rb_gst_element_set_time, 1);
    //rb_define_method (c, "set_time_delay", rb_gst_element_set_time_delay, 2);
    //rb_define_method (c, "adjust_time", rb_gst_element_adjust_time, 1);
    rb_define_method (c, "num_pads", rb_gst_element_num_pads, 0);
    rb_define_method (c, "each_pad", rb_gst_element_each_pad, 0);
    rb_define_method (c, "get_pad", rb_gst_element_get_pad, 1);
    rb_define_alias (c, "[]", "get_pad");
    rb_define_method (c, "get_static_pad", rb_gst_element_get_static_pad, 1);
    rb_define_method (c, "get_request_pad", rb_gst_element_get_request_pad, 1);
    rb_define_method (c, "release_request_pad",
                      rb_gst_element_release_request_pad, 1);
    rb_define_method (c, "get_compatible_pad",
                      rb_gst_element_get_compatible_pad, -1);
    //rb_define_method (c, "get_pad_template", rb_gst_element_get_pad_template,
    //                  1);
    //rb_define_method (c, "pad_templates", rb_gst_element_get_pad_templates, 0);
    //rb_define_method (c, "each_pad_template", rb_gst_element_each_pad_template,
    //                  0);
    rb_define_method (c, "get_compatible_pad_template",
                      rb_gst_element_get_compatible_pad_template, 1);
    rb_define_method (c, "link_pads", rb_gst_element_link_pads, 1);
    rb_define_method (c, "unlink_pads", rb_gst_element_unlink_pads, 1);
    rb_define_method (c, "add_pad", rb_gst_element_add_pad, 1);
    rb_define_method (c, "remove_pad", rb_gst_element_remove_pad, 1);
    rb_define_alias (c, "remove_ghost_pad", "remove_pad");
    rb_define_method (c, "indexable?", rb_gst_element_is_indexable, 0);
    //rb_define_method (c, "query", rb_gst_element_query, -1);
    rb_define_method (c, "send_event", rb_gst_element_send_event, 1);
    rb_define_method (c, "seek", rb_gst_element_seek, 7);
    rb_define_method (c, "index", rb_gst_element_get_index, 0);
    rb_define_method (c, "set_index", rb_gst_element_set_index, 1);
    rb_define_method (c, "locked_state?", rb_gst_element_is_locked_state, 0);
    rb_define_method (c, "set_locked_state", rb_gst_element_set_locked_state,
                      1);
    rb_define_method (c, "sync_state_with_parent",
                      rb_gst_element_sync_state_with_parent, 0);
    rb_define_method (c, "no_more_pads", rb_gst_element_no_more_pads, 0);
    rb_define_method (c, "query_position", rb_gst_element_query_position, -1);
    rb_define_method (c, "query_duration", rb_gst_element_query_duration, -1);

    rb_define_method(c, "base_time", rb_gst_element_get_basetime, 0);
    rb_define_method(c, "base_time=", rb_gst_element_set_basetime, 1);

    G_DEF_SETTERS (c);

    G_DEF_CLASS (GST_TYPE_STATE_CHANGE_RETURN, "StateReturn", c);
    G_DEF_CONSTANTS (c, GST_TYPE_STATE_CHANGE_RETURN, "GST_");
    G_DEF_CLASS (GST_TYPE_STATE, "State", c);
    G_DEF_CONSTANTS (c, GST_TYPE_STATE, "GST_");
    G_DEF_CLASS (GST_TYPE_ELEMENT_FLAGS, "Types", c);
    G_DEF_CONSTANTS (c, GST_TYPE_ELEMENT_FLAGS, "GST_ELEMENT_");

    /*
     * TODO:
     * gst_element_clock_wait () 
     */

    G_DEF_SIGNAL_FUNC (c, "found-tag", 
                       (GValToRValSignalFunc)rb_gst_element_found_tag_sig);
}
