# Upgrading from rspec-rails-2.x to rspec-rails-3

For detailed information on the general RSpec 3.x upgrade process see the [RSpec
Upgrade docs](https://relishapp.com/rspec/docs/upgrade).

There are several changes specific to `rspec-rails` to be aware of:

- [Default helper files created in RSpec 3.x have changed](#default-helper-files)

- [File-type inference disabled by default](#file-type-inference-disabled)

- [Rails 4.x `ActiveRecord::Migration` pending migration checks](#pending-migration-checks)

- [Extraction of `stub_model` and `mock_model` to `rspec-activemodel-mocks`](#extract-stub-model)

<a name="default-helper-files"></a>
## Default helper files created in RSpec 3.x have changed

In prior versions, only a single `spec_helper.rb` file was generated. This file
has been moved to `rails_helper.rb`. The new `spec_helper.rb` is the same
standard helper generated by running `rspec --init`.

This change was made to accomplish two general goals:

- Keep the installation process in sync with regular RSpec changes

- Provide an out-of-the-box way to avoid loading Rails for those specs that do
  not require it

<a name="generators"></a>
### Generators

Generators run in RSpec 3.x will require `rails_helper` and not `spec_helper`.

<a name="upgrading-an-existing-app"></a>
### Upgrading an Existing App

For most existing apps, one of the following upgrade paths is sufficient to
switch to the new helpers:

#### I need to move things over in stages

1. Create a new `rails_helper.rb` with the following content:

    ```ruby
    require 'spec_helper'
    ```

2. As necessary, replace `require 'spec_helper'` with `require 'rails_helper'`
   in the specs.

3. When ready, move any Rails specific code and setup from `spec_helper.rb` to
   `rails_helper.rb`.

#### I'm ready to just switch completely

1. Move the existing `spec_helper.rb` to `rails_helper.rb`:

    ```ruby
    git mv spec/spec_helper.rb spec/rails_helper.rb
    ```

2. Run the installation rake task opting to not replace `rails_helper.rb`:

    ```console
    $ bin/rails generate rspec:install
          create  .rspec
           exist  spec
          create  spec/spec_helper.rb
        conflict  spec/rails_helper.rb
    Overwrite my_app/spec/rails_helper.rb? (enter "h"for help) [Ynaqdh] n
            skip  spec/rails_helper.rb
    ```

3. Move any non-Rails RSpec configurations and customizations from your
   `rails_helper.rb` to `spec_helper.rb`.

4. Find/replace instances of `require 'spec_helper'` with
   `require 'rails_helper'` in any specs which rely on Rails.

<a name="file-type-inference-disabled"></a>
## File-type inference disabled by default

Previously we automatically inferred spec type from a file location, this
was a surprising behaviour for new users and undesirable for some veteran users
so from RSpec 3 onwards this behaviour must be explicitly opted into with:

```Ruby
RSpec.configure do |config|
  config.infer_spec_type_from_file_location!
end
```

This change was made to accomplish our general goals of acting with the principle
of least surprise and making RSpec configuration more explicit. See [the
directory structure documentation](https://www.relishapp.com/rspec/rspec-rails/v/3-0/docs/directory-structure) for more details.

<a name="pending-migration-checks"></a>
## Rails 4.x `ActiveRecord::Migration` pending migration checks

If you are not using `ActiveRecord` you do not need to worry about these
settings.

Users of Rails 4.x can now take advantage of improved schema migration and sync
abilities. Prior to RSpec 3, users were required to manually run migrations in
both the development and test environments. Additionally, the behavior differed
depending on if the specs were run via `rake` or via the standalone `rspec`
command.

With the release of Rails 4, new APIs have been exposed on
`ActiveRecord::Migration`. This allows RSpec to take advantage of these new
standard migration checks, mirroring behavior across the board.

  - Rails 4.0.x

    Add the following to the top of the `rails_helper` file after Rails has
    been required:

    ```ruby
    ActiveRecord::Migration.check_pending!
    ```

    This will raise an exception if there are any pending schema changes. Users
    will still be required to manually keep the development and test
    environments in sync.

  - Rails 4.1+

    With this release there was an exciting new feature. Users no longer need
    to keep the development and test environments in sync. To take advantage of
    this add the following to the top of the `rails_helper` file after Rails
    has been required:

    ```ruby
    ActiveRecord::Migration.maintain_test_schema!
    ```

    What this does is that rather than just raising when the test schema has
    pending migrations, Rails will try to load the schema. An exception will
    now only be raised if there are pending migrations afterwards the schema
    has been loaded.

    There are a few caveates to be aware of when using this:

      - Migrations still need to be run manually; although now this only has to
        be done in the 'development' environment
      - An exception will be raised If the schema has not been initialized. The
        exception will provide instructions stating `rake db:migrate` needs to
        be run.

It is possible to opt-out of checking for pending migrations. Since this is
actually a feature of Rails, the change needs to be done as part of the Rails
configuration. To do this, add the following to your
`config/environments/test.rb` file:

```ruby
config.active_record.maintain_test_schema = false
```

New RSpec projects don't need to worry about these commands as the `rails
generate rspec:install` will add them automatically.

<a name="extract-stub-model"></a>
## Extraction of `stub_model` and `mock_model` to [`rspec-activemodel-mocks`](https://github.com/rspec/rspec-activemodel-mocks)

Historically, `stub_model` and `mock_model` have been difficult to maintain.
They are tightly coupled to `ActiveRecord` which isn't always the ORM of choice.
This maintainence coupling has lead to delays with previous releases.
Additionally, the objects generated by these methods hide important
`ActiveRecord` behavior complexity which would otherwise be good to expose.

Some alternatives are:

- Wrap calls to `ActiveRecord` objects in more specific domain models and
  services
- Use new unsaved `ActiveRecord` instances (e.g. `Model.new`)
- Consider [partial mocks](https://www.relishapp.com/rspec/rspec-mocks/docs/basics/partial-test-doubles) on an `ActiveRecord` instance
- Let the specs hit database directly where appropriate
