/*
 * hyptop - Show hypervisor performance data on System z
 *
 * System data module: Provide backend independent database for system data
 *                     (e.g. for CPU and memory data)
 *
 * Copyright IBM Corp. 2010
 * Author(s): Michael Holzheu <holzheu@linux.vnet.ibm.com>
 */

#include <string.h>
#include <time.h>
#include "sd.h"
#include "hyptop.h"
#include "helper.h"
#include "opts.h"

/*
 * Internal globals for system data
 */
static u32		l_cpu_type_selected_mask;
static int		l_cpu_type_cnt;
static int		l_sys_item_cnt;
static int		l_cpu_item_cnt;
static struct sd_sys	*l_root_sys;

/*
 * External globals for system data
 */
struct sd_globals sd;

/*
 * Get root system
 */
struct sd_sys *sd_sys_root_get(void)
{
	return l_root_sys;
}

/*
 * Get CPU type by it's ID
 */
struct sd_cpu_type *sd_cpu_type_by_id(const char *id)
{
	struct sd_cpu_type *type;
	unsigned int i;

	sd_cpu_type_iterate(type, i) {
		if (strcasecmp(id, type->id) == 0)
			return type;
	}
	return NULL;
}

/*
 * Is CPU type selected?
 */
int sd_cpu_type_selected(struct sd_cpu_type *cpu_type)
{
	return l_cpu_type_selected_mask & cpu_type->idx;
}

/*
 * Toggle selection of CPU type
 */
void sd_cpu_type_select_toggle(struct sd_cpu_type *cpu_type)
{
	if (l_cpu_type_selected_mask & cpu_type->idx)
		l_cpu_type_selected_mask &= ~cpu_type->idx;
	else
		l_cpu_type_selected_mask |= cpu_type->idx;
}

/*
 * Select exactly specified CPU type
 */
void sd_cpu_type_select(struct sd_cpu_type *cpu_type)
{
	l_cpu_type_selected_mask = cpu_type->idx;
}

/*
 * Select all available CPU types
 */
void sd_cpu_type_select_all(void)
{
	l_cpu_type_selected_mask = (u32)-1;
}

/*
 * Deselect all CPU types
 */
void sd_cpu_type_select_none(void)
{
	l_cpu_type_selected_mask = 0;
}

/*
 * Setup CPU types specified on command line
 */
static void l_opts_cpu_types_init(void)
{
	struct sd_cpu_type *type;
	unsigned int i;

	if (!g.o.cpu_types.specified)
		return;

	sd_cpu_type_select_none();
	for (i = 0; i < g.o.cpu_types.cnt; i++) {
		type = sd_cpu_type_by_id(g.o.cpu_types.vec[i]);
		if (!type)
			ERR_EXIT("Invalid CPU type \"%s\"\n",
				 g.o.cpu_types.vec[i]);
		sd_cpu_type_select_toggle(type);
	}
}

/*
 * Init CPU count for all CPU types
 */
static void l_cpu_types_init(void)
{
	struct sd_sys *sys = sd_sys_root_get();
	struct sd_cpu_type *cpu_type;
	unsigned int i;

	sd_cpu_type_iterate(cpu_type, i) {
		sd_cpu_type_select(cpu_type);
		cpu_type->cpu_cnt = sd_sys_item_u64(sys, &sd_sys_item_cpu_cnt);
	}
	sd_cpu_type_select_all();
	l_opts_cpu_types_init();
}

/*
 * Update system data using the data gatherer
 */
void sd_update(void)
{
	sd.dg->update_sys();
}

/*
 * Register a data gatherer
 */
void sd_dg_register(struct sd_dg *dg)
{
	struct timespec ts = {0, SD_DG_INIT_INTERVAL_MS * 1000000};
	struct sd_sys_item *sys_item;
	struct sd_cpu_item *cpu_item;
	unsigned int i;

	sd.dg = dg;

	for (i = 0; dg->cpu_type_vec[i]; i++)
		dg->cpu_type_vec[i]->idx = (1UL << i);
	l_cpu_type_cnt = i;
	sd_sys_item_iterate(sys_item, i)
		l_sys_item_cnt++;
	sd_cpu_item_iterate(cpu_item, i)
		l_cpu_item_cnt++;

	sd_update();
	nanosleep(&ts, NULL);
	sd_update();

	l_cpu_types_init();
}

/*
 * Get CPU from sys by ID
 */
struct sd_cpu *sd_cpu_get(struct sd_sys *sys, const char* id)
{
	struct sd_cpu *cpu;

	list_iterate(cpu, &sys->cpu_list, list) {
		if (strcmp(cpu->id, id) == 0)
			return cpu;
	}
	return NULL;
}

/*
 * Get CPU type by ID
 */
static struct sd_cpu_type *l_cpu_type_by_id(const char *id)
{
	struct sd_cpu_type **cpu_type_vec = sd.dg->cpu_type_vec;
	int i;

	for (i = 0; i < l_cpu_type_cnt; i++) {
		if (strcmp(cpu_type_vec[i]->id, id) == 0)
			return cpu_type_vec[i];
	}
	return NULL;
}

/*
 * Allocate and initialize new CPU
 */
struct sd_cpu *sd_cpu_new(struct sd_sys *parent, const char *id,
			  const char *type, int cnt)
{
	struct sd_cpu *cpu;

	cpu = ht_zalloc(sizeof(*cpu));
	cpu->i.parent = parent;
	strncpy(cpu->id, id, sizeof(cpu->id));
	cpu->type = l_cpu_type_by_id(type);
	cpu->d_cur = &cpu->d1;
	cpu->cnt = cnt;

	list_add_end(&cpu->list, &parent->cpu_list);

	return cpu;
}

/*
 * Get system by ID
 */
struct sd_sys *sd_sys_get(struct sd_sys *parent, const char* id)
{
	struct sd_sys *sys;

	list_iterate(sys, &parent->child_list, list) {
		if (strcmp(sys->id, id) == 0)
			return sys;
	}
	return NULL;
}

/*
 * Allocate and initialize new system
 */
struct sd_sys *sd_sys_new(struct sd_sys *parent, const char *id)
{
	struct sd_sys *sys_new;

	sys_new = ht_zalloc(sizeof(*sys_new));
	strncpy(sys_new->id, id, sizeof(sys_new->id));
	list_init(&sys_new->child_list);
	list_init(&sys_new->cpu_list);
	list_init(&sys_new->list);

	if (parent) {
		sys_new->i.parent = parent;
		parent->child_cnt++;
		list_add_end(&sys_new->list, &parent->child_list);
	}
	return sys_new;
}

/*
 * Free system
 */
static void sd_sys_free(struct sd_sys *sys)
{
	ht_free(sys);
}

/*
 * Free CPU
 */
static void sd_cpu_free(struct sd_cpu *cpu)
{
	ht_free(cpu);
}

/*
 * Start update cycle for CPU
 */
static void l_cpu_update_start(struct sd_cpu *cpu)
{
	struct sd_cpu_info *tmp;

	cpu->i.active = 0;
	if (!cpu->d_prev) {
		cpu->d_prev = &cpu->d1;
		cpu->d_cur = &cpu->d2;
	} else {
		tmp = cpu->d_prev;
		cpu->d_prev = cpu->d_cur;
		cpu->d_cur = tmp;
	}
}

/*
 * Start update cycle for system
 */
void sd_sys_update_start(struct sd_sys *sys)
{
	struct sd_sys *child;
	struct sd_cpu *cpu;

	sys->i.active = 0;
	sys->child_cnt_active = 0;
	sys->cpu_cnt_active = 0;

	list_iterate(cpu, &sys->cpu_list, list)
		l_cpu_update_start(cpu);
	list_iterate(child, &sys->child_list, list)
		sd_sys_update_start(child);
}

/*
 * End update cycle for CPUs of a system
 */
static void l_cpu_update_end(struct sd_sys *sys)
{
	struct sd_cpu *cpu, *tmp;

	/* Has system not lost any CPU? */
	if (sys->cpu_cnt_active == sys->cpu_cnt)
		return;

	list_iterate_safe(cpu, &sys->cpu_list, list, tmp) {
		if (!cpu->i.active) {
			/* CPU has not been updated, remove it */
			list_del(&cpu->list);
			sd_cpu_free(cpu);
			continue;
		}
	}
}

/*
 * End update cycle for system
 */
static void l_sys_update_end(struct sd_sys *sys)
{
	struct sd_sys *child, *tmp;

	if (sys->child_cnt_active == sys->child_cnt)
		return;

	l_cpu_update_end(sys);

	list_iterate_safe(child, &sys->child_list, list, tmp) {
		if (!child->i.active) {
			/* child has not been updated, remove it */
			list_del(&child->list);
			sd_sys_free(child);
			continue;
		}
		/* Recursively update child */
		l_sys_update_end(child);
	}
	sys->child_cnt = sys->child_cnt_active;
}

/*
 * End update cycle for system
 */
void sd_sys_update_end(struct sd_sys *sys, u64 update_time_us)
{
	sys->update_time_us = update_time_us;
	l_sys_update_end(sys);
}

/*
 * Is system item available?
 */
int sd_sys_item_available(struct sd_sys_item *item)
{
	struct sd_sys_item *ptr;
	unsigned int i;

	sd_sys_item_iterate(ptr, i) {
		if (item == ptr)
			return 1;
	}
	return 0;
}

/*
 * Number of system items
 */
int sd_sys_item_cnt(void)
{
	return l_sys_item_cnt;
}

/*
 * Is CPU item avaiable?
 */
int sd_cpu_item_available(struct sd_cpu_item *item)
{
	struct sd_cpu_item *ptr;
	unsigned int i;

	sd_cpu_item_iterate(ptr, i) {
		if (item == ptr)
			return 1;
	}
	return 0;
}

/*
 * Number of CPU items
 */
int sd_cpu_item_cnt(void)
{
	return l_cpu_item_cnt;
}

/*
 * Init system data module
 */
void sd_init(void)
{
	l_root_sys = sd_sys_new(NULL, "root");
}

/*
 * CPU Types
 */
struct sd_cpu_type sd_cpu_type_ifl = {
	.id	= SD_CPU_TYPE_STR_IFL,
	.desc	= "Integrated Facility for Linux",
	.hotkey	= 'i',
};

struct sd_cpu_type sd_cpu_type_cp = {
	.id	= SD_CPU_TYPE_STR_CP,
	.desc	= "Central processor",
	.hotkey	= 'p',
};

struct sd_cpu_type sd_cpu_type_un = {
	.id	= SD_CPU_TYPE_STR_UN,
	.desc	= "Unspecified processor type",
	.hotkey	= 'u',
};
