/*
 * Copyright (C) 1993-2004 Robert & Jeremy Lain
 * See AUTHORS file for a full list of contributors.
 * 
 * $Id: saildispgl.cpp,v 1.3 2004/10/26 19:30:50 jeremy_laine Exp $
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
  #include "config.h"
#endif

#ifdef HAVE_GL

#include "saildispgl.h"

/** Constructs an OpenGL view area
 */
CSailDispGL::CSailDispGL( QWidget * parent )
        :  QGLWidget( parent )
{
    if ( !QGLFormat::hasOpenGL() )
        throw CException(tr("This system has no OpenGL support."));
}


/* Clears the display.
 */
void CSailDispGL::clear()
{
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
}


/** Draws a panel of a sail.
 */
void CSailDispGL::draw( const CPanel &panel )
{
    CPoint3d pt;
    unsigned int i;

    glBegin(GL_TRIANGLE_STRIP);
    for (i =0; i < panel.top.nbpoints(); i++)
    {
        pt = panel.top.point[i];
        glVertex3d(pt.x(),pt.y(),pt.z());
        pt = panel.bottom.point[i];
        glVertex3d(pt.x(),pt.y(),pt.z());
    }
    glEnd();

    glBegin(GL_TRIANGLE_FAN);
    pt = (panel.left.point[0]+panel.left.point[panel.left.nbpoints()-1])*0.5;
    glVertex3d(pt.x(),pt.y(),pt.z());
    for (i =0; i < panel.left.nbpoints(); i++)
    {
        pt = panel.left.point[i];
        glVertex3d(pt.x(),pt.y(),pt.z());
    }
    glEnd();

    glBegin(GL_TRIANGLE_FAN);
    pt = (panel.right.point[0]+panel.right.point[panel.right.nbpoints()-1])*0.5;
    glVertex3d(pt.x(),pt.y(),pt.z());
    for (i =0; i < panel.right.nbpoints(); i++)
    {
        pt = panel.right.point[i];
        glVertex3d(pt.x(),pt.y(),pt.z());
    }
    glEnd();
}


/** Draws a complete sail.
 */
void CSailDispGL::draw( const CSail &sail )
{
    for( unsigned int i = 0; i < sail.nbpanels(); i++ )
    {
        // alternate light yellow / yellow panels
        glColor3f(1, 1, 0.4 + 0.4*(i%2));
        draw( sail.panel[i] );
    }
}


/** Forces a redraw of the view area.
 */
void CSailDispGL::redraw()
{
    updateGL();
}


/** InitialiZes the OpenGL subsystem.
 */
void CSailDispGL::initializeGL()
{
    // Set up the rendering context, define display lists etc.:
    glClearColor( 0.0f, 0.0f, 0.0f, 0.5f );
    glEnable(GL_DEPTH_TEST);
    glDepthFunc(GL_LEQUAL);
    glEnable(GL_COLOR_MATERIAL);
    glEnable(GL_LIGHTING);
    glClearDepth(1.0f);
    // lighting
    GLfloat LightAmbient[]= { 0.1f, 0.1f, 0.1f, 1.0f };
    GLfloat LightDiffuse[]= { 0.00015f, 0.00015f, 0.00015f, 1.0f };
    GLfloat LightPosition[]= { 1.0f, 1.0f, 1.0f, 1.0f };

    // Setup the ambient & diffuse lamps
    glLightfv(GL_LIGHT1, GL_AMBIENT, LightAmbient);
    glLightfv(GL_LIGHT1, GL_DIFFUSE, LightDiffuse);
    glLightfv(GL_LIGHT1, GL_POSITION,LightPosition);

    // Enable lighting
    glEnable(GL_LIGHT1);
}


/** We received a mouse click.
 */
void CSailDispGL::mousePressEvent ( QMouseEvent *event )
{
    if (event->button() == LeftButton)
    {
        setCenter(screenToLogical(event->pos().x(),event->pos().y()));
        redraw();
    }
}


void CSailDispGL::paintGL()
{
    clear();

    // set coordinate system to match the logical viewport
    glLoadIdentity();
    real zs = real(2) / sqrt(lRect.width()*lRect.width() + lRect.height()*lRect.height());
    glScaled(real(2) / lRect.width(),real(2) / lRect.height(),zs);
    glTranslated(-center.x(),-center.y(),-center.z());

    draw( sailDisp );
}


/** The draw area has been resized
 */
void CSailDispGL::resizeGL( int w, int h )
{
    glViewport( 0, 0, (GLint)w, (GLint)h );
    setVRect(w,h);
}


/** We received a mouse wheel movement
 */
void CSailDispGL::wheelEvent( QWheelEvent *event)
{
    if (event->delta()>0)
        zoomIn();
    else
        zoomOut();
    redraw();
}

#endif
