/*
 * Copyright (C) 1993-2004 Robert & Jeremy Lain
 * See AUTHORS file for a full list of contributors.
 * 
 * $Id: sailviewer-panel.cpp,v 1.4 2004/10/26 19:30:51 jeremy_laine Exp $
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "sailviewer-panel.h"
#include "sail.h"

#include <qgroupbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qpushbutton.h>


/**
 * Constructs a CSailViewerPanel object.
 *
 * @param parent The parent widget
 * @param viewMode The viewing mode (shaded or wireframe)
 * @param show_sliders Should the elevation and azimuth sliders be displayed?
 * @param show_labeling Should the "Labeling" button be displayed?
 */
CSailViewerPanel::CSailViewerPanel(QWidget *parent, enumViewMode viewMode, bool show_sliders, bool show_labeling)
        : QWidget(parent, "CSailViewerPanel" ), showSliders(show_sliders)
{

    /* the drawing area */
    if (showSliders)
    {
        /* display parameters groupbox */
        grpParams = new QGroupBox( 0, Qt::Vertical, this, "grpParams" );
        QGridLayout* grpParamsLayout = new QGridLayout( grpParams->layout() );
        lblAzimuthStatic = new QLabel( grpParams, "lblAzimuthStatic" );
        grpParamsLayout->addWidget( lblAzimuthStatic, 0, 0 );
        lblAzimuth = new QLabel( grpParams, "lblAzimuth" );
        grpParamsLayout->addWidget( lblAzimuth, 0, 1 );
        lblElevationStatic = new QLabel( grpParams, "lblElevationStatic" );
        grpParamsLayout->addWidget( lblElevationStatic, 1, 0 );
        lblElevation = new QLabel( grpParams, "lblElevation" );
        grpParamsLayout->addWidget( lblElevation, 1, 1 );
    }
    else
    {
        grpParams = NULL;
        lblAzimuthStatic = NULL;
        lblAzimuth = NULL;
        lblElevationStatic = NULL;
        lblElevation = NULL;
    }

    sailDisp = new CSailViewer(this, viewMode, showSliders);

    /* controls groupbox */
    grpControls = new QGroupBox( 0, Qt::Vertical, this, "grpControls" );
    grpControls->layout()->setAutoAdd(true);
    btnResetView = new QPushButton( grpControls, "btnResetView" );
    btnZoomIn = new QPushButton( grpControls, "btnZoomIn" );
    btnZoomOut = new QPushButton( grpControls, "btnZoomOut" );
    if (viewMode == WIREFRAME && show_labeling)
    {
        btnLabeling = new QPushButton( grpControls, "btnLabeling" );
    }
    else
    {
        btnLabeling = NULL;
    }

    /* put it all together */
    layout = new QGridLayout( this, 1, 1, 11, 6 );
    layout->addWidget( grpControls, 0, 2 );
    layout->addItem( new QSpacerItem( 20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding ),
                     1, 2 );
    if (showSliders)
    {
        layout->addWidget( grpParams, 2, 2 );
        layout->addItem( new QSpacerItem( 20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding ),
                         3, 2 );
        layout->addMultiCellWidget( sailDisp, 0, 3, 0, 1 );
    }
    else
    {
        layout->addMultiCellWidget( sailDisp, 0, 1, 0, 1 );
    }

    /* set language and size */
    languageChange();

    /* connect signals */
    connect( sailDisp, SIGNAL( azimuthChanged(real) ), this, SLOT(slotAzimuth(real)) );
    connect( sailDisp, SIGNAL( elevationChanged(real) ), this, SLOT(slotElevation(real)) );

    if (btnLabeling)
    {
        connect( (QWidget*) btnLabeling, SIGNAL( clicked() ), sailDisp, SLOT( slotLabeling() ));
    }
    connect( (QWidget*) btnResetView, SIGNAL( clicked() ), sailDisp, SLOT( slotResetView() ));
    connect( (QWidget*) btnZoomIn, SIGNAL( clicked() ), sailDisp, SLOT( slotZoomIn() ));
    connect( (QWidget*) btnZoomOut, SIGNAL( clicked() ), sailDisp, SLOT( slotZoomOut() ));
}


CSailViewerPanel::~CSailViewerPanel()
{}


/**
 * We received a keypress, we pass it down to the CSailViewer.
 */
void CSailViewerPanel::keyPressEvent ( QKeyEvent * e )
{
    sailDisp->keyPressEvent(e);
}


/**
 * Sets the strings of the subwidgets using the current language.
 */
void CSailViewerPanel::languageChange()
{
    if (showSliders)
    {
        grpParams->setTitle( tr( "Display parameters" ) );
        lblElevationStatic->setText( tr( "elevation" ) );
        lblAzimuthStatic->setText( tr( "azimuth" ) );
    }
    grpControls->setTitle( tr( "Controls" ) );
    btnResetView->setText( tr( "Reset view" ) );
    if (btnLabeling)
    {
        btnLabeling->setText( tr( "Labeling" ) );
    }
    btnZoomIn->setText( tr( "Zoom in" ) );
    btnZoomOut->setText( tr( "Zoom out" ) );
}


/**
 * Change the sail to be displayed.
 *
 * @param sail The new sail
 */
void CSailViewerPanel::setSail(const CSail& sail)
{
    sailDisp->setSail(sail);
}


/**
 * The azimuth was changed, update the corresponding label.
 *
 * @param azimuth
 */
void CSailViewerPanel::slotAzimuth(real azimuth)
{
    if (lblAzimuth)
    {
        lblAzimuth->setText(QString::number(azimuth) + " " +tr("deg"));
    }
}


/**
 * The elevation changed, update the corresponding label.
 *
 * @param elevation
 */
void CSailViewerPanel::slotElevation(real elevation)
{
    if (lblElevation)
    {
        lblElevation->setText(QString::number(elevation) + " " +tr("deg"));
    }
}
