/*
 * Copyright (C) 1993-2004 Robert & Jeremy Lain
 * See AUTHORS file for a full list of contributors.
 *
 * $Id: widgetprofile.cpp,v 1.11 2005/05/08 22:09:43 jeremy_laine Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "widgetprofile.h"
#include "widgetprofilevert.h"
#include "sailmould.h"
#include <qpushbutton.h>
#include <qgroupbox.h>
#include <qspinbox.h>
#include <qlayout.h>
#include <qpainter.h>


/**************************************************************************

                         CLabelProfile class

**************************************************************************/

/** The constructor.
 *
 * @param parent the parent dialog
 * @param ptr pointer to the CProfile
 */
CLabelProfile::CLabelProfile( QWidget *parent, CProfile *ptr)
        : QLabel(parent), CDispArea(0.95), profile(ptr)
{
    QPalette pal;
    QColorGroup cg;
    cg.setColor( QColorGroup::Background, white );
    pal.setActive( cg );
    pal.setInactive( cg );
    setPalette( pal );
}


/** Clear the drawing area.
 */
void CLabelProfile::clear()
{
    QPainter painter( this );
    QRect rect = painter.viewport();
    painter.eraseRect(rect);
}


/** Display the CProfile in the drawing area.
 */
void CLabelProfile::paintEvent( QPaintEvent *)
{
    QPainter painter( this );
    QPoint prev, next;
    unsigned int nbpoints = 32;
    real x, z;

    QRect view = painter.viewport();
    real scale =  view.height() / lRect.height();
    
    // do a translation to have from z=0 to z=scale centered
    painter.translate( (lRect.width() / 2 - center.x()) * scale,  view.height() + (center.y() - lRect.height()/2) * scale );

    // flip coordinate system to have the z axis pointing up
    painter.scale(1,-1);

    prev = QPoint( 0, int(scale * profile->z(0)) );
    for( unsigned i = 1; i < nbpoints; i++)
    {
        x = real(i) / (nbpoints-1);
        z = profile->z(x);

        //qDebug( "( " + QString::number(x) + ", " + QString::number(z) + " )" );
        next = QPoint( int(x*scale) , int(z*scale) );
        painter.drawLine(prev, next);
        prev = next;
    }
}


/** The draw area has been resized.
 */
void CLabelProfile::resizeEvent( QResizeEvent * )
{
    QPainter painter(this);
    QRect vRect = painter.viewport();
    CRect3d viewRect;
    viewRect.max = CPoint3d(vRect.width(), vRect.height());
    
    CRect3d objRect; 
    objRect.max = CPoint3d(1, 0.2);
    center = objRect.center();
    
    lRect = calcLRect(viewRect, objRect);
}




/**************************************************************************

                         CWidgetProfile class

**************************************************************************/

/** The constructor.
 *
 * @param parent the parent dialog
 * @param ptr a pointer to the profile we are displaying / editing
 * @param caption caption for the profile view (e.g. top,middle..)
 * @param vert the vertical repartition widget
 */
CWidgetProfile::CWidgetProfile( QWidget *parent, CProfile *ptr,
                                QString caption , CWidgetProfileVert *vert)
        : QWidget(parent), widgetVert(vert)
{
    profile = ptr;

    QGridLayout *layout = new QGridLayout( this, 1, 1, 0, 0 );

    QGroupBox *grpProfile = new QGroupBox( this );
    grpProfile->setAlignment( int( QGroupBox::AlignCenter | QGroupBox::AlignRight | QGroupBox::AlignLeft ) );
    grpProfile->setColumnLayout(0, Qt::Vertical );
    grpProfile->layout()->setSpacing( 6 );
    grpProfile->layout()->setMargin( 11 );
    QGridLayout *grpProfileLayout = new QGridLayout( grpProfile->layout() );
    grpProfileLayout->setAlignment( Qt::AlignTop );
    grpProfile->setTitle(caption);

    // add the spin boxes
    QHBoxLayout *spinBoxesLayout = new QHBoxLayout( 0, 0, 6 );

    // luff
    lblLuff = new QLabel( grpProfile, "lblLuff" );
    lblLuff->setAlignment( int( QLabel::AlignVCenter | QLabel::AlignRight ) );
    spinBoxesLayout->addWidget( lblLuff );

    spinLuff = new QSpinBox( grpProfile, "spinLuff" );
    spinLuff->setMaxValue( 12 );
    spinLuff->setMinValue( 1 );
    spinLuff->setValue( 1 );
    spinBoxesLayout->addWidget( spinLuff );
    QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::MinimumExpanding, QSizePolicy::Minimum );
    spinBoxesLayout->addItem( spacer );

    // depth
    lblDepth = new QLabel( grpProfile, "lblDepth" );
    lblDepth->setAlignment( int( QLabel::AlignVCenter | QLabel::AlignRight ) );
    spinBoxesLayout->addWidget( lblDepth );

    spinDepth = new QSpinBox( grpProfile, "spinDepth" );
    spinDepth->setMaxValue( 20 );
    spinDepth->setMinValue( 1 );
    spinDepth->setValue( 10 );
    spinBoxesLayout->addWidget( spinDepth );

    lblPercent = new QLabel( grpProfile, "lblPercent" );
    spinBoxesLayout->addWidget( lblPercent );
    QSpacerItem* spacer_2 = new QSpacerItem( 20, 20, QSizePolicy::MinimumExpanding, QSizePolicy::Minimum );
    spinBoxesLayout->addItem( spacer_2 );

    // leech
    lblLeech = new QLabel( grpProfile, "lblLeech" );
    spinBoxesLayout->addWidget( lblLeech );

    spinLeech = new QSpinBox( grpProfile, "spinLeech" );
    spinLeech->setMinValue( -4 );
    spinLeech->setMaxValue( 8 );
    spinBoxesLayout->addWidget( spinLeech );

    grpProfileLayout->addLayout( spinBoxesLayout, 1, 0 );
    QSpacerItem* spacer_3 = new QSpacerItem( 20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding );
    grpProfileLayout->addItem( spacer_3, 0, 0 );

    layout->addWidget( grpProfile, 0, 0 );
    languageChange();
    resize( QSize(527, 159).expandedTo(minimumSizeHint()) );
    clearWState( WState_Polished );

    // signals and slots connections
    connect( spinLuff, SIGNAL( valueChanged(int) ), this, SLOT( slotChanged() ) );
    connect( spinDepth, SIGNAL( valueChanged(int) ), this, SLOT( slotChanged() ) );
    connect( spinLeech, SIGNAL( valueChanged(int) ), this, SLOT( slotChanged() ) );


    // create the drawing area
    lblDraw = new CLabelProfile( grpProfile, profile );
    grpProfileLayout->addWidget( lblDraw, 0, 0 );

    // we set the "active" flag to false so that we can set the initial
    // values without slotChanged being triggered
    active = false;
    spinLeech->setValue( int(profile->getLeech()*50) );
    spinLuff->setValue( int(profile->getLuff()) );
    spinDepth->setValue( int(profile->getDepth()*100) );
    active = true;
}


/** Sets the strings of the subwidgets using the current
 *  language.
 */
void CWidgetProfile::languageChange()
{
    lblLuff->setText( tr( "Luff shape" ) );
    lblDepth->setText( tr( "Depth" ) );
    lblPercent->setText( tr( "%" ) );
    lblLeech->setText( tr( "Leech shape" ) );
}


/** Called when one of the spinboxes changes value.
 */
void CWidgetProfile::slotChanged()
{
    if (active == false)
        return;

    // store the new values
    *profile = CProfile( real(spinDepth->value())/100, real(spinLeech->value())/50, spinLuff->value() );

    // update profile view
    lblDraw->clear();
    lblDraw->paintEvent( NULL );

    // update the vertical repartition view
    widgetVert->refresh();
}

