//  Copyright (C) 2007-2008  CEA/DEN, EDF R&D, OPEN CASCADE
//
//  Copyright (C) 2003-2007  OPEN CASCADE, EADS/CCR, LIP6, CEA/DEN,
//  CEDRAT, EDF R&D, LEG, PRINCIPIA R&D, BUREAU VERITAS
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
//
//  See http://www.salome-platform.org/ or email : webmaster.salome@opencascade.com
//
//  File   : VisuGUI_TableDlg.cxx
//  Author : Vadim SANDLER
//  Module : SALOME
//
#include "VisuGUI_TableDlg.h"

#include "SUIT_Tools.h"
#include "SUIT_MessageBox.h"
#include "SUIT_ResourceMgr.h"
#include "SUIT_Session.h"

#include "LightApp_Application.h"

#include "CAM_Module.h"

#include "SALOMEDSClient_Study.hxx"
#include "SALOMEDSClient_GenericAttribute.hxx"
#include "SALOMEDSClient_AttributeTableOfInteger.hxx"
#include "SALOMEDSClient_AttributeTableOfReal.hxx"
#include "SALOMEDSClient_StudyBuilder.hxx"

#include <QLayout>
#include <QValidator>
#include <QTableWidget>
#include <QTabWidget>
#include <QList>
#include <QVector>
#include <QInputDialog>
#include <QLabel>
#include <QIntValidator>
#include <QDoubleValidator>
#include <QKeyEvent>
#include <QHeaderView>

#include "utilities.h"

using namespace std;

#define MARGIN_SIZE       11
#define SPACING_SIZE      6
#define SPACER_SIZE       5
#define MIN_TABLE_WIDTH   200
#define MIN_TABLE_HEIGHT  200

NumDelegateItem::NumDelegateItem( QObject* parent, NumValidator mode )
  : QItemDelegate( parent ),
    myMode( mode )
{
}

NumDelegateItem::~NumDelegateItem()
{
}

QWidget * NumDelegateItem::createEditor( QWidget * parent,
                                         const QStyleOptionViewItem & option,
                                         const QModelIndex & index ) const
{
  QLineEdit *editor = new QLineEdit(parent);
  switch ( myMode )
  {
  case NV_Int:
    editor->setValidator( new QIntValidator( editor ) );
    break;
  case NV_Real:
    editor->setValidator( new QDoubleValidator( editor ) );
    break;
  default:
    editor->setText( "No validator!" );
  }
  
  return editor;
}

void NumDelegateItem::setEditorData( QWidget * editor,
                                     const QModelIndex & index ) const
{
  QLineEdit *aLE = qobject_cast<QLineEdit*>(editor);
  if ( !aLE )
    return;
  
  switch ( myMode )
  {
  case NV_Int:
    {
      int value = index.model()->data(index, Qt::DisplayRole).toInt();
      aLE->setText( QString("%1").arg( value ) );
      break;
    }
  case NV_Real:
    {
      double value = index.model()->data(index, Qt::DisplayRole).toDouble();
      aLE->setText( QString("%1").arg( value ) );
      break;
    }
  default:
    aLE->setText( "No validator!!!" );
  }  
}





/*class VisuGUI_Table : public QTableWidget {
public:
  VisuGUI_Table( Orientation orient, QWidget* parent = 0 ) 
    : QTableWidget( parent ), myValidator( 0 ), myOrientation( orient ) {}
  VisuGUI_Table( Orientation orient, int numRows, int numCols, QWidget* parent = 0 )
    : QTableWidget( numRows, numCols, parent ), myValidator( 0 ), myOrientation( orient ) {}
  
  void setValidator( QValidator* v = 0 ) { myValidator = v;  }
  bool isEditing() const { return QTable::isEditing(); }
  
protected:
  QWidget* createEditor ( int row, int col, bool initFromCell ) const
    {
      bool testUnits = ( myOrientation == Qt::Horizontal && col == 0 ) || ( myOrientation == Qt::Vertical && row == 0 );
      QWidget* wg = QTable::createEditor( row, col, initFromCell );
      if ( wg && wg->inherits("QLineEdit") && myValidator && !testUnits ) 
	(( QLineEdit*)wg)->setValidator( myValidator );
      return wg;
    }

protected:
  QValidator* myValidator;
  Orientation myOrientation;
  };*/

/*!
  Constructor
*/
VisuGUI_TableDlg::VisuGUI_TableDlg( QWidget* parent, 
				    _PTR(SObject) obj, 
				    bool edit,
				    int which,
				    Qt::Orientation orient,
				    bool showColumnTitles )
     : QDialog( parent, Qt::WindowTitleHint | Qt::WindowSystemMenuHint ),
       myIntTable( 0 ), myRealTable( 0 )
{
  setWindowTitle( edit ? tr( "EDIT_TABLE_TLT" ) : tr( "VIEW_TABLE_TLT" ) );
  setSizeGripEnabled( true );

  myObject = obj;
  bool bHasIntTable = false;
  bool bHasRealTable = false;
  if ( myObject ) {
    _PTR(GenericAttribute) anAttr;
    bHasIntTable  = myObject->FindAttribute( anAttr, "AttributeTableOfInteger");
    bHasRealTable = myObject->FindAttribute( anAttr, "AttributeTableOfReal");
  }
  
  QVBoxLayout* mainLayout = new QVBoxLayout( this );
  mainLayout->setMargin( MARGIN_SIZE );
  mainLayout->setSpacing( SPACING_SIZE );

  bool bDoInt  = which == ttInt  || which == ttBoth || which == ttAuto && bHasIntTable;
  bool bDoReal = which == ttReal || which == ttBoth || which == ttAuto && bHasRealTable;

  QWidget* top;
  QVBoxLayout* tl;
  if ( bDoInt && bDoReal ) {
    top = new QTabWidget( this );
    //( ( QTabWidget* ) top) ->setMargin( MARGIN_SIZE );
  }
  else {
    top = new QWidget( this );
    tl  = new QVBoxLayout( top ); tl->setMargin( 0 ); tl->setSpacing( SPACING_SIZE );
  }

  if ( bDoInt ) {
    myIntTable = new VisuGUI_TableWidget( top, "myIntTable", edit, orient, showColumnTitles );
    //myIntTable->getTable()->setValidator( new QIntValidator( this ) );
    //QAbstractItemDelegate* item = myIntTable->getTable()->itemDelegate();
    myIntTable->getTable()->setItemDelegate( new NumDelegateItem( myIntTable, NumDelegateItem::NV_Int ) );
    //item->deleteLater();
                                 
    if ( bDoInt && bDoReal )
      ( ( QTabWidget* )top )->addTab( myIntTable, tr( "TABLE_OF_INTEGER_TLT" ) );
    else
      tl->addWidget( myIntTable );
  }
  if ( bDoReal ) {
    myRealTable = new VisuGUI_TableWidget( top, "myRealTable", edit, orient, showColumnTitles );
    //myRealTable->getTable()->setValidator( new QDoubleValidator( this ) );
    //QAbstractItemDelegate* item = myIntTable->getTable()->itemDelegate();
    myRealTable->getTable()->setItemDelegate( new NumDelegateItem( myRealTable, NumDelegateItem::NV_Real ) );
    //item->deleteLater();
    if ( bDoInt && bDoReal )
      ( ( QTabWidget* )top )->addTab( myRealTable, tr( "TABLE_OF_REAL_TLT" ) );
    else
      tl->addWidget( myRealTable );
  }
  if ( !bDoInt && !bDoReal ) {
    QLabel *dumbLabel = new QLabel( tr( "ERR_TABLE_NOT_AVAILABLE" ), top );
    dumbLabel->setAlignment( Qt::AlignCenter );
    tl->addWidget( dumbLabel );
  }

  QHBoxLayout* btnLayout = new QHBoxLayout; 
  btnLayout->setMargin( 0 ); btnLayout->setSpacing( SPACING_SIZE );
  
  myOKBtn = new QPushButton( tr( "BUT_OK" ), this );
  myHelpBtn = new QPushButton( tr( "BUT_HELP" ), this );
  if ( edit ) {
    myCancelBtn = new QPushButton( tr( "BUT_CANCEL" ), this );
    btnLayout->addWidget( myOKBtn );
    btnLayout->addItem( new QSpacerItem( SPACER_SIZE, SPACER_SIZE, QSizePolicy::Expanding, QSizePolicy::Minimum ) );
    btnLayout->addWidget( myCancelBtn );
    btnLayout->addWidget( myHelpBtn );
    connect( myOKBtn,     SIGNAL( clicked() ), this, SLOT( onOK() ) );
    connect( myCancelBtn, SIGNAL( clicked() ), this, SLOT( reject() ) );
  }
  else {
    btnLayout->addWidget( myOKBtn );
    btnLayout->addItem( new QSpacerItem( SPACER_SIZE, SPACER_SIZE, QSizePolicy::Expanding, QSizePolicy::Minimum ) );
    btnLayout->addWidget( myHelpBtn );
    connect( myOKBtn,     SIGNAL( clicked() ), this, SLOT( accept() ) );
  }
  connect( myHelpBtn,     SIGNAL( clicked() ), this, SLOT( onHelp() ) );

  mainLayout->addWidget( top );
  mainLayout->addLayout( btnLayout );

  initDlg();
  resize( 500, 400 );
  SUIT_Tools::centerWidget( this, parent );
}

/*!
  Destructor
*/
VisuGUI_TableDlg::~VisuGUI_TableDlg()
{
}

/*!
  <OK> button slot, saves table(s)
  Called only in create/edit mode ( <edit> parameter for constructor is true )
*/
void VisuGUI_TableDlg::onOK()
{
  myOKBtn->setFocus(); // accept possible changes
  bool done = true;

  if ( myObject ) {
    _PTR(Study) study = myObject->GetStudy();
    _PTR(AttributeTableOfInteger) tblIntAttr;
    _PTR(AttributeTableOfReal)    tblRealAttr;

    if ( study ) {
      _PTR(StudyBuilder) builder = study->NewBuilder();
      builder->NewCommand(); // start transaction !!!!!!!!!!!!!!!!!!!!!!!!!!!!!
      try {
	if ( myIntTable ) {
	  builder->RemoveAttribute( myObject, "AttributeTableOfInteger" );
	  tblIntAttr = builder->FindOrCreateAttribute( myObject, "AttributeTableOfInteger" );

	  int i;
	  int nbRows  = myIntTable->getNumRows();
	  int nbCols  = myIntTable->getNumCols();
	  QString tlt = myIntTable->getTableTitle();
	  QStringList rowTitles, colTitles, units;
	  myIntTable->getRowTitles( rowTitles );
	  myIntTable->getColTitles( colTitles );
	  myIntTable->getUnits( units );
	  
	  if ( nbRows > 0) {
	    // data
	    int nRow = 0;
	    tblIntAttr->SetNbColumns( nbCols );
	    for ( i = 0; i < nbRows; i++ ) {
	      QStringList data;
	      myIntTable->getRowData( i, data );
	      bool bEmptyRow = true;
	      for ( int j = 0; j < data.count(); j++ ) {
		if ( !data[ j ].isNull() ) {
		  tblIntAttr->PutValue( data[ j ].toInt(), nRow+1, j+1 );
		  bEmptyRow = false;
		}
	      }
	      if ( !bEmptyRow ) {  // Skip rows with no data !!!
		// set row title
		tblIntAttr->SetRowTitle( nRow+1, rowTitles[ i ].isNull() ? "" : (const char*)rowTitles[ i ].toLatin1() ); 
		// set row unit
		tblIntAttr->SetRowUnit( nRow+1, units[ i ].isNull() ? "" : (const char*)units[ i ].toLatin1() ); 
		nRow++;
	      }
	    }
	    if ( nRow > 0 ) { // Set columns only if table is not empty, otherwise exception is raised !!!
	      // column titles
	      for ( i = 0; i < colTitles.count(); i++ )
		tblIntAttr->SetColumnTitle( i+1, colTitles[ i ].isNull() ? "" : (const char*)colTitles[ i ].toLatin1() );
	    }
	  }
	  // title
	  tblIntAttr->SetTitle( (const char*)myIntTable->getTableTitle().toLatin1() );
	}
	if ( myRealTable ) {
	  builder->RemoveAttribute( myObject, "AttributeTableOfReal" );
	  tblRealAttr = builder->FindOrCreateAttribute( myObject, "AttributeTableOfReal" );

	  int i;
	  int nbRows  = myRealTable->getNumRows();
	  int nbCols  = myRealTable->getNumCols();
	  QString tlt = myRealTable->getTableTitle();
	  QStringList rowTitles, colTitles, units;
	  myRealTable->getRowTitles( rowTitles );
	  myRealTable->getColTitles( colTitles );
	  myRealTable->getUnits( units );
	  
	  if ( nbRows > 0) {
	    // data
	    int nRow = 0;
	    tblRealAttr->SetNbColumns( nbCols );
	    for ( i = 0; i < nbRows; i++ ) {
	      QStringList data;
	      myRealTable->getRowData( i, data );
	      bool bEmptyRow = true;
	      for ( int j = 0; j < data.count(); j++ ) {
		if ( !data[ j ].isNull() ) {
		  tblRealAttr->PutValue( data[ j ].toDouble(), nRow+1, j+1 );
		  bEmptyRow = false;
		}
	      }
	      if ( !bEmptyRow ) {  // Skip rows with no data !!!
		// set row title
		tblRealAttr->SetRowTitle( nRow+1, rowTitles[ i ].isNull() ? "" : (const char*)rowTitles[ i ].toLatin1() ); 
		// set row unit
		tblRealAttr->SetRowUnit( nRow+1, units[ i ].isNull() ? "" : (const char*)units[ i ].toLatin1() );
		nRow++;
	      }
	    }
	    if ( nRow > 0 ) { // Set columns only if table is not empty, otherwise exception is raised !!!
	      // column titles
	      for ( i = 0; i < colTitles.count(); i++ )
		tblRealAttr->SetColumnTitle( i+1, colTitles[ i ].isNull() ? "" : (const char*)colTitles[ i ].toLatin1() );
	    }
	  }
	  // title
	  tblRealAttr->SetTitle( (const char*)myRealTable->getTableTitle().toLatin1() );
	}
	if ( myIntTable || myRealTable)
	  builder->CommitCommand(); // commit transaction !!!!!!!!!!!!!!!!!!!!!!!!!!!
	else
	  builder->AbortCommand();  // abort transaction  !!!!!!!!!!!!!!!!!!!!!!!!!!!
      }
      catch( ... ) {
	MESSAGE("VisuGUI_TableDlg::onOK : Exception has been caught !!!");
	builder->AbortCommand();  // abort transaction  !!!!!!!!!!!!!!!!!!!!!!!!!!!
	done = false;
	SUIT_MessageBox::critical ( this, tr("ERR_ERROR"), tr("ERR_APP_EXCEPTION") );
      }
    }
  }
  if ( done ) 
    accept();
}

/*!
  <Help> button slot, shows corresponding help page
*/
void VisuGUI_TableDlg::onHelp()
{
  QString aHelpFileName = "table_presentations_page.html";
  LightApp_Application* app = (LightApp_Application*)(SUIT_Session::session()->activeApplication());
  if (app)
    app->onHelpContextModule(app->activeModule() ? app->moduleName(app->activeModule()->moduleName()) : QString(""), aHelpFileName);
  else {
		QString platform;
#ifdef WIN32
		platform = "winapplication";
#else
		platform = "application";
#endif
    SUIT_MessageBox::warning(0, QObject::tr("WRN_WARNING"),
                             QObject::tr("EXTERNAL_BROWSER_CANNOT_SHOW_PAGE").
                             arg(app->resourceMgr()->stringValue("ExternalBrowser", platform)).arg(aHelpFileName) );
  }
}

/*!
   Populates table with data
*/
void VisuGUI_TableDlg::initDlg()
{
  int i, j;
  if ( myObject ) {
    _PTR(GenericAttribute) anAttr;
    _PTR(AttributeTableOfInteger) tblIntAttr;
    _PTR(AttributeTableOfReal)    tblRealAttr;
    if ( myObject->FindAttribute( anAttr, "AttributeTableOfInteger") ) {
      tblIntAttr = anAttr;
    }
    if ( myObject->FindAttribute( anAttr, "AttributeTableOfReal") ) {
      tblRealAttr = anAttr;
    }
    // Table of integer
    if ( tblIntAttr && myIntTable ) {
      try {
	// title
	myIntTable->setTableTitle( tblIntAttr->GetTitle().c_str() );
	// nb of rows & cols
	int nbRows = tblIntAttr->GetNbRows() ; 
	int nbCols = tblIntAttr->GetNbColumns();
	myIntTable->setNumRows( nbRows );
	myIntTable->setNumCols( nbCols );
	// rows titles
	QStringList strlist;
	vector<string> rowTitles = tblIntAttr->GetRowTitles();
	for ( i = 0; i < nbRows; i++ ) {
	  if ( rowTitles.size() > 0 )
	    strlist.append( rowTitles[i].c_str() );
	  else
	    strlist.append( "" );
	}
	myIntTable->setRowTitles( strlist );
	// columns titles
	strlist.clear();
	vector<string> colTitles = tblIntAttr->GetColumnTitles();
	for ( i = 0; i < nbCols; i++ ) {
	  if ( colTitles.size() > 0 )
	    strlist.append( colTitles[i].c_str() );
	  else
	    strlist.append( "" );
	}
	myIntTable->setColTitles( strlist );
	// units
	strlist.clear();
	vector<string> rowUnits = tblIntAttr->GetRowUnits();
	if ( rowUnits.size() > 0 ) {
	  for ( i = 0; i < nbRows; i++ )
	    strlist.append( rowUnits[i].c_str() );
	  myIntTable->setUnits( strlist );
	}
	// data
	for ( i = 1; i <= nbRows; i++ ) {
	  strlist.clear();
	  for ( j = 1; j <= nbCols; j++ ) {
	    if ( tblIntAttr->HasValue( i, j ) )
	      strlist.append( QString::number( tblIntAttr->GetValue( i, j ) ) );
	    else
	      strlist.append( QString::null );
	  }
	  myIntTable->setRowData( i-1, strlist );
	}
	myIntTable->adjustTable();
      }
      catch( ... ) {
	MESSAGE("VisuGUI_TableDlg::initDlg : Exception has been caught !!!");
      }
    } 
    // Table of real
    if ( tblRealAttr && myRealTable ) {
      try {
	// title
	myRealTable->setTableTitle( tblRealAttr->GetTitle().c_str() );
	// nb of rows & cols
	int nbRows = tblRealAttr->GetNbRows() ; 
	int nbCols = tblRealAttr->GetNbColumns();
	myRealTable->setNumRows( nbRows );
	myRealTable->setNumCols( nbCols );
	// rows titles
	QStringList strlist;
	vector<string> rowTitles = tblRealAttr->GetRowTitles();
	for ( i = 0; i < nbRows; i++ ) {
	  if ( rowTitles.size() > 0 )
	    strlist.append( rowTitles[i].c_str() );
	  else
	    strlist.append( "" );
	}
	myRealTable->setRowTitles( strlist );
	// columns titles
	strlist.clear();
	vector<string> colTitles = tblRealAttr->GetColumnTitles();
	for ( i = 0; i < nbCols; i++ ) {
	  if ( colTitles.size() > 0 )
	    strlist.append( colTitles[i].c_str() );
	  else
	    strlist.append( "" );
	}
	myRealTable->setColTitles( strlist );
	// units
	strlist.clear();
	vector<string> rowUnits = tblRealAttr->GetRowUnits();
	if ( rowUnits.size() > 0 ) {
	  for ( i = 0; i < nbRows; i++ )
	    strlist.append( rowUnits[i].c_str() );
	  myRealTable->setUnits( strlist );
	}
        // data
	for ( i = 1; i <= nbRows; i++ ) {
	  strlist.clear();
	  for ( j = 1; j <= nbCols; j++ ) {
	    if ( tblRealAttr->HasValue( i, j ) )
	      strlist.append( QString::number( tblRealAttr->GetValue( i, j ) ) );
	    else
	      strlist.append( QString::null );
	  }
	  myRealTable->setRowData( i-1, strlist );
	}
	myRealTable->adjustTable();
      }
      catch( ... ) {
	MESSAGE("VisuGUI_TableDlg::initDlg : Exception has been caught !!!");
      }
    } 
  }
}

/*!
  Provides help on F1 button click
*/
void VisuGUI_TableDlg::keyPressEvent( QKeyEvent* e )
{
  QDialog::keyPressEvent( e );
  if ( e->isAccepted() )
    return;

  if ( e->key() == Qt::Key_F1 )
    {
      e->accept();
      onHelp();
    }
}

/*!
  Constructor
*/
VisuGUI_TableWidget::VisuGUI_TableWidget( QWidget* parent, 
					      const char* name, 
					      bool edit, 
					      Qt::Orientation orient,
					      bool showColumnTitles )
     : QWidget( parent ), myOrientation( orient )
{
  QGridLayout* mainLayout = new QGridLayout( this );
  mainLayout->setMargin( 0 );
  mainLayout->setSpacing( SPACING_SIZE );

  myTitleEdit = new QLineEdit( this );
  myTitleEdit->setAlignment( Qt::AlignCenter );
  myTitleEdit->setReadOnly( !edit );
  QFont fnt = myTitleEdit->font();
  fnt.setBold( true ); 
  myTitleEdit->setFont( fnt );

  //myTable = new VisuGUI_Table( orient, this );
  myTable = new QTableWidget( 5, 5, this );
  //myTable->setNumRows( 5 );
  //myTable->setNumCols( 5 );
  myTable->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Expanding ) );
  myTable->setMinimumSize( MIN_TABLE_WIDTH, MIN_TABLE_HEIGHT );
  myTable->setSelectionMode( QAbstractItemView::SingleSelection );
  myTable->setShowGrid( true );
  myTable->horizontalHeader()->setMovable( false );
  myTable->verticalHeader()->setMovable( false );
  //myTable->setColumnMovingEnabled( false );
  //myTable->setRowMovingEnabled( false );
  myTable->setDragEnabled( false );
  //myTable->setReadOnly( !edit );
  myTable->setEditTriggers( edit ? QAbstractItemView::AllEditTriggers : QAbstractItemView::NoEditTriggers );

  setUnitsTitle( tr( "UNITS_TLT" ) );

  if ( !showColumnTitles ) {
    if ( myOrientation == Qt::Horizontal ) {
      myTable->horizontalHeader()->hide();
      //myTable->setTopMargin( 0 );
    }
    else {
      myTable->verticalHeader()->hide();
      //myTable->setLeftMargin( 0 );
    }
  }

  mainLayout->addWidget( myTitleEdit, 0, 0 );
  mainLayout->addWidget( myTable, 1, 0 );

  if ( edit ) {
    myAddRowBtn    = new QPushButton( tr( "ADD_ROW_BTN" ), this );
    myDelRowBtn    = new QPushButton( tr( "REMOVE_ROW_BTN" ), this );
    myAddColBtn    = new QPushButton( tr( "ADD_COLUMN_BTN" ), this );
    myDelColBtn    = new QPushButton( tr( "REMOVE_COLUMN_BTN" ), this );
    myAdjustBtn    = new QPushButton( tr( "ADJUST_CELLS_BTN" ), this );
    mySelectAllBtn = new QPushButton( tr( "SELECT_ALL_BTN" ), this );
    myClearBtn     = new QPushButton( tr( "CLEAR_BTN"), this );
    QVBoxLayout* btnLayout = new QVBoxLayout; btnLayout->setMargin( 0 ); btnLayout->setSpacing( SPACING_SIZE );
    btnLayout->addWidget( myAddRowBtn );
    btnLayout->addWidget( myDelRowBtn );
    btnLayout->addWidget( myAddColBtn );
    btnLayout->addWidget( myDelColBtn );
    btnLayout->addStretch();
    btnLayout->addWidget( myAdjustBtn );
    btnLayout->addStretch();
    btnLayout->addWidget( mySelectAllBtn );
    btnLayout->addWidget( myClearBtn );
    mainLayout->addLayout( btnLayout, 1, 1 );
    connect( myTable, SIGNAL( selectionChanged() ),        this, SLOT( updateButtonsState() ) );
    connect( myTable, SIGNAL( currentChanged( int, int) ), this, SLOT( updateButtonsState() ) );
    connect( myAddRowBtn,    SIGNAL( clicked() ),   this, SLOT( addRow() ) );
    connect( myAddColBtn,    SIGNAL( clicked() ),   this, SLOT( addCol() ) );
    connect( myDelRowBtn,    SIGNAL( clicked() ),   this, SLOT( delRow() ) );
    connect( myDelColBtn,    SIGNAL( clicked() ),   this, SLOT( delCol() ) );
    connect( myAdjustBtn,    SIGNAL( clicked() ),   this, SLOT( adjustTable() ) );
    connect( mySelectAllBtn, SIGNAL( clicked() ),   this, SLOT( selectAll() ) );
    connect( myClearBtn,     SIGNAL( clicked() ),   this, SLOT( clearTable() ) );
    myTable->horizontalHeader()->installEventFilter( this );
    myTable->verticalHeader()->installEventFilter( this );
    myTable->installEventFilter( this );
  }
  updateButtonsState();
}
/*!
  Destructor
*/
VisuGUI_TableWidget::~VisuGUI_TableWidget()
{
}
/*!
  Sets table title
*/
void VisuGUI_TableWidget::setTableTitle( const QString& title )
{
  myTitleEdit->setText( title );
}
/*!
  Gets table title
*/
QString VisuGUI_TableWidget::getTableTitle()
{
  return myTitleEdit->text();
}
/*!
  Sets total number of rows
*/
void VisuGUI_TableWidget::setNumRows( const int num )
{
  myOrientation == Qt::Horizontal ? myTable->setRowCount( num ) : myTable->setColumnCount( num );
}
/*!
  Gets total number of rows
*/
int VisuGUI_TableWidget::getNumRows()
{
  return myOrientation == Qt::Horizontal ? myTable->rowCount() : myTable->columnCount();
}
/*!
  Sets total number of columns
*/
void VisuGUI_TableWidget::setNumCols( const int num )
{
  // !!! first column contains units !!!
  myOrientation == Qt::Horizontal ? myTable->setColumnCount( num+1 ) : myTable->setRowCount( num+1 );
//  myOrientation == Qt::Horizontal ? myTable->setColumnReadOnly( 0, true ) : myTable->setRowReadOnly( 0, true );
}
/*!
  Gets total number of columns
*/
int VisuGUI_TableWidget::getNumCols()
{
  // !!! first column contains units !!!
  return myOrientation == Qt::Horizontal ? myTable->columnCount()-1 : myTable->rowCount()-1;
}
/*!
  Sets rows titles
*/
void VisuGUI_TableWidget::setRowTitles( QStringList& tlts )
{
  QStringList aLabels;
  for ( int i = 0; i < tlts.count(); i++ )
    tlts[i].isNull() ? aLabels.append("") : aLabels.append( tlts[i] );

  myOrientation == Qt::Horizontal ?
    myTable->setVerticalHeaderLabels( aLabels ) :
    myTable->setHorizontalHeaderLabels( aLabels );
    
    //  myTable->verticalHeader()->setLabel( i, tlts[i] ) : 
    //myTable->horizontalHeader()->setLabel( i, tlts[i] );
  //}
}
/*!
  Gets rows titles
*/
void VisuGUI_TableWidget::getRowTitles( QStringList& tlts )
{
  tlts.clear();
  if ( myOrientation == Qt::Horizontal ) {
    for ( int i = 0; i < myTable->rowCount(); i++ ) {
      tlts.append( myTable->verticalHeaderItem(i) ? myTable->verticalHeaderItem(i)->text() : "" );
    }
  }
  else {
    for ( int i = 0; i < myTable->columnCount(); i++ ) {
      tlts.append( myTable->horizontalHeaderItem(i) ? myTable->horizontalHeaderItem(i)->text() : "" );
    }
  }
}
/*!
  Sets columns titles
*/
void VisuGUI_TableWidget::setColTitles( QStringList& tlts )
{
  QStringList aLabels;

  // !!! first column contains units !!!
  aLabels.append(""); // it'll be initialized below - in setUnitsTitle() method

  for ( int i = 0; i < tlts.count(); i++ )
    tlts[i].isNull() ? aLabels.append("") : aLabels.append( tlts[i] );

  myOrientation == Qt::Horizontal ?
    myTable->setHorizontalHeaderLabels( aLabels ) :
    myTable->setVerticalHeaderLabels( aLabels );
  
  setUnitsTitle( tr( "UNITS_TLT" ) );
}
/*!
  Sets columns titles
*/
void VisuGUI_TableWidget::getColTitles( QStringList& tlts )
{
  // !!! first column contains units !!!
  tlts.clear();
  if ( myOrientation == Qt::Horizontal ) {
    for ( int i = 1; i < myTable->columnCount(); i++ ) {
      tlts.append( myTable->horizontalHeaderItem(i) ? myTable->horizontalHeaderItem(i)->text() : "" );
    }    
  }
  else {
    for ( int i = 1; i < myTable->rowCount(); i++ ) {
      tlts.append( myTable->verticalHeaderItem(i) ? myTable->verticalHeaderItem(i)->text() : "" );
    }
  }
}
/*!
  Sets units title
*/
void VisuGUI_TableWidget::setUnitsTitle( const QString& tlt ) {
  // !!! first column contains units !!!
  myTable->model()->setHeaderData( 0, myOrientation, QVariant(tlt.isNull() ? "" : tlt), Qt::DisplayRole );
}
/*!
  Sets units
*/
void VisuGUI_TableWidget::setUnits( QStringList& units )
{
  QAbstractTableModel* aModel = qobject_cast<QAbstractTableModel*>( myTable->model() );
  if ( aModel )
  {
    QModelIndex anIndex;
    for ( int i = 0; i < units.count(); i++ )
    {
      myOrientation == Qt::Horizontal ?
        anIndex = aModel->index( i, 0 ) :
        anIndex = aModel->index( 0, i );

      aModel->setData( anIndex, QVariant( units[i].isNull() ? "" : units[i] ) );      
    }
  }
}
/*!
  Gets units
*/
void VisuGUI_TableWidget::getUnits( QStringList& units )
{
  units.clear();
  QAbstractTableModel* aModel = qobject_cast<QAbstractTableModel*>( myTable->model() );
  if ( aModel )
  {
    if ( myOrientation == Qt::Horizontal )
    {
      for ( int i = 0; i < myTable->rowCount(); i++ )
        units.append( aModel->index( i, 0 ).data().toString() );
    }
    else {
      for ( int i = 0; i < myTable->columnCount(); i++ )
        units.append( aModel->index( 0, i ).data().toString() );
    }
  }
}
/*!
  Sets row data
*/
void VisuGUI_TableWidget::setRowData( int row, QStringList& data )
{
  QAbstractTableModel* aModel = qobject_cast<QAbstractTableModel*>( myTable->model() );
  if ( aModel )
  {
    QModelIndex anIndex; 
    if ( row >= 0 && row < getNumRows() ) {
      for ( int i = 0; i < data.count(); i++ )
      {
        myOrientation == Qt::Horizontal ? anIndex = aModel->index( row, i+1 ) :
                                          anIndex = aModel->index( i+1, row );
        aModel->setData( anIndex, QVariant( data[i] ) );
          
      }
    }
  }
}
/*!
  Gets row data
*/
void VisuGUI_TableWidget::getRowData( int row, QStringList& data )
{
  data.clear();
  QAbstractTableModel* aModel = qobject_cast<QAbstractTableModel*>( myTable->model() );
  if ( aModel )
  {
    if ( row >= 0 && row < getNumRows() )
    {
      if ( myOrientation == Qt::Horizontal )
      {
        for ( int i = 1; i < myTable->columnCount(); i++ )
          data.append( aModel->index( row, i ).data().toString() );
      }
      else {
        for ( int i = 1; i < myTable->rowCount(); i++ )
          data.append( aModel->index( i, row ).data().toString() );
      }
    }
  }
}
/*!
  Adjusts table cell to see contents, <Adjust Cells> button slot
*/
void VisuGUI_TableWidget::adjustTable()
{
  myTable->resizeRowsToContents();
  myTable->resizeColumnsToContents();
}
/*!
  Called when selection changed in table
*/
void VisuGUI_TableWidget::updateButtonsState()
{
  if ( myTable->editTriggers() == QAbstractItemView::NoEditTriggers )
    return;
  bool bDR = false; // <Delete Row(s)>
  bool bDC = false; // <Delete Column(s)>
  bool bSA = false; // <Select All>
  bool bCT = false; // <Clear>
  int i;
  //TO DO column/row selection check
  /*int c = myOrientation == Qt::Horizontal ? 0 : 1;
  for ( i = c; i < myTable->rowCount(); i++ ) {
    
    if ( myTable->isRowSelected( i, true ) )
      bDR = true;
    else 
      bSA = true;
  }
  c = myOrientation == Qt::Horizontal ? 1 : 0;
  for ( i = c; i < myTable->columnCount(); i++ ) {
    if ( myTable->isColumnSelected( i, true ) )
      bDC = true;
    else 
      bSA = true;
      }*/
  /*int nbSel = myTable->numSelections();
  for ( i = 0; i < nbSel; i++ ) {
    QTableSelection ts = myTable->selection( i );
    for ( int j = ts.topRow(); j < ts.bottomRow()+1; j++) {
      for ( int k = ts.leftCol(); k < ts.rightCol()+1; k++) {
	if ( myTable->item( j, k ) )
	  bCT = true;
      }
    }
    }*/
  QList<QTableWidgetItem*> aSelection = myTable->selectedItems();
  QList<QTableWidgetItem*>::ConstIterator anIt = aSelection.constBegin(),
    anEndIt = aSelection.constEnd();
  for ( ; anIt !=  anEndIt; anIt++ )
  {
    if( *anIt )
    {
      bCT = true;
      break;
    }
  }
  
  if ( myTable->item( myTable->currentRow(), myTable->currentColumn() ) )
    bCT = true;
  myDelRowBtn->setEnabled( bDR );
  myDelColBtn->setEnabled( bDC );
  mySelectAllBtn->setEnabled( bSA );
  myClearBtn->setEnabled( bCT );
}
/*!
  <Add row> button slot
*/
void VisuGUI_TableWidget::addRow()
{
  myTable->insertRow( myTable->rowCount() );
  updateButtonsState();
}
/*!
  <Add column> button slot
*/
void VisuGUI_TableWidget::addCol()
{
  myTable->insertColumn( myTable->columnCount() );
  updateButtonsState();
}
/*!
  <Delete row(s)> button slot
*/
void VisuGUI_TableWidget::delRow()
{
  //TODO
  /*int c = myOrientation == Qt::Horizontal ? 0 : 1;
  QList<int> il;
  int i;
  for ( i = c; i < myTable->rowCount(); i++ )
    if ( myTable->isRowSelected( i, true ) )
      il.append( i );
  if ( il.count() > 0 ) {
    QMemArray<int> ildel( il.count() );
    for ( i = 0; i < il.count(); i++ )
      ildel[ i ] = il[ i ];
    myTable->removeRows( ildel );
    }*/
  
  updateButtonsState();
}
/*!
  <Delete column(s)> button slot
*/
void VisuGUI_TableWidget::delCol()
{
  //TODO
  /*int c = myOrientation == Qt::Horizontal ? 1 : 0;
  QValueList<int> il;
  int i;
  for ( i = c; i < myTable->numCols(); i++ )
    if ( myTable->isColumnSelected( i, true ) )
      il.append( i );
  if ( il.count() > 0 ) {
    QMemArray<int> ildel( il.count() );
    for ( i = 0; i < il.count(); i++ )
      ildel[ i ] = il[ i ];
    myTable->removeColumns( ildel );
    }*/
  updateButtonsState();
}
/*!
  <Select All> button slot
*/
void VisuGUI_TableWidget::selectAll()
{
  /*myTable->clearSelection();
  QTableSelection ts;
  ts.init( 0, 0 ); ts.expandTo( myTable->numRows()-1, myTable->numCols()-1 );
  myTable->addSelection( ts );*/
  myTable->selectAll();
  updateButtonsState();
}
/*!
  <Clear> button slot
*/
void VisuGUI_TableWidget::clearTable()
{
  /*int nbSel = myTable->numSelections();
  for ( int i = 0; i < nbSel; i++ ) {
    QTableSelection ts = myTable->selection( i );
    for ( int j = ts.topRow(); j < ts.bottomRow()+1; j++) {
      if ( myOrientation == Qt::Vertical && j == 0 ) {
//	continue;      // UNITS
      }
      for ( int k = ts.leftCol(); k < ts.rightCol()+1; k++) {
	if ( myOrientation == Qt::Horizontal && k == 0 ) {
//	  continue;   // UNITS
	}
	myTable->clearCell( j, k );
      }
    }
  }
  if ( nbSel == 0 )
    myTable->clearCell( myTable->currentRow(), myTable->currentColumn() );
    myTable->clearSelection();*/
  myTable->clearContents();
  updateButtonsState();
}
/*!
  Event filter - handles titles editing
*/
bool VisuGUI_TableWidget::eventFilter( QObject* o, QEvent* e )
{
  if ( e->type() == QEvent::MouseButtonDblClick) {
    //TODO
    /*QMouseEvent* me = ( QMouseEvent* )e;
    if ( me->button() == Qt::LeftButton && (myTable->editTriggers() != QAbstractItemView::NoEditTriggers ) ) {
      if ( o == myTable->horizontalHeader() ) {
	for ( int i = 0; i < myTable->horizontalHeader()->count(); i++ ) {
	  QRect rect = myTable->horizontalHeader()->sectionRect( i );
	  rect.addCoords( 1, 1, -1, -1 );
	  if ( rect.contains( myTable->horizontalHeader()->mapFromGlobal( me->globalPos() ) ) ) {
	    if ( myOrientation == Qt::Vertical || i != 0 ) {
	      bool bOk;
	      QString tlt = QInputDialog::getText( tr( "SET_TITLE_TLT" ), 
						   tr( "TITLE_LBL" ),
						   QLineEdit::Normal,
						   myTable->horizontalHeader()->label( i ),
						   &bOk,
						   this );
	      if ( bOk && !tlt.isNull() )
		myTable->horizontalHeader()->setLabel( i, tlt );
	      break;
	    }
	  }
	}
      }
      if ( o == myTable->verticalHeader() ) {
	for ( int i = 0; i < myTable->verticalHeader()->count(); i++ ) {
	  QRect rect = myTable->verticalHeader()->sectionRect( i );
	  rect.addCoords( 1, 1, -1, -1 );
	  if ( rect.contains( myTable->verticalHeader()->mapFromGlobal( me->globalPos() ) ) ) {
	    if ( myOrientation == Qt::Horizontal || i != 0 ) {
	      bool bOk;
	      QString tlt = QInputDialog::getText( tr( "SET_TITLE_TLT" ), 
						   tr( "TITLE_LBL" ),
						   QLineEdit::Normal,
						   myTable->verticalHeader()->label( i ),
						   &bOk,
						   this );
	      if ( bOk && !tlt.isNull() )
		myTable->verticalHeader()->setLabel( i, tlt );
	      break;
	    }
	  }
	}
      }
      }*/    
  }     
  else if ( e->type() == QEvent::KeyRelease && o == myTable ) {
    QKeyEvent* ke = (QKeyEvent*)e;
    if ( ke->key() == Qt::Key_Delete && (myTable->editTriggers() != QAbstractItemView::NoEditTriggers) ) {
      clearTable();
    }
    else if ( ke->key() == Qt::Key_Backspace && (myTable->editTriggers() != QAbstractItemView::NoEditTriggers) ) {
      clearTable();
      int i = myTable->currentRow();
      int j = myTable->currentColumn() - 1;
      if ( j < 0 ) { j = myTable->columnCount()-1; i--; }
      if ( i >= 0 && j >= 0 )
	myTable->setCurrentCell( i, j );
    }
  }
  return QWidget::eventFilter( o, e );
}
