//  Copyright (C) 2007-2008  CEA/DEN, EDF R&D, OPEN CASCADE
//
//  Copyright (C) 2003-2007  OPEN CASCADE, EADS/CCR, LIP6, CEA/DEN,
//  CEDRAT, EDF R&D, LEG, PRINCIPIA R&D, BUREAU VERITAS
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
//
//  See http://www.salome-platform.org/ or email : webmaster.salome@opencascade.com
//
//  SALOME VTKViewer : build VTK viewer into Salome desktop
//  File   :
//  Author :
//  Module :
//  $Header$
//
#include "VVTK_Recorder.h"

#include "VVTK_ImageWriter.h"
#include "VVTK_ImageWriterMgr.h"

#include <vtkObjectFactory.h>
#include <vtkObject.h>
#include <vtkCallbackCommand.h>
#include <vtkRenderWindow.h>
#include <vtkTimerLog.h>
#include <vtkWindowToImageFilter.h>
#include <vtkJPEGWriter.h>
#include <vtkImageData.h>

#include <sstream>
#include <iomanip>
#include <iostream>

#ifndef WIN32
#include <unistd.h>
#endif

#include <QApplication>
#include <QFileInfo>
#include <QDir>

#include "utilities.h"

#ifdef _DEBUG_
static int MYDEBUG = 0;
#else
static int MYDEBUG = 0;
#endif


namespace
{
  //----------------------------------------------------------------------------
  inline
  void
  GetNameJPEG(const std::string& thePreffix,  
	      const int theIndex,
	      std::string& theName)
  {
    using namespace std;
    ostringstream aStream;
    aStream<<thePreffix<<"_"<<setw(6)<<setfill('0')<<theIndex<<".jpeg";
    theName = aStream.str();
  }
}

//----------------------------------------------------------------------------
vtkCxxRevisionMacro(VVTK_Recorder,"$Revision$");
vtkStandardNewMacro(VVTK_Recorder);


//----------------------------------------------------------------------------
VVTK_Recorder
::VVTK_Recorder():
  myRenderWindow(NULL),
  myState(VVTK_Recorder_Stop),
  myNbFPS(5.5),
  myQuality(100),
  myProgressiveMode(true),
  myUseSkippedFrames(true),
  myErrorStatus(0),
  myCommand(vtkCallbackCommand::New()),
  myPriority(0.0),
  myTimeStart(0.0),
  myFrameIndex(0),
  myPaused(0),
  myFilter(vtkWindowToImageFilter::New()),
  myWriterMgr(new VVTK_ImageWriterMgr),
  myNbWrittenFrames(0),
  myNameAVIMaker("jpeg2yuv")
{
  myCommand->SetClientData(this); 
  myCommand->SetCallback(VVTK_Recorder::ProcessEvents);
}


//----------------------------------------------------------------------------
VVTK_Recorder
::~VVTK_Recorder()
{
  myCommand->Delete();
  myFilter->Delete();
  delete myWriterMgr;
}


//----------------------------------------------------------------------------
void
VVTK_Recorder
::CheckExistAVIMaker()
{
  myErrorStatus = 0;
  using namespace std;
  ostringstream aStream;
  aStream<<"which "<<myNameAVIMaker<<" >& /dev/null";
  std::string anAVIMakeCheck = aStream.str();
  int iErr = system(anAVIMakeCheck.c_str());
  if(iErr != 0)
    myErrorStatus = 127;
}


//----------------------------------------------------------------------------
void
VVTK_Recorder
::SetName(const char* theName)
{
  myName = theName;
}

const char* 
VVTK_Recorder::Name() const
{
  return myName.c_str();
}


//----------------------------------------------------------------------------
void
VVTK_Recorder
::SetNbFPS(const double theNbFPS)
{
  myNbFPS = theNbFPS;
}

double
VVTK_Recorder
::NbFPS() const
{
  return myNbFPS;
}


//----------------------------------------------------------------------------
void
VVTK_Recorder
::SetQuality(int theQuality)
{
  myQuality = theQuality;
}

int
VVTK_Recorder
::GetQuality() const
{
  return myQuality;
}


//----------------------------------------------------------------------------
void 
VVTK_Recorder
::SetRenderWindow(vtkRenderWindow* theRenderWindow)
{
  myRenderWindow = theRenderWindow;
}

vtkRenderWindow* 
VVTK_Recorder
::RenderWindow()
{
  return myRenderWindow;
}


//----------------------------------------------------------------------------
void
VVTK_Recorder
::SetProgressiveMode(bool theProgressiveMode)
{
  myProgressiveMode = theProgressiveMode;
}

bool
VVTK_Recorder
::GetProgressiveMode() const
{
  return myProgressiveMode;
}


//----------------------------------------------------------------------------
void
VVTK_Recorder
::SetUseSkippedFrames(bool theUseSkippedFrames)
{
  myUseSkippedFrames = theUseSkippedFrames;
}

bool
VVTK_Recorder
::UseSkippedFrames() const
{
  return myUseSkippedFrames;
}


//----------------------------------------------------------------------------
int
VVTK_Recorder
::ErrorStatus() const
{
  return myErrorStatus;
}

int
VVTK_Recorder
::State() const
{
  return myState;
}


//----------------------------------------------------------------------------
void
VVTK_Recorder
::ProcessEvents(vtkObject* vtkNotUsed(theObject), 
		unsigned long theEvent,
		void* theClientData, 
		void* vtkNotUsed(theCallData))
{
  if(vtkObject* anObj = reinterpret_cast<vtkObject*>(theClientData)){ 
    if(VVTK_Recorder* aSelf = dynamic_cast<VVTK_Recorder*>(anObj)){
      if(theEvent==vtkCommand::EndEvent){
	if(aSelf->State() == VVTK_Recorder::VVTK_Recorder_Record){
	  aSelf->DoRecord();
	}
      }
    }
  }
}


//----------------------------------------------------------------------------
void
VVTK_Recorder
::Record()
{
  if(myState == VVTK_Recorder_Stop){
    if(myRenderWindow){
      myState = VVTK_Recorder_Record;
      myFilter->SetInput(myRenderWindow);
      myFrameIndex = -1;
      myNbWrittenFrames = 0;
      myRenderWindow->RemoveObserver(myCommand);
      myRenderWindow->AddObserver(vtkCommand::EndEvent,
                                  myCommand,
                                  myPriority);
      myRenderWindow->Render();
    }
  }
}


//----------------------------------------------------------------------------
void
VVTK_Recorder
::Stop()
{
  QApplication::setOverrideCursor( Qt::WaitCursor );

  if(myState == VVTK_Recorder_Record){ 
    if(!myPaused)
      DoRecord();

    myWriterMgr->Stop();

    if(myUseSkippedFrames)
      AddSkippedFrames();

    myFrameIndexes.clear();

    MakeFileAVI();
  }
  myState = VVTK_Recorder_Stop;
  myPaused = 0;

  QApplication::restoreOverrideCursor();
}


//----------------------------------------------------------------------------
void
VVTK_Recorder
::Pause()
{
  myPaused = myPaused ? 0 : 1;
  if(myPaused && !myFrameIndexes.empty()){
    myFrameIndexes.back() *= -1;
    if(MYDEBUG) MESSAGE("VVTK_Recorder::Pause - myFrameIndexes.back() = "<<myFrameIndexes.back());
  }
}


//----------------------------------------------------------------------------
inline 
int
GetFrameIndex(double theStartTime,
	      double theFPS)
{
  double aTimeNow = vtkTimerLog::GetCurrentTime();
  double aDelta = aTimeNow - theStartTime;
  return int(aDelta*theFPS);
}

void
VVTK_Recorder
::DoRecord()
{
  if(myPaused)
    return;

  if(myFrameIndex < 0){
    myFrameIndex = 0;
    myTimeStart = vtkTimerLog::GetCurrentTime();
  }else{
    int aFrameIndex = GetFrameIndex(myTimeStart,myNbFPS);
    if(aFrameIndex <= myFrameIndex)
      return;

    // If there was a "pause" we correct the myTimeStart
    int aLastFrameIndex = myFrameIndexes.back();
    if(aLastFrameIndex < 0){
      myFrameIndexes.back() = abs(myFrameIndexes.back());
      double aPauseTime = fabs((double)(aFrameIndex - myFrameIndex - 1)) / myNbFPS;
      if(MYDEBUG) 
	MESSAGE("VVTK_Recorder::DoRecord - aFrameIndex = "<<aFrameIndex<<
	  "; aPauseTime = "<<aPauseTime);
      myTimeStart += aPauseTime;
    }

    aFrameIndex = GetFrameIndex(myTimeStart,myNbFPS);
    if(aFrameIndex <= myFrameIndex)
      return;

    myFrameIndex = aFrameIndex;
  }

  myFrameIndexes.push_back(myFrameIndex);
  if(MYDEBUG) MESSAGE("VVTK_Recorder::DoRecord - myFrameIndex = "<<myFrameIndex);

  myRenderWindow->RemoveObserver(myCommand);
  myFilter->Modified();

  std::string aName;
  GetNameJPEG(myName,myFrameIndex,aName);

  PreWrite();

  vtkImageData *anImageData = vtkImageData::New(); 
  anImageData->DeepCopy(myFilter->GetOutput());

  myWriterMgr->StartImageWriter(anImageData,aName,myProgressiveMode,myQuality);
  myNbWrittenFrames++;

  myRenderWindow->AddObserver(vtkCommand::EndEvent,
                              myCommand,
                              myPriority);
}


//----------------------------------------------------------------------------
void
VVTK_Recorder
::PreWrite()
{
  vtkImageData *anImageData = myFilter->GetOutput();
  //
  if(!anImageData){
    myErrorStatus = 20;
    return;
  }
  anImageData->UpdateInformation();
  int *anExtent = anImageData->GetWholeExtent();
  anImageData->SetUpdateExtent(anExtent[0], anExtent[1],
			       anExtent[2], anExtent[3],
			       0,0);
  anImageData->UpdateData();
}


//----------------------------------------------------------------------------
void
VVTK_Recorder
::AddSkippedFrames()
{
  myErrorStatus = 0;

  if(myFrameIndexes.size() < 2)
    return;

  size_t anId = 0, anEnd = myFrameIndexes.size() - 1;
  for(; anId < anEnd; anId++){
    int aStartIndex = myFrameIndexes[anId];
    if(aStartIndex < 0)
      continue;

    int aFinishIndex = abs(myFrameIndexes[anId + 1]);
    if(aStartIndex + 1 == aFinishIndex)
      continue;

    std::string anInitialName;
    std::ostringstream aStream;
    GetNameJPEG(myName,aStartIndex,anInitialName);
    for(int anIndex = aStartIndex + 1; anIndex < aFinishIndex; anIndex++){
      myNbWrittenFrames++;
      std::string anCurrentName;
      GetNameJPEG(myName,anIndex,anCurrentName);
      aStream<<"ln -s "<< anInitialName<<" "<<anCurrentName<<";";
      if(anIndex + 1 < aFinishIndex)
	aStream<<" \\";
      aStream<<endl;
    }
    std::string aString(aStream.str());
    system(aString.c_str());
    if(MYDEBUG) MESSAGE("VVTK_Recorder::AddSkippedFrames - "<<aString);
  }
}


//----------------------------------------------------------------------------
void
VVTK_Recorder
::MakeFileAVI()
{
  myErrorStatus = 0;
  std::ostringstream aStream;
  aStream<<myNameAVIMaker<<
    " -I p"<<
    " -v 0"<<
    //" -f "<<int(myNbFPS)<<" "<<
    " -f "<<myNbFPS<<" "<<
    " -n "<<myNbWrittenFrames<<" "<<
    " -j "<<myName<<"_\%06d.jpeg "<<
    "| yuv2lav"<<
    " -o "<<myName;
   
  std::string aString(aStream.str());
  myErrorStatus = system(aString.c_str());

  if(MYDEBUG) MESSAGE("VVTK_Recorder::MakeFileAVI - "<<aString);

  QFileInfo aFileInfo(myName.c_str());
  QString aDirPath = aFileInfo.absoluteDir().path();
  QString aBaseName = aFileInfo.fileName();
  QString aCommand = 
    QString("(cd ") + aDirPath + 
    "; ls " +
    " | egrep '" + aBaseName + "_[0-9]*.jpeg'" +
    " | xargs rm " +
    ")";

  aCommand = 
    QString("rm ") + aDirPath + "/" + aBaseName + "*.jpeg";

  if(MYDEBUG) MESSAGE("VVTK_Recorder::MakeFileAVI - "<<aCommand.toLatin1().data() );
  system((const char*)aCommand.toLatin1());
}
