//  Copyright (C) 2007-2008  CEA/DEN, EDF R&D, OPEN CASCADE
//
//  Copyright (C) 2003-2007  OPEN CASCADE, EADS/CCR, LIP6, CEA/DEN,
//  CEDRAT, EDF R&D, LEG, PRINCIPIA R&D, BUREAU VERITAS
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
//
//  See http://www.salome-platform.org/ or email : webmaster.salome@opencascade.com
//
//  SALOME VTKViewer : build VTK viewer into Salome desktop
//  File   :
//  Author :
//  Module :
//  $Header$
//
#include "VVTK_Renderer.h"

#include "VISU_GaussPtsAct.h"
#include "VISU_GaussPointsPL.hxx"
#include "VISU_WidgetCtrl.hxx"
#include "VISU_PlanesWidget.hxx"
#include "VISU_SphereWidget.hxx"

#include <VTKViewer_Algorithm.h>

#include <vtkObjectFactory.h>
#include <vtkProperty.h>
#include <vtkPointPicker.h>

#include <vtkRenderWindowInteractor.h>
#include <vtkCallbackCommand.h>
#include <vtkCommand.h>
#include <vtkPlane.h>

#include <vtkPropCollection.h>
#include <vtkProp.h>
#include <vtkActor.h>
#include <vtkMapper.h>
#include <vtkPolyDataMapper.h>
#include <vtkPolyData.h>
#include <vtkTextMapper.h>
#include <vtkTextActor.h> 
#include <vtkTextProperty.h>
#include <vtkRenderer.h>

#include <vtkPropCollection.h>
#include <vtkProp.h>
#include <vtkActor.h>
#include <vtkMapper.h>
#include <vtkPolyDataMapper.h>
#include <vtkPolyData.h>
#include <vtkTextMapper.h>
#include <vtkTextActor.h> 
#include <vtkTextProperty.h>
#include <vtkRenderer.h>

#include "utilities.h"

#ifdef _DEBUG_
static int MYDEBUG = 0;
#else
static int MYDEBUG = 0;
#endif

//======================================================================
class VISU_FPSActor : public vtkTextActor 
{
public:
  vtkTypeMacro( VISU_FPSActor, vtkTextActor);
  static
  VISU_FPSActor* 
  New();

  virtual
  int
  RenderOpaqueGeometry(vtkViewport *theViewport);
};

//======================================================================
vtkStandardNewMacro(VISU_FPSActor);

//======================================================================
// function: RenderOpaqueGeometry
// purpose :
//======================================================================
int
VISU_FPSActor
::RenderOpaqueGeometry(vtkViewport *theViewport) 
{
  // It's impossible to render opaque geometry of text actor
  // if the size of the viewport is less than 1.0
  int *size = theViewport->GetSize();
  if( size[0] <= 1.0 || size[1] <= 1.0 )
    return 1;

  if(vtkRenderer *aRenderer = dynamic_cast<vtkRenderer*>(theViewport)){
    static float aTol = 1.e-6;
    float aLastRenderTimeInSeconds = aRenderer->GetLastRenderTimeInSeconds();
    if(aLastRenderTimeInSeconds > aTol){
      size_t aNumberOfCells = 0;
      VTK::ActorCollectionCopy aCopy(aRenderer->GetActors());
      if(vtkActorCollection *anActorCollection = aCopy.GetActors()){
	anActorCollection->InitTraversal();
	while(vtkActor *anActor = anActorCollection->GetNextActor()){
	  if(anActor->GetVisibility()){
	    if(SALOME_Actor *aSActor = dynamic_cast<SALOME_Actor*>(anActor)){
	      if(vtkMapper *aMapper = aSActor->GetMapper()){
		if(vtkDataSet *aDataSet = aMapper->GetInput()){
		  aNumberOfCells += aDataSet->GetNumberOfCells();
		}
	      }
	    }
	  }
	}
      }
      std::ostringstream aStr;
      float aFPS = 1.0 / aLastRenderTimeInSeconds;
      aStr<<"FPS: "<<aFPS<<"\n NumberOfCells: "<<aNumberOfCells;
      std::string anInput = aStr.str();
      SetInput(anInput.c_str());
      return Superclass::RenderOpaqueGeometry(theViewport);
    }
  }
  return 1;
}

//----------------------------------------------------------------------------
vtkStandardNewMacro(VVTK_Renderer);

//----------------------------------------------------------------------------
VVTK_Renderer
::VVTK_Renderer():
  myFPSActor(VISU_FPSActor::New()),
  myInsideCursorSettings(NULL),
  myPickingSettings(NULL),
  myGaussPointPicker(vtkPointPicker::New()),
  myGaussPreHighlightProperty(vtkProperty::New()),
  myGaussHighlightProperty(vtkProperty::New())
{
  if(MYDEBUG) INFOS("VVTK_Renderer() - "<<this);

  myFPSActor->Delete();

  vtkTextMapper* aTextMapper = vtkTextMapper::New();
  vtkTextProperty *aTextProperty = aTextMapper->GetTextProperty();
  aTextProperty->SetJustificationToRight();
  aTextProperty->SetVerticalJustificationToTop();
  aTextProperty->SetFontSize(10);

  myFPSActor->SetPickable(false); 
  myFPSActor->ScaledTextOff();
  myFPSActor->SetAlignmentPoint(8);
  myFPSActor->SetPosition2 (1., 1.);
  myFPSActor->SetMapper(aTextMapper);
  aTextMapper->Delete();

  //GetDevice()->AddActor2D(myFPSActor.GetPointer());

  myGaussPointPicker->Delete();

  myGaussPreHighlightProperty->Delete();
  myGaussPreHighlightProperty->SetColor(0,1,1);

  myGaussHighlightProperty->Delete();
  myGaussHighlightProperty->SetColor(1,1,0);

}

VVTK_Renderer
::~VVTK_Renderer()
{
  if(MYDEBUG) INFOS("~VVTK_Renderer() - "<<this);
}

//----------------------------------------------------------------------------
void
VVTK_Renderer
::AddActor(VTKViewer_Actor* theActor)
{
  Superclass::AddActor(theActor);
  if(VISU_GaussPtsAct* anActor = dynamic_cast<VISU_GaussPtsAct*>(theActor)){
    anActor->SetPointPicker(myGaussPointPicker.GetPointer());
    anActor->SetPreHighlightProperty(myGaussPreHighlightProperty.GetPointer());
    anActor->SetHighlightProperty(myGaussHighlightProperty.GetPointer());

    anActor->SetInsideCursorSettings(myInsideCursorSettings);
    //anActor->SetPickingSettings(myPickingSettings);
  }
}

//----------------------------------------------------------------------------
void
VVTK_Renderer
::RemoveActor(VTKViewer_Actor* theActor)
{
  Superclass::RemoveActor(theActor);
  if(VISU_GaussPtsAct* anActor = dynamic_cast<VISU_GaussPtsAct*>(theActor)){
    anActor->SetPointPicker(NULL);
    anActor->SetPreHighlightProperty(NULL);
    anActor->SetHighlightProperty(NULL);

    anActor->SetInsideCursorSettings(NULL);
    //anActor->SetPickingSettings(NULL);
  }
}

//----------------------------------------------------------------------------
void
VVTK_Renderer
::SetInsideCursorSettings(VISU_InsideCursorSettings* theInsideCursorSettings)
{
  myInsideCursorSettings = theInsideCursorSettings;
}

//----------------------------------------------------------------------------
void
VVTK_Renderer
::SetPickingSettings(VISU_PickingSettings* thePickingSettings)
{
  myPickingSettings = thePickingSettings;
}


//----------------------------------------------------------------------------
vtkStandardNewMacro(VVTK_Renderer1);

//----------------------------------------------------------------------------
VVTK_Renderer1::VVTK_Renderer1():
  myWidgetCtrl(VISU_WidgetCtrl::New()),
  myOutsideCursorSettings(NULL)
{
  if(MYDEBUG) INFOS("VVTK_Renderer1() - "<<this);

  myWidgetCtrl->SetPlaceFactor(1.1);
  //
  VISU_PlanesWidget *aPlanesWidget = myWidgetCtrl->GetPlanesWidget();
  aPlanesWidget->SetOutlineTranslation(false);
  vtkProperty* aSelectedPlaneProperty = aPlanesWidget->GetSelectedPlaneProperty();
  vtkProperty* aPlaneProperty = aPlanesWidget->GetPlaneProperty();
  aPlaneProperty->SetOpacity(aSelectedPlaneProperty->GetOpacity()*1.5);
  //
  //myWidgetCtrl->Delete();
}

VVTK_Renderer1
::~VVTK_Renderer1()
{
  if(MYDEBUG) INFOS("~VVTK_Renderer1() - "<<this);
  myWidgetCtrl->SetInteractor(NULL);
}

//----------------------------------------------------------------------------
void
VVTK_Renderer1
::AddActor(VTKViewer_Actor* theActor)
{
  Superclass::AddActor(theActor);
  if(VISU_GaussPtsAct1* anActor = dynamic_cast<VISU_GaussPtsAct1*>(theActor)){
    anActor->SetWidgetCtrl(GetWidgetCtrl());
    anActor->SetOutsideCursorSettings(myOutsideCursorSettings);
    AdjustWidgetCtrl();
  }
}


//----------------------------------------------------------------------------
void
VVTK_Renderer1
::RemoveActor(VTKViewer_Actor* theActor)
{
  Superclass::RemoveActor(theActor);
  if(VISU_GaussPtsAct1* anActor = dynamic_cast<VISU_GaussPtsAct1*>(theActor)){
    anActor->SetWidgetCtrl(NULL);
    anActor->SetOutsideCursorSettings(NULL);
    AdjustWidgetCtrl();
  }
}


//----------------------------------------------------------------------------
void
VVTK_Renderer1
::AdjustWidgetCtrl()
{
  VISU_PlanesWidget *aPlanesWidget = myWidgetCtrl->GetPlanesWidget();
  aPlanesWidget->InitialPlaceWidget(myBndBox);
  aPlanesWidget->SetOrigin(0.5*(myBndBox[1] + myBndBox[0]),
			   0.5*(myBndBox[3] + myBndBox[2]),
			   0.5*(myBndBox[5] + myBndBox[4]));
  //
  VISU_SphereWidget *aSphereWidget = myWidgetCtrl->GetSphereWidget();
  aSphereWidget->SetCenter(0.5*(myBndBox[1] + myBndBox[0]),
			   0.5*(myBndBox[3] + myBndBox[2]),
			   0.5*(myBndBox[5] + myBndBox[4]));
  
  float aMinLength = VTK_LARGE_FLOAT;
  for (int i=0; i<3; ++i) {
    float aLength = myBndBox[2*i+1]-myBndBox[2*i];
    aMinLength = std::min(aMinLength,aLength);
  }
  aSphereWidget->SetRadius(aMinLength);
}

//----------------------------------------------------------------------------
void 
VVTK_Renderer1
::Initialize(vtkRenderWindowInteractor* theInteractor,
	     SVTK_Selector* theSelector)
{
  SVTK_Renderer::Initialize(theInteractor,theSelector);
  myWidgetCtrl->SetInteractor(theInteractor);
}

//----------------------------------------------------------------------------
void
VVTK_Renderer1
::SetOutsideCursorSettings(VISU_OutsideCursorSettings* theOutsideCursorSettings)
{
  myOutsideCursorSettings = theOutsideCursorSettings;
}

//----------------------------------------------------------------------------
VISU_WidgetCtrl* 
VVTK_Renderer1
::GetWidgetCtrl()
{
  return myWidgetCtrl;//.GetPointer();
}

//----------------------------------------------------------------------------
bool
VVTK_Renderer1
::OnAdjustActors()
{
  return SVTK_Renderer::OnAdjustActors();
}


//----------------------------------------------------------------------------
vtkStandardNewMacro(VVTK_Renderer2);

//----------------------------------------------------------------------------
VVTK_Renderer2
::VVTK_Renderer2():
  myEventCallbackCommand(vtkCallbackCommand::New())
{
  if(MYDEBUG) INFOS("VVTK_Renderer2() - "<<this);
  myEventCallbackCommand->Delete();

  myPriority = 0.0;
  myEventCallbackCommand->SetClientData(this); 
  myEventCallbackCommand->SetCallback(VVTK_Renderer2::ProcessEvents);
}

VVTK_Renderer2
::~VVTK_Renderer2()
{
  if(MYDEBUG) INFOS("~VVTK_Renderer2() - "<<this);
}

//----------------------------------------------------------------------------
void VVTK_Renderer2::SetWidgetCtrl(VISU_WidgetCtrl* theWidgetCtrl)
{
  theWidgetCtrl->AddObserver(vtkCommand::EndInteractionEvent, 
			     myEventCallbackCommand.GetPointer(), 
			     myPriority);
  theWidgetCtrl->AddObserver(vtkCommand::EnableEvent, 
			     myEventCallbackCommand.GetPointer(), 
			     myPriority);
  theWidgetCtrl->AddObserver(vtkCommand::DisableEvent, 
			     myEventCallbackCommand.GetPointer(), 
			     myPriority);
  myWidgetCtrl = theWidgetCtrl;
}

void 
VVTK_Renderer2
::ProcessEvents(vtkObject* vtkNotUsed(theObject), 
		unsigned long theEvent,
		void* theClientData, 
		void* vtkNotUsed(theCallData))
{
  VVTK_Renderer2* self = reinterpret_cast<VVTK_Renderer2*>(theClientData);

  switch(theEvent){
  case vtkCommand::EnableEvent:
  case vtkCommand::EndInteractionEvent:
    self->OnEndInteractionEvent();  
    break;
  }
}

void
VVTK_Renderer2
::OnEndInteractionEvent()
{
  AdjustActors();
  myInteractor->Render();
}


//----------------------------------------------------------------------------
void VVTK_Renderer2::AddActor(VTKViewer_Actor* theActor)
{
  if(VISU_GaussPtsAct1* anActor = dynamic_cast<VISU_GaussPtsAct1*>(theActor)){
    if(VISU::TGaussPtsActorFactory* aFactory = anActor->GetGaussPtsFactory()){
      if(VISU_GaussPtsAct2* anActor2 = aFactory->CloneActor(anActor)){
	anActor2->SetWidgetCtrl(myWidgetCtrl);
	Superclass::AddActor(anActor2);
      }
    }
  }
}

//----------------------------------------------------------------------------
void
VVTK_Renderer2
::RemoveActor(VTKViewer_Actor* theActor)
{
  using namespace VISU;  
  if(VISU_GaussPtsAct2* anActor = dynamic_cast<VISU_GaussPtsAct2*>(theActor)){
    anActor->SetWidgetCtrl(NULL);
    Superclass::RemoveActor(theActor);
  }
}
