//  Copyright (C) 2007-2008  CEA/DEN, EDF R&D, OPEN CASCADE
//
//  Copyright (C) 2003-2007  OPEN CASCADE, EADS/CCR, LIP6, CEA/DEN,
//  CEDRAT, EDF R&D, LEG, PRINCIPIA R&D, BUREAU VERITAS
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
//
//  See http://www.salome-platform.org/ or email : webmaster.salome@opencascade.com
//
//  File   : VVTK_SegmentationCursorDlg.cxx
//  Author : Oleg Uvarov
//  Module : VISU
//
#include "VVTK_SegmentationCursorDlg.h"
#include "VVTK_PrimitiveBox.h"
#include "VVTK_SizeBox.h"

#include "VISU_GaussPtsAct.h"
#include "VISU_GaussPtsSettings.h"

#include "VISU_WidgetCtrl.hxx"
#include "VISU_PlanesWidget.hxx"
#include "VISU_SphereWidget.hxx"

#include "VISU_GaussPointsPL.hxx"
#include "VISU_OpenGLPointSpriteMapper.hxx"

#include <LightApp_Application.h>
#include <SUIT_MessageBox.h>
#include <SUIT_ResourceMgr.h>
#include <SUIT_Session.h>

#include <SVTK_RenderWindowInteractor.h>
#include <VTKViewer_Algorithm.h>

#include <vtkActorCollection.h>
#include <vtkCallbackCommand.h>
#include <vtkObjectFactory.h>
#include <vtkRenderer.h>
#include <vtkRenderWindowInteractor.h>
#include <vtkSmartPointer.h>
#include <vtkImageData.h>

#include "utilities.h"

#include <QButtonGroup>
#include <QColorDialog>
#include <QFileDialog>
#include <QGroupBox>
#include <QKeyEvent>
#include <QLabel>
#include <QLayout>
#include <QLineEdit>
#include <QPushButton>
#include <QRadioButton>
#include <QTabWidget>

#include <QtxAction.h>
#include <QtxDoubleSpinBox.h>
#include <QtxIntSpinBox.h>
#include <CAM_Module.h>

//----------------------------------------------------------------
VVTK_SegmentationCursorDlg::VVTK_SegmentationCursorDlg( QWidget* parent, const char* name )
  :QDialog( parent,
	    Qt::WindowTitleHint | Qt::WindowSystemMenuHint ),
   myEventCallbackCommand( vtkCallbackCommand::New() ),
   myInsideCursorSettings( VISU_InsideCursorSettings::New() ),
   myOutsideCursorSettings( VISU_OutsideCursorSettings::New() ),
   myIsPlaneSegmentation( true ),
   myWidgetCtrl(NULL),
   myInteractor(NULL)
{
  setAccessibleName( name );

  myPriority = 0.0;
  myEventCallbackCommand->Delete();
  myEventCallbackCommand->SetClientData(this); 
  myEventCallbackCommand->SetCallback(VVTK_SegmentationCursorDlg::ProcessEvents);

  myInsideCursorSettings->AddObserver(VISU::UpdateFromSettingsEvent, 
				      myEventCallbackCommand.GetPointer(), 
				      myPriority);

  setWindowTitle( tr( "SEGMENTATION_CURSOR_DLG_TITLE" ) );
  setSizeGripEnabled(TRUE);

  QVBoxLayout* TopLayout = new QVBoxLayout( this );
  TopLayout->setSpacing(6);
  TopLayout->setMargin(11);

  myTabBox = new QTabWidget( this );

  // Segmentation cursor pane
  mySegmentationCursorBox = new QWidget( this );
  QVBoxLayout* SCBoxLayout = new QVBoxLayout( mySegmentationCursorBox );
  SCBoxLayout->setMargin(11);
  SCBoxLayout->setSpacing(6);
  SCBoxLayout->setAlignment(Qt::AlignTop);

  // Origin
  myOriginGroup = new QGroupBox( tr( "ORIGIN_TITLE" ), mySegmentationCursorBox );
  //myOriginGroup->setColumnLayout( 0, Qt::Vertical );
  //myOriginGroup->layout()->setSpacing( 0 );
  //myOriginGroup->layout()->setMargin( 0 );

  QGridLayout* OriginGroupLayout = new QGridLayout( myOriginGroup );
  OriginGroupLayout->setAlignment(Qt::AlignTop | Qt::AlignCenter);
  OriginGroupLayout->setSpacing(6);
  OriginGroupLayout->setMargin(11);

  QLabel* XOriginLabel = new QLabel( tr( "ORIGIN_X" ), myOriginGroup );
  myXOriginSpinBox = new QtxDoubleSpinBox( -1000.0, 1000.0, 0.1, myOriginGroup );
  myXOriginSpinBox->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed ) );
  myXOriginSpinBox->setMinimumWidth( 100 );
  myXOriginSpinBox->setValue( 0.0 );

  QLabel* YOriginLabel = new QLabel( tr( "ORIGIN_Y" ), myOriginGroup );
  myYOriginSpinBox = new QtxDoubleSpinBox( -1000.0, 1000.0, 0.1, myOriginGroup );
  myYOriginSpinBox->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed ) );
  myYOriginSpinBox->setMinimumWidth( 100 );
  myYOriginSpinBox->setValue( 0.0 );

  QLabel* ZOriginLabel = new QLabel( tr( "ORIGIN_Z" ), myOriginGroup );
  myZOriginSpinBox = new QtxDoubleSpinBox( -1000.0, 1000.0, 0.1, myOriginGroup );
  //myZOriginSpinBox->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed ) );
  myZOriginSpinBox->setMinimumWidth( 100 );
  myZOriginSpinBox->setValue( 1.0 );

  OriginGroupLayout->addWidget(   XOriginLabel,   0, 0 );
  OriginGroupLayout->addWidget( myXOriginSpinBox, 0, 1 );
  OriginGroupLayout->addWidget(   YOriginLabel,   0, 2 );
  OriginGroupLayout->addWidget( myYOriginSpinBox, 0, 3 );
  OriginGroupLayout->addWidget(   ZOriginLabel,   0, 4 );
  OriginGroupLayout->addWidget( myZOriginSpinBox, 0, 5 );

  SCBoxLayout->addWidget( myOriginGroup );

  // Direction ( Plane Segmentation )
  myDirectionGroup = new QGroupBox( tr( "DIRECTION_TITLE" ), mySegmentationCursorBox );
  //myDirectionGroup->setColumnLayout( 0, Qt::Vertical );
  //myDirectionGroup->layout()->setSpacing( 0 );
  //myDirectionGroup->layout()->setMargin( 0 );

  QGridLayout* DirectionGroupLayout = new QGridLayout( myDirectionGroup );
  DirectionGroupLayout->setAlignment(Qt::AlignTop | Qt::AlignCenter);
  DirectionGroupLayout->setSpacing(6);
  DirectionGroupLayout->setMargin(11);

  QLabel* DXDirectionLabel = new QLabel( tr( "DIRECTION_DX" ), myDirectionGroup );
  myDXDirectionSpinBox = new QtxDoubleSpinBox( -1.0, 1.0, 0.1, myDirectionGroup );
  myDXDirectionSpinBox->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed ) );
  myDXDirectionSpinBox->setMinimumWidth( 100 );
  myDXDirectionSpinBox->setValue( 0.0 );

  QLabel* DYDirectionLabel = new QLabel( tr( "DIRECTION_DY" ), myDirectionGroup );
  myDYDirectionSpinBox = new QtxDoubleSpinBox( -1.0, 1.0, 0.1, myDirectionGroup );
  myDYDirectionSpinBox->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed ) );
  myDYDirectionSpinBox->setMinimumWidth( 100 );
  myDYDirectionSpinBox->setValue( 0.0 );

  QLabel* DZDirectionLabel = new QLabel( tr( "DIRECTION_DZ" ), myDirectionGroup );
  myDZDirectionSpinBox = new QtxDoubleSpinBox( -1.0, 1.0, 0.1, myDirectionGroup );
  myDZDirectionSpinBox->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed ) );
  myDZDirectionSpinBox->setMinimumWidth( 100 );
  myDZDirectionSpinBox->setValue( 1.0 );

  DirectionGroupLayout->addWidget(   DXDirectionLabel,   0, 0 );
  DirectionGroupLayout->addWidget( myDXDirectionSpinBox, 0, 1 );
  DirectionGroupLayout->addWidget(   DYDirectionLabel,   0, 2 );
  DirectionGroupLayout->addWidget( myDYDirectionSpinBox, 0, 3 );
  DirectionGroupLayout->addWidget(   DZDirectionLabel,   0, 4 );
  DirectionGroupLayout->addWidget( myDZDirectionSpinBox, 0, 5 );

  SCBoxLayout->addWidget( myDirectionGroup );

  // Depth ( Plane Segmentation )
  myDepthGroup = new QGroupBox( tr( "DEPTH_TITLE" ), mySegmentationCursorBox );
  //myDepthGroup->setColumnLayout( 0, Qt::Vertical );
  //myDepthGroup->layout()->setSpacing( 0 );
  //myDepthGroup->layout()->setMargin( 0 );

  QGridLayout* DepthGroupLayout = new QGridLayout( myDepthGroup );
  DepthGroupLayout->setAlignment(Qt::AlignTop | Qt::AlignCenter);
  DepthGroupLayout->setSpacing(6);
  DepthGroupLayout->setMargin(11);

  QLabel* DepthLabel = new QLabel( tr( "DEPTH" ), myDepthGroup );
  myDepthSpinBox = new QtxDoubleSpinBox( 0.0, VTK_LARGE_FLOAT, 0.1, myDepthGroup );
  myDepthSpinBox->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed ) );
  myDepthSpinBox->setMinimumWidth( 100 );
  myDepthSpinBox->setValue( 1.0 );

  DepthGroupLayout->addWidget(   DepthLabel,   0, 0 );
  DepthGroupLayout->addWidget( myDepthSpinBox, 0, 1 );

  SCBoxLayout->addWidget( myDepthGroup );

  // Radius ( Sphere Segmentation )
  myRadiusGroup = new QGroupBox( tr( "RADIUS_TITLE" ), mySegmentationCursorBox );
  //myRadiusGroup->setColumnLayout( 0, Qt::Vertical );
  //myRadiusGroup->layout()->setSpacing( 0 );
  //myRadiusGroup->layout()->setMargin( 0 );

  QGridLayout* RadiusGroupLayout = new QGridLayout( myRadiusGroup );
  RadiusGroupLayout->setAlignment(Qt::AlignTop | Qt::AlignCenter);
  RadiusGroupLayout->setSpacing(6);
  RadiusGroupLayout->setMargin(11);

  QLabel* RadiusLabel = new QLabel( tr( "RADIUS" ),myRadiusGroup  );
  myRadiusSpinBox = new QtxDoubleSpinBox( 0.0, 1000.0, 1.0,myRadiusGroup  );
  myRadiusSpinBox->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed ) );
  myRadiusSpinBox->setMinimumWidth( 100 );
  myRadiusSpinBox->setValue( 100.0 );

  QLabel* RatioLabel = new QLabel( tr( "RATIO" ), myRadiusGroup );
  myRatioSpinBox = new QtxDoubleSpinBox( 0.1, 10.0, 0.1,myRadiusGroup  );
  myRatioSpinBox->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed ) );
  myRatioSpinBox->setMinimumWidth( 100 );
  myRatioSpinBox->setValue( 2.0 );

  RadiusGroupLayout->addWidget( RadiusLabel,     0, 0 );
  RadiusGroupLayout->addWidget( myRadiusSpinBox, 0, 1 );
  RadiusGroupLayout->addWidget( RatioLabel,      0, 2 );
  RadiusGroupLayout->addWidget( myRatioSpinBox,  0, 3 );

  SCBoxLayout->addWidget( myRadiusGroup );

  myTabBox->addTab( mySegmentationCursorBox, tr( "SEGMENTATION_CURSOR_TAB" ) );

  // Gauss points pane
  myGaussPointsBox = new QWidget( this );
  QVBoxLayout* GPBoxLayout = new QVBoxLayout( myGaussPointsBox );
  GPBoxLayout->setMargin(11);
  GPBoxLayout->setSpacing(6);

  // Inside Gauss points
  QGroupBox* anInsideGroup = new QGroupBox( tr( "INSIDE_GAUSS_POINTS" ), myGaussPointsBox );
  //anInsideGroup->setColumnLayout( 0, Qt::Vertical );
  //anInsideGroup->layout()->setSpacing( 0 );
  //anInsideGroup->layout()->setMargin( 0 );

  QGridLayout* anInsideGroupLayout = new QGridLayout( anInsideGroup );
  anInsideGroupLayout->setAlignment(Qt::AlignTop | Qt::AlignCenter);
  anInsideGroupLayout->setSpacing(6);
  anInsideGroupLayout->setMargin(11);

  // Primitive
  myInsidePrimitiveBox = new VVTK_PrimitiveBox( anInsideGroup );

  // Size
  myInsideSizeBox = new VVTK_SizeBox( anInsideGroup );
  myInsideSizeBox->setType( VVTK_SizeBox::Inside );

  anInsideGroupLayout->addWidget( myInsidePrimitiveBox, 0, 0 );
  anInsideGroupLayout->addWidget( myInsideSizeBox, 1, 0 );

  GPBoxLayout->addWidget( anInsideGroup );

  // Outside Gauss points
  QGroupBox* anOutsideGroup = new QGroupBox( tr( "OUTSIDE_GAUSS_POINTS" ), myGaussPointsBox );
  //anOutsideGroup->setColumnLayout( 0, Qt::Vertical );
  //anOutsideGroup->layout()->setSpacing( 0 );
  //anOutsideGroup->layout()->setMargin( 0 );

  QGridLayout* anOutsideGroupLayout = new QGridLayout( anOutsideGroup );
  anOutsideGroupLayout->setAlignment(Qt::AlignTop | Qt::AlignCenter);
  anOutsideGroupLayout->setSpacing(6);
  anOutsideGroupLayout->setMargin(11);

  // Primitive
  myOutsidePrimitiveBox = new VVTK_PrimitiveBox( anOutsideGroup );

  // Size
  myOutsideSizeBox = new VVTK_SizeBox( anOutsideGroup );
  myOutsideSizeBox->setType( VVTK_SizeBox::Outside );

  anOutsideGroupLayout->addWidget( myOutsidePrimitiveBox, 0, 0 );
  anOutsideGroupLayout->addWidget( myOutsideSizeBox, 1, 0 );

  GPBoxLayout->addWidget( anOutsideGroup );

  // Magnification
  QGroupBox* MagnificationGroup = new QGroupBox ( tr( "MAGNIFICATION_TITLE" ), myGaussPointsBox );
  //MagnificationGroup->setColumnLayout(0, Qt::Vertical );
  //MagnificationGroup->layout()->setSpacing( 0 );
  //MagnificationGroup->layout()->setMargin( 0 );

  QGridLayout* MagnificationGroupLayout = new QGridLayout ( MagnificationGroup );
  MagnificationGroupLayout->setAlignment(Qt::AlignTop | Qt::AlignCenter);
  MagnificationGroupLayout->setSpacing(6);
  MagnificationGroupLayout->setMargin(11);

  myMagnificationLabel = new QLabel( tr( "MAGNIFICATION" ), MagnificationGroup );
  myMagnificationSpinBox = new QtxIntSpinBox( 1, 10000, 10, MagnificationGroup );
  myMagnificationSpinBox->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed ) );

  MagnificationGroupLayout->addWidget( myMagnificationLabel, 0, 0 );
  MagnificationGroupLayout->addWidget( myMagnificationSpinBox, 0, 1 );

  // Increment
  myIncrementLabel = new QLabel( tr( "INCREMENT" ), MagnificationGroup );
  myIncrementSpinBox = new QtxDoubleSpinBox( 0.01, 10, 0.1, MagnificationGroup );
  myIncrementSpinBox->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed ) );

  MagnificationGroupLayout->addWidget( myIncrementLabel, 0, 2 );
  MagnificationGroupLayout->addWidget( myIncrementSpinBox, 0, 3 );


  myTabBox->addTab( myGaussPointsBox, tr( "GAUSS_POINTS_TAB" ) );
  //myTabBox->showPage( myGaussPointsBox );
  myTabBox->setCurrentWidget( myGaussPointsBox );

  GPBoxLayout->addWidget( MagnificationGroup );


  // Common buttons ===========================================================
  QGroupBox* GroupButtons = new QGroupBox( this );
  //GroupButtons->setColumnLayout(0, Qt::Vertical );
  //GroupButtons->layout()->setSpacing( 0 );
  //GroupButtons->layout()->setMargin( 0 );
  QGridLayout* GroupButtonsLayout = new QGridLayout( GroupButtons );
  GroupButtonsLayout->setAlignment( Qt::AlignTop );
  GroupButtonsLayout->setSpacing( 6 );
  GroupButtonsLayout->setMargin( 11 );

  QPushButton* buttonApply = new QPushButton( tr( "&Apply" ), GroupButtons );
  buttonApply->setAutoDefault( TRUE );
  buttonApply->setDefault( TRUE );
  GroupButtonsLayout->addWidget( buttonApply, 0, 0 );
  GroupButtonsLayout->addItem( new QSpacerItem( 5, 5, QSizePolicy::Expanding, QSizePolicy::Minimum ), 0, 1 );

  QPushButton* buttonClose = new QPushButton( tr( "&Close" ) , GroupButtons );
  buttonClose->setAutoDefault( TRUE );
  GroupButtonsLayout->addWidget( buttonClose, 0, 2 );

  QPushButton* buttonHelp = new QPushButton( tr( "&Help" ) , GroupButtons );
  buttonHelp->setAutoDefault( TRUE );
  GroupButtonsLayout->addWidget( buttonHelp, 0, 3 );

  TopLayout->addWidget( myTabBox );
  TopLayout->addWidget( GroupButtons );

  connect( buttonApply, SIGNAL( clicked() ), this, SLOT( onClickApply() ) );
  connect( buttonClose, SIGNAL( clicked() ), this, SLOT( onClickClose() ) );
  connect( buttonHelp,  SIGNAL( clicked() ), this, SLOT( onClickHelp() ) );

  connect(parent, SIGNAL(Show( QShowEvent * )), this, SLOT(onParentShow()));
  connect(parent, SIGNAL(Hide( QHideEvent * )), this, SLOT(onParentHide()));
}

VVTK_SegmentationCursorDlg::~VVTK_SegmentationCursorDlg()
{
  SetWidgetCtrl(NULL);
}

float VVTK_SegmentationCursorDlg::getMagnification() const
{
  return myMagnificationSpinBox->value() / 100.0;
}

void VVTK_SegmentationCursorDlg::setMagnification( float theMagnification )
{
  myMagnificationSpinBox->setValue( ( int )( theMagnification * 100 ) );
}

float VVTK_SegmentationCursorDlg::getIncrement() const
{
  return myIncrementSpinBox->value();
}

void VVTK_SegmentationCursorDlg::setIncrement( float theIncrement )
{
  myIncrementSpinBox->setValue( theIncrement );
}


void VVTK_SegmentationCursorDlg::SetWidgetCtrl( VISU_WidgetCtrl* theWidgetCtrl )
{
  if(myWidgetCtrl == theWidgetCtrl)
    return;

  if(myWidgetCtrl)
    myWidgetCtrl->RemoveObserver(myEventCallbackCommand.GetPointer());

  myWidgetCtrl = theWidgetCtrl;

  if(theWidgetCtrl)
    theWidgetCtrl->AddObserver(vtkCommand::EndInteractionEvent, 
			       myEventCallbackCommand.GetPointer(), 
			       myPriority);
}

void VVTK_SegmentationCursorDlg::ProcessEvents(vtkObject* vtkNotUsed(theObject), 
					       unsigned long theEvent,
					       void* theClientData, 
					       void* vtkNotUsed(theCallData))
{
  VVTK_SegmentationCursorDlg* self = reinterpret_cast<VVTK_SegmentationCursorDlg*>(theClientData);

  switch(theEvent){
  case vtkCommand::EndInteractionEvent:
    self->UpdateSegmentation();
    break;
  case VISU::UpdateFromSettingsEvent:
    self->GetOutsideCursorSettings()->SetMagnification( self->GetInsideCursorSettings()->GetMagnification() );
    self->GetOutsideCursorSettings()->SetIncrement( self->GetInsideCursorSettings()->GetIncrement() );

    self->UpdateInsideGaussPoints();
    self->UpdateOutsideGaussPoints();

    self->GetInsideCursorSettings()->InvokeEvent(VISU::UpdateInsideSettingsEvent,NULL);
    self->GetOutsideCursorSettings()->InvokeEvent(VISU::UpdateOutsideSettingsEvent,NULL);
    break;
  }
}

void VVTK_SegmentationCursorDlg::UpdateSegmentation()
{
  if( myIsPlaneSegmentation )
  {
    myDirectionGroup->show();
    myDepthGroup->show();
    myRadiusGroup->hide();

    VISU_PlanesWidget *pPlanesWidget=myWidgetCtrl->GetPlanesWidget();
    vtkFloatingPointType origin[3];
    pPlanesWidget->GetOrigin( origin );
    myXOriginSpinBox->setValue( origin[0] );
    myYOriginSpinBox->setValue( origin[1] );
    myZOriginSpinBox->setValue( origin[2] );

    vtkFloatingPointType normal[3];
    pPlanesWidget->GetNormal( normal );
    myDXDirectionSpinBox->setValue( normal[0] );
    myDYDirectionSpinBox->setValue( normal[1] );
    myDZDirectionSpinBox->setValue( normal[2] );

    myDepthSpinBox->setValue( pPlanesWidget->Distance() );
  }
  else
  {
    myDirectionGroup->hide();
    myDepthGroup->hide();
    myRadiusGroup->show();
    
    VISU_SphereWidget *pSphereWidget=myWidgetCtrl->GetSphereWidget();
    vtkFloatingPointType origin[3], aRadius;
    pSphereWidget->GetCenter(origin);
    myXOriginSpinBox->setValue( origin[0] );
    myYOriginSpinBox->setValue( origin[1] );
    myZOriginSpinBox->setValue( origin[2] );
    aRadius=pSphereWidget->GetRadius();
    myRadiusSpinBox->setValue(aRadius);
    myRatioSpinBox->setValue(pSphereWidget->GetRatio());
  }
}

void VVTK_SegmentationCursorDlg::UpdateInsideGaussPoints()
{
  int aPrimitiveType = VISU_OpenGLPointSpriteMapper::PointSprite;
  vtkFloatingPointType aClamp = 200.0;
  QString aMainTexture = QString( getenv( "VISU_ROOT_DIR") ) + "/share/salome/resources/visu/sprite_texture.bmp";
  QString anAlphaTexture = QString( getenv( "VISU_ROOT_DIR") ) + "/share/salome/resources/visu/sprite_alpha.bmp";
  vtkFloatingPointType anAlphaThreshold = 0.1;
  int aResolution = 8;
  int aMinSize = 3;
  int aMaxSize = 33;
  int aMagnification = 100;
  vtkFloatingPointType anIncrement = 2.0;

  if( !myInsideCursorSettings->GetInitial() )
  {
    myInsidePrimitiveBox->setPrimitiveType( myInsideCursorSettings->GetPrimitiveType() );
    myInsidePrimitiveBox->setClamp( myInsideCursorSettings->GetClamp() );
    myInsidePrimitiveBox->setMainTexture( myInsideMainTexture );
    myInsidePrimitiveBox->setAlphaTexture( myInsideAlphaTexture );
    myInsidePrimitiveBox->setAlphaThreshold( myInsideCursorSettings->GetAlphaThreshold() );
    myInsidePrimitiveBox->setResolution( myInsideCursorSettings->GetResolution() );

    myInsideSizeBox->setMinSize( myInsideCursorSettings->GetMinSize() );
    myInsideSizeBox->setMaxSize( myInsideCursorSettings->GetMaxSize() );

    this->setMagnification( myInsideCursorSettings->GetMagnification() );
    this->setIncrement( myInsideCursorSettings->GetIncrement() );

    return;
  }

  SUIT_ResourceMgr* aResourceMgr = SUIT_Session::session()->resourceMgr();

  aPrimitiveType = aResourceMgr->integerValue( "VISU", "inside_point_sprite_primitive_type", aPrimitiveType );
  myInsidePrimitiveBox->setPrimitiveType( aPrimitiveType );

  aClamp = aResourceMgr->doubleValue( "VISU", "inside_point_sprite_clamp", aClamp );
  myInsidePrimitiveBox->setClamp( aClamp );

  aMainTexture = aResourceMgr->stringValue( "VISU", "inside_point_sprite_main_texture", aMainTexture );
  myInsidePrimitiveBox->setMainTexture( aMainTexture );

  anAlphaTexture = aResourceMgr->stringValue( "VISU", "inside_point_sprite_alpha_texture", anAlphaTexture );
  myInsidePrimitiveBox->setAlphaTexture( anAlphaTexture );

  anAlphaThreshold = aResourceMgr->doubleValue( "VISU", "inside_point_sprite_alpha_threshold", anAlphaThreshold );
  myInsidePrimitiveBox->setAlphaThreshold( anAlphaThreshold );

  aResolution = aResourceMgr->integerValue( "VISU", "inside_geom_sphere_resolution", aResolution );
  myInsidePrimitiveBox->setResolution( aResolution );

  aMinSize = aResourceMgr->integerValue( "VISU", "inside_point_sprite_min_size", aMinSize );
  myInsideSizeBox->setMinSize( aMinSize / 100.0 );

  aMaxSize = aResourceMgr->integerValue( "VISU", "inside_point_sprite_max_size", aMaxSize );
  myInsideSizeBox->setMaxSize( aMaxSize / 100.0 );

  aMagnification = aResourceMgr->integerValue( "VISU", "inside_point_sprite_magnification", aMagnification );
  this->setMagnification( aMagnification / 100.0 );

  anIncrement = aResourceMgr->doubleValue( "VISU", "inside_point_sprite_increment", anIncrement );
  this->setIncrement( anIncrement );

  myInsidePrimitiveBox->setFaceLimit( 50000 );

  ApplyInsideGaussPoints();
}

void VVTK_SegmentationCursorDlg::UpdateOutsideGaussPoints()
{
  vtkFloatingPointType aClamp = 256.0;
  int aPrimitiveType = VISU_OpenGLPointSpriteMapper::PointSprite;
  QString aMainTexture = QString( getenv( "VISU_ROOT_DIR") ) + "/share/salome/resources/visu/sprite_texture.bmp";
  QString anAlphaTexture = QString( getenv( "VISU_ROOT_DIR") ) + "/share/salome/resources/visu/sprite_alpha.bmp";
  vtkFloatingPointType anAlphaThreshold = 0.1;
  int aResolution = 8;
  int aSize = 25;
  bool aUniform = false;
  QColor aColor = Qt::blue;

  if( !myOutsideCursorSettings->GetInitial() )
  {
    myOutsidePrimitiveBox->setPrimitiveType( myOutsideCursorSettings->GetPrimitiveType() );
    myOutsidePrimitiveBox->setClamp( myOutsideCursorSettings->GetClamp() );
    myOutsidePrimitiveBox->setMainTexture( myOutsideMainTexture );
    myOutsidePrimitiveBox->setAlphaTexture( myOutsideAlphaTexture );
    myOutsidePrimitiveBox->setAlphaThreshold( myOutsideCursorSettings->GetAlphaThreshold() );
    myOutsidePrimitiveBox->setResolution( myOutsideCursorSettings->GetResolution() );

    myOutsideSizeBox->setOutsideSize( myOutsideCursorSettings->GetSize() );
    myOutsideSizeBox->setUniform( myOutsideCursorSettings->GetUniform() );

    vtkFloatingPointType* aColor = myOutsideCursorSettings->GetColor();
    myOutsideSizeBox->setColor( QColor( ( int )( aColor[0] * 255.0 ),
					( int )( aColor[1] * 255.0 ),
					( int )( aColor[2] * 255.0 ) ) );

    return;
  }

  SUIT_ResourceMgr* aResourceMgr = SUIT_Session::session()->resourceMgr();

  aPrimitiveType = aResourceMgr->integerValue( "VISU", "outside_point_sprite_primitive_type", aPrimitiveType );
  myOutsidePrimitiveBox->setPrimitiveType( aPrimitiveType );

  aClamp = aResourceMgr->doubleValue( "VISU", "outside_point_sprite_clamp", aClamp );
  myOutsidePrimitiveBox->setClamp( aClamp );

  aMainTexture = aResourceMgr->stringValue( "VISU", "outside_point_sprite_main_texture", aMainTexture );
  myOutsidePrimitiveBox->setMainTexture( aMainTexture );

  anAlphaTexture = aResourceMgr->stringValue( "VISU", "outside_point_sprite_alpha_texture", anAlphaTexture );
  myOutsidePrimitiveBox->setAlphaTexture( anAlphaTexture );

  anAlphaThreshold = aResourceMgr->doubleValue( "VISU", "outside_point_sprite_alpha_threshold", anAlphaThreshold );
  myOutsidePrimitiveBox->setAlphaThreshold( anAlphaThreshold );

  aResolution = aResourceMgr->integerValue( "VISU", "outside_geom_sphere_resolution", aResolution );
  myOutsidePrimitiveBox->setResolution( aResolution );

  aSize = aResourceMgr->integerValue( "VISU", "outside_point_sprite_size", aSize );
  myOutsideSizeBox->setOutsideSize( aSize / 100.0 );

  aUniform = aResourceMgr->booleanValue( "VISU", "outside_point_sprite_uniform", aUniform );
  myOutsideSizeBox->setUniform( aUniform );

  aColor = aResourceMgr->colorValue( "VISU", "outside_point_sprite_color", aColor );
  myOutsideSizeBox->setColor( aColor );

  myOutsidePrimitiveBox->setFaceLimit( 50000 );

  ApplyOutsideGaussPoints();
}

VISU_InsideCursorSettings* VVTK_SegmentationCursorDlg::GetInsideCursorSettings()
{
  return myInsideCursorSettings.GetPointer();
}

VISU_OutsideCursorSettings* VVTK_SegmentationCursorDlg::GetOutsideCursorSettings()
{
  return myOutsideCursorSettings.GetPointer();
}

VISU::TTextureValue 
VVTK_SegmentationCursorDlg
::MakeImageData( bool theInside,
		 const QString& theMainTexture, 
		 const QString& theAlphaTexture )
{
  if( theInside )
  {
    bool updateMainTexture = myInsideMainTexture != theMainTexture;
    bool updateAlphaTexture = myInsideAlphaTexture != theAlphaTexture;
    if( !updateMainTexture && !updateAlphaTexture )
      return 0;

    myInsideMainTexture = theMainTexture;
    myInsideAlphaTexture = theAlphaTexture;
  }
  else
  {
    bool updateMainTexture = myOutsideMainTexture != theMainTexture;
    bool updateAlphaTexture = myOutsideAlphaTexture != theAlphaTexture;
    if( !updateMainTexture && !updateAlphaTexture )
      return 0;

    myOutsideMainTexture = theMainTexture;
    myOutsideAlphaTexture = theAlphaTexture;
  }

  return VISU::GetTexture( (const char*)theMainTexture.toLatin1(), 
			   (const char*)theAlphaTexture.toLatin1());
}

void VVTK_SegmentationCursorDlg::onClickApply()
{
  if( myTabBox->currentWidget() == mySegmentationCursorBox )
    ApplySegmentationCursor();
  else
  {
    QString aWarning = "The number of faces needed to perform the 'Geometrical Sphere' primitive\n";
    aWarning.append( "presentation might be too important to ensure an acceptable frame rate.\n\n" );
    aWarning.append( "Can you please confirm that you want to continue anyway?" );
    bool toApply = CheckNumberOfFaces() ||
      SUIT_MessageBox::warning( this, tr( "Warning" ), aWarning,
			      tr( "&OK" ), tr( "&Cancel" ), 0, 1, 1 ) == 0;

    if( toApply )
    {
      ApplyInsideGaussPoints();
      ApplyOutsideGaussPoints();
    }
  }
}

void VVTK_SegmentationCursorDlg::ApplySegmentationCursor()
{
  if( myIsPlaneSegmentation )
  {
    VISU_PlanesWidget *pPlanesWidget=myWidgetCtrl->GetPlanesWidget();
    vtkFloatingPointType origin[3];
    origin[0] = myXOriginSpinBox->value();
    origin[1] = myYOriginSpinBox->value();
    origin[2] = myZOriginSpinBox->value();
    pPlanesWidget->SetOrigin( origin );

    vtkFloatingPointType normal[3];
    normal[0] = myDXDirectionSpinBox->value();
    normal[1] = myDYDirectionSpinBox->value();
    normal[2] = myDZDirectionSpinBox->value();
    
    if( normal[0] == 0.0 && normal[1] == 0.0 && normal[2] == 0.0 )
    {
      normal[2] = 1.0;
      myDZDirectionSpinBox->setValue( 1.0 );
    }
    pPlanesWidget->SetNormal( normal );

    pPlanesWidget->SetDistance( myDepthSpinBox->value() );
    
    myWidgetCtrl->InvokeEvent(vtkCommand::EndInteractionEvent,NULL);
    myWidgetCtrl->GetInteractor()->Render();
  }
  else
  {
    VISU_SphereWidget *pSphereWidget=myWidgetCtrl->GetSphereWidget();
    vtkFloatingPointType origin[3], aRadius;
    origin[0] = myXOriginSpinBox->value();
    origin[1] = myYOriginSpinBox->value();
    origin[2] = myZOriginSpinBox->value();
    pSphereWidget->SetCenter(origin);
    
    aRadius=myRadiusSpinBox->value();
    pSphereWidget->SetRadius(aRadius);

    pSphereWidget->SetRatio(myRatioSpinBox->value());

    myWidgetCtrl->InvokeEvent(vtkCommand::EndInteractionEvent,NULL);
    myWidgetCtrl->GetInteractor()->Render();
  }
}

void VVTK_SegmentationCursorDlg::ApplyInsideGaussPoints()
{
  QString anInsideMainTexture = myInsidePrimitiveBox->getMainTexture();
  QString anInsideAlphaTexture = myInsidePrimitiveBox->getAlphaTexture();
  VISU::TTextureValue aTexture = MakeImageData( true, anInsideMainTexture, anInsideAlphaTexture );

  if( aTexture.GetPointer() )
    myInsideCursorSettings->SetTexture( aTexture.GetPointer() );

  myInsideCursorSettings->SetInitial( false );
  myInsideCursorSettings->SetPrimitiveType( myInsidePrimitiveBox->getPrimitiveType() );
  myInsideCursorSettings->SetClamp( myInsidePrimitiveBox->getClamp() );
  myInsideCursorSettings->SetAlphaThreshold( myInsidePrimitiveBox->getAlphaThreshold() );
  myInsideCursorSettings->SetResolution( myInsidePrimitiveBox->getResolution() );

  myInsideCursorSettings->SetMinSize( myInsideSizeBox->getMinSize() );
  myInsideCursorSettings->SetMaxSize( myInsideSizeBox->getMaxSize() );

  myInsideCursorSettings->SetMagnification( this->getMagnification() );
  myInsideCursorSettings->SetIncrement( this->getIncrement() );

  myInsideCursorSettings->InvokeEvent(VISU::UpdateInsideSettingsEvent,NULL);
}

void VVTK_SegmentationCursorDlg::ApplyOutsideGaussPoints()
{
  QString anOutsideMainTexture = myOutsidePrimitiveBox->getMainTexture();
  QString anOutsideAlphaTexture = myOutsidePrimitiveBox->getAlphaTexture();
  VISU::TTextureValue aTexture = MakeImageData( false, anOutsideMainTexture, anOutsideAlphaTexture );

  if( aTexture.GetPointer() )
    myOutsideCursorSettings->SetTexture( aTexture.GetPointer() );

  myOutsideCursorSettings->SetInitial( false );
  myOutsideCursorSettings->SetPrimitiveType( myOutsidePrimitiveBox->getPrimitiveType() );
  myOutsideCursorSettings->SetClamp( myOutsidePrimitiveBox->getClamp() );
  myOutsideCursorSettings->SetAlphaThreshold( myOutsidePrimitiveBox->getAlphaThreshold() );
  myOutsideCursorSettings->SetResolution( myOutsidePrimitiveBox->getResolution() );

  myOutsideCursorSettings->SetSize( myOutsideSizeBox->getOutsideSize() );
  myOutsideCursorSettings->SetUniform( myOutsideSizeBox->getUniform() );

  QColor aButtonColor = myOutsideSizeBox->getColor();
  vtkFloatingPointType aColor[3];
  aColor[0] = aButtonColor.red() / 255.0;
  aColor[1] = aButtonColor.green() / 255.0;
  aColor[2] = aButtonColor.blue() / 255.0;
  myOutsideCursorSettings->SetColor( aColor );

  myOutsideCursorSettings->SetMagnification( this->getMagnification() );
  myOutsideCursorSettings->SetIncrement( this->getIncrement() );

  myOutsideCursorSettings->InvokeEvent(VISU::UpdateOutsideSettingsEvent,NULL);
}

void VVTK_SegmentationCursorDlg::onClickClose()
{
  myPlaneAction->setChecked( false );
  mySphereAction->setChecked( false );

  emit scgClose();

  reject();
}

void VVTK_SegmentationCursorDlg::onClickHelp()
{
  QString aHelpFileName = "segmentation.htm";
  LightApp_Application* app = (LightApp_Application*)(SUIT_Session::session()->activeApplication());
  if (app)
    app->onHelpContextModule(app->activeModule() ? app->moduleName(app->activeModule()->moduleName()) : QString(""), aHelpFileName);
  else {
		QString platform;
#ifdef WIN32
		platform = "winapplication";
#else
		platform = "application";
#endif
    SUIT_MessageBox::warning( 0, QObject::tr("WRN_WARNING"),
			      QObject::tr("EXTERNAL_BROWSER_CANNOT_SHOW_PAGE").
			      arg(app->resourceMgr()->stringValue("ExternalBrowser", platform)).arg(aHelpFileName),
			      QObject::tr("BUT_OK"));
  }
}

void VVTK_SegmentationCursorDlg::done( int r )
{
  myPlaneAction->setChecked( false );
  mySphereAction->setChecked( false );

  emit scgClose();

  QDialog::done( r );
}

void VVTK_SegmentationCursorDlg::onParentShow()
{
  if(myPlaneAction->isChecked() || mySphereAction->isChecked())
    show();
  else
    hide();
}

void VVTK_SegmentationCursorDlg::onParentHide()
{
  hide();
}

bool VVTK_SegmentationCursorDlg::CheckNumberOfFaces()
{
  if( !myInteractor )
    return false;

  vtkRenderer* aRenderer = myInteractor->getRenderer();

  if( !aRenderer )
    return false;

  int aNumberOfPoints = 0;

  vtkActor* anActor;
  VTK::ActorCollectionCopy aCopy(aRenderer->GetActors());
  vtkActorCollection* anActColl = aCopy.GetActors();
  for( anActColl->InitTraversal(); ( anActor = anActColl->GetNextActor() ) != NULL; )
  {
    if( VISU_GaussPtsAct1* aGaussActor = VISU_GaussPtsAct1::SafeDownCast( anActor ) )
      aNumberOfPoints += aGaussActor->GetInput()->GetNumberOfCells();
  }
  return !( myInsidePrimitiveBox->getPrimitiveType() == VISU_OpenGLPointSpriteMapper::GeomSphere &&
	    aNumberOfPoints * myInsidePrimitiveBox->getFaceNumber() * 2 > myInsidePrimitiveBox->getFaceLimit() ||
	    myOutsidePrimitiveBox->getPrimitiveType() == VISU_OpenGLPointSpriteMapper::GeomSphere &&
	    aNumberOfPoints * myOutsidePrimitiveBox->getFaceNumber() > myOutsidePrimitiveBox->getFaceLimit() );
}

void VVTK_SegmentationCursorDlg::keyPressEvent( QKeyEvent* e )
{
  QDialog::keyPressEvent( e );
  if ( e->isAccepted() )
    return;

  if ( e->key() == Qt::Key_F1 )
    {
      e->accept();
      onClickHelp();
    }
}
