# Samizdat time-limited FIFO cache
#
#   Copyright (c) 2002-2005  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'sync'

module Samizdat

class CacheEntry
  def initialize(value)
    @value = value
    @time = Time.now
    @sync = Sync.new
  end

  attr_reader :value, :time, :sync
end

class Cache

  # _timeout_ is time limit until cache entry is expired
  # (set to _nil_ to disable time limit)
  # 
  # _max_size_ is max number of objects in cache
  #
  def initialize(timeout=60*60, max_size=5000)
    @timeout = timeout
    @max_size = max_size
    @sync = Sync.new
    @cache = {}
  end

  # remove all values from cache
  #
  # if +base+ is given, only values with keys matching the base are removed
  #
  def flush(base=nil)
    @sync.synchronize(:EX) do
      if base
        @cache.delete_if {|key, value| base === key }
      else
        @cache = {}
      end
    end
  end

  # check availability
  #
  def has_key?(key)
    entry = nil   # scope fix
    @sync.synchronize(:SH) do
      unless @cache.has_key? key and @cache[key].kind_of? CacheEntry
        return nil
      end
      entry = @cache[key]
      entry.sync.lock(:EX)
    end
    begin
      if @timeout.nil? or Time.now < entry.time + @timeout
        return true
      else
        @cache.delete(key)
        return nil
      end
    ensure
      entry.sync.unlock
    end
  end

  # store value in cache
  #
  def []=(key, value)
    @sync.synchronize(:EX) do
      @cache[key] = CacheEntry.new(value)
      truncate if @cache.size > @max_size
    end
    value
  end

  # retrieve value from cache if it's still fresh
  #
  def [](key)
    @sync.synchronize(:SH) { has_key?(key) ? @cache[key].value : nil }
  end

  # initialize missing cache entry from supplied block
  #
  def fetch_or_add(key)
    @sync.synchronize(:EX) { self[key] or self[key] = yield }
  end

private

  # remove oldest item from cache
  #
  # (always run inside :EX lock)
  #
  def truncate
    oldest = @cache.keys.min {|a, b| @cache[a].time <=> @cache[b].time }
    @cache.delete(oldest)
  end
end

end   # module Samizdat
