# Samizdat message model
#
#   Copyright (c) 2002-2008  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'

class Message

  def Message.cached(id)
    return nil unless id.kind_of? Integer and id > 0
    cache.fetch_or_add(%{message/#{id}}) do
      Message.new(id)
    end
  end

  def initialize(id = nil)
    if id.kind_of? Integer
      @id = Resource.validate_id(id)

      load_from_db

      @creator = Member.cached(@creator)
      @content = Content.new(@id, @creator.login)

      @nversions, = rdf.select_one %{
SELECT count(?version)
WHERE (dct::isVersionOf ?version #{@id})}

      @translations = rdf.select_all(%{
SELECT ?lang, ?msg, ?rating
WHERE (rdf::predicate ?stmt dc::relation)
      (rdf::subject ?stmt ?msg)
      (rdf::object ?stmt focus::Translation)
      (s::inReplyTo ?msg #{@id})
      (dc::language ?msg ?lang)
      (s::rating ?stmt ?rating)
      (s::hidden ?msg ?hidden)
LITERAL ?rating > 0 #{exclude_hidden}
ORDER BY ?rating DESC}, limit_page).collect {|l, m, r| [l, m, r] }   # DBI::Row to Array
      @nreplies, = rdf.select_one %{
SELECT count(?msg)
WHERE (s::inReplyTo ?msg #{@id})
      (s::hidden ?msg ?hidden)
LITERAL ?msg IS NOT NULL #{exclude_hidden}}
      @nreplies -= @translations.size   # don't count translations

      @focuses = rdf.select_all(%{
SELECT ?focus
WHERE (rdf::subject ?stmt #{@id})
      (rdf::predicate ?stmt dc::relation)
      (rdf::object ?stmt ?focus)
      (s::rating ?stmt ?rating)
LITERAL ?rating > 0
ORDER BY ?rating DESC}, limit_page).collect {|f,| f }
      @nrelated, = rdf.select_one(%{
SELECT count(?related)
WHERE (rdf::subject ?stmt ?related)
      (rdf::predicate ?stmt dc::relation)
      (rdf::object ?stmt #{@id})
      (s::rating ?stmt ?rating)
LITERAL ?rating > 0})

    else
      @date = Time.now
      @version_of = nil
      @nversions = 0
      @translations = []
      @nreplies = 0
      @focuses = []
      @nrelated = 0
    end
  end

  attr_accessor :id, :date, :creator, :content, :lang, :desc, :parent, :open, :focuses

  attr_reader :version_of, :nversions, :translations, :nreplies, :nrelated

  def to_i
    @id.to_i
  end

  def assert_current_version
    if @version_of
      raise UserError, sprintf(
        _('Only <a href="%s">current version</a> may be used for this action'),
        @version_of)
    end
  end

  def validate_parent(parent)
    parent = Resource.validate_id(parent)
    raise UserError, _('Bad input') if parent.nil?
    parent
  end

  def validate_lang(lang)
    raise UserError, _('Specified language is not supported on this site') if
      lang and not config['locale']['languages'].include?(lang)
    lang
  end

  # expects valid reference to a current version of a message that is different
  # from this message
  #
  def validate_reference(ref)
    begin
      ref = Resource.validate_id(ref)
      (ref.kind_of?(Integer) and ref > 0) or raise ResourceNotFoundError

      message = Message.cached(ref)
      if message.id == @id
        raise UserError, _('Recursive message reference not allowed')
      end
      message.assert_current_version

    rescue ResourceNotFoundError
      raise UserError, _('Invalid message reference')
    end

    ref
  end

  def validate_desc(desc)
    validate_reference(desc) if desc
  end

  def validate_open(open)
    case open
    when 'true', true
      true
    when 'false', false, nil
      false
    else
      raise UserError, _('Invalid openForAll value')
    end
  end

  def hidden?
    if @version_of
      # inherit hidden status from current version
      Message.cached(@version_of).hidden?
    else
      @hidden
    end
  end

  def is_translation?
    @parent and @focuses.include?(Focus.translation_focus)
  end

  # find available translation to the most preferred language
  #
  # returns Message object or self, when no translation is suitable
  #
  def select_translation(accept_language)
    return self unless @lang   # don't translate messages with no language

    t_row = nil
    accept_language.each do |l|
      break if l == @lang   # message already in preferred language
      t_row = @translations.assoc(l)   # [l, m, r]
      break unless t_row.nil?
    end
    t_row ? Message.cached(t_row[1]) : self
  end

  # fixme: refactor it out to MessageHelper
  #
  # add RSS item of the message to _maker_ feed
  #
  # _maker_ is assumed to provide RSS::Maker API
  #
  def rss(maker, request)
    item = maker.items.new_item
    item.link = request.base + @id.to_s
    item.date = @date.to_time
    t = select_translation(request.accept_language)
    item.title = t.content.title
    item.dc_language = t.lang
  end

  def hide!(hide)
    db.do 'UPDATE Message SET hidden = ? WHERE id = ?', hide, @id
  end

  def reparent!(new_parent)
    # todo: validate new_parent

    rdf.assert( %{
UPDATE ?parent = :new_parent
WHERE (s::inReplyTo #{@id} ?parent)},
      { :new_parent => new_parent } )

    @parent = new_parent
  end

  def insert!
    @id, = rdf.assert( %{
INSERT ?msg
WHERE (dc::creator ?msg :creator)
      (dc::title ?msg :title)
      (dc::language ?msg :language)
      (dc::format ?msg :format)
      (dc::description ?msg :desc)
      (s::content ?msg :content)
      (s::openForAll ?msg :open)
      (s::inReplyTo ?msg :parent)},
      { :creator => @creator.id, :title => @content.title, :language => @lang,
        :format => @content.format, :desc => @desc, :content => @content.body,
        :open => (@open or false), :parent => @parent } )

    update_content
  end

  # returns id under which old version of the message is saved
  #
  def edit!
    @id or raise UserError, _('Reference to previous version lost')

    # save current version at new id
    db.do('
      INSERT INTO Message (
        version_of, creator, title, language, format, description, content,
        open, html_full, html_short)
      SELECT
        id, creator, title, language, format, description, content,
        open, html_full, html_short
      FROM Message
      WHERE id = ?', @id)
    version, = db.select_one "SELECT MAX(id) FROM Resource"
    db.do('
      UPDATE Resource
      SET published_date = (
        SELECT published_date
        FROM Resource
        WHERE id = ?)
      WHERE id = ?', @id, version)

    # write new version at old id
    db.do '
      UPDATE Message
      SET creator = ?, title = ?, format = ?, content = ?,
          language = ?, description = ?
      WHERE id = ?',
      @creator.id, @content.title, @content.format, @content.body, @lang, @desc, @id
    db.do '
      UPDATE Resource
      SET published_date = CURRENT_TIMESTAMP
      WHERE id = ?', @id

    unless @open.nil?   # change s:openForAll
      db.do 'UPDATE Message SET open = ? WHERE id = ?',
        (@open or false), @id
    end

    update_content

    version
  end

  # replace content without saving previous version
  #
  def replace!
    db.do "UPDATE Message SET creator = ?, title = ?, language = ?,
      format = ?, description = ?, content = ?, open = ? WHERE id = ?",
      @creator.id, @content.title, @lang, @content.format, @desc,
      @content.body, false, @id

# todo: make Samizdat::RDF#assert grok this 
#
#        rdf.assert( %{
#ASSERT (dc::creator ?msg :creator)
#       (dc::title ?msg :title)
#       (dc::language ?msg :lang)
#       (dc::format ?msg :format)
#       (dc::description ?msg :desc)
#       (s::content ?msg :content)
#       (s::openForAll ?msg ?open)
#WHERE (s::id ?msg #{@id})},
#          { :creator => @creator.id, :title => @content.title, :lang => @lang,
#            :format => @content.format, :desc => @desc, :content => @content.body,
#            :open => false } )

    update_content
  end

  private

  def load_from_db
    @date, @creator, @lang, @desc, @parent, @version_of, @hidden, @open =
      db.select_one(%{
SELECT r.published_date AS date,
       m.creator AS creator,
       m.language AS lang,
       m.description AS desc,
       m.parent,
       m.version_of,
       m.hidden,
       m.open
FROM Resource AS r
INNER JOIN Message AS m ON (m.id = r.id)
WHERE r.id = ?}, @id)
    raise ResourceNotFoundError, @id.to_s if @date.nil?
  end

  def load_from_rdf
    @date = rdf.get_property(@id, 'dc::date')
    raise ResourceNotFoundError, @id.to_s if @date.nil?

    @creator = rdf.get_property(@id, 'dc::creator')

    @lang = rdf.get_property(@id, 'dc::language')

    @desc = rdf.get_property(@id, 'dc::description')
    @parent = rdf.get_property(@id, 's::inReplyTo')
    @version_of = rdf.get_property(@id, 'dct::isVersionOf')

    @hidden = rdf.get_property(@id, 's::hidden')
    @open = rdf.get_property(@id, 's::openForAll')
  end

  def update_content
    @content = Content.new(@id, @creator.login, @content.title, @content.format, @content.body)
    @content.update_html
  end
end
