#!/usr/bin/env ruby
#
# Samizdat message helper tests
#
#   Copyright (c) 2002-2008  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'test/unit'
require 'rexml/document'
require 'samizdat/engine'
# let mocks overwrite originals
require 'test/mock'
require 'test/util'

require 'samizdat/helpers/message_helper'

class TC_MessageHelper < Test::Unit::TestCase
  include MessageHelper

  def setup
    # create a dummy session
    @request = MockRequest.new
    @session = @request.session
  end

  def teardown
    @request = nil
    @session = nil
  end

  def test_message_buttons
    m = MockMessage.new(:id => 'upload')
    @request.permissions['post'] = true
    root = parse(message_buttons(m))
    assert_equal ['message/upload/reply', 'message/upload/request_moderation'],
      elements(root, '/div[@class="foot"]/a[@class="action"]', 'href')
  end

  def test_message_info_date
    m = MockMessage.new(:id => 'upload')
    message_info(m, :full)
    m = MockMessage.new(:id => 'upload', :date => '2004-03-11')
    message_info(m, :full)
    m = MockMessage.new(:id => 'upload', :date => nil)
    message_info(m, :full)
    date = 'current time'
    class << date   # check duck typing
      def to_time
        Time.now
      end
    end
    m = MockMessage.new(:id => 'upload', :date => date)
    message_info(m, :full) =~ /<\/a> on (.*)$/
    assert_not_equal date, $1
  end

  def test_message_info_parent
    m = MockMessage.new(:id => 'upload', :parent => 2)
    assert_equal true,
      message_info(m, :full).split(",\n ")[1].include?('href="2"')
  end

  def test_message_content
    m = MockMessage.new
    assert_equal nil, message_content(m, :list)

    m.content = MockContent.new(:body => "line 1\n\nline 2")
    root = parse(box(nil, message_content(m, :full)))
    assert_equal ['line 1', 'line 2'], elements(root, 'div/p', nil)

    m.content = MockContent.new(:body => 'test', :format => 'text/plain')
    assert_equal 'pre', parse(message_content(m, :full)).name

    m.content = MockContent.new(:format => 'application/x-squish', :body => 'SELECT')
    root = parse(message_content(m, :full))
    assert_equal ['q', nil, 'run'], elements(root, 'div/*', 'name')

    m.content = MockContent.new(:format => 'image/jpeg', :title => 'A', :file_path => 'anarchy.png')
    root = parse(message_content(m, :full))
    assert_equal 'A', root.attributes['alt']
    assert_equal 'content/anarchy.png', root.attributes['src']

    m.content = MockContent.new(:format => 'application/pdf')
    root = parse(message_content(m, :full))
    assert_equal ['application/pdf content'], elements(root, 'a')
  end
end
