# Samizdat antispam filter
#
#   Copyright (c) 2002-2009  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 3 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'open-uri'

class Antispam

  # only supply option values from safe source
  #
  # valid options:
  #
  # * check_roles - list of roles that should be subjected to antispam
  #   filtering
  #
  # * url - a URL from where a list of filtering regexps can be downloaded
  #
  # * exclude - a list of regexps used to drop items from regexp list
  #
  def initialize(options)
    if options.kind_of?(Hash) and options['url'] and
      options['check_roles'].kind_of?(Array) and options['check_roles'].size > 0

      @roles = options['check_roles']
      @regexp_list = load_regexp_list(options)
    else
      @roles = []
    end
  end

  # check against a list of known spam patterns if it's enabled for this role
  #
  def spam?(role, text)
    return false unless @roles.include?(role)

    @regexp_list.each do |re|
      if text =~ re
        return true
      end
    end

    false
  end

  private

  # NB: define Kernel#log_exception to report failure to load the list of
  # regexps from the supplied URL
  #
  def load_regexp_list(options)
    if options['exclude'].kind_of? Array
      exclude = Regexp.new(
        options['exclude'].collect {|s| Regexp.escape(s) }.join('|')
      ).freeze
    end

    begin
      Kernel.open(options['url'].untaint) {|f| f.read }

    rescue => error
      # report error when running from under Samizdat
      log_exception(error) if Kernel.respond_to? :log_exception

      # default to 68-char "GTUBE" spamstring, see
      # http://spamassassin.apache.org/gtube/
      'XJS\*C4JDBQADN1\.NSBN3\*2IDNEN\*GTUBE-STANDARD-ANTI-UBE-TEST-EMAIL\*C\.34X'
    end.split(/[\r\n]+/).collect do |line|
      line.gsub!(/\s*#.*\z/, '')
      if line.size > 6 and
        line !~ /\(\?\</ and   # (?< ) not supported in Ruby
        (exclude.nil? or line !~ exclude)

        line
      end
    end.compact.collect do |line|
      Regexp.new(line).freeze
    end
  end
end
