#  Copyright (C) 1999-2004
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc HVLoadURL {tt url query} {
    global hv
    global debug
    global message

    # this assumes the url has been already resolved
    if {$debug(tcl,hv)} {
	puts "HVLoadURL $tt $url $query"
    }

    # do we have anything?
    if {$url == ""} {
	return
    }

    # set the cursor to busy
    $hv($tt,widget) configure -cursor watch

    # parse url
    ParseURL $url r
    if {$debug(tcl,hv)} {
	puts "HVLoadURL parse url ;$r(scheme);$r(authority);$r(path);$r(query);$r(fragment);$query"
    }

    # check if have we been here before
    set prev ";$r(scheme);$r(authority);$r(path);$r(query);$query"
    if {$hv($tt,previous) == $prev} {
	if {$debug(tcl,hv)} {
	    puts "HVLoadURL found prev $prev"
	}
	HVSet $tt $url $query $r(fragment)
	HVGotoHTML $tt
	return
    }
    set hv($tt,previous) $prev

    switch -- $r(scheme) {
	file -
	{} {HVProcessURLFile $tt $url $query r}
	ftp {HVProcessURLFTP $tt $url $query r}
	http {HVProcessURLHTTP $tt $url $query r}
	default {
	    $hv($tt,widget) configure -cursor {}
	    Error "$message(error,hvsup,scheme) $r(scheme)"
	}
    }
}

proc HVProcessURLFile {tt url query rr} {
    upvar $rr r
    global hv
    global debug
    global ds9

    if {$debug(tcl,hv)} {
	puts "HVProcessURLFile"
    }

    if [file exists $r(path)] {
	if [file isdirectory $r(path)] {
	    HVSetAll $tt $url {} {} {} \
		[HVFileHtmlList $r(path) [HVDirList $r(path)]] {} \
		"text/html" {} 200 {}

	    set hv($tt,active) 1
	    set hv($tt,delete) 0
	    HVParse $tt
	} else {
	    HVSet $tt $url {} $r(fragment)
	    set hv($tt,active) 1
	    set hv($tt,delete) 0
	    HVLoadFile $tt $r(path)
	}
    }
}

proc HVProcessURLFTP {tt url query rr} {
    upvar $rr r
    global hv
    global debug
    global ds9

    if {$debug(tcl,hv)} {
	puts "HVProcessURLFTP"
    }

    set fn "$ds9(tmpdir)/[file tail $r(path)]"
    set ftp [ftp::Open $r(authority) "ftp" "ds9@"]
    if {$ftp > -1} {
	# first try to get as file
	set ftp::VERBOSE $debug(tcl,ftp)
	set "ftp::ftp${ftp}(Output)" FTPLog
	ftp::Type $ftp binary
	if [ftp::Get $ftp $r(path) "$fn"] {
	    ftp::Close $ftp

	    HVSet $tt $url {} $r(fragment)

	    set hv($tt,active) 1
	    set hv($tt,delete) 1
	    HVLoadFile $tt "$fn"
	} else {
	    # from the prev attempt
	    catch {file delete -force "$fn"}

	    # now as a directory
	    set list [ftp::List $ftp $r(path)]
	    ftp::Close $ftp

	    HVSetAll $tt $url {} {} {} \
		[HVFTPHtmlList $r(authority) $r(path) $list] {} \
		"text/html" {} 200 {}

	    set hv($tt,active) 1
	    set hv($tt,delete) 0
	    HVParse $tt
	}
    }
}

proc HVProcessURLHTTP {tt url query rr} {
    upvar $rr r
    global hv
    global debug
    global ds9
    global message

    if {$debug(tcl,hv)} {
	puts "HVProcessURLHTTP"
    }

    HVSet $tt $url $query $r(fragment)

    # cmd
    set cmd "http::geturl \{$url\} -progress \"HVProgress $tt\" -binary 1 -headers \{[ProxyHTTP]\}" 
    if {[string length $query] != 0} {
	append cmd " -query \{$query\}"
    }

    # stop any refresh
    if [info exists hv($tt,refresh,id)] {
	if {$hv($tt,refresh,id)>0} {
	    after cancel $hv($tt,refresh,id)
	}
    }
    set hv($tt,html) {}
    set hv($tt,meta) {}
    set hv($tt,refresh,time) 0
    set hv($tt,refresh,url) {}
    set hv($tt,refresh,id) 0
    set hv($tt,fn) {}
    set hv($tt,token) {}
    set hv($tt,ch) {}

    # do we have html? if so, use a var
    ParseURL $url r
    switch -- [file extension $r(path)] {
	".html" -
	".htm" {
	    # geturl in var
	    if {![catch {set hv($tt,token) [eval $cmd -command HVProcessURLHTTPVarFinish]}]} {
		set hv($tt,active) 1
		set hv($tt,delete) 0

		# set $tt
		upvar #0 $hv($tt,token) t
		set hv($t(url)) $tt

		$hv($tt,mb).view entryconfig "Stop" -state normal
	    } else {
		Error "$message(error,hvsup,url) $url"
	    }
	}
	default {
	    # geturl as file
	    set hv($tt,fn) [tmpnam ds9 .http]
	    if [catch {open "$hv($tt,fn)" w} hv($tt,ch)] {
		Error "Unable to open tmp file $hv($tt,fn) for writing"
		return
	    }

	    if {![catch {set hv($tt,token) [eval $cmd -channel $hv($tt,ch) -command HVProcessURLHTTPFileFinish]}]} {
		set hv($tt,active) 1
		set hv($tt,delete) 1

		# set $tt
		upvar #0 $hv($tt,token) t
		set hv($t(url)) $tt

		$hv($tt,mb).view entryconfig "Stop" -state normal
	    } else {
		catch {close $hv($tt,ch)}
		Error "$message(error,hvsup,url) $url"
		return
	    }
	}
    }
}

proc HVProcessURLHTTPVarFinish {token} {
    global hv
    global debug
    global message

    if {$debug(tcl,hv)} {
	puts "HVLoadURLVarFinish"
    }

    upvar #0 $token t

    # find $tt
    if {![info exists hv($t(url))]} {
	Error "$message(error,hvsup,http)"
	return
    }
    set tt $hv($t(url))
    unset hv($t(url))

    # Code
    set hv($tt,code) [http::ncode $token]
    if {$debug(tcl,hv)} {
	puts "HVLoadURLVarFinish code: $hv($tt,code)"
    }

    # Meta
    set hv($tt,meta) $t(meta)
    if {$debug(tcl,hv)} {
	puts "HVLoadURLVarFinish meta: $hv($tt,meta)"
    }

    # Mime-type
    # we want to strip and extra info after ';'
    regexp -nocase {([^;])*} $t(type) hv($tt,mime)
    if {$debug(tcl,hv)} {
	puts "HVLoadURLVarFinish mime: $hv($tt,mime)"
    }

    # Content-Encoding / Refresh
    foreach {name value} $hv($tt,meta) {
	if {[regexp -nocase ^content-encoding $name]} {
	    switch -- $value {
		gzip -
		x-gzip {set hv($tt,encoding) gzip}
		compress -
		Z {set hv($tt,encoding) compress}
		pack -
		z {set hv($tt,encoding) pack}
		default {}
	    }
	}

	if {[regexp -nocase ^Refresh $name]} {
	    set f [split $value \;]
	    set hv($tt,refresh,time) [lindex $f 0]
	    set hv($tt,refresh,url) [string [lindex $f 1] 4 end]
	    if {$hv($tt,refresh,url) != {} & $hv($tt,refresh,time) != {}} {
		if {$debug(tcl,hv)} {
		    puts "HVLoadURLVarFinish Refresh $hv($tt,refresh,time) $hv($tt,refresh,url)"
		}
		set hv($tt,previous) {}
		set hv($tt,refresh,id) [after [expr $hv($tt,refresh,time)*1000] "HVLoadURL $tt \{$hv($tt,refresh,url)\} {}"]

	    } else {
		set hv($tt,refresh,id) 0
	    }
	}
    }
	   
    # html
    set hv($tt,html) [http::data $token]

    HTTPLog $token
    http::cleanup $token
    
    $hv($tt,mb).view entryconfig "Stop" -state disabled

    # do this last, since we may reenter
    HVParse $tt
}

proc HVProcessURLHTTPFileFinish {token} {
    global hv
    global debug
    global message

    if {$debug(tcl,hv)} {
	puts "HVLoadURLFileFinish"
    }

    upvar #0 $token t

    # find $tt
    if {![info exists hv($t(url))]} {
	Error "$message(error,hvsup,http)"
	return
    }
    set tt $hv($t(url))
    unset hv($t(url))

    catch {close $hv($tt,ch)}

    # Code
    set hv($tt,code) [http::ncode $token]
    if {$debug(tcl,hv)} {
	puts "HVLoadURLFileFinish code: $hv($tt,code)"
    }

    # Meta
    set hv($tt,meta) $t(meta)
    if {$debug(tcl,hv)} {
	puts "HVLoadURLFileFinish meta: $hv($tt,meta)"
    }

    # Mime-type
    # we want to strip and extra info after ';'
    regexp -nocase {([^;])*} $t(type) hv($tt,mime)
    if {$debug(tcl,hv)} {
	puts "HVLoadURLFileFinish mime: $hv($tt,mime)"
    }

    # Content-Encoding
    foreach {name value} $hv($tt,meta) {
	if {[regexp -nocase ^content-encoding $name]} {
	    switch -- $value {
		gzip -
		x-gzip {set hv($tt,encoding) gzip}
		compress -
		Z {set hv($tt,encoding) compress}
		pack -
		z {set hv($tt,encoding) pack}
		default {}
	    }
	}

	if {[regexp -nocase ^Refresh $name]} {
	    set f [split $value \;]
	    set hv($tt,refresh,time) [lindex $f 0]
	    set hv($tt,refresh,url) [string range [lindex $f 1] 4 end]
	    if {$hv($tt,refresh,url) != {} & $hv($tt,refresh,time) != {}} {
		if {$debug(tcl,hv)} {
		    puts "HVLoadURLVarFinish Refresh $hv($tt,refresh,time) $hv($tt,refresh,url)"
		}
		set hv($tt,previous) {}
		set hv($tt,refresh,id) [after [expr $hv($tt,refresh,time)*1000] "HVLoadURL $tt \{$hv($tt,refresh,url)\} {}"]

	    } else {
		set hv($tt,refresh,id) 0
	    }
	}
    }

    HTTPLog $token
    http::cleanup $token
    
    $hv($tt,mb).view entryconfig "Stop" -state disabled

    # do this last, since we may reenter
    HVParse $tt
}

proc HVCancel {tt} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVCancel"
    }

    # stop any refresh
    if {$hv($tt,refresh,id)>0} {
	after cancel $hv($tt,refresh,id)
	set hv($tt,refresh,id) 0
    }

    if {$hv($tt,active)} {
	# clean up
	HVClearTmpFile $tt
	set hv($tt,active) 0

	# reset willl call FinishURL and we can't feed extra params
	http::reset $hv($tt,token)

	$hv($tt,widget) configure -cursor {}
    }
}

proc HVLoadFile {tt path} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVLoadFile $path"
    }

    set hv($tt,html) {}
    set hv($tt,fn) $path
    set hv($tt,code) 200
    set hv($tt,meta) {}
    set hv($tt,refresh,time) 0
    set hv($tt,refresh,url) {}
    set hv($tt,refresh,id) 0

    # set content-encoding
    switch -- [string tolower [file extension $path]] {
	.gz {
	    set path [file rootname $path]
	    set hv($tt,encoding) gzip
	}
	.Z {
	    set path [file rootname $path]
	    set hv($tt,encoding) compress
	}
	.z {
	    set path [file rootname $path]
	    set hv($tt,encoding) pack
	}
	default {set hv($tt,encoding) {}}
    }

    # set mime-type
    switch -- [string tolower [file extension $path]] {
	.html -
	.htm {set hv($tt,mime) "text/html"}
	.gif {set hv($tt,mime) "image/gif"}
	.jpeg -
	.jpg {set hv($tt,mime) "image/jpeg"}
	.tiff -
	.tif {set hv($tt,mime) "image/tiff"}
	.png {set hv($tt,mime) "image/png"}
	.bmp {set hv($tt,mime) "image/bmp"}
	.ppm {set hv($tt,mime) "image/ppm"}
	.xbm {set hv($tt,mime) "image/xbm"}

	.fits -
	.fit -
	.fts {set hv($tt,mime) "image/fits"}

	.ftz -
	.fits.gz -
	.fgz {
	    set hv($tt,mime) "image/fits"
	    set hv($tt,encoding) "gzip"
	}

	.text -
	.txt {set hv($tt,mime) "text/plain"}
	default {
	    switch -- $hv($tt,encoding) {
		gzip -
		compress -
		pack {set hv($tt,mime) "application/octet-stream"}
		default {set hv($tt,mime) "text/plain"}
	    }
	}
    }

    HVParse $tt
}

proc HVParse {tt} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVParse mime-type:$hv($tt,mime) content-encoding:$hv($tt,encoding)"
    }

    if {!$hv($tt,active)} {
	if {$debug(tcl,hv)} {
	    puts "HVParse not active- abort"
	}

	$hv($tt,widget) configure -cursor {}
	return
    }

    set result 1
    switch -- "$hv($tt,mime)" {
	"multipart/x-mixed-replace" -
	"text/html" {set result [HVParseHTML $tt]}

	"text/plain" {HVParseText $tt}

	"image/gif" -
	"image/jpeg" -
	"image/tiff" -
	"image/png" -
	"image/bmp" -
	"image/ppm" -
	"image/x-portable-pixmap" -
	"image/xbm" -
	"image/x-xbitmap" {HVParseImg $tt}

	"image/fits" -
	"application/fits" {HVParseFITS $tt}

	"application/fits-image" -
	"application/fits-table" -
	"application/fits-group" {HVParseFITS $tt}

	"image/x-fits" -
	"binary/x-fits" -
	"application/x-fits" {HVParseFITS $tt}

	"image/x-gfits" -
	"binary/x-gfits" -
	"image/gz-fits" -
	"application/x-gzip" -
	"display/gz-fits" {
	    set hv($tt,encoding) gzip
	    HVParseFITS $tt
	}

	"image/x-cfits" -
	"binary/x-cfits" {
	    set hv($tt,encoding) compress
	    HVParseFITS $tt
	}

	"image/fits-hcompress" -
	"image/x-fits-h" {HVParseSave $tt}

	"application/octet-stream" {
	    # one last chance to grap it as a fits file

	    ParseURL $hv($tt,url) r
	    set path [file tail $r(path)]  

	    # set content-encoding
	    switch -- [file extension $path] {
		.gz {
		    set path [file rootname $path]
		    set hv($tt,encoding) gzip
		}
		.Z {
		    set path [file rootname $path]
		    set hv($tt,encoding) compress
		}
		.z {
		    set path [file rootname $path]
		    set hv($tt,encoding) pack
		}
		default {set hv($tt,encoding) {}}
	    }

	    # set mime-type
	    switch -- [file extension $path] {
		.fits -
		.fit -
		.fts {
		    set hv($tt,mime) "image/fits"
		    HVParseFITS $tt
		}

		.ftz -
		.fgz {
		    set hv($tt,mime) "image/fits"
		    set hv($tt,encoding) "gzip"
		    HVParseFITS $tt
		}
		default {HVParseSave $tt}
	    }
	}

	default {HVParseSave $tt}
    }

    # something wrong?
    if {!$result} {
	return
    }

    HVClearStatus $tt
    set hv($tt,active) 0
    HVClearTmpFile $tt
}

proc HVParseHTML {tt} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVParseHTML"
    }

    if {[string length $hv($tt,html)] == 0} {
	if [file exists "$hv($tt,fn)"] {
	    if [catch {open "$hv($tt,fn)" r} ch] {
		Error "Unable to open file $hv($tt,fn) for reading"
		return
	    }
	    set hv($tt,html) [read $ch]
	    close $ch
	}
    }

    # figure out the base
    # we don't want any query or fragments
    ParseURL $hv($tt,url) r

    set base {}
    # scheme
    if {[string length $r(scheme)] != 0} {
	append base "$r(scheme)://"
    }
    # authority
    if {[string length $r(authority)] != 0} {
	append base "$r(authority)"
    }
    # path
    if {[string length $r(path)] != 0} {
	append base "$r(path)"
    } else {
	append base "/"
    }
    # query
    if {[string length $r(query)] != 0} {
	append base "?$r(query)"
    }
    $hv($tt,widget) config -base $base

    if {$debug(tcl,hv)} {
	DumpURL r
	puts "HVParseHTML base [$hv($tt,widget) cget -base]"
    }

    # is it a redirection?
    switch -- $hv($tt,code) {
	301 -
	302 -
	303 -
	304 {
	    # look for Location in meta data
	    foreach {name value} $hv($tt,meta) {
		if {[regexp -nocase ^location$ $name]} {
		    if {$debug(tcl,hv)} {
		      puts "HVParse redirect $hv($tt,code) from meta to $value"
		    }
		    HVClearTmpFile $tt
		    HVLoadURL $tt [$hv($tt,widget) resolve $value] {}
		    return 0
		}
	    }
	    # check html page
	    if {[regexp -nocase {.*<a href=\"([^\"]+)} $hv($tt,html) x url]} {
		if {$debug(tcl,hv)} {
		    puts "HVParse redirect $hv($tt,code) from html to $url"
		}
		HVClearTmpFile $tt
		HVLoadURL $tt [$hv($tt,widget) resolve $url] {}
		return 0
	    }
	}
    }

    # we have a valid html
    HVClearWidget $tt

    # fix forms with no action
    HVFixHTMLForm $tt

    # and now, parse it
    $hv($tt,widget) parse $hv($tt,html)

    HVGotoHTML $tt

    # success
    return 1
}

proc HVParseImg {tt} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVParseImg"
    }

    # fake html 
    set html \
	"<html>\n<body>\n<img src=\"$hv($tt,fn)\" border=0>\n</body>\n</html>"
    HVSetAll $tt "$hv($tt,fn)" {} {} {} $html "$hv($tt,fn)" "text/html" {} 200 {}

    HVClearWidget $tt

    $hv($tt,widget) config -base {}
    $hv($tt,widget) parse $hv($tt,html)
    HVGotoHTML $tt
}

proc HVParseFITS {tt} {
    global hv
    global debug
    global ds9

    if {$debug(tcl,hv)} {
	puts "HVParseFITS mime-type:$hv($tt,mime) content-encoding:$hv($tt,encoding)"
    }

    if {$hv($tt,delete) && $hv($tt,save)} {
	switch -- $hv($tt,encoding) {
	    gzip {FileLast hvfitsfbox "ds9.fits.gz"}
	    compress {FileLast hvfitsfbox "ds9.fits.Z"}
	    pack {FileLast hvfitsfbox "ds9.fits.z"}
	    default {FileLast hvfitsfbox "ds9.fits"}
	}

	set fn [SaveFileDialog hvfitsfbox]
	if {[string length "$fn"] != 0} {
	    if {![catch {file rename -force "$hv($tt,fn)" "$fn"}]} {
		set hv($tt,fn) "$fn"
		set hv($tt,delete) 0
	    }
	}
    }

    switch -- $hv($tt,frame) {
	new {
	    set ds9(display,user) tile
	    DisplayMode
	    CreateFrame
	}
	current {}
    }

    StartLoad
    global loadParam
    set loadParam(load,type) allocgz
    set loadParam(file,type) fits
    set loadParam(file,mode) "$hv($tt,file,mode)"
    set loadParam(file,name) "$hv($tt,fn)"

    # may have to convert the file, based on content-encoding
    switch -- "$hv($tt,encoding)" {
	compress {
	    catch {set ch [open "| uncompress < $hv($tt,fn) " r]}
	    set loadParam(load,type) channel
	    set loadParam(channel,name) $ch
	}
	pack {
	    catch {set ch [open "| pcat $hv($tt,fn) " r]}
	    set loadParam(load,type) channel
	    set loadParam(channel,name) $ch
	}
    }

    ProcessLoad
    FinishLoad

    HVClearTmpFile $tt
    HVClearAll $tt
    HVUpdateDialog $tt
}

proc HVParseText {tt} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVParseText"
    }

    if {[string length $hv($tt,html)] == 0} {
	if [file exists "$hv($tt,fn)"] {
	    catch {
		if [catch {open "$hv($tt,fn)" r} ch] {
		    Error "Unable to open file $hv($tt,fn) for reading"
		    return
		}
		set hv($tt,html) [read $ch]
		close $ch
	    }
	}
    }

    set hv($tt,html) \
	"<html>\n<body>\n<pre>\n$hv($tt,html)\n</pre>\n</body>\n</html>"
    set hv($tt,mime) "text/html"
    set hv($tt,encoding) {}
    set hv($tt,code) 200
    set hv($tt,meta) {}
    set hv($tt,refresh,time) 0
    set hv($tt,refresh,url) {}
    set hv($tt,refresh,id) 0
    HVParseHTML $tt
}

proc HVParseSave {tt} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVParseSave"
    }

    ParseURL $hv($tt,url) r
    FileLast hvfitsfbox [file tail $r(path)]
    set fn [SaveFileDialog hvfitsfbox]
    if {[string length "$fn"] != 0} {
	if {![catch {file rename -force "$hv($tt,fn)" "$fn"}]} {
	    set hv($tt,delete) 0
	}
    }

    HVClearAll $tt
    HVUpdateDialog $tt
}

proc HVGotoHTML {tt} {
    global hv
    global debug

    incr hv($tt,index)
    set hv($tt,index,$hv($tt,index)) "$hv($tt,url) $hv($tt,query)"

    if {$debug(tcl,hv)} {
	puts "HVGotoHTML $hv($tt,index) $hv($tt,fragment)"
    }

    if {[string length $hv($tt,fragment)] != 0} {
	global debug
	if {$debug(tcl,idletasks)} {
	    puts "HVGotoHTML"
	}
	update idletasks

	$hv($tt,widget) yview $hv($tt,fragment)
    } else {
	$hv($tt,widget) yview moveto 0
    }

    HVUpdateDialog $tt
}

proc HVClearWidget {tt} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVClearWidget"
    }

    $hv($tt,widget) clear
    foreach x [array names hv] {
	set f [split $x ,]
	if {[lindex $f 0] == $tt && [lindex $f 1] == "images"} {
	    image delete $hv($x)
	    unset hv($x)
	}
    }
}

proc HVClearIndex {tt n} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVClearIndex $tt $n"
    }

    foreach x [array names hv] {
	set f [split $x ,]
	if {[lindex $f 0] == $tt && 
	    [lindex $f 1] == "index" &&
	    [lindex $f 2] > $n} {
	    unset hv($x)
	}
    }
    set hv($tt,index) $n
}

proc HVClearTmpFile {tt} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVClearTmpFile"
    }

    if {$hv($tt,delete) && [string length "$hv($tt,fn)"] != 0} {
	if [file exists "$hv($tt,fn)"] {
	    if {$debug(tcl,hv)} {
		puts "HVClearTmpFile delete $hv($tt,fn)"
	    }
	    file delete "$hv($tt,fn)"
	}
	set hv($tt,fn) {}
	set hv($tt,delete) 0
    }
}

proc HVClearStatus {tt} {
    global hv
    global debug

    $hv($tt,top).s.status configure -text {}
}

proc HVUpdateDialog {tt} {
    global hv
    global debug

    # in case we've set the cursor
    $hv($tt,widget) configure -cursor {}

    if {$debug(tcl,hv)} {
	puts "HVUpdateDialog\n"
    }

    set id $hv($tt,index)
    set id [incr id -1]
    if {[info exists hv($tt,index,$id)]} {
	$hv($tt,mb).view entryconfig "Back" -state normal    
    } else {
	$hv($tt,mb).view entryconfig "Back" -state disabled
    }

    set id $hv($tt,index)
    set id [incr id 1]
    if {[info exists hv($tt,index,$id)]} {
	$hv($tt,mb).view entryconfig "Forward" -state normal    
    } else {
	$hv($tt,mb).view entryconfig "Forward" -state disabled
    }
}

proc HVRefresh {tt} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVRefresh"
    }

    set hv($tt,active) 1
    set hv($tt,delete) 0
    HVParse $tt
}

proc HVStatus {tt str} {
    global hv
    global debug

    if {[string length $str] > 0} {
	$hv($tt,top).s.status configure -text $str
    } else {
	$hv($tt,top).s.status configure -text {}
    }
}

proc HVProgress {tt token totalsize currentsize} {
    global hv
    global debug
    
    # HVProgress can be called, even after the window has been destroyed
    if {![info exist hv($tt,top)]} {
	return
    }

    if {!$hv($tt,active)} {
	return
    }

    if {$totalsize != 0} {
	HVStatus $tt "$currentsize bytes of $totalsize bytes [expr int(double($currentsize)/$totalsize*100)]%"
    } else {
	HVStatus $tt "$currentsize bytes"
    }
}

proc HVFTPHtmlList {host path list} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVFTPHtmlList $host $path"
    }
    if {[string range $path end end] != "/"} {
	append path {/}
    }

    set html {}
    append html "<html>\n"
    append html "<head>\n"
    append html "<title>Index of $path on $host</title>\n"
    append html "</head>\n"
    append html "<body>\n"
    append html "<h1>Index on $path on $host</h1>\n"
    append html "<hr>\n"
    append html "<pre>\n"
    foreach l $list {
	switch -- [llength $l] {
	    8 {set offset 4}
	    9 {set offset 5}
	    10 {set offset 4}
	    11 {set offset 5}
	    default {set offset 5}
	}

	set ii [lindex $l [expr $offset+3]]
	switch -- [string range $l 0 0] {
	    d {
		set new "<a href=\"ftp://$host$path$ii/\">$ii</A>"
	    }
	    l {
		set new "<a href=\"ftp://$host$path$ii\">$ii</A>"
	    }
	    default {
		set new "<a href=\"ftp://$host$path$ii\">$ii</A>"
	    }
	}

	regsub $ii $l $new l
	append html "$l\n"
    }
    append html "</pre>\n"
    append html "</hr>\n"
    append html "</body>\n"

    return $html
}

proc HVFileHtmlList {path list} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVFileHtmlList $path"
    }

    if {[string range $path end end] != "/"} {
	append path {/}
    }

    set html {}
    append html "<html>\n"
    append html "<head>\n"
    append html "<title>Index of $path</title>\n"
    append html "</head>\n"
    append html "<body>\n"
    append html "<h1>Index on $path</h1>\n"
    append html "<hr>\n"
    append html "<pre>\n"
    foreach l $list {
	switch -- [llength $l] {
	    8 {set offset 4}
	    9 {set offset 5}
	    10 {set offset 4}
	    11 {set offset 5}
	    default {set offset 5}
	}

	set ii [lindex $l [expr $offset+3]]
	switch -- [string range $l 0 0] {
	    d {
		set new "<a href=\"file:$path$ii/\">$ii</A>"
	    }
	    l {
		set new "<a href=\"file:$path$ii\">$ii</A>"
	    }
	    default {
		set new "<a href=\"file:$path$ii\">$ii</A>"
	    }
	}

	regsub $ii $l $new l
	append html "$l\n"
    }
    append html "</pre>\n"
    append html "</hr>\n"
    append html "</body>\n"

    return $html
}

proc HVDirList {path} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVDirList $path"
    }
    return [split [exec ls -l $path] \n]
}

proc HVSet {tt url query fragment} {
    global hv
    global debug

    set hv($tt,url) $url
    set hv($tt,query) $query
    set hv($tt,fragment) $fragment
}

proc HVSetAll {tt url query fragment prev html fn mime encoding code meta} {
    global hv
    global debug

    set hv($tt,url) $url
    set hv($tt,query) $query
    set hv($tt,fragment) $fragment
    set hv($tt,previous) $prev
    set hv($tt,html) $html
    set hv($tt,fn) "$fn"
    set hv($tt,mime) $mime
    set hv($tt,encoding) $encoding
    set hv($tt,code) $code
    set hv($tt,meta) $meta
    set hv($tt,refresh,time) 0
    set hv($tt,refresh,url) {}
    set hv($tt,refresh,id) 0
}

proc HVClearAll {tt} {
    global hv
    global debug

    set hv($tt,url) {}
    set hv($tt,query) {}
    set hv($tt,fragment) {}
    set hv($tt,previous) {}
    set hv($tt,html) {}
    set hv($tt,fn) {}
    set hv($tt,mime) {}
    set hv($tt,encoding) {}
    set hv($tt,code) {}
    set hv($tt,meta) {}
    set hv($tt,refresh,time) 0
    set hv($tt,refresh,url) {}
    set hv($tt,refresh,id) 0
}

# Bindings

proc HVMotionBind {tt x y} {
    global hv 
    global debug
    global tcl_platform

    set url [$hv($tt,widget) href $x $y] 

    if {!$hv($tt,active)} {
	if {[string length $url] > 0} {
	    $hv($tt,widget) configure -cursor hand2
	} else {
	    $hv($tt,widget) configure -cursor {}
	}
    }

    HVStatus $tt $url
}

proc HVLinkBind {tt x y} {
    global hv
    global debug
    global xpa

    if {$debug(tcl,hv)} {
	puts "HVLinkBind"
    }

    HVClearIndex $tt $hv($tt,index)
    HVClearStatus $tt
    set url [$hv($tt,widget) href $x $y]
    if {[string length $url] != 0} {
	# sub xpa method
	set exp {%40%40XPA_METHOD%40%40|@@XPA_METHOD@@}
	if {[regexp $exp $url]} {
	    regsub -all $exp $url "[xparec $xpa method]" url
	    if {$debug(tcl,hv)} {
		puts "HVLinkBind XPA_METHOD $url"
	    }
	}

	# already resolved
	HVLoadURL $tt $url {}
    }
}

proc HVLinkNewBind {tt x y} {
    global hv
    global debug
    global xpa

    if {$debug(tcl,hv)} {
	puts "HVLinkNewBind"
    }

    set url [$hv($tt,widget) href $x $y]
    if {[string length $url] != 0} {
	# sub xpa method
	set exp {%40%40XPA_METHOD%40%40|@@XPA_METHOD@@}
	if {[regexp $exp $url]} {
	    regsub -all $exp $url "[xparec $xpa method]" url
	}

	incr hv(incr)
	HV "$tt$hv(incr)" "$hv($tt,title)" $url 2 0 {}
    }
}

# CallBacks

proc HVImageCB {tt args} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVImageCB args: $tt $args"
    }

    set url [lindex $args 0]
    if {$debug(tcl,hv)} {
	puts "HVImageCB $url"
    }

    # do we have anything?
    if {[string length $url] == 0} {
	return
    }

    # do we have a width/height?
    set aa [lindex $args 3]
    set width [HVattrs width $aa 0]
    set height [HVattrs height $aa 0]

    # check for percent (100%) in width/height
    if {![string is integer $width]} {
	set width 0
    }
    if {![string is integer $height]} {
	set height 0
    }

    if {$debug(tcl,hv)} {
	puts "HVImageCB width $width height $height"
    }

    set img [HVImageURL $tt $url $width $height]

    if {[string length $img] != 0} {
	return $img
    } else {
	if {$debug(tcl,hv)} {
	    puts "HVImageCB FAIL $url"
	}
	return biggray
    }
}

proc HVImageURL {tt url width height} {
    global hv
    global debug
    global http

    if {$debug(tcl,hv)} {
	puts "HVImageURL $tt $url $width $height"
    }

    # do we already have the image?
    if {[info exists hv($tt,images,$url)]} {
	if {$debug(tcl,hv)} {
	    puts "HVImageURL found image a $url"
	}
	return $hv($tt,images,$url)
    }

    ParseURL $url r

    set fn {}
    switch -- $r(scheme) {
	{} -
	file {
	    if [file exists $r(path)] {
		catch {image create photo -file $r(path)} img
	    }
	}
	ftp {
	    set fn [tmpnam ds9 [file extension $r(path)]]
	    set ftp [ftp::Open $r(authority) "ftp" "ds9@"]
	    if {$ftp > -1} {
		set ftp::VERBOSE $debug(tcl,ftp)
		set "ftp::ftp${ftp}(Output)" FTPLog
		ftp::Type $ftp binary
		if [ftp::Get $ftp $r(path) "$fn"] {
		    ftp::Close $ftp

		    if {[file size "$fn"] == 0} {
			catch {file delete -force "$fn"}
			return {}
		    }
		    if {[catch {image create photo -file "$fn"} img]} {
			catch {file delete -force "$fn"}
			return {}
		    }
		}
	    }
	}
	http {
	    set fn [tmpnam ds9 [file extension $r(path)]]
	    catch {
		if [catch {open "$fn" w} ch] {
		    Error "Unable to open tmp file $fn for writing"
		    return {}
		}
		set token [http::geturl $url -progress "HVProgress $tt" \
			       -channel $ch -timeout $hv(timeout) \
			       -binary 1 -headers "[ProxyHTTP]"]
		close $ch
		http::cleanup $token
	    }

	    if {[file size "$fn"] == 0} {
		catch {file delete -force "$fn"}
		return {}
	    }
	    if {[catch {image create photo -file "$fn"} img]} {
		catch {file delete -force "$fn"}
		return {}
	    }
	}
    }

    # do we have an img?
    if {![info exists img]} {
	return {}
    }
    if {$debug(tcl,hv)} {
	puts "HVImageURL got image $img"
    }

    # adjust image size if needed
    if {$width != 0 || $height != 0} {
	set iw [image width $img]
	set ih [image height $img]

	set doit 1
	# check for one dimension of 0. calculate to maintain aspect ratio
	if {$width == 0} {
	    set width [expr $iw*$height/$ih]

	    # see if we have a bad resample dimension
	    set wf [expr double($iw)*$height/$ih]
	    if {$width != $wf} {
		if {$debug(tcl,hv)} {
		    puts "HVImageURL abort resample image $img width $wf"
		}
		set doit 0
	    }
	}
	if {$height == 0} {
	    set height [expr $ih*$width/$iw]

	    # see if we have a bad resample dimension
	    set hf [expr double($ih)*$width/$iw]
	    if {$height != $hf} {
		if {$debug(tcl,hv)} {
		    puts "HVImageURL abort resample image $img height $hf"
		}
		set doit 0
	    }
	}

	# check to resample
	if {$doit && ($width != $iw || $height != $ih)} {
	    if {$debug(tcl,hv)} {
		puts "HVImageURL resample image $iw->$width $ih->$height"
	    }

	    set img2 \
		[image create photo -width $width -height $height]
	    if {[catch {blt::winop image resample $img $img2 box} ]} {
		# just use existing img
		if {$debug(tcl,hv)} {
		    puts "HVImageURL error resample image $img"
		}
	    } else {
		set tmp $img
		set img $img2
		catch {image delete $tmp}
	    }
	}
    }

    # delete any tmp files
    if {"$fn" != {}} {
	catch {file delete -force "$fn"}
    }

    set hv($tt,images,$url) $img
    return $img
}

proc HVFontCB {tt sz args} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVFont $tt $sz $args"
    }

    set family helvetica
    set param {}
    set size 12

    foreach f [concat [lindex $args 0]] {
	switch -- $f {
	    fixed {set family courier}
	    default {append param "$f "} 
	}
    }
    
    switch -- $sz {
	1 {set size 6}
	2 {set size 8}
	3 {set size 10}
	4 {set size 12}
	5 {set size 16}
	6 {set size 20}
	7 {set size 24}
	default {set size 12}
    }

#	1 {set size 8}
#	2 {set size 10}
#	3 {set size 12}
#	4 {set size 16}
#	5 {set size 20}
#	6 {set size 24}
#	7 {set size 36}
#	default {set size 16}

    incr size $hv($tt,font,size)

    return "$family $size $param"
}

proc HVNoScriptCB {tt n tag args} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVNoScript $tt $n $tag $args"
    }
}

proc HVScriptCB {tt args} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVScriptCB $tt $args"
    }
}

proc HVFrameCB {tt args} {
    if {$debug(tcl,hv)} {
	puts "HVFrameCB $tt $args"
    }
}

proc HVAppletCB {tt w args} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVAppletCB $tt $w $args"
    }
}

