// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#ifndef __fitshdu_h__
#define __fitshdu_h__

#include <stdlib.h>

#include "column.h"

#define FTY_MAXAXES 7

class FitsHead;

class FitsHDU {
 protected:
  char* extname_;             // EXTNAME keyword
  int extver_;                // EXTVER keyword

  int bitpix_;                // BITPIX keyword
  int naxes_;                 // NAXIS keyword
  int naxis_[FTY_MAXAXES];    // NAXIS[i] keywords

  unsigned long pixnum_;      // number of pixels
  unsigned long pixbytes_;    // number of pixel bytes
  unsigned long realbytes_;   // Number of real data bytes (no padding)
  unsigned long databytes_;   // Number of bytes in data segment (padding)
  unsigned long datablocks_;  // Number of data blocks

  char keybuf[9];
  char* keycat(const char*, int);

public:
  FitsHDU(FitsHead*);
  virtual ~FitsHDU();

  virtual void updateCards(FitsHead*);

  const char* extname() {return extname_;}
  int extver() {return extver_;}
  int bitpix() {return bitpix_;}
  int naxes() {return naxes_;}
  int naxis(int i) {return i<=naxes_ ? naxis_[i-1] : 1;}

  unsigned long pixnum() {return pixnum_;}
  unsigned long pixbytes() {return pixbytes_;}
  unsigned long realbytes() {return realbytes_;}
  unsigned long databytes() {return databytes_;}
  unsigned long datablocks() {return datablocks_;}
};

class FitsImageHDU : public FitsHDU {
private:
  unsigned long datapixels_;

  double bscale_;
  double bzero_;
  int hasblank_;
  int blank_;

public:
  FitsImageHDU(FitsHead*);

  void updateCards(FitsHead*);

  unsigned long datapixels() {return datapixels_;}

  double bscale() {return bscale_;}
  double bzero() {return bzero_;}
  int hasscaling() {return bscale_ != 1 || bzero_ != 0;}
  int hasblank() {return hasblank_;}
  int blank() {return blank_;}
};

class FitsTableHDU : public FitsHDU {
protected:
  int tfields_;
  FitsColumn** cols_;

public:
  FitsTableHDU(FitsHead*);
  virtual ~FitsTableHDU();

  int tfields() {return tfields_;}
  int rows() {return naxis_[1];}
  int cols() {return tfields_;}
  int width() {return naxis_[0];}
  char* list();
  FitsColumn* find(const char*);
  Vector dimension(const char*);
};

class FitsBinTableHDU : public FitsTableHDU {
public:
  FitsBinTableHDU(FitsHead*);
};

class FitsAsciiTableHDU : public FitsTableHDU {
public:
  FitsAsciiTableHDU(FitsHead*);
};

#endif


